/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.utils;

import java.awt.Container;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageDecoder;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

/**
 * Utility functions for processing JPEG images. 
 *
 * @author Erik Brakkee
 */
public class JpegUtils {
    /**
     * Scales an image preserving the aspect ratio. 
     * 
     * @param aMaxWidth Maximum width. 
     * @param aMaxHeight Maximum height. 
     * @param aImage Image to scale. 
     * @return Scaled image. 
     */
    public static BufferedImage scaleImage(int aMaxWidth, int aMaxHeight, Image aImage) {
        double thumbRatio = (double) aMaxWidth / (double) aMaxHeight;
        int imageWidth = aImage.getWidth(null);
        int imageHeight = aImage.getHeight(null);
        double imageRatio = (double) imageWidth / (double) imageHeight;
        if (thumbRatio < imageRatio) {
            aMaxHeight = (int) (aMaxWidth / imageRatio);
        } else {
            aMaxWidth = (int) (aMaxHeight * imageRatio);
        }
        BufferedImage thumbImage = new BufferedImage(aMaxWidth, aMaxHeight,
                BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics2D = thumbImage.createGraphics();
        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics2D.drawImage(aImage, 0, 0, aMaxWidth, aMaxHeight, null);
        return thumbImage;
    }

    /**
     * Loads a jpeg image from an input stream.
     *  
     * @param aInput Input stream. 
     * @return JPEG image. 
     * @throws IOException In case of IO problems.  
     * @throws InterruptedException When execution is interrupted. 
     */
    public static BufferedImage loadJpegImage(InputStream aInput) throws IOException, InterruptedException {
        JPEGImageDecoder decoder = JPEGCodec.createJPEGDecoder(aInput);
        BufferedImage image = decoder.decodeAsBufferedImage();
        MediaTracker mediaTracker = new MediaTracker(new Container());
        mediaTracker.addImage(image, 0);
        mediaTracker.waitForID(0);
        return image;
    }

    /**
     * Writes a JPEG image. 
     * 
     * @param aOutput Output stream to write to. 
     * @param aQuality Quality of the JPEG image in the range 0..100
     * @param aThumbImage
     * @throws IOException
     */
    public static void writeJpegImage(OutputStream aOutput, int aQuality, BufferedImage aThumbImage) throws IOException {
        // save thumbnail image to OUTFILE
        
        if ( aQuality < 0 || aQuality > 100 ) {
            throw new IllegalArgumentException("Argument quality must be in range 0.100: " + aQuality); 
        }
            
        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(aOutput);
        JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(aThumbImage);
        aQuality = Math.max(0, Math.min(aQuality, 100));
        param.setQuality((float) aQuality / 100.0f, false);
        encoder.setJPEGEncodeParam(param);
        encoder.encode(aThumbImage);
    }

}

