/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.security.authorization;

import org.wamblee.persistence.Persistent;

/**
 * Service to determine if access to a certain resource is allowed. 
 *
 * @author Erik Brakkee
 */
public interface AuthorizationService extends Persistent {

    /**
     * Checks whether an operation is allowed on a resource. 
     * @param aResource Resource. 
     * @param aOperation Operation. 
     * @return Checks whether the operation is allowed on a resource. 
     */
    boolean isAllowed(Object aResource, Operation aOperation);
    
    /**
     * Same as {@link #isAllowed(Object, Operation)} but throws a 
     * <code>RuntimeException</code> in case access is not allowed. 
     * @param aResource Resource to check. 
     * @param aOperation Operation to perform.
     * @return Resource that was checked. 
     */
    <T> T check(T aResource, Operation aOperation); 
    
    /**
     * Gets the authorization rules. 
     * @return Rules. 
     */
    AuthorizationRule[] getRules(); 
    
    /**
     * Appends a new authorization rule to the end. 
     * @param aRule Rule to append. 
     */
    void appendRule(AuthorizationRule aRule); 
    
    /**
     * Removes a rule. 
     * @param aRule Index of the rule to remove.  
     */
    void removeRule(int aIndex); 
        
    /**
     * Inserts a rule. 
     * @param aIndex Index of the position of the rule after insertion. 
     * @param aRule Rule to insert. 
     */
    void insertRuleAfter(int aIndex, AuthorizationRule aRule);
}
