/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.security.authorization;

import java.util.ArrayList;
import java.util.List;

import org.wamblee.persistence.AbstractPersistent;
import org.wamblee.usermgt.User;
import org.wamblee.usermgt.UserAccessor;

/**
 * Default implementation of an authorization service. 
 * To determine whether access to a resource is allowed, the service consults a number
 * of authorization rules in a fixed order. The first rule that gives a result GRANTED or 
 * DENIED determines the result of the evaluation. Rules that return any other result are 
 * ignoed. If none of the rules match, than access is denied. 
 *
 * @author Erik Brakkee
 */
public class DefaultAuthorizationService extends AbstractPersistent implements AuthorizationService {
    
    /**
     * List of ordered authorization rules. 
     */
    private List<AuthorizationRule> _rules; 
    
    /**
     * User accessor used to obtain the current user. 
     */
    private UserAccessor _userAccessor; 
    
    /**
     * Name for this instance of the authorization service. 
     */
    private String _name;
    
    /**
     * Constructs the service. 
     * @param aAccessor User accessor. 
     * @param aName Name of this instance of the service. 
     */
    public DefaultAuthorizationService(UserAccessor aAccessor, String aName) {
        _rules = new ArrayList<AuthorizationRule>();
        _userAccessor = aAccessor;
        _name = aName; 
    }
    
    /**
     * Constructs the authorization service.  
     */
    public DefaultAuthorizationService() {
        _rules = new ArrayList<AuthorizationRule>();
        _userAccessor = null;
        _name = null; 
    }
    
    /**
     * Sets the user accessor. 
     * @param aUserAccessor User accessor. 
     */
    public void setUserAccessor(UserAccessor aUserAccessor) { 
        _userAccessor = aUserAccessor; 
    }

    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.AuthorizationService#isAllowed(java.lang.Object, org.wamblee.security.authorization.Operation)
     */
    public boolean isAllowed(Object aResource, Operation aOperation) {
        User user = _userAccessor.getCurrentUser(); 
        for (AuthorizationRule rule: _rules) { 
            switch ( rule.isAllowed(aResource, aOperation, user)) {
            case DENIED: { return false; } 
            case GRANTED: { return true; }
            }
        }
        return false; 
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.AuthorizationService#check(T, org.wamblee.security.authorization.Operation)
     */
    public <T> T check(T aResource, Operation aOperation) {
        if ( !isAllowed(aResource, aOperation)) { 
            throw new AuthorizationException(aResource, aOperation);
        }
        return aResource;
    }
    
    protected String getName() { 
        return _name; 
    }
    
    public void setName(String aName) { 
        _name = aName; 
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.AuthorizationService#getRules()
     */
    public AuthorizationRule[] getRules() {
        return _rules.toArray(new AuthorizationRule[0]); 
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.AuthorizationService#appendRule(org.wamblee.security.authorization.AuthorizationRule)
     */
    public void appendRule(AuthorizationRule aRule) {
        _rules.add(aRule); 
    }

    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.AuthorizationService#insertRuleAfter(int, org.wamblee.security.authorization.AuthorizationRule)
     */
    public void insertRuleAfter(int aIndex, AuthorizationRule aRule) {
       _rules.add(aIndex, aRule);
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.AuthorizationService#removeRule(int)
     */
    public void removeRule(int aIndex) {
       _rules.remove(aIndex);
    }
    
    /**
     * For OR mapping. 
     * @return The rules. 
     */
    protected List<AuthorizationRule> getMappedRules() {
        return _rules; 
    }
    
    /**
     * For OR mapping.
     * @param aRules The rules. 
     */
    protected void setMappedRules(List<AuthorizationRule> aRules) { 
        _rules = aRules; 
    }
}
