/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.security.authorization;

import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;

/**
 * Operation registry implementation. 
 * This implementation ignores the distinction between different types of resources
 * and simply assumes that every operation is applicable to every type of resource. 
 *
 * @author Erik Brakkee
 */
public class DefaultOperationRegistry implements OperationRegistry {

    private Map<String,Operation> _operations; 
    
    public DefaultOperationRegistry(Operation[] aOperations) { 
        _operations = new TreeMap<String, Operation>();
        for (Operation operation: aOperations) { 
            _operations.put(operation.getName(), operation);
        }
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.OperationRegistry#getOperations(java.lang.Class)
     */
    public Operation[] getOperations(Class aResourceClass) {
        return _operations.values().toArray(new Operation[0]); 
    }

    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.OperationRegistry#encode(org.wamblee.security.authorization.Operation[])
     */
    public String encode(Operation[] aOperations) {
        StringBuffer buffer = new StringBuffer(); 
        for (Operation operation: aOperations) {
            if ( buffer.length() > 0 ) { 
                buffer.append(',');
            }
            buffer.append(operation.getName());
        }
        return buffer.toString();
    }

    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.OperationRegistry#decode(java.lang.Class, java.lang.String)
     */
    public Operation[] decode(Class aResourceClass, String aOperationsString) {
        return decode(aOperationsString); 
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.security.authorization.OperationRegistry#decode(java.lang.String)
     */
    public Operation[] decode(String aOperationsString) {
        if ( aOperationsString.length() == 0 ) { 
            return new Operation[0]; 
        }
        String[] names = aOperationsString.split(",");
        ArrayList<Operation> result = new ArrayList<Operation>(); 
        for (String name: names) { 
            Operation operation = _operations.get(name);
            if ( operation == null ) { 
                throw new IllegalArgumentException("Unknown operation '" + name + "'");
            }
            result.add(operation);
        }
        return result.toArray(new Operation[0]); 
    }

}
