/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.security.authorization;

import org.wamblee.persistence.AbstractPersistent;

/**
 * Determiens if an operation is a subclass of a specified operation.
 */
public class IsaOperationCondition extends AbstractPersistent implements
        OperationCondition {

    /**
     * Operation that the other operation must be a subclass of.
     */
    private Class<? extends Operation> _operation;

    /**
     * Constructs the condition.
     * 
     * @param aOperation
     *            Operation that an operation must be an instance of.
     */
    public IsaOperationCondition(Class<? extends Operation> aOperation) {
        _operation = aOperation;
    }

    /**
     * For OR mapping.
     * 
     */
    public IsaOperationCondition() {
        _operation = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.security.authorization.OperationCondition#matches(org.wamblee.security.authorization.Operation)
     */
    public boolean matches(Operation aOperation) {
        return _operation.isInstance(aOperation);
    }

    /**
     * Gets the operation as a string. For OR mapping only.
     * 
     * @return Operation string.
     */
    protected String getOperationString() {
        if (_operation == null) {
            return null; 
        }
        return _operation.getName(); 
    }

    /**
     * Sets the operation as a string. For OR mapping only.
     * 
     * @param aOperation
     *            Operation string.
     */
    protected void setOperationString(String aOperation) {
        if  (aOperation == null ) { 
            return;
        }
        try { 
            _operation = (Class<? extends Operation>)Class.forName(aOperation);
        } catch (Exception e) { 
            throw new IllegalArgumentException("Unknown class '" + aOperation + "'");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "IsaOperationCondition(operation=" + _operation.getName() + ")";
    }

}
