/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt;

import java.util.Set;
import java.util.TreeSet;

import org.wamblee.security.encryption.MessageDigester;

/**
 * In-memory user set.
 *
 * @author Erik Brakkee
 */
public class InMemoryUserSet extends AbstractUserSet {

    /**
     * Users. All users in this set have their password validator and encoder set. 
     */
    private Set<User> _users;

    /**
     * Constructs an empty user set.
     */
    public InMemoryUserSet(NameValidator aPasswordValidator, MessageDigester aPasswordEncoder) {
        super(aPasswordValidator, aPasswordEncoder);
        _users = new TreeSet<User>();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#userModified(org.wamblee.usermgt.User)
     */
    public void userModified(User aUser) {
        _users.remove(aUser);
       setPasswordInfo(aUser); 
        _users.add(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#find(java.lang.String)
     */
    public User find(String aName) {
        for (User user : _users) {
            if (user.getName().equals(aName)) {
                return new User(user);
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#add(org.wamblee.usermgt.User)
     */
    public boolean add(User aUser) {
        setPasswordInfo(aUser);
        return _users.add(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#contains(org.wamblee.usermgt.User)
     */
    public boolean contains(User aUser) {
        return _users.contains(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#remove(org.wamblee.usermgt.User)
     */
    public boolean remove(User aUser) {
        return _users.remove(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#list()
     */
    public Set<User> list() {
        Set<User> list = new TreeSet<User>();
        for (User user : _users) {
            list.add(new User(user));
        }
        return list;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#list(org.wamblee.usermgt.Group)
     */
    public Set<User> list(Group aGroup) {
        Set<User> result = new TreeSet<User>();
        for (User user : _users) {
            if (user.getGroups().contains(aGroup)) {
                result.add(new User(user));
            }
        }
        return result;
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.usermgt.UserSet#size()
     */
    public int size() {
        return _users.size(); 
    }

}
