/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt;

import java.security.AccessController;
import java.security.Principal;
import java.util.Set;

import javax.security.auth.Subject;

/**
 * Implementation of the user accessor that retrieves user information
 * from JAAS. 
 *
 * @author Erik Brakkee
 */
public class JaasUserAccessor implements UserAccessor {

    /**
     * User administration to use. 
     */
    private UserAdministration _admin;

    /**
     * Class of the JAAS user principal. 
     */
    private Class _userPrincipalClass;

    /**
     * Constructs user accessor. 
     * @param aAdmin User administration. 
     * @param aUserClassName Class name of the user principal. 
     */
    public JaasUserAccessor(UserAdministration aAdmin, String aUserClassName) {
        _admin = aAdmin;
        try {
            _userPrincipalClass = Class.forName(aUserClassName);
            if ( !Principal.class.isAssignableFrom(_userPrincipalClass)) {
                throw new IllegalArgumentException("Specified class '" + aUserClassName + "' is not a subclass of '" +
                        Principal.class.getName());
            }
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAccessor#getCurrentUser()
     */
    public User getCurrentUser() {
        Subject subject = Subject.getSubject(AccessController.getContext());
        if (subject == null) {
            return null;
        }
        Principal userPrincipal = getUserPrincipal(subject);
      
        return _admin.getUser(userPrincipal.getName());
    }

    /**
     * Gets the user principal from the subject. 
     * @param subject Subject. 
     * @return User principal. 
     * @throws IllegalArgumentException In case there is a duplicate principal or the principal was not found. 
     */
    private Principal getUserPrincipal(Subject subject) {
        Set<Principal> principals = subject.getPrincipals();
        Principal userPrincipal = null;  
        for ( Principal principal: principals) { 
            if ( principal.getClass().equals(_userPrincipalClass)) { 
                if ( userPrincipal != null ) { 
                    throw new IllegalArgumentException(
                            "Multiple principals for class '" + _userPrincipalClass + "', subject: " + subject);                   
                }
                userPrincipal = principal; 
            }
        }
        if ( userPrincipal == null ) { 
            throw new IllegalArgumentException(
                    "No user principal found for class '" + _userPrincipalClass + "', subject: " + subject);
        }
        return userPrincipal;
    }

}
