/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.usermgt;

import org.wamblee.usermgt.UserMgtException.Reason;

/**
 * Validation of names based on a regular expression. 
 *
 * @author Erik Brakkee
 */
public class RegexpNameValidator implements NameValidator {
    
    /**
     * Convenience pattern for an id. 
     */
    public static final String ID_PATTERN = "[a-zA-Z]+[a-zA-Z0-9]*";
    
    /**
     * Convenience pattern for a password consisting of at least 6 characters. 
     */
    public static final String PASSWORD_PATTERN = ".{6}.*";
    
    /**
     * Pattern to use. 
     */
    private String _pattern;
    
    /**
     * Reason to use when validation fails. 
     */
    private Reason _reason; 
    
    /**
     * Message to report. 
     */
    private String _message; 
    
    /**
     * Validates a regular expression. 
     * @param aPattern Pattern that names must comply to. 
     * @param aReason Reason to report when validation fails. 
     * @param aMessage Message to report. 
     */
    public RegexpNameValidator(String aPattern, Reason aReason, String aMessage) { 
        _pattern = aPattern; 
        _reason = aReason;
        _message = aMessage; 
    }
    
    /**
     * Convenience constructor with all string parameters. Useful for configuration 
     * in Spring. 
     * @param aPattern Pattern to use.
     * @param aReason Reason. 
     * @param aMessage Message. 
     */
    public RegexpNameValidator(String aPattern, String aReason, String aMessage) { 
        this(aPattern, Reason.valueOf(aReason), aMessage);
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.NameValidator#validate(java.lang.String)
     */
    public void validate(String aName) throws UserMgtException {
        if ( !aName.matches(_pattern)) { 
            throw new UserMgtException(_reason, _message); 
        }
    }

}
