/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt;

import java.io.Serializable;
import java.util.Set;
import java.util.TreeSet;

import org.wamblee.persistence.AbstractPersistent;
import org.wamblee.security.encryption.MessageDigester;
import org.wamblee.usermgt.UserMgtException.Reason;

/**
 * Represents a user.
 * The methods for managing the groups of the user have package scope. 
 * Managing the groups of the user should be done through the 
 * {@link org.wamblee.usermgt.UserAdministration} interface. 
 */
public class User extends AbstractPersistent implements Serializable, Comparable {

    /**
     * User name. 
     */
    private String _name;

    /**
     * Password. 
     */
    private String _password;

    /**
     * Groups the user belongs to. 
     */
    private Set<Group> _groups;
    
    /**
     * Password validator. 
     */
    private NameValidator _passwordValidator; 

    /**
     * Password encoder. 
     */
    private MessageDigester _passwordEncoder;
    
    /**
     * Constructs the user. 
     * @param aName User name. 
     * @param aPassword Password. 
     * @param aGroup Group the user belongs to.  
     */
    User(String aName, String aPassword, Group aGroup, NameValidator aPasswordValidator, 
            MessageDigester aPasswordEncoder) throws UserMgtException {
        super(); 
        _name = aName;
        aPasswordValidator.validate(aPassword);
        _password = aPasswordEncoder.hash(aPassword);
        _groups = new TreeSet<Group>(); 
        _groups.add(aGroup);
        _passwordValidator = aPasswordValidator;
        _passwordEncoder = aPasswordEncoder;
    }
    
    public User(User aUser) { 
        super(aUser); 
        _name = aUser._name;
        _password = aUser._password;
        _groups = new TreeSet<Group>(); 
        for (Group group: aUser._groups) {
            _groups.add(new Group(group)); 
        }
        _passwordValidator = aUser._passwordValidator; 
        _passwordEncoder = aUser._passwordEncoder;
    }
    
    User() { 
        super(); 
        _name = null; 
        _password = null; 
        _groups = null; 
        _passwordValidator = null; 
        _passwordEncoder = null;
    }
    
    /**
     * Sets the password validator. 
     * @param aPasswordValidator Validator. 
     */
    public void setPasswordValidator(NameValidator aPasswordValidator) { 
        _passwordValidator = aPasswordValidator;
    }
    
    /**
     * Sets the password encoder. 
     * @param aPasswordEncoder Encoder. 
     */
    public void setPasswordEncoder(MessageDigester aPasswordEncoder) { 
        _passwordEncoder = aPasswordEncoder; 
    }

    /**
     * @return Returns the _password.
     */
    String getPassword() {
        return _password;
    }
    
    /**
     * Checks the password. 
     * @param aPassword Password to check. 
     * @throws UserMgtException In case the password is incorrect. 
     */
    public void checkPassword(String aPassword) throws UserMgtException {
        String encoded = _passwordEncoder.hash(aPassword);
        if ( !_password.equals(encoded) ) { 
            throw new UserMgtException(Reason.INVALID_PASSWORD, this);
        }
    }
    
    /**
     * Changes the password. 
     * @param aOldPassword Old password. 
     * @param aNewPassword New password. 
     * @throws UserMgtException In case the old password is incorrect. 
     */
    public void changePassword(String aOldPassword, String aNewPassword) throws UserMgtException {
        checkPassword(aOldPassword);
        _passwordValidator.validate(aNewPassword);
        setPassword(aNewPassword);
    }

    /**
     * @param aPassword
     *            The password to set.
     */
    public void setPassword(String aPassword) throws UserMgtException {
        _passwordValidator.validate(aPassword);
        _password = _passwordEncoder.hash(aPassword);
    }
    
    /**
     * For OR mapping. 
     * @return Password. 
     */
    protected String getPasswordString() { 
        return _password;
    }
    
    /**
     * For OR mapping. 
     * @param aPassword Password. 
     */
    protected void setPasswordString(String aPassword) { 
        _password = aPassword; 
    }

    /**
     * @return Returns the _user.
     */
    public String getName() {
        return _name;
    }

    /**
     * @param aName
     *            The username to set.
     */
    void setName(String aName) {
        _name = aName;
    }

    /**
     * Gets the groups the user belongs to. 
     * @return Groups.
     */
    public Set<Group> getGroups() {
        Set<Group> result = new TreeSet<Group>(); 
        result.addAll(_groups);
        return result; 
    }
    
    /**
     * Checks whether the user belongs to the given group. 
     * @param aGroup Group. 
     * @return True if the user belongs to the group.
     */
    public boolean isInGroup(Group aGroup) {
        return _groups.contains(aGroup);
    }
    
    /**
     * Checks whether the user belongs to the given group. 
     * @param aGroup Group. 
     * @return True if the user belongs to the group.
     */
    public boolean isInGroup(String aGroup) {
        return _groups.contains(new Group(aGroup));
    }
    
    /**
     * Gets the group set. For OR mapping. 
     * @return set of groups. 
     */
    Set<Group> getGroupSet() { 
        return _groups;
    }

    /**
     * Sets the groups the user belongs to, for OR mapping. 
     * @param aGroups Groups.
     */
    void setGroupSet(Set<Group> aGroups) {
        _groups = aGroups;
    }

    /**
     * Adds the user to a group. 
     * @param aGroup Group to add the user to.
     * @throws UserMgtException In case the user already belongs to the group.  
     */
    void addGroup(Group aGroup) throws UserMgtException {
        if (_groups.contains(aGroup)) {
            throw new UserMgtException(Reason.USER_ALREADY_IN_GROUP, aGroup);
        }
        _groups.add(aGroup);
    }

    /**
     * Removes the user from a group. 
     * @param aGroup Group. 
     * @throws UserMgtException In case the user does not belong to the group.
     */
    void removeGroup(Group aGroup) throws UserMgtException {
        if (!_groups.contains(aGroup)) {
            throw new UserMgtException(Reason.USER_NOT_IN_GROUP, this, aGroup);
        }
        if ( _groups.size() == 1 ) { 
            throw new UserMgtException(Reason.USER_MUST_BE_IN_A_GROUP, this, aGroup); 
        }
        _groups.remove(aGroup);
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object aUser) {
        if ( !(aUser instanceof User)) {
            return false; 
        }
        User user = (User)aUser; 
        return _name.equals(user._name);
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return _name.hashCode(); 
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        String result = "User(name=" + _name + ", password=" + _password;
        for (Group group: _groups) { 
            result += ", group=" + group; 
        }
        return result + ")"; 
    }
    
    /* (non-Javadoc)
     * @see java.lang.Comparable#compareTo(T)
     */
    public int compareTo(Object aUser) {
        return _name.compareTo(((User)aUser)._name);
    }
}
