/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt;

import java.security.NoSuchAlgorithmException;

import org.apache.log4j.Logger;

/**
 * User administration initializer. It populates the user administration with a
 * number of groups and users but only in case no users exist.
 *
 * @author Erik Brakkee
 */
public class UserAdminInitializer {
    
    private static final Logger LOGGER = Logger.getLogger(UserAdminInitializer.class);
    
    /**
     * Initializes the user administration in case no users are present.
     * 
     */
    public UserAdminInitializer(UserAdministration aAdmin, String[] aUsers,
            String[] aGroups, String[] aPasswords) throws UserMgtException, NoSuchAlgorithmException {
        if (aUsers.length != aGroups.length
                || aUsers.length != aPasswords.length) {
            throw new IllegalArgumentException(
                    "Array sizes for users, groups, and passwords differ: "
                            + aUsers.length + "," + aGroups.length + ","
                            + aPasswords.length);

        }
        if (aAdmin.getUserCount() == 0) {
            initialize(aAdmin, aUsers, aGroups, aPasswords);
        }
       
    }

    /**
     * Adds the specified users and groups to the user administration. 
     * @param aAdmin User administration. 
     * @param aUsers Users. 
     * @param aGroups Groups. 
     * @param aPasswords Passwords. 
     * @throws UserMgtException In case of a problem creating users or groups. 
     */
    private void initialize(UserAdministration aAdmin, String[] aUsers,
            String[] aGroups, String[] aPasswords) throws UserMgtException {
        for (int i = 0; i < aUsers.length; i++) {
            String user = aUsers[i];
            String group = aGroups[i];
            String password = aPasswords[i];
            
            if (aAdmin.getUser(user) == null) {
                // must create user.
                Group groupObj = aAdmin.getGroup(group);
                if (groupObj == null) {
                    // must create group
                    LOGGER.info("Creating group: " + group);
                    groupObj = aAdmin.createGroup(group);
                }
                assert groupObj != null; 
                
                LOGGER.info("Creating user: " + user + " password: " + password);
                aAdmin.createUser(user, password, groupObj);
            }
        }
    }
}
