/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt;

import java.util.Set;

/**
 * Interface for user administration. Manages the users and groups. 
 *
 * @author Erik Brakkee
 */
public interface UserAdministration {
    
    /**
     * Creates a new user. 
     * @param aUser Username. 
     * @param aPassword Password. 
     * @param aGroup Group. 
     * @return User. 
     * @throws UserMgtException In case there is a conflict with an existing user. 
     */
    User createUser(String aUser, String aPassword, Group aGroup) throws UserMgtException; 
    
    /**
     * Creates a new group. 
     * @param aName Group name. 
     * @return Group
     * @throws UserMgtException In case there is a conflict with an existing group. 
     */
    Group createGroup(String aName) throws UserMgtException;
    
    /** 
     * @return Number of users. 
     */
    int getUserCount(); 
    
    /**
     * @return Number of groups. 
     */
    int getGroupCount(); 
    
    /**
     * Must be called when the user is modified. 
     * @param aUser User.
     */
    void userModified(User aUser);
    
    /**
     * Must be called when the group is modified. 
     * @param aGroup Group. 
     */
    void groupModified(Group aGroup); 

    /**
     * Gets the user for a given name. 
     * @param aName User name. 
     * @return User or null if not found.   
     */
    User getUser(String aName); 
    
    /**
     * Gets the group for a given group name. 
     * @param aName Group name. 
     * @return Group or null if not found. 
     */
    Group getGroup(String aName); 
    
    /**
     * Get the users. 
     * @return All known users. 
     */
    Set<User> getUsers(); 
    
    /**
     * Gets the users for a given group. 
     * @param aGroup Group. 
     * @return Set of users (always non-null).
     */
    Set<User> getUsers(Group aGroup);
    
    /**
     * Gets all known groups. 
     * @return Groups. 
     */
    Set<Group> getGroups();
    
    /**
     * Renames a user. 
     * @param aUser User object for which user name must be changed. 
     * @param aUserName New user name. 
     * @throws UserMgtException In case the user is not known or the new user 
     *    name is already in use by another user.  
     */
    void renameUser(User aUser, String aUserName) throws UserMgtException;
    
    /**
     * Renames a group. 
     * @param aGroup Group to rename. 
     * @param aGroupName New name for the group. 
     * @throws UserMgtException In case the new group name is already used by 
     *    another group of if the existing group is unknown.  
     */
    void renameGroup(Group aGroup, String aGroupName) throws UserMgtException;
    
    /**
     * Removes the user. 
     * @param aUser User to remove. 
     * @throws UserMgtException In case the user does not exist.  
     */
    void removeUser(User aUser) throws UserMgtException;
    
    /**
     * Removes the group. 
     * @param aGroup Group to remove. 
     * @throws UserMgtException In case there are still users that are in the given group. 
     */
    void removeGroup(Group aGroup) throws UserMgtException;
    
    /**
     * Adds a user to a group. 
     * @param aUser User. 
     * @param aGroup Group. 
     * @throws UserMgtException In case the user or group or not known or if the user 
     *   is already part of the group. 
     */
    void addUserToGroup(User aUser, Group aGroup) throws UserMgtException;
    
    /**
     * Removes a user from a group. 
     * @param aUser User
     * @param aGroup Group
     * @throws UserMgtException In case the user or group are unknown or if the user 
     *    is not part of the group. 
     */
    void removeUserFromGroup(User aUser, Group aGroup) throws UserMgtException; 
}

