/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt;

import static org.wamblee.usermgt.UserMgtException.Reason.DUPLICATE_GROUP;
import static org.wamblee.usermgt.UserMgtException.Reason.DUPLICATE_USER;
import static org.wamblee.usermgt.UserMgtException.Reason.GROUP_STILL_OCCUPIED;
import static org.wamblee.usermgt.UserMgtException.Reason.TRIVIAL_RENAME;
import static org.wamblee.usermgt.UserMgtException.Reason.UNKNOWN_GROUP;
import static org.wamblee.usermgt.UserMgtException.Reason.UNKNOWN_USER;

import java.util.Set;

/**
 * Administration of users and groups.
 *
 * @author Erik Brakkee
 */
public class UserAdministrationImpl implements UserAdministration {
   
    /**
     * All known users.
     */
    private UserSet _users;

    /**
     * All known groups.
     */
    private GroupSet _groups;
    
    /**
     * Validator for user names. 
     */
    private NameValidator _userValidator;
    
    /**
     * Validator for group names. 
     */
    private NameValidator _groupValidator;
    
    /**
     * Constructs empty user administration.
     * 
     */
    public UserAdministrationImpl(UserSet aUsers, GroupSet aGroups, NameValidator aUserValidator, 
              NameValidator aGroupValidator) { 
        _users = aUsers;
        _groups = aGroups;
        _userValidator = aUserValidator;
        _groupValidator = aGroupValidator;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#createUser(java.lang.String,
     *      java.lang.String)
     */
    public User createUser(String aUser, String aPassword, Group aGroup)
            throws UserMgtException {
        _userValidator.validate(aUser);
        checkGroup(aGroup);
        User user = _users.createUser(aUser, aPassword, aGroup);
        return new User(user);
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#createGroup(java.lang.String)
     */
    public Group createGroup(String aName) throws UserMgtException {
        _groupValidator.validate(aName);
        Group group = new Group(aName);
        if (_groups.contains(group)) {
            throw new UserMgtException(DUPLICATE_GROUP, group);
        }
        _groups.add(group);
        return new Group(group);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#userModified(org.wamblee.usermgt.User)
     */
    public void userModified(User aUser) {
        _users.userModified(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#groupModified(org.wamblee.usermgt.Group)
     */
    public void groupModified(Group aGroup) {
        _groups.groupModified(aGroup);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#getUser(java.lang.String)
     */
    public User getUser(String aName) {
        return _users.find(aName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#getGroup(java.lang.String)
     */
    public Group getGroup(String aName) {
        return _groups.find(aName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#getUsers()
     */
    public Set<User> getUsers() {
        return _users.list();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#getUsers(org.wamblee.usermgt.Group)
     */
    public Set<User> getUsers(Group aGroup) {
        return _users.list(aGroup);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#getGroups()
     */
    public Set<Group> getGroups() {
        return _groups.list();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#removeUser(org.wamblee.usermgt.User)
     */
    public void removeUser(User aUser) throws UserMgtException {
        checkUser(aUser);
        _users.remove(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#removeGroup(org.wamblee.usermgt.Group)
     */
    public void removeGroup(Group aGroup) throws UserMgtException {
        checkGroup(aGroup);
        if (getUsers(aGroup).size() > 0) {
            throw new UserMgtException(GROUP_STILL_OCCUPIED, aGroup);
        }
        _groups.remove(aGroup);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#renameUser(org.wamblee.usermgt.User,
     *      java.lang.String)
     */
    public void renameUser(User aUser, String aUserName)
            throws UserMgtException {
        checkUser(aUser);
        if (aUser.getName().equals(aUserName)) {
            throw new UserMgtException(TRIVIAL_RENAME, aUser);
        }
        if (_users.find(aUserName) != null) {
            throw new UserMgtException(DUPLICATE_USER, aUser);
        }
        _userValidator.validate(aUserName);
        // we are modifying the user so we should re-insert it into the set
        // after renaming it.
        _users.remove(aUser);
        aUser.setName(aUserName);
        _users.add(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#renameGroup(org.wamblee.usermgt.Group,
     *      java.lang.String)
     */
    public void renameGroup(Group aGroup, String aGroupName)
            throws UserMgtException {
        checkGroup(aGroup);
        if (aGroup.getName().equals(aGroupName)) {
            throw new UserMgtException(TRIVIAL_RENAME, aGroup);
        }
        if (_groups.find(aGroupName) != null) {
            throw new UserMgtException(DUPLICATE_GROUP, aGroup);
        }
        _groupValidator.validate(aGroupName);
        // we are renaming the group so we should re-insert it into the set
        // after renaming it.
        _groups.remove(aGroup);
        aGroup.setName(aGroupName);
        _groups.add(aGroup);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#addUserToGroup(org.wamblee.usermgt.User,
     *      org.wamblee.usermgt.Group)
     */
    public void addUserToGroup(User aUser, Group aGroup)
            throws UserMgtException {
        checkUser(aUser);
        checkGroup(aGroup);
        aUser.addGroup(aGroup);
        _users.userModified(aUser);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserAdministration#removeUserFromGroup(org.wamblee.usermgt.User,
     *      org.wamblee.usermgt.Group)
     */
    public void removeUserFromGroup(User aUser, Group aGroup)
            throws UserMgtException {
        checkUser(aUser);
        checkGroup(aGroup);
        aUser.removeGroup(aGroup);
        _users.userModified(aUser);
    }

    /**
     * @param aUser
     * @throws UserMgtException
     */
    private void checkUser(User aUser) throws UserMgtException {
        if (!_users.contains(aUser)) {
            throw new UserMgtException(UNKNOWN_USER, aUser);
        }
    }

    /**
     * @param aGroup
     * @throws UserMgtException
     */
    private void checkGroup(Group aGroup) throws UserMgtException {
        if (!_groups.contains(aGroup)) {
            throw new UserMgtException(UNKNOWN_GROUP, aGroup);
        }
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.usermgt.UserAdministration#getUserCount()
     */
    public int getUserCount() {
        return _users.size(); 
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.usermgt.UserAdministration#getGroupCount()
     */
    public int getGroupCount() {
        return _groups.size(); 
    }
}
