/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.usermgt;

import java.util.EnumMap;

/**
 * User management exception. 
 *
 * @author Erik Brakkee
 */
public class UserMgtException extends Exception {
    
    static final long serialVersionUID = 5585349754997507529L; 
    
    /**
     * Possible causes for the exception. 
     *
     */
    public enum Reason { 
        UNKNOWN_USER, 
        UNKNOWN_GROUP,
        DUPLICATE_USER, 
        DUPLICATE_GROUP, 
        USER_ALREADY_IN_GROUP,
        USER_NOT_IN_GROUP,
        TRIVIAL_RENAME,
        INVALID_PASSWORD, 
        GROUP_STILL_OCCUPIED, 
        USER_MUST_BE_IN_A_GROUP,
        INVALID_USERNAME,
        INVALID_GROUPNAME
    }
   
    /**
     * Mapping of enum to exception message text. 
     */
    private static final EnumMap<Reason,String> MESSAGES = new EnumMap<Reason,String>(Reason.class);
    
    static {
        MESSAGES.put(Reason.UNKNOWN_USER, "Unknown user");
        MESSAGES.put(Reason.UNKNOWN_GROUP, "Unknown group");
        MESSAGES.put(Reason.DUPLICATE_USER, "Duplicate user");
        MESSAGES.put(Reason.DUPLICATE_GROUP, "Duplicate group");
        MESSAGES.put(Reason.USER_ALREADY_IN_GROUP, "User already in group");
        MESSAGES.put(Reason.USER_NOT_IN_GROUP, "User not in group");
        MESSAGES.put(Reason.TRIVIAL_RENAME, "Trivial rename");
        MESSAGES.put(Reason.INVALID_PASSWORD, "Invalid password");
        MESSAGES.put(Reason.GROUP_STILL_OCCUPIED, "Group still occupied");
        MESSAGES.put(Reason.USER_MUST_BE_IN_A_GROUP, "User must be in at least one group");
        MESSAGES.put(Reason.INVALID_USERNAME, "Invalid user name");
        MESSAGES.put(Reason.INVALID_GROUPNAME, "Invalid group name");
    }
    
    /**
     * Cause of the exception. 
     */
    private Reason _cause; 
    
    /**
     * User or null if no user is relevant for the problem. 
     */
    private User _user; 
    
    /**
     * Group or null if no group is relevant for the problem. 
     */
    private Group _group; 

    public UserMgtException(Reason aCause, String aMessage) {
        super(MESSAGES.get(aCause) + ": " + aMessage);
        _cause = aCause; 
    }
    
    public UserMgtException(Reason aCause, User aUser) {
        this(aCause, "for user '" + aUser.getName() + "'");
        _user = aUser; 
    }
    
    public UserMgtException(Reason aCause, Group aGroup) {
        this(aCause, "for group '" + aGroup.getName() + "'");
        _group = aGroup; 
    }
    
    public UserMgtException(Reason aCause, User aUser, Group aGroup) {
        this(aCause, "for user '" + aUser.getName() + "' and group '" + aGroup.getName() + "'");
        _user = aUser; 
        _group = aGroup; 
    }

    /**
     * Gets the cause of the problem. 
     * @return Cause. 
     */
    public Reason getReason() {
        return _cause; 
    }
    
    /**
     * Gets the user for which the problem occurred. 
     * @return User or null if not applicable.  
     */
    public User getUser() {
        return _user; 
    }
    
    /**
     * Gets the group for which the problem occured. 
     * @return Group or null if not applicable. 
     */
    public Group getGroup() {
        return _group;
    }   
}
