/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.usermgt;

import java.util.Set;

/**
 * Represents a set of users.
 * Typical implementations would be an implementation based on a static configuration file or 
 * an implementation backed by a database.  
 *
 * @author Erik Brakkee
 */
public interface UserSet {
    
    /**
     * Creates a user. 
     * @param aUsername User name. 
     * @param aPassword Password. 
     * @param aGroup Group.
     * @return New user.  
     * @throws UserMgtException In case the user cannot be created. 
     */ 
    User createUser(String aUsername, String aPassword, Group aGroup) throws UserMgtException; 
    
    /**
     * Must be called whenever a user object has been modified to notify the
     * user set.  
     * @param aUser Modified user. 
     */
    void userModified(User aUser); 
    
    /**
     * Finds  user. 
     * @param aName Username. 
     * @return User or null if not found. 
     */
    User find(String aName);
    
    /**
     * Checks if a user exists. 
     * @param aUser User. 
     * @return True iff the user exists. 
     */
    boolean contains(User aUser); 
    
    /**
     * Adds a user. If the user already exists, the user details are updated with that
     * of the specified user object. 
     * @param aUser User to add. 
     */
    boolean add(User aUser);
    
    /**
     * Removes a user. If the user does not exist, nothing happens. 
     * @param aUser
     */
    boolean remove(User aUser);
    
    /**
     * Lists the current users. 
     * @return Users. 
     */
    Set<User> list(); 
    
    /**
     * Lists the users belonging to a particular group. 
     * @param aGroup Group. 
     * @return Groups. 
     */
    Set<User> list(Group aGroup);

    /**
     * 
     * @return The number of users. 
     */
    int size(); 
}
