/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.wamblee.usermgt.hibernate;

import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.wamblee.persistence.hibernate.HibernateSupport;
import org.wamblee.usermgt.Group;
import org.wamblee.usermgt.GroupSet;

/**
 * Set of groups backed by the database. 
 *
 * @author Erik Brakkee
 */
public class HibernateGroupSet extends HibernateSupport implements GroupSet {
    
    
    private static final String QUERY_FIND_BY_NAME = "findGroupByName";
    
    private static final String PARAM_NAME = "name";
    
    private static final String QUERY_COUNT_GROUPS = "countGroups";
    
    public HibernateGroupSet() {
        // Empty
    }
    
    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#groupModified(org.wamblee.usermgt.Group)
     */
    public void groupModified(Group aGroup) {
        assert aGroup.getPrimaryKey() != null; 
        super.merge(aGroup);        
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#find(java.lang.String)
     */
    public Group find(String aName) {
        List list = getHibernateTemplate().findByNamedQueryAndNamedParam(QUERY_FIND_BY_NAME, PARAM_NAME, aName);
        if ( list.size() > 1 ) { 
            throw new RuntimeException("More than one group with the same name '" + aName + "'");
        }
        if ( list.size() == 0 ) { 
            return null; 
        }
        return new Group((Group)list.get(0));
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#contains(org.wamblee.usermgt.Group)
     */
    public boolean contains(Group aGroup) {
        return find(aGroup.getName()) != null; 
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#add(org.wamblee.usermgt.Group)
     */
    public boolean add(Group aGroup) {
        assert aGroup.getPrimaryKey() == null; 
        if ( contains(aGroup) ) { 
            return false; 
        }
        super.merge(aGroup);
        return true; 
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#remove(org.wamblee.usermgt.Group)
     */
    public boolean remove(Group aGroup) {
        assert aGroup.getPrimaryKey() != null; 
        if ( !contains(aGroup)) { 
            return false; 
        }
        Group group = (Group) getHibernateTemplate().merge(aGroup); 
        getHibernateTemplate().delete(group);
        aGroup.setPrimaryKey(null); 
        aGroup.setPersistedVersion(-1);
        return true; 
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#list()
     */
    public Set<Group> list() {
        Set<Group> groups = new TreeSet<Group>(); 
        List<Group> list = getHibernateTemplate().loadAll(Group.class);
        for (Group group: list) { 
            groups.add(new Group(group));
        }
        return groups; 
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.GroupSet#size()
     */
    public int size() {
        Long result = (Long) getHibernateTemplate().findByNamedQuery(QUERY_COUNT_GROUPS).get(0);
        return result.intValue(); 
    }
}
