/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.usermgt.hibernate;

import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.hibernate.SessionFactory;
import org.springframework.orm.hibernate3.HibernateTemplate;
import org.wamblee.cache.Cache;
import org.wamblee.persistence.hibernate.HibernateSupport;
import org.wamblee.security.encryption.MessageDigester;
import org.wamblee.usermgt.AbstractUserSet;
import org.wamblee.usermgt.Group;
import org.wamblee.usermgt.NameValidator;
import org.wamblee.usermgt.User;

/**
 * User set backed by the database.
 *
 * @author Erik Brakkee
 */
public class HibernateUserSet extends AbstractUserSet {

    private static final String QUERY_FIND_BY_NAME = "findUserByName";

    private static final String QUERY_FIND_BY_GROUP_NAME = "findUserByGroupName";

    private static final String PARAM_NAME = "name";
    
    private static final String QUERY_COUNT_USERS = "countUsers"; 

    /**
     * Cache of users. Every user in the cache has its password validator and encoder set.  
     */
    private Cache<String, User> _cache;
    
    /** 
     * Spring hibernate support. 
     */
    private HibernateSupport _hibernateSupport; 

    /**
     * Constructs a user set backed by the database.
     * @param aCache User cache to use. 
     */
    public HibernateUserSet(Cache<String,User> aCache, 
            NameValidator aPasswordValidator, MessageDigester aPasswordEncoder) {
        super(aPasswordValidator, aPasswordEncoder);
        _cache = aCache;
        _hibernateSupport = new HibernateSupport();
    }
    
    /**
     * Sets the session factory. 
     * @param aFactory Session factory. 
     */
    public void setSessionFactory(SessionFactory aFactory) { 
        _hibernateSupport.setSessionFactory(aFactory);
    }
    
    /**
     * Gets the hibernate template. 
     * @return Hibernate template. 
     */
    private HibernateTemplate getHibernateTemplate() { 
        return _hibernateSupport.getHibernateTemplate();
    }
   
    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#userModified(org.wamblee.usermgt.User)
     */
    public void userModified(User aUser) {
        assert aUser.getPrimaryKey() != null;
        _hibernateSupport.merge(aUser);
        _cache.remove(aUser.getName());
        setPasswordInfo(aUser);
        _cache.put(aUser.getName(), new User(aUser));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#find(java.lang.String)
     */
    public User find(String aName) {
        User user = _cache.get(aName);
        if (user != null) {
            return user;
        }
        List result = getHibernateTemplate().findByNamedQueryAndNamedParam(
                QUERY_FIND_BY_NAME, PARAM_NAME, aName);
        if (result.size() > 1) {
            throw new RuntimeException(
                    "Implementation problem, more than one user with the same name!");
        }
        if (result.size() == 0) {
            return null;
        }
        user = (User) result.get(0);
        setPasswordInfo(user);
        _cache.put(aName, user);
        return new User(user);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#contains(org.wamblee.usermgt.User)
     */
    public boolean contains(User aUser) {
        return find(aUser.getName()) != null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#add(org.wamblee.usermgt.User)
     */
    public boolean add(User aUser) {
        assert aUser.getPrimaryKey() == null;
        if (contains(aUser)) {
            return false;
        }
        getHibernateTemplate().saveOrUpdate(aUser);
        setPasswordInfo(aUser);
        _cache.put(aUser.getName(), aUser);
        return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#remove(org.wamblee.usermgt.User)
     */
    public boolean remove(User aUser) {
        assert aUser.getPrimaryKey() != null;
        if (!contains(aUser)) {
            return false;
        }
        User user = (User) getHibernateTemplate().merge(aUser);
        getHibernateTemplate().delete(user);
        aUser.setPersistedVersion(-1);
        aUser.setPrimaryKey(null);
        _cache.remove(aUser.getName());
        return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#list()
     */
    public Set<User> list() {
        Set<User> users = new TreeSet<User>();
        List<User> list = getHibernateTemplate().loadAll(User.class);
        for (User user : list) {
            setPasswordInfo(user);
            users.add(new User(user));
        }
        return users;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.usermgt.UserSet#list(org.wamblee.usermgt.Group)
     */
    public Set<User> list(Group aGroup) {
        Set<User> users = new TreeSet<User>();
        List<User> list = getHibernateTemplate().findByNamedQueryAndNamedParam(
                QUERY_FIND_BY_GROUP_NAME, PARAM_NAME, aGroup.getName());
        for (User user : list) {
            setPasswordInfo(user);
            users.add(new User(user));
        }
        return users;
    }

    /* (non-Javadoc)
     * @see org.wamblee.usermgt.UserSet#size()
     */
    public int size() {
        Long result = (Long)getHibernateTemplate().findByNamedQuery(QUERY_COUNT_USERS).get(0);
        return result.intValue(); 
    }
}
