/*
 * Copyright 2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.cache;

import java.io.Serializable;

import org.apache.log4j.Logger;

/**
 * Represents a cached object. The object is either retrieved from the cache if
 * the cache has it, or a call back is invoked to get the object (and put it in
 * the cache).
 *
 * @author Erik Brakkee
 */
public class CachedObject<KeyType extends Serializable, ValueType extends Serializable> {

    private static final Logger LOGGER = Logger.getLogger(CachedObject.class);

    /**
     * Callback invoked to compute an object if it was not found in the cache.
     * 
     * @param <T>
     *            Type of the object
     */
    public static interface Computation<Key extends Serializable, Value extends Serializable> {
        /**
         * Gets the object. Called when the object is not in the cache.
         * 
         * @param aObjectKey
         *            Id of the object in the cache.
         * @return Object, must be non-null.
         */
        Value getObject(Key aObjectKey);
    }

    /**
     * Cache to use.
     */
    private Cache<KeyType, ValueType> _cache;

    /**
     * Key of the object in the cache.
     */
    private KeyType _objectKey;

    /**
     * Computation used to obtain the object if it is not found in the cache.
     */
    private Computation<KeyType, ValueType> _computation;

    /**
     * Constructs the cached object.
     * 
     * @param aCache
     *            Cache to use.
     * @param aObjectKey
     *            Key of the object in the cache.
     * @param aComputation
     *            Computation to get the object in case the object is not in the
     *            cache.
     */
    public CachedObject(Cache<KeyType, ValueType> aCache, KeyType aObjectKey,
            Computation<KeyType, ValueType> aComputation) {
        _cache = aCache;
        _objectKey = aObjectKey;
        _computation = aComputation;
    }

    /**
     * Gets the object. Since the object is cached, different calls to this
     * method may return different objects.
     * 
     * @return Object.
     */
    public ValueType get() {
        ValueType object = (ValueType) _cache.get(_objectKey); // the used
                                                                // cache is
                                                                // thread safe.
        if (object == null) {
            // synchronize the computation to make sure that the object is only
            // computed
            // once when multiple concurrent threads detect that the entry must
            // be
            // recomputed.
            synchronized (this) {
                object = (ValueType) _cache.get(_objectKey);
                if (object == null) {
                    // No other thread did a recomputation so we must do this
                    // now.
                    LOGGER.debug("Refreshing cache for '" + _objectKey + "'");
                    object = _computation.getObject(_objectKey);
                    _cache.put(_objectKey, object);
                }
            }
        }
        return object;
    }

    /**
     * Invalidates the cache for the object so that it is recomputed the next
     * time it is requested.
     * 
     */
    public void invalidate() {
        _cache.remove(_objectKey);
    }

    /**
     * Gets the cache.
     * 
     * @return Cache.
     */
    public Cache getCache() {
        return _cache;
    }
}
