/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wamblee.general;

import java.io.IOException;
import java.io.File;
import java.net.URLClassLoader;
import java.net.URL;
import java.lang.reflect.Method;

/**
 * Utility for working with the class loader. Based on the ClassPathHacker
 * example found on the internet.   
 */
public class ClassLoaderUtils {
    
    // No logging in this class to keep the required class libraries
    // limited to the standard java classes. This allows use of the
    // utilities in an environment with a very limited classpath. 

    private static final String JAR_SUFFIX = ".jar";

    /**
     * Adds all jars in the given directory to the class path. 
     * @param aDirectory Directory. 
     * @throws IOException
     */
    public static void addJarsInDirectory(File aDirectory) throws IOException {
        System.out.println("directory '" + aDirectory + "'");

        for (File aFile : aDirectory.listFiles()) {
            System.out
                    .println("Considering '" + aFile.getCanonicalPath() + "'");
            if (aFile.getName().toLowerCase().endsWith(JAR_SUFFIX)) {
                System.out.println("Adding '" + aFile.getCanonicalPath()
                        + "' to classpath.");
                addFile(aFile);
            }
        }
    }

    /**
     * Adds a file to the classpath.
     * @param aFilename Filename to add. 
     * @throws IOException
     */
    public static void addFile(String aFilename) throws IOException {
        File f = new File(aFilename);
        addFile(f);
    }

    /**
     * Adds a file to the classpath. 
     * @param aFile File to add. 
     * @throws IOException
     */
    public static void addFile(File aFile) throws IOException {
        addURL(aFile.toURL());
    }

    /**
     * Adds a url to the classpath.
     * @param aUrl Url to add. 
     * @throws IOException
     */
    public static void addURL(URL aUrl) throws IOException {

        URLClassLoader sysloader = (URLClassLoader) ClassLoader
                .getSystemClassLoader();
        Class sysclass = URLClassLoader.class;

        try {
            Method method = sysclass.getDeclaredMethod("addURL", new Class[]{ URL.class } );
            method.setAccessible(true);
            method.invoke(sysloader, new Object[] { aUrl });
        } catch (Throwable t) {
            t.printStackTrace();
            throw new IOException(
                    "Error, could not add URL to system classloader");
        }

    }

}
