/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.conditions;

import org.apache.commons.beanutils.PropertyUtils;

import java.lang.reflect.InvocationTargetException;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Condition to check whether a given property value matches a certain regular
 * expression.
 * 
 * @author Erik Brakkee
 * 
 */
public class PropertyRegexCondition<T> implements Condition<T> {
    /**
     * Property name.
     */
    private String property;

    /**
     * Regular expression.
     */
    private Pattern regex;

    /**
     * Whether or not to convert the value to lowercase before matching.
     */
    private boolean tolower;

    /**
     * Constructs the condition.
     * 
     * @param aProperty
     *            Name of the property to examine.
     * @param aRegex
     *            Regular expression to use.
     * @param aTolower
     *            Whether or not to convert the value to lowercase before
     *            matching.
     */
    public PropertyRegexCondition(String aProperty, String aRegex,
        boolean aTolower) {
        property = aProperty;
        regex = Pattern.compile(aRegex);
        tolower = aTolower;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.conditions.Condition#matches(T)
     */
    public boolean matches(T aObject) {
        try {
            String value = PropertyUtils.getProperty(aObject, property) + "";

            if (tolower) {
                value = value.toLowerCase();
            }

            Matcher matcher = regex.matcher(value);

            return matcher.matches();
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e.getMessage(), e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e.getMessage(), e);
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }
}
