/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.io;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.FileFilter;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Monitors a directory for changes.
 * 
 * @author Erik Brakkee
 */
public class DirectoryMonitor {
    private static final Log LOG = LogFactory.getLog(DirectoryMonitor.class);

    private File directory;

    private FileFilter filter;

    private Listener listener;

    private Map<File, Date> contents;

    /**
     * Creates a new DirectoryMonitor object.
     * 
     */
    public DirectoryMonitor(File aDirectory, FileFilter aFilefilter,
        Listener aListener) {
        directory = aDirectory;

        if (!directory.isDirectory()) {
            throw new IllegalArgumentException("Directory '" + directory +
                "' does not exist");
        }

        filter = aFilefilter;
        listener = aListener;
        contents = new HashMap<File, Date>();
    }

    /**
     * Polls the directory for changes and notifies the listener of any changes.
     * In case of any exceptions thrown by the listener while handling the
     * changes, the next call to this method will invoked the listeners again
     * for the same changes.
     */
    public void poll() {
        LOG.debug("Polling " + directory);

        Map<File, Date> newContents = new HashMap<File, Date>();
        File[] files = directory.listFiles(filter);

        // Check deleted files.
        Set<File> deletedFiles = new HashSet<File>(contents.keySet());

        for (File file : files) {
            if (file.isFile()) {
                if (contents.containsKey(file)) {
                    deletedFiles.remove(file);
                }
            }
        }

        for (File file : deletedFiles) {
            listener.fileDeleted(file);
        }

        for (File file : files) {
            if (file.isFile()) {
                if (contents.containsKey(file)) {
                    Date oldDate = contents.get(file);

                    if (file.lastModified() != oldDate.getTime()) {
                        listener.fileChanged(file);
                    } else {
                        // No change.
                    }

                    newContents.put(file, new Date(file.lastModified()));
                } else {
                    listener.fileCreated(file);
                    newContents.put(file, new Date(file.lastModified()));
                }
            }
        }

        contents = newContents;
    }

    public static interface Listener {
        void fileChanged(File aFile);

        void fileCreated(File aFile);

        void fileDeleted(File aFile);
    }
}
