/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.cache;

import java.io.Serializable;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Represents a cached object identified by the key it has in a certain
 * {@link Cache}. The object is either retrieved from the cache if the cache has
 * it, or a call back is invoked to get the object (and put it in the cache).
 * 
 * @author Erik Brakkee
 * 
 */
public class CachedObject<KeyType extends Serializable, ValueType extends Serializable> {
    private static final Logger LOGGER = Logger.getLogger(CachedObject.class
        .getName());

    /**
     * Cache to use.
     */
    private Cache<KeyType, ValueType> cache;

    /**
     * Key of the object in the cache.
     */
    private KeyType objectKey;

    /**
     * Last known value.
     */
    private ValueType value;

    /**
     * Are we now computing the value or not?
     */
    private boolean computing;

    /**
     * Computation used to obtain the object if it is not found in the cache.
     */
    private Computation<KeyType, ValueType> computation;

    /**
     * Constructs the cached object.
     * 
     * @param aCache
     *            Cache to use.
     * @param aObjectKey
     *            Key of the object in the cache.
     * @param aComputation
     *            Computation to get the object in case the object is not in the
     *            cache.
     */
    public CachedObject(Cache<KeyType, ValueType> aCache, KeyType aObjectKey,
        Computation<KeyType, ValueType> aComputation) {
        cache = aCache;
        objectKey = aObjectKey;
        computation = aComputation;
    }

    /**
     * Gets the object. Since the object is cached, different calls to this
     * method may return different objects.
     * 
     * If the object is expired from the cache it is recomputed using the
     * callback. In case the callback throws an exception the
     * last known value is used. In case an exception is thrown, the problem is
     * also logged. In case a recomputation is already being done by another
     * thread, the last known value is immediately returned.
     * 
     * @return Object.
     */
    public ValueType get() {
        synchronized (this) {
            if (computing) {
                // always return old value while computing.
                return value;
            }

            ValueType cachedValue = cache.get(objectKey);
            if (cachedValue == null) {
                // expired
                computing = true;
            } else {
                // no change.
                return value;
            }
        }
        try {

            // we only get here if we are computing
            // do the computation without holding the lock.
            LOGGER.fine("Refreshing cache for '" + objectKey + "'");
            ValueType object = computation.getObject(objectKey);
            cache.put(objectKey, object);

            synchronized (this) {
                value = object;
            } 
        } catch (Exception e) {
            LOGGER.log(Level.INFO,
                "Recomputation of cached item failed for key '" + objectKey +
                    "'", e);
        } finally {
            synchronized (this) {
                computing = false;
            }
        }
        synchronized(this) { 
            return value; 
        }
    }

    /**
     * Invalidates the cache for the object so that it is recomputed the next
     * time it is requested.
     */
    public void invalidate() {
        cache.remove(objectKey);
    }

    /**
     * Gets the cache.
     * 
     * @return Cache.
     */
    public Cache getCache() {
        return cache;
    }

    /**
     * Callback invoked to compute an object if it was not found in the cache.
     * 
     * @param <T>
     *            Type of the object
     */
    public static interface Computation<Key extends Serializable, Value extends Serializable> {
        /**
         * Gets the object. Called when the object is not in the cache.
         * In case computation fails, an exception should be thrown to ensure that the last 
         * known value will be used. 
         * 
         * @param aObjectKey
         *            Id of the object in the cache.
         * 
         * @return Object, must be non-null.
         */
        Value getObject(Key aObjectKey) throws Exception;
    }
}
