/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.cache;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;


import org.wamblee.io.InputResource;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.logging.Logger;

/**
 * Cache implemented on top of EhCache.
 * 
 * @author Erik Brakkee
 * 
 */
public class EhCache<KeyType extends Serializable, ValueType extends Serializable>
    implements org.wamblee.cache.Cache<KeyType, ValueType> {
    private static final Logger LOGGER = Logger.getLogger(EhCache.class.getName());

    /**
     * EH Cache manager.
     */
    private CacheManager manager;

    /**
     * EH cache.
     */
    private Cache cache;

    /**
     * Constructs a cache based on EHCache.
     * 
     * @param aResource
     *            Resource containing the configuration file for EHCache.
     * @param aCacheName
     *            Name of the cache to use. If a cache with this name does not
     *            exist, one is created based on default settings.
     * @throws IOException
     * @throws CacheException
     */
    public EhCache(InputResource aResource, String aCacheName)
        throws IOException, CacheException {
        InputStream is = aResource.getInputStream();

        try {
            manager = new CacheManager(is);
            cache = manager.getCache(aCacheName);

            if (cache == null) {
                LOGGER.warning("Creating cache '" + aCacheName +
                    "' because it is not configured");
                manager.addCache(aCacheName);
                cache = manager.getCache(aCacheName);
            }
            assert cache != null;
        } finally {
            is.close();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.cache.Cache#put(KeyType, ValueType)
     */
    public void put(KeyType aKey, ValueType aValue) {
        cache.put(new Element(aKey, aValue));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.cache.Cache#get(KeyType)
     */
    public ValueType get(KeyType aKey) {
        try {
            Element element = cache.get(aKey);

            if (element == null) {
                return null;
            }

            return (ValueType) element.getValue();
        } catch (CacheException e) {
            throw new RuntimeException("Cache problem key = '" + aKey + "'", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.cache.Cache#remove(KeyType)
     */
    public void remove(KeyType aKey) {
        cache.remove(aKey);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.cache.Cache#clear()
     */
    public void clear() {
        cache.removeAll();
    }
}
