/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.persistence;

/**
 * Detachable implementation that takes care of the basic logic for detachable
 * objects. All that needs to be done is to implement {@link #load()}.
 * 
 * @param <T>
 *            The type of the object to be attached/detached
 * @param <Ref>
 *            The type of the reference to store when the object is detached.
 */
public abstract class AbstractDetachable<T, Ref> implements Detachable<T> {

    private T object;
    private Ref reference;

    /**
     * Constructs the detachable.
     * 
     * @param aObject
     *            Object.
     * @throws IllegalArgumentException
     *             When the object passed in is null.
     */
    protected AbstractDetachable(T aObject) {
        if (aObject == null) {
            throw new IllegalArgumentException("Object '" + aObject +
                "' is null");
        }
        object = aObject;
        reference = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.persistence.Detachable#detach()
     */
    public void detach() {
        if (object == null) {
            return; // Nothing to do.
        }
        reference = getReference(object);
        if (reference == null) {
            throw new IllegalStateException("Object '" + object +
                "' not persisted yet'");
        }
        object = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.persistence.Detachable#get()
     */
    public T get() {
        if (object == null) {
            object = load(reference);
            reference = null;
        }
        return object;
    }

    /**
     * For testing.
     * 
     * @return current object.
     */
    T getObject() {
        return object;
    }

    /**
     * For testing.
     * 
     * @return The reference.
     */
    Ref getReference() {
        return reference;
    }

    /**
     * Loads the object based on a reference.
     * 
     * @param aReference
     *            Reference.
     * @return Object (may be null ).
     */
    protected abstract T load(Ref aReference);

    /**
     * Obtains the reference for a given object.
     * 
     * @param aObject
     *            Object.
     * @return Reference.
     */
    protected abstract Ref getReference(T aObject);

}
