/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.general;

import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.naming.InitialContext;
import javax.naming.NamingException;

/**
 * Proxy factory that can provide contextual references to objects retrieved
 * through a lookup mechanism. The returned proxies are serializable.
 * 
 * @param T
 *            Interface to proxy.
 * @author Erik Brakkee
 * 
 */
public class LookupProxyFactory<T> {

    /**
     * Interface to lookup the object to delegate to.
     * 
     * @author Erik Brakkee
     */
    public static interface Lookup extends Serializable {
        /**
         * Looks up the object.
         * 
         * @return Object (non-null)
         * @throws Any
         *             exception in case the object cannot be found.
         */
        Object lookup() throws Exception;
    }

    /**
     * Exception thrown in case an object cannot be retrieved from JNDI.
     * 
     * @author Erik Brakkee
     */
    public static class LookupException extends RuntimeException {
        public LookupException(String aMsg, Throwable aCause) {
            super(aMsg, aCause);
        }

        public LookupException(String aMsg) {
            super(aMsg);
        }
    }

    /**
     * Invocation handler that does a lookup in JNDI and invokes the method on
     * the object it found.
     * 
     * @author Erik Brakkee
     */
    private static class LookupInvocationHandler<T> implements
        InvocationHandler, Serializable {

        private Class clazz;
        private Lookup lookup;

        /**
         * Constructs the invocation handler.
         * 
         * @param aLookup
         *            Lookup class.
         */
        public LookupInvocationHandler(Class aClass, Lookup aLookup) {
            clazz = aClass;
            lookup = aLookup;
        }

        @Override
        /**
         * @throws JndiWiringException in case the object could not be retrieved from JNDI. 
         */
        public Object invoke(Object aProxy, Method aMethod, Object[] aArgs)
            throws Throwable {
            Object svcObj = null;
            try {
                svcObj = lookup.lookup();
            } catch (Exception e) {
                throw new LookupException("Error looking up object", e);
            }
            if (svcObj == null) {
                throw new LookupException("Object is null");
            }
            if (!clazz.isInstance(svcObj)) {
                throw new LookupException("Object '" + svcObj +
                    "' is not of type " + clazz.getName() + " but of type " +
                    svcObj.getClass().getName());
            }
            T svc = (T) svcObj;
            try {
                return aMethod.invoke(svc, aArgs);
            } catch (InvocationTargetException e) {
                throw e.getCause();
            }
        }
    }

    private Lookup lookup;
    private Class clazz;

    /**
     * Constructs the factory.
     * 
     * @param aClass
     *            Interface class of the service to proxy.
     * @param aJndi
     *            JNDI name of the object to lookup.
     * 
     */
    public LookupProxyFactory(Class<T> aClass, Lookup aLookup) {
        if (!aClass.isInterface()) {
            throw new IllegalArgumentException("Class " + aClass.getName() +
                " is not an interface");
        }
        clazz = aClass;
        lookup = aLookup;
    }

    /**
     * Gets the proxy that delegates to the thread-specific instance set by
     * {@link #set(Object)}
     * 
     * When at runtime the proxy cannot find lookup the object in JNDI, it
     * throws {@link LookupException}.
     * 
     * @return Proxy.
     */
    public T getProxy() {
        InvocationHandler handler = new LookupInvocationHandler(clazz, lookup);
        Class proxyClass = Proxy.getProxyClass(clazz.getClassLoader(),
            new Class[] { clazz });
        T proxy;
        try {
            proxy = (T) proxyClass.getConstructor(
                new Class[] { InvocationHandler.class }).newInstance(
                new Object[] { handler });
            return proxy;
        } catch (Exception e) {
            throw new RuntimeException("Could not create proxy for " +
                clazz.getName(), e);
        }
    }
}
