/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.reflection;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Some utilities for reflection.
 * 
 * @author Erik Brakkee
 */
public class ReflectionUtils {

    public static final List<Class> PRIMITIVE_WRAPPERS = createPrimitiveWrappers();

    private static final List<Class> createPrimitiveWrappers() {
        Class[] vals = { Boolean.class, Byte.class, Character.class,
            Short.class, Integer.class, Long.class, Float.class, Double.class };
        return Arrays.asList(vals);
    }

    /**
     * Wraps a type by the corresponding wrapper type if it is a primitive type.
     * 
     * @param aClass
     *            Type to wrap.
     * @return Wrapped type for primitives or the provided argument value.
     */
    public static Class wrapIfNeeded(Class aClass) {
        if (aClass == boolean.class) {
            return Boolean.class;
        }

        if (aClass == byte.class) {
            return Byte.class;
        }

        if (aClass == char.class) {
            return Character.class;
        }

        if (aClass == short.class) {
            return Short.class;
        }

        if (aClass == int.class) {
            return Integer.class;
        }

        if (aClass == long.class) {
            return Long.class;
        }

        if (aClass == float.class) {
            return Float.class;
        }

        if (aClass == double.class) {
            return Double.class;
        }

        if (aClass == void.class) {
            return Void.class;
        }

        return aClass;
    }

    public static List<Method> getAllMethods(Class aClass,
        Class... aExcludedClasses) {
        if (aClass.isInterface()) {
            throw new IllegalArgumentException(aClass.getName() +
                " is not an interface.");
        }
        Map<String, Method> found = new HashMap<String, Method>();
        getAllMethods(aClass, found, Arrays.asList(aExcludedClasses));

        return new ArrayList<Method>(found.values());
    }

    private static void getAllMethods(Class aClass, Map<String, Method> aFound,
        List<Class> aExcludedClasses) {
        List<Method> declared = Arrays.asList(aClass.getDeclaredMethods());

        for (Method method : declared) {
            Method superMethod = aFound.get(method.getName());

            if (superMethod == null) {
                // no subclass method
                aFound.put(method.getName(), method);
            } else {
                // subclass method. Check for override.
                if (!Arrays.equals(superMethod.getParameterTypes(), method
                    .getParameterTypes())) {
                    // parameters differ so this is a new method.
                    aFound.put(method.getName(), method);
                }
            }
        }

        Class superClass = aClass.getSuperclass();

        if (superClass != null && !aExcludedClasses.contains(superClass)) {
            getAllMethods(superClass, aFound, aExcludedClasses);
        }
    }

    public static List<Field> getAllFields(Class aClass,
        Class... aExcludedClasses) {
        if (aClass.isInterface()) {
            throw new IllegalArgumentException(aClass.getName() +
                " is an interface.");
        }
        List<Field> found = new ArrayList<Field>();
        getAllFields(aClass, found, Arrays.asList(aExcludedClasses));

        return found;
    }

    private static void getAllFields(Class aClass, List<Field> aFound,
        List<Class> aExcludedClasses) {
        List<Field> declared = Arrays.asList(aClass.getDeclaredFields());

        for (Field field : declared) {
            aFound.add(field);
        }

        Class superClass = aClass.getSuperclass();

        if (superClass != null && !aExcludedClasses.contains(superClass)) {
            getAllFields(superClass, aFound, aExcludedClasses);
        }
    }

    /**
     * Checks if a class is a primitive type or wrapper type.
     * 
     * @param aClass
     * @return
     */
    public static boolean isPrimitive(Class aClass) {
        if (aClass.isPrimitive()) {
            return true;
        }
        return PRIMITIVE_WRAPPERS.contains(aClass);
    }
}
