/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.xml;

import org.w3c.dom.Document;

import org.wamblee.io.FileResource;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.URIResolver;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

/**
 * XSL transformer for simplified usage of XSL transformations.
 * 
 * @author Erik Brakkee
 */
public class XslTransformer {
    private TransformerFactory factory;

    /**
     * Constructs the URL resolver.
     * 
     * @param aResolver
     *            URI resolver to use.
     */
    public XslTransformer(URIResolver aResolver) {
        factory = TransformerFactory.newInstance();
        factory.setURIResolver(aResolver);
    }

    /**
     * Constructs the XSLT processor.
     * 
     */
    public XslTransformer() {
        factory = TransformerFactory.newInstance();
    }

    /**
     * Resolves an XSLT based on URI.
     * 
     * @param aXslt
     *            XSLT to resolve,
     * 
     * @return Source for the XSLT
     * 
     * @throws TransformerException
     *             In case the XSLT cannot be found.
     */
    public Source resolve(String aXslt) throws TransformerException {
        URIResolver resolver = factory.getURIResolver();

        if (resolver == null) {
            if (new File(aXslt).canRead()) {
                try {
                    return new StreamSource(new FileResource(new File(aXslt))
                        .getInputStream());
                } catch (IOException e) {
                    throw new TransformerException(e.getMessage(), e);
                }
            }
            throw new TransformerException("Cannot read '" + aXslt + "'");
        }

        return resolver.resolve(aXslt, "");
    }

    /**
     * Transforms a DOM document into another DOM document using a given XSLT
     * transformation.
     * 
     * @param aDocument
     *            Document to transform.
     * @param aXslt
     *            XSLT to use.
     * 
     * @return Transformed document.
     * 
     * @throws IOException
     *             In case of problems reading resources.
     * @throws TransformerException
     *             In case transformation fails.
     */
    public Document transform(Document aDocument, Source aXslt)
        throws IOException, TransformerException {
        Source source = new DOMSource(aDocument);
        DOMResult result = new DOMResult();
        transform(source, result, aXslt);

        return (Document) result.getNode();
    }

    /**
     * Transforms a document using XSLT.
     * 
     * @param aDocument
     *            Document to transform.
     * @param aXslt
     *            XSLT to use.
     * 
     * @return Transformed document.
     * 
     * @throws IOException
     *             In case of problems reading resources.
     * @throws TransformerException
     *             In case transformation fails.
     */
    public Document transform(byte[] aDocument, Source aXslt)
        throws IOException, TransformerException {
        Source source = new StreamSource(new ByteArrayInputStream(aDocument));
        DOMResult result = new DOMResult();
        transform(source, result, aXslt);

        return (Document) result.getNode();
    }

    /**
     * Transforms a document to a text output. This supports XSLT
     * transformations that result in text documents.
     * 
     * @param aDocument
     *            Document to transform.
     * @param aXslt
     *            XSL transformation.
     * 
     * @return Transformed document.
     * 
     */
    public String textTransform(byte[] aDocument, Source aXslt)
        throws IOException, TransformerException {
        Source source = new StreamSource(new ByteArrayInputStream(aDocument));
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        StreamResult result = new StreamResult(os);
        transform(source, result, aXslt);

        return new String(os.toByteArray());
    }

    /**
     * Transforms a document using XSLT.
     * 
     * @param aSource
     *            Document to transform.
     * @param aResult
     *            Result of the transformation.
     * @param aXslt
     *            XSLT to use.
     * 
     * @throws IOException
     *             In case of problems reading resources.
     * @throws TransformerException
     *             In case transformation fails.
     */
    public void transform(Source aSource, Result aResult, Source aXslt)
        throws IOException, TransformerException {
        try {
            Transformer transformer = factory.newTransformer(aXslt);
            transformer.transform(aSource, aResult);
        } catch (TransformerConfigurationException e) {
            throw new TransformerException(
                "Configuration problem of XSLT transformation", e);
        }
    }
}
