/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.general;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;

/**
 * <p>
 * Thread-specific proxy is used to create implementations of interfaces that
 * delegate to a thread-specific implementation of the service.
 * </p>
 * 
 * <p>
 * It can be used for instance to create a contextual reference to an entity
 * manager that delegates to a thread-specific instance.
 * </p>
 * 
 * <p>
 * The {@link #set(Object)} method sets the current service instance for the
 * current thread. The {@link #get()} method gets the current service instance
 * for the current thread. The {@link #getProxy()} method gets a proxy that will
 * delegate at runtime to the thread-specific instance. The result from this
 * method can be passed at construction of an object that will be used by
 * multiple threads.
 * </p>
 * 
 * <p>
 * This class is mostly used by infrastructure code (utilities) and test tools.
 * </p>
 * 
 * <p>
 * Care has been taken so that the invocation handler is serializable. However,
 * it is only serializable within one virtual machine. It cannot be used in a
 * distributed context where it can be sent to another JVM.
 * </p>
 * 
 * <p>
 * This class currently does not do any cleanup. So it should not be used in production code
 * but only in test utilities. 
 * </p>
 * 
 * @param T
 *            Interface to proxy.
 * @author Erik Brakkee
 * 
 */
public class ThreadSpecificProxyFactory<T> {

    /**
     * Optional callback invoked to create the thread-specific object when there
     * is no object yet associated with the current thread.
     * 
     * @author Erik Brakkee
     * 
     */
    public static interface CreationCallback<T> {
        /**
         * Creates the object.
         * 
         * @return Object.
         */
        T create();
    }

    private ThreadLocal<T> svc;
    private Class clazz;
    private T proxy;

    /**
     * Constructs the factory.
     * 
     * @param aClass
     *            Interface class of the service to proxy.
     */
    public ThreadSpecificProxyFactory(Class<T> aClass) {
        this(aClass, null);
    }

    /**
     * Constructs the factory with a callback to create thread-specific objects
     * automatically.
     * 
     * @param aClass
     *            Interface class of the service to proxy.
     * @param aCallback
     *            Callback to create the object if it does not exist. When null,
     *            then no initialization is done.
     */
    public ThreadSpecificProxyFactory(Class<T> aClass,
        final CreationCallback<T> aCallback) {
        if (!aClass.isInterface()) {
            throw new IllegalArgumentException("Class " + aClass.getName() +
                " is not an interface");
        }
        svc = new ThreadLocal<T>() {
            @Override
            protected T initialValue() {
                if (aCallback != null) {
                    return aCallback.create();
                }
                return null;
            }
        };
        clazz = aClass;
        proxy = createProxy();

    }

    /**
     * Sets the thread-specific service.
     * 
     * @param aService
     *            Service, use null value to reset.
     */
    public void set(T aService) {
        svc.set(aService);
    }

    /**
     * Gets the current thread-specific service. To get a contextual reference
     * that can be used by any thread but delegates to a thread-specific
     * instance, use {@link #getProxy()}.
     * 
     * @return Service.
     */
    public T get() {
        return svc.get();
    }

    /**
     * Gets the proxy that delegates to the thread-specific instance set by
     * {@link #set(Object)}
     * 
     * @return Proxy.
     */
    public T getProxy() {
        return proxy;
    }

    private T createProxy() {
        InvocationHandler handler = new ThreadSpecificInvocationHandler(svc,
            clazz);
        Class proxyClass = Proxy.getProxyClass(clazz.getClassLoader(),
            new Class[] { clazz });
        T proxyObj;
        try {
            proxyObj = (T) proxyClass.getConstructor(
                new Class[] { InvocationHandler.class }).newInstance(
                new Object[] { handler });
            return proxyObj;
        } catch (Exception e) {
            throw new RuntimeException("Could not create proxy for " +
                clazz.getName(), e);
        }
    }
}
