/*
 * Copyright 2005-2011 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.xml;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.transform.URIResolver;
import javax.xml.transform.dom.DOMSource;

import org.w3c.dom.Document;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSException;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSParser;
import org.w3c.dom.ls.LSResourceResolver;

/**
 * Convenience class for representing an XML document. Together with
 * {@link XMLProcessor}, {@link XMLSchema}, {@link XMLTextProcessor}, and
 * {@link XSLTransformation} it provides a simple API for XML parsing,
 * validation, and transformation.
 * 
 * @author Erik Brakkee
 */
public class XMLDocument {

    private static final Logger LOG = Logger.getLogger(XMLDocument.class
        .getName());

    private DOMSource doc;

    public XMLDocument(DOMSource aDoc) {
        doc = aDoc;
    }

    public XMLDocument(URI aUri) throws IOException, XMLException {
        this(aUri.toString(), aUri.toURL().openStream());
    }
    
    public XMLDocument(String aSystemId, String aContents) throws XMLException { 
        this(aSystemId, new ByteArrayInputStream(aContents.getBytes()));
    }

    public XMLDocument(String aSystemId, InputStream aStream)
        throws XMLException {
        LSInput input = DomUtils.getDomImplementationLS().createLSInput();
        input.setByteStream(aStream);
        input.setSystemId(aSystemId);
        doc = read(input, null);
    }

    public XMLDocument(String aSystemId, LSResourceResolver aResolver)
        throws XMLException {
        LSInput input = aResolver.resolveResource(null, null, null, aSystemId,
            null);
        doc = read(input, aResolver);
    }

    public static XMLDocument xmldocument(URI aUri) throws IOException,
        XMLException {
        return new XMLDocument(aUri);
    }
    
    public static XMLDocument xmldocument(String aSystemId, String aContents) throws XMLException { 
        return new XMLDocument(aSystemId, aContents);
    }

    public static XMLDocument xmldocument(String aSystemId, InputStream aStream)
        throws XMLException {
        return new XMLDocument(aSystemId, aStream);
    }

    public static XMLDocument xmldocument(String aSystemId,
        LSResourceResolver aResolver) throws XMLException {
        return new XMLDocument(aSystemId, aResolver);
    }

    public DOMSource getDOMSource() {
        return doc;
    }
    
    public Document getDocument() { 
        return (Document)doc.getNode();
    }

    public XMLDocument process(XMLProcessor aProcessor) throws XMLException {
        return new XMLDocument(aProcessor.process(doc));
    }

    public void write(XMLTextProcessor aProcessor, OutputStream aOs)
        throws XMLException {
        aProcessor.write(doc, aOs);
    }
    
    public String write(XMLTextProcessor aProcessor) throws XMLException { 
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        write(aProcessor, bos); 
        return bos.toString();
    }

    public XMLDocument transform(URI aUri) throws XMLException {
        return process(new XSLTransformation(aUri));
    }

    public XMLDocument transform(String aSystemId, URIResolver aResolver)
        throws XMLException {
        return process(new XSLTransformation(aSystemId, aResolver));
    }

    public XMLDocument validate(URI aUri) throws XMLException {
        return process(new XMLSchema(aUri));
    }

    public XMLDocument validate(String aSystemId,
        LSResourceResolver aResolver) throws XMLException {
        return process(new XMLSchema(aSystemId, aResolver));
    }

    /**
     * Prints an XML document.
     * 
     * @param aOs
     *            Output stream to print on.
     * @param aPrettyPrint
     *            Pretty print or not.
     */
    public void print(OutputStream aOs, boolean aPrettyPrint) {
        try {
            write(new XSLTransformation().setPrettyPrint(aPrettyPrint), aOs);
        } catch (XMLException e) {
            throw new RuntimeException(
                "Programming error, cannot print a DOM tree?: " +
                    doc.getSystemId(), e);
        }
    }

    /**
     * Prints an XML document.
     * 
     * @param aPrettyPrint
     *            Pretty print or not.
     * @return XML string.
     */
    public String print(boolean aPrettyPrint) {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        print(bos, aPrettyPrint);
        return bos.toString();
    }

    private static DOMSource read(LSInput aInput, LSResourceResolver aResolver)
        throws XMLException {
        try {
            DOMImplementationLS impl = DomUtils.getDomImplementationLS();

            LSParser builder = impl.createLSParser(
                DOMImplementationLS.MODE_SYNCHRONOUS, null);
            if (aResolver != null) {
                builder.getDomConfig().setParameter("resource-resolver",
                    aResolver);
            }
            Document docraw = builder.parse(aInput);
            DOMSource result = new DOMSource(docraw);
            result.setSystemId(aInput.getSystemId());
            return result;
        } catch (LSException e) {
            throw new XMLException(e.getMessage(), e);
        }
    }

}
