/*
 * Copyright 2005-2011 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.xml;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.w3c.dom.Document;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;
import org.xml.sax.SAXException;

/**
 * Represents an XML Schema.
 * 
 * @author Erik Brakkee
 */
public class XMLSchema implements XMLProcessor {

    private Schema schema;
    private Validator validator;

    public XMLSchema(URI aUri) throws XMLException {
        try {
            initialize(aUri.toString(), aUri.toURL().openStream(), null);
        } catch (MalformedURLException e) {
            throw new XMLException("Could not read '" + aUri + "'", e);
        } catch (IOException e) {
            throw new XMLException("Could not read '" + aUri + "'", e);
        }
    }

    public XMLSchema(String aSystemId, LSResourceResolver aResolver)
        throws XMLException {
        LSInput input = aResolver.resolveResource(null, null, null, aSystemId,
            null);
        if (input == null) {
            throw new XMLException("Schema classpath resource '" + aSystemId +
                "' not found");
        }
        InputStream is = input.getByteStream();
        initialize(aSystemId, is, aResolver);
    }

    private void initialize(String aSystemId, InputStream aIs,
        LSResourceResolver aResolver) throws XMLException {
        try {
            SchemaFactory factory = SchemaFactory
                .newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            if (aResolver != null) {
                factory.setResourceResolver(aResolver);
            }
            Source source = new StreamSource(aIs);
            source.setSystemId(aSystemId);
            schema = factory.newSchema(source);
            validator = schema.newValidator();
        } catch (SAXException e) {
            throw new XMLException(e.getMessage(), e);
        }
    }

    public static XMLSchema schema(URI aUri) throws XMLException {
        return new XMLSchema(aUri);
    }

    public static XMLSchema schema(String aSystemId,
        LSResourceResolver aResolver) throws XMLException {
        return new XMLSchema(aSystemId, aResolver);
    }

    private void validateImpl(DOMSource aDoc) throws XMLException {
        try {
            validator.validate(aDoc);
        } catch (SAXException e) {
            throw new XMLException(e.getMessage(), e);
        } catch (IOException e) {
            throw new XMLException(e.getMessage(), e);
        }
    }

    public void validate(XMLDocument aDocument) throws XMLException {
        validateImpl(aDocument.getDOMSource());
    }

    public void validate(DOMSource aDocument) throws XMLException {
        validateImpl(aDocument);
    }

    @Override
    public DOMSource process(DOMSource aDocument) throws XMLException {
        validate(aDocument);
        return aDocument;
    }

}
