/*
 * Copyright 2005-2011 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.xml;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * XPath expression. 
 * 
 * @author Erik Brakkee
 */
public class XPathExpression {
    private String expressionString;
    private javax.xml.xpath.XPathExpression expression;

    /**
     * Constructs the expression. Users typically use {@link XPathContext#createExpression(String)} instead
     * of this constructor.
     * 
     * @param aXpath Xpath interface.
     * @param aExpression Expression.
     * @throws XMLException In case of problems. 
     */
    public XPathExpression(XPath aXpath, String aExpression)
        throws XMLException {
        expressionString = aExpression;
        try {
            expression = aXpath.compile(aExpression);
        } catch (XPathExpressionException e) {
            throw new XMLException("Could not compile xpath expression '" +
                aExpression + "'", e);
        }
    }

    /**
     * Evaluates the xpath expression to a string. 
     * @param aDocument Document
     * @return Value.
     * @throws XMLException In case of problems. 
     */
    public String stringEval(XMLDocument aDocument) throws XMLException {
        return (String)evaluateImpl(aDocument, XPathConstants.STRING);
    }
    
    /**
     * Evaluates the xpath expression to a boolean. 
     * @param aDocument Document
     * @return Value.
     * @throws XMLException In case of problems. 
     */
    public boolean booleanEval(XMLDocument aDocument) throws XMLException {
        return (Boolean)evaluateImpl(aDocument, XPathConstants.BOOLEAN);
    }
    
    /**
     * Evaluates the xpath expression to a node list. 
     * @param aDocument Document
     * @return Value.
     * @throws XMLException In case of problems. 
     */
    public NodeList nodelistEval(XMLDocument aDocument) throws XMLException {
        return (NodeList)evaluateImpl(aDocument, XPathConstants.NODESET);
    }
    
    /**
     * Evaluates the xpath expression to a Node. 
     * @param aDocument Document
     * @return Value.
     * @throws XMLException In case of problems. 
     */
    public Node nodeEval(XMLDocument aDocument) throws XMLException {
        return (Node)evaluateImpl(aDocument, XPathConstants.NODE);
    }
    
    /**
     * Evaluates the xpath expression to a number. 
     * @param aDocument Document
     * @return Value.
     * @throws XMLException In case of problems. 
     */
    public Double numberEval(XMLDocument aDocument) throws XMLException {
        return (Double)evaluateImpl(aDocument, XPathConstants.NUMBER);
    }
    
    /**
     * @return Low-level xpath expression object.
     */
    public javax.xml.xpath.XPathExpression getExpression() {
        return expression;
    }
    
    /**
     * @return XPath expression. 
     */
    public String getExpressionString() {
        return expressionString;
    }
    

    private Object evaluateImpl(XMLDocument aDocument, QName aResultType) throws XMLException {
        try {
            return expression.evaluate(aDocument.getDocument(),
                aResultType);
        } catch (XPathExpressionException e) {
            throw new XMLException("Problem evaluating expression '" +
                expressionString + "' on document '" + aDocument.print(true) +
                "'", e);
        }
    }
    
}
