/*
 * Copyright 2005-2011 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.xml;

import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URI;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.URIResolver;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;

/**
 * Represents an XSL Transformation.
 * @author Erik Brakkee
 *
 */
public class XSLTransformation implements XMLProcessor, XMLTextProcessor {

    private TransformerFactory factory;
    private Transformer transformer;
    private String systemId; 

    /**
     * Identity transform.
     */
    public XSLTransformation() throws XMLException {
        factory = TransformerFactory.newInstance();
        try {
            transformer = factory.newTransformer();
            systemId = "identityTransform";
        } catch (TransformerConfigurationException e) {
            throw new XMLException(e.getMessage(), e);
        }
    }

    public XSLTransformation(URI aUri) throws XMLException {
        try {
            factory = TransformerFactory.newInstance();
            Source source = new StreamSource(aUri.toURL().openStream());
            source.setSystemId(aUri.toString());
            systemId = aUri.toString();
            transformer = factory.newTransformer(source);
        } catch (MalformedURLException e) {
            throw new XMLException(e.getMessage(), e);
        } catch (TransformerConfigurationException e) {
            throw new XMLException(e.getMessage(), e);
        } catch (TransformerFactoryConfigurationError e) {
            throw new XMLException(e.getMessage(), e);
        } catch (IOException e) {
            throw new XMLException(e.getMessage(), e);
        }

    }

    public XSLTransformation(String aSystemId, URIResolver aResolver)
        throws XMLException {
        try {
            factory = TransformerFactory.newInstance();
            factory.setURIResolver(aResolver);
            Source source = aResolver.resolve(aSystemId, null);
            source.setSystemId(aSystemId);
            systemId = aSystemId;
            transformer = factory.newTransformer(source);
        } catch (TransformerConfigurationException e) {
            throw new XMLException(e.getMessage(), e);
        } catch (TransformerFactoryConfigurationError e) {
            throw new XMLException(e.getMessage(), e);
        } catch (TransformerException e) { 
            throw new XMLException(e.getMessage(), e);
        }
    }
    
    /**
     * Configure the transformation to do pretty-printing. 
     */
    public XSLTransformation setPrettyPrint(boolean aPrettyPrint) { 
        transformer.setOutputProperty(OutputKeys.INDENT, aPrettyPrint ? "yes": "no");
        return this; 
    }
    
  
    public XSLTransformation transformation(URI aUri) throws XMLException {
        return new XSLTransformation(aUri);
    }

    public XSLTransformation transformation(String aSystemId, URIResolver aResolver)
        throws XMLException {
        return new XSLTransformation(aSystemId, aResolver);
    }

    @Override
    public DOMSource process(DOMSource aDocument) throws XMLException {
        DOMResult result = new DOMResult();
        try {
            transform(aDocument, result);
        } catch (IOException e) {
            throw new XMLException(e.getMessage(), e);
        } catch (TransformerException e) {
            throw new XMLException(e.getMessage(), e);
        }
        Document docraw = (Document)result.getNode();
        DOMSource resultSource = new DOMSource(docraw); 
        resultSource.setSystemId(aDocument.getSystemId() + "/transformed(" + systemId + ")");
        return resultSource;
    }

    @Override
    public void write(DOMSource aDocument, OutputStream aStream) throws XMLException {
        StreamResult result = new StreamResult(aStream);
        try {
            transform(aDocument, result);
        } catch (IOException e) {
            throw new XMLException(e.getMessage(), e);
        } catch (TransformerException e) {
            throw new XMLException(e.getMessage(), e);
        }
    }

    /**
     * Transforms a document using XSLT.
     * 
     * @param aSource
     *            Document to transform.
     * @param aResult
     *            Result of the transformation.
     * 
     * @throws IOException
     *             In case of problems reading resources.
     * @throws TransformerException
     *             In case transformation fails.
     */
    private void transform(Source aSource, Result aResult) throws IOException,
        TransformerException {
        try {
            transformer.transform(aSource, aResult);
        } catch (TransformerConfigurationException e) {
            throw new TransformerException(
                "Configuration problem of XSLT transformation", e);
        }
    }
}
