/*
 * Copyright 2005-2011 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.xml;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;

/**
 * Implementation of {@link NamespaceContext} for binding namespace prefixes to 
 * namespaces. This class implements the full contract as defined by that interface. 
 * 
 * It provides a fluent interface style. In its simplest use, it can be constructed
 * empty or with a single prefix using a constructor. In addition, the methods
 * {@link #addPrefix(String, String)} and {@link #defaultNamespace(String)} can be used
 * to add prefixes and to define the default namespace respectively. 
 * 
 * @author Erik Brakkee
 */
public class SimpleNamespaceContext implements NamespaceContext {

    private String defaultNs;
    private Map<String, String> prefixMap;

    /**
     * Constructs an empty context without prefixes. 
     */
    public SimpleNamespaceContext() {
        defaultNs = null;
        prefixMap = new HashMap<String, String>();
    }
    
    /**
     * Creation of the namespace context for use with static imports.
     * @return Empty namespace context. 
     */
    public static SimpleNamespaceContext namespaces() { 
        return new SimpleNamespaceContext();
    }

    /**
     * Constructs a context with a single prefix. 
     * @param aPrefix Prefix. 
     * @param aNamespace Namespace. 
     */
    public SimpleNamespaceContext(String aPrefix, String aNamespace) {
        this();
        addPrefix(aPrefix, aNamespace);
    }

    /**
     * Constructs empty context (for use with static imports).
     * @return Namespace context. 
     */
    public static SimpleNamespaceContext namespaceContext() {
        return new SimpleNamespaceContext();
    }

    /**
     * Constructs a context with a single prefix (for use with static imports).  
     * @param aPrefix Prefix
     * @param aNamespace Namespace. 
     * @return
     */
    public static SimpleNamespaceContext namespaceContext(String aPrefix,
        String aNamespace) {
        return new SimpleNamespaceContext(aPrefix, aNamespace);
    }

    /**
     * Sets the default namespace. 
     * @param aDefaultNs Default namespace. 
     * @return Current object for method chaining. 
     */
    public SimpleNamespaceContext defaultNamespace(String aDefaultNs) {
        defaultNs = aDefaultNs;
        return this;
    }

    /**
     * Adds a prefix. 
     * @param aPrefix Prefix to add. 
     * @param aNamespace Namespace to bind prefix to. 
     * @return Current object for method chaining. 
     */
    public SimpleNamespaceContext addPrefix(String aPrefix, String aNamespace) {
        if (aPrefix == null) {
            throw new IllegalArgumentException("prefix is null");
        }
        if (aNamespace == null) {
            throw new IllegalArgumentException("namespace is null");
        }
        prefixMap.put(aPrefix, aNamespace);
        return this;
    }

    @Override
    public String getNamespaceURI(String aPrefix) {
        if (XMLConstants.DEFAULT_NS_PREFIX.equals(aPrefix)) {
            if (defaultNs == null) {
                return XMLConstants.NULL_NS_URI;
            }
            return defaultNs;
        }
        if (XMLConstants.XML_NS_PREFIX.equals(aPrefix)) {
            return XMLConstants.XML_NS_URI;
        }
        if (XMLConstants.XMLNS_ATTRIBUTE.equals(aPrefix)) {
            return XMLConstants.XMLNS_ATTRIBUTE_NS_URI;
        }
        String ns = prefixMap.get(aPrefix);
        if (ns == null) {
            return XMLConstants.NULL_NS_URI;
        }
        return ns;
    }

    @Override
    public String getPrefix(String aNamespaceURI) {
         List<String> prefixes = getPrefixList(aNamespaceURI);
         if (prefixes.isEmpty()) { 
             return null; 
         }
         return prefixes.get(0);
    }

    @Override
    public Iterator getPrefixes(String aNamespaceURI) {
        return Collections.unmodifiableList(getPrefixList(aNamespaceURI)).iterator();
    }
    
    public List<String> getPrefixList(String aNamespaceURI) {
        List<String> result = new ArrayList<String>(); 
        
        if ((aNamespaceURI == null && defaultNs == null) ||
            (defaultNs != null && defaultNs.equals(aNamespaceURI))) {
            result.add(XMLConstants.DEFAULT_NS_PREFIX);
            return result; // make sure not more prefixes added.
        }
        if (XMLConstants.XML_NS_URI.equals(aNamespaceURI)) {
            result.add(XMLConstants.XML_NS_PREFIX);
            return result; // make sure no more prefixes added. 
        }
        if (XMLConstants.XMLNS_ATTRIBUTE_NS_URI.equals(aNamespaceURI)) {
            result.add(XMLConstants.XMLNS_ATTRIBUTE);
            return result; 
        }
        for (Map.Entry<String,String> entry: prefixMap.entrySet()) {
            if (entry.getValue().equals(aNamespaceURI)) { 
                result.add(entry.getKey());
            }
        }
        return result; 
    }

}
