/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package org.wamblee.general.spring;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.access.BeanFactoryLocator;
import org.springframework.beans.factory.access.BeanFactoryReference;
import org.springframework.context.access.ContextSingletonBeanFactoryLocator;
import org.wamblee.ioc.BeanFactory;
import org.wamblee.ioc.BeanFactoryException;

/**
 * Bean factory which uses Spring. This bean factory cannot be configured
 * directly in the {@link org.wamblee.general.BeanKernel} because it does not
 * provide a default no-arg constructor. Therefore, it must be delegated to or
 * it must tbe subclassed to provide a default constructor.
 */
public class SpringBeanFactory implements BeanFactory {
    private BeanFactoryReference factoryReference;

    /**
     * Constructs the bean factory.
     * 
     * @param aSelector
     *            Selector to find the appropriate bean ref context.
     * @param aFactoryName
     *            Spring bean factory to use.
     */
    public SpringBeanFactory(String aSelector, String aFactoryName) {
        try {
            BeanFactoryLocator locator = ContextSingletonBeanFactoryLocator
                .getInstance(aSelector);
            factoryReference = locator.useBeanFactory(aFactoryName);
        } catch (BeansException e) {
            throw new BeanFactoryException(
                "Could not load bean factory: selector = '" + aSelector +
                    "', factory = '" + aFactoryName + "'", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.general.BeanFactory#find(java.lang.String)
     */
    public Object find(String aId) {
        return find(aId, Object.class);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.general.BeanFactory#find(java.lang.Class)
     */
    public <T> T find(Class<T> aClass) {
        return find(aClass.getName(), aClass);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wamblee.general.BeanFactory#find(java.lang.String,
     * java.lang.Class)
     */
    public <T> T find(String aId, Class<T> aClass) {
        try {
            Object obj = factoryReference.getFactory().getBean(aId, aClass);
            assert obj != null;

            return aClass.cast(obj);
        } catch (BeansException e) {
            throw new BeanFactoryException(e.getMessage(), e);
        }
    }

    /**
     * Gets the spring bean factory.
     * 
     * @return Spring bean factory.
     */
    public org.springframework.beans.factory.BeanFactory getSpringBeanFactory() {
        return factoryReference.getFactory();
    }
}
