/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.support;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * Thread-specific proxy is used to create implementations of interfaces that
 * delegate to a thread-specific implementation of the service.
 * 
 * It is used for instance to pass a transaction scoped entity manager around.
 * 
 * @param T
 *            Interface to proxy.
 * @author Erik Brakkee
 * 
 */
public class ThreadSpecificProxyFactory<T> {
    private class ThreadSpecificInvocationHandler implements InvocationHandler {

        @Override
        public Object invoke(Object aProxy, Method aMethod, Object[] aArgs)
            throws Throwable {
            try {
                return aMethod.invoke(svc.get(), aArgs);
            } catch (InvocationTargetException e) {
                throw e.getCause();
            }
        }
    }

    private ThreadLocal<T> svc = new ThreadLocal<T>();
    private Class clazz;

    /**
     * Constructs the factory.
     * 
     * @param aClass
     *            Interface class of the service to proxy.
     */
    public ThreadSpecificProxyFactory(Class<T> aClass) {
        if (!aClass.isInterface()) {
            throw new IllegalArgumentException("Class " + aClass.getName() +
                " is not an interface");
        }
        clazz = aClass;
    }

    /**
     * Sets the thread-specific service.
     * 
     * @param aService
     *            Service, use null value to reset.
     */
    public void set(T aService) {
        svc.set(aService);
    }

    /**
     * Gets the proxy that delegates to the thread-specific instance set by
     * {@link #set(Object)}
     * 
     * @return Proxy.
     */
    public T getProxy() {
        InvocationHandler handler = new ThreadSpecificInvocationHandler();
        Class proxyClass = Proxy.getProxyClass(clazz.getClassLoader(),
            new Class[] { clazz });
        T proxy;
        try {
            proxy = (T) proxyClass.getConstructor(
                new Class[] { InvocationHandler.class }).newInstance(
                new Object[] { handler });
            return proxy;
        } catch (Exception e) {
            throw new RuntimeException("Could not create proxy for " +
                clazz.getName(), e);
        }
    }
}
