/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.support.persistence;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.persistence.EntityManager;

import org.wamblee.support.ThreadSpecificProxyFactory;
import org.wamblee.support.persistence.JpaBuilder.JpaUnitOfWork;

/**
 * This utility makes sure that each invocation on a certain interface is
 * carried out within a JPA unit of work.
 * 
 * Use {@link #getTransactionScopedEntityManager()} to get the transaction
 * scoped entity manager to pass to services.
 * 
 * @param T
 *            Type of interface to proxy.
 * 
 * @author Erik Brakkee
 */
public class TransactionProxyFactory<T> {

    private class UnitOfWorkInvocationHandler<T> implements InvocationHandler {

        private T service;

        public UnitOfWorkInvocationHandler(T aService) {
            service = aService;
        }

        @Override
        public Object invoke(Object aProxy, final Method aMethod,
            final Object[] aArgs) throws Throwable {
            return TransactionProxyFactory.this.jpaBuilder
                .execute(new JpaUnitOfWork<Object>() {
                    @Override
                    public Object execute(EntityManager aEm) throws Exception {
                        try {
                            ENTITY_MANAGER.set(aEm);
                            return aMethod.invoke(service, aArgs);
                        } catch (InvocationTargetException e) {
                            Throwable cause = e.getCause();
                            if (cause instanceof Exception) {
                                throw (Exception) cause;
                            } else if (cause instanceof Error) {
                                throw (Error) cause;
                            }
                            // last resort.
                            throw new RuntimeException(e);
                        } finally {
                            ENTITY_MANAGER.set(null);
                        }
                    }
                });
        }

    }

    private static final ThreadSpecificProxyFactory<EntityManager> ENTITY_MANAGER = new ThreadSpecificProxyFactory<EntityManager>(
        EntityManager.class);

    private JpaBuilder jpaBuilder;
    private Class<T> clazz;

    /**
     * Constructs the transaction proxy.
     * 
     * @param aJpaBuilder
     */
    public TransactionProxyFactory(JpaBuilder aJpaBuilder, Class<T> aClass) {
        jpaBuilder = aJpaBuilder;
        clazz = aClass;
    }

    public EntityManager getTransactionScopedEntityManager() {
        return ENTITY_MANAGER.getProxy();
    }

    public T getProxy(T aService) {
        InvocationHandler handler = new UnitOfWorkInvocationHandler<T>(aService);
        Class proxyClass = Proxy.getProxyClass(clazz.getClassLoader(),
            new Class[] { clazz });
        T proxy;
        try {
            proxy = (T) proxyClass.getConstructor(
                new Class[] { InvocationHandler.class }).newInstance(
                new Object[] { handler });
            return proxy;
        } catch (Exception e) {
            throw new RuntimeException("Could not create proxy for " +
                clazz.getName(), e);
        }
    }
}
