/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.inject;

import javax.persistence.EntityManager;

import org.wamblee.inject.Injector;
import org.wamblee.inject.InjectorFactory;
import org.wamblee.inject.InjectorFactoryBuilder;
import org.wamblee.test.persistence.JpaBuilder;

/**
 * <p>
 * The test injector factory provides dependency injection of a contextual entity manager
 * using the support/inject mini framework. It supports dependency injection of fields
 * annoted with <code>&#064;PersistenceContext</code>. It only supports one persistence context
 * at the moment. This injector can be easily used together with {@link JpaBuilder#getContextualEntityManager()}
 * for obtaining an entity manager in unit test. 
 * </p>
 * 
 * <p>
 * The reason it is needed is because standard injection mechanisms (such as weld CDI) do not support
 * entity manager injection in a Java SE environment out of the box.  
 * </p>
 *
 * <p>
 * To use it, construct the factory using one of the available constructors and set 
 * <code>InjectorFactoryBuilder.setInjectorFactory(InjectorFactory</code>.
 * </p>
 * 
 * @author Erik Brakkee
 *
 */
public class JavaEETestInjectorFactory implements InjectorFactory {

    private EntityManager entityManager;
    private InjectorFactory delegate;
    
    /**
     * Constructs the factory. 
     * @param aEntityManager Contextual entity manager to inject. 
     * @param aInjectorFactory Injector factory to delegate to. 
     */
    public JavaEETestInjectorFactory(EntityManager aEntityManager, InjectorFactory aInjectorFactory) { 
        entityManager = aEntityManager; 
        delegate = aInjectorFactory;
    }

    /**
     * Constructs the factory with the default injector factory obtained from 
     * {@link InjectorFactoryBuilder#getInjector()}. 
     * @param aEntityManager Contextual entity manager to inject. 
     */
    public JavaEETestInjectorFactory(EntityManager aEntityManager) {
        this(aEntityManager,  getDefaultInjectorFactory());
    }

    private static InjectorFactory getDefaultInjectorFactory() {
        InjectorFactoryBuilder.setInjectorFactory(null);
        return InjectorFactoryBuilder.getInjectorFactory();
    }

    @Override
    public Injector create(Class aClass) {
        return new JavaEETestInjector(aClass, entityManager, delegate
            .create(aClass));
    }
}
