/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.persistence;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.sql.DataSource;

import junit.framework.Assert;

import org.apache.commons.dbcp.ConnectionFactory;
import org.apache.commons.dbcp.DriverManagerConnectionFactory;
import org.apache.commons.dbcp.PoolableConnectionFactory;
import org.apache.commons.dbcp.PoolingDataSource;
import org.apache.commons.pool.impl.GenericObjectPool;

/**
 * Abstract database class providing the creation of the datasource, 
 * preventing duplicate starts of the same database, and checking
 * for connection leaks when the database is stopped. 
 * 
 * @author Erik Brakkee
 *
 */
public abstract class AbstractDatabase implements Database {

    /**
     * Set this system property to a non-null value to ignore connection leaks
     * when {@link #stop()} is called.
     */
    private static final String IGNORE_CONNECTION_LEAK_PROPERTY = "org.wamblee.database.ignoreconnectionleaks";

    private static final Logger LOGGER = Logger
        .getLogger(AbstractDatabase.class.getName());

    private static final int CONNECTION_POOL_SIZE = 16;

    private DataSource itsDataSource;

    private GenericObjectPool connectionPool;

    private boolean started;

    /**
     * Constructs the database. 
     */
    protected AbstractDatabase() {
        started = false;
    }

    /**
     * To be implemented by subclasses to start the database. 
     */
    protected abstract void doStart();

    /**
     * To be implemented by subclasses to stop the database. 
     */
    protected abstract void doStop();

    /**
     * This method must be called from the start method.
     */
    protected final void createDataSource() {
        connectionPool = new GenericObjectPool(null);
        connectionPool.setMaxActive(CONNECTION_POOL_SIZE);
        ConnectionFactory connectionFactory = new DriverManagerConnectionFactory(
            getJdbcUrl(), getUsername(), getPassword());
        // The following line must be kept in although it does not appear to be
        // used, the constructor regsiters the
        // constructed object at the connection pool.
        PoolableConnectionFactory poolableConnectionFactory = new PoolableConnectionFactory(
            connectionFactory, connectionPool, null, null, false, true);
        ingoredVariable(poolableConnectionFactory);
        itsDataSource = new PoolingDataSource(connectionPool);
    }

    @Override
    public int getActiveConnections() {
        return connectionPool.getNumActive();
    }

    private static void ingoredVariable(PoolableConnectionFactory aFactory) {
        // Empty
    }

    // / BELOW THIS LINE IS NOT OF INTEREST TO SUBCLASSES.

    /**
     * Starts the database. 
     */
    public final DataSource start() {
        if (started) {
            throw new RuntimeException("Database already started");
        }
        started = true;
        doStart();
        return getDatasource();
    }

    /**
     * Stops the database and tests for connection leaks. 
     * In cast the system property with the name given by {@link #IGNORE_CONNECTION_LEAK_PROPERTY}
     * is set then the connection leaks are  not checked. 
     */
    public final void stop() {
        if (!started) {
            return; // nothing to do.
        }
        started = false;
        try {
            if (connectionPool.getNumActive() > 0) {
                String msg = "JDBC connection pool still has " +
                    connectionPool.getNumActive() +
                    " active connection(s), this is a potential resource leak in the code\n";
                // backdoor to ignore connection leaks. Use this system property only if you 
                // can safely ignore the connection leaks. 
                if (System.getProperty(IGNORE_CONNECTION_LEAK_PROPERTY) == null) {
                    Assert.fail(msg);
                }
            }
            connectionPool.close();
            connectionPool.close();
        } catch (Exception e) {
            LOGGER.log(Level.WARNING, "Could not close pool", e);
        }
        doStop();
    }

    private final DataSource getDatasource() {
        if (!started) {
            throw new RuntimeException("Database is not started!");
        }
        return itsDataSource;
    }

    protected String getProperty(String aName) {
        String value = System.getProperty(aName);
        if (value != null) {
            return value;
        }
        value = System.getenv(aName);
        if (value != null) {
            return value;
        }
        throw new RuntimeException("This class expects the '" + aName +
            "' property to be set");
    }

}
