/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.persistence;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import org.wamblee.test.jndi.StubInitialContextFactory;

/**
 * This class is the entry point for JPA tests. Test code should construct a
 * JpaTester in the <code>@Before</code> method and call {@link #start()} on it
 * in that method. Also, test code should call {@link #stop()} on it in the
 * <code>@After</code> method.
 * 
 * This class is constructed with a description of the persistence unit to be
 * tested. The principle is that an existing <code>persistence.xml</code> can be
 * tested without change in unit test code.
 * 
 * It then takes care of the following:
 * <ul>
 * <li>Creating an inmemory database for testing (default) or connecting to an
 * external database. See {@link DatabaseBuilder} for more information on how a
 * databse is obtained.</li>
 * <li>Drop all database tables that are related to the persistence unit under
 * test, including JPA provider specific tables.</li>
 * <li>Creating a datasource for the database and make the datasource available
 * through JNDI.</li>
 * <li>Creating the entity manager factory for JPA and configuring it in such a
 * way that schema creation happens. (Typically, schema creation will be
 * disabled in the persistence.xml but this utility enables it for unit test).</li>
 * <li>Creating a DBUnit database tester which is appropriately configured for
 * the persistence unit under test.</li>
 * </ul>
 * 
 * The main entry point for all this functionality is the
 * {@link PersistenceUnitDescription} which describes the persistence unit and
 * must be provided at construction of the <code>JpaTester</code>
 * 
 * NOTE: Persistence XML files should be explicitly configured with the classes
 * that are part of the persistence unit since scanning of classes does not work
 * correctly in a unit test environment. This is currently the only limitation.
 */
public class JpaTester {

    private PersistenceUnitDescription persistenceUnit;
    private Database db;
    private DataSource dataSource;
    private DatabaseUtils dbUtils;
    private JpaBuilder jpaBuilder;
   
    /**
     * Constructs the tester.
     * 
     * @param aPersistenceUnit
     *            Persistence unit under test.
     */
    public JpaTester(PersistenceUnitDescription aPersistenceUnit) {
        persistenceUnit = aPersistenceUnit;
    }

    /**
     * Starts the tester. This must be called prior to running the test.
     * 
     * @throws Exception
     */
    public void start() throws Exception {
        db = DatabaseBuilder.getDatabase();
        dataSource = db.start();
        
        // NOTE: adding datasource to JNDI is no longer needed for 
        //       JPA testing, but is nice to have available for other uses.
        StubInitialContextFactory.register();
        try {
            InitialContext ctx = new InitialContext();
            ctx.bind(persistenceUnit.getJndiName(), dataSource);
        } catch (NamingException e) {
            throw new RuntimeException("JNDI problem", e);
        }

        dbUtils = new DatabaseUtils(dataSource);
        dbUtils.start();
        dbUtils.dropTables(JpaCustomizerBuilder.getCustomizer().getJpaTables());

        jpaBuilder = new JpaBuilder(db.getJdbcUrl(), db.getUsername(), db.getPassword(), persistenceUnit);
        jpaBuilder.start();
    }

    /**
     * Stops the tester. This must be called after the test.
     */
    public void stop() {
        if (jpaBuilder != null) {
            jpaBuilder.stop();
        }
        if (dbUtils != null) {
            dbUtils.stop();
        }
        if (db != null) {
            db.stop();
        }
    }

    /**
     * Gets the database. 
     * @return Database. 
     */
    public Database getDb() {
        return db;
    }

    /**
     * Gets the datasource. 
     * @return Datasource. 
     */
    public DataSource getDataSource() {
        return dataSource;
    }

    /**
     * Gets the database utilities. 
     * @return Database utilities. 
     */
    public DatabaseUtils getDbUtils() {
        return dbUtils;
    }

    /**
     * Gets the jpa builder. 
     * @return JPA builder. 
     */
    public JpaBuilder getJpaBuilder() {
        return jpaBuilder;
    }

    /**
     * Gets the persistence unit. 
     * @return Persistence unit. 
     */
    public PersistenceUnitDescription getPersistenceUnit() {
        return persistenceUnit;
    }

}
