/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.inject;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.wamblee.inject.Injector;
import org.wamblee.inject.InjectorBuilder;
import org.wamblee.inject.InjectorFactory;
import org.wamblee.test.persistence.JpaBuilder;

/**
 * <p>
 * The test injector factory provides dependency injection of a contextual
 * entity manager using the support/inject mini framework. It supports
 * dependency injection of fields annoted with
 * <code>&#064;PersistenceContext</code>. It only supports one persistence
 * context at the moment. This injector can be easily used together with
 * {@link JpaBuilder#getContextualEntityManager()} for obtaining an entity
 * manager in unit test.
 * </p>
 * 
 * <p>
 * The reason it is needed is because standard injection mechanisms (such as
 * weld CDI) do not support entity manager injection in a Java SE environment
 * out of the box.
 * </p>
 * 
 * <p>
 * To use it, construct the factory using one of the available constructors and
 * set <code>InjectorBuilder.setInjectorFactory(InjectorFactory)</code>.
 * </p>
 * 
 * @author Erik Brakkee
 * 
 */
public class JavaEETestInjectorFactory implements InjectorFactory {

    private List<Binding> bindings;
    private InjectorFactory delegate;

    /**
     * Constructs the factory.
     * 
     * @param aInjectorFactory
     *            Injector factory to delegate to.
     */
    public JavaEETestInjectorFactory(InjectorFactory aInjectorFactory) {
        bindings = new ArrayList<Binding>();
        delegate = aInjectorFactory;
    }

    /**
     * Adds default entity manager binding. Any field annotated with @PersistenceContext
     * and of type entity manager will get injected.
     * 
     * @param aEntityManager
     *            Entitymanager object to inject.
     * @return Factory to allow chaining.
     */
    public JavaEETestInjectorFactory addEntityManagerBinding(
        EntityManager aEntityManager) {
        Binding em = new Binding(EntityManager.class, PersistenceContext.class,
            aEntityManager);
        addBinding(em);
        return this;
    }

    /**
     * Adds another custom injection binding.
     * 
     * @param aBinding
     *            Injection binding to use.
     * @return the factoryto allow chaining.
     */
    public JavaEETestInjectorFactory addBinding(Binding aBinding) {
        bindings.add(aBinding);
        return this;
    }

    /**
     * Constructs the factory with the default injector factory obtained from
     * {@link InjectorBuilder#getInjector()}.
     */
    public JavaEETestInjectorFactory() {
        this(getDefaultInjectorFactory());
    }

    private static InjectorFactory getDefaultInjectorFactory() {
        InjectorBuilder.setInjectorFactory(null);
        return InjectorBuilder.getInjectorFactory();
    }

    @Override
    public Injector create(Class aClass) {
        return new JavaEETestInjector(aClass, bindings, delegate.create(aClass));
    }
}
