/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.jndi;

import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.spi.InitialContextFactory;

/**
 * Test initial context factory used for testing software in a Java SE
 * environnment that uses JNDI to retrieve objects.
 * 
 * See {@link #bind(String, Object)} to resp. register the initial context.
 * 
 * To start mocking the JNDI tree, call {@link #register()}.
 * 
 * To bind objects in the JNDI tree simply use the standard JNDI api:
 * 
 * <pre>
 *   InitialContext context = new InitialContext();
 *   MyClass myObj = ...; 
 *   context.bind("a/b", myObj);
 * </pre>
 * 
 * When finished with a test case, call {@link #unregister()} to unregister the
 * JNDI tree again.
 */
public class StubInitialContextFactory implements InitialContextFactory {

    private static Context CONTEXT;

    private static void initialize() {
        try {
            CONTEXT = new StubInitialContext();
        } catch (NamingException e) { // can't happen.
            throw new RuntimeException(e);
        }
    }

    /**
     * This method must be called to register this initial context factory as
     * the default implementation for JNDI.
     * 
     * @throws Exception
     */
    public static void register() {
        // sets up the InitialContextFactoryForTest as default factory.
        System.setProperty(Context.INITIAL_CONTEXT_FACTORY,
            StubInitialContextFactory.class.getName());
        if (CONTEXT == null) {
            initialize();
        }
    }

    /**
     * Unregisters the initial context factory
     */
    public static void unregister() {
        System.setProperty(Context.INITIAL_CONTEXT_FACTORY, "");
        CONTEXT = null;
    }

    public Context getInitialContext(Hashtable<?, ?> aEnvironment)
        throws NamingException {
        return CONTEXT;
    }
}
