/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.persistence;

/**
 * This class is used for starting the database as a main program.
 */
public class DatabaseStarter {

    /**
     * Database class which encapsulates management of the database.
     */
    private Class databaseClass;

    /**
     * Execution as a main program. Commandline
     * 
     * <pre>
     * 
     *    DatabaseStarter &lt;databaseClassName&gt;
     * 
     * </pre>
     * 
     * where the database class name must be the name of a concrete subclass of
     * {@link Database}.
     * 
     * @param aArgs
     */
    public static void main(String[] aArgs) throws Exception {
        String clazz = aArgs[0];
        try {
            new DatabaseStarter(Class.forName(clazz)).start();
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("\nUsage: ant dbClass ");
        }
    }

    /**
     * Constructs the database starter.
     * 
     * @param aClassName
     *            Classname of the database class to use.
     * @throws Exception
     */
    public DatabaseStarter(Class aClass) throws Exception {
        if (!Database.class.isAssignableFrom(aClass)) {
            throw new IllegalArgumentException("Class '" + aClass.getName() +
                "' is not a subclass of Database");
        }
        databaseClass = aClass;
    }

    /**
     * Constructs a database starter with the derby database.
     * 
     * @throws Exception
     */
    public DatabaseStarter() throws Exception {
        this(DerbyDatabase.class);
    }

    /**
     * Starts the database.
     * 
     * @throws Exception
     */
    public void start() throws Exception {
        Database lDatabase = (Database) databaseClass.newInstance();
        lDatabase.start();
        System.out.println("Database has been started. ");
        System.out.println();
        System.out
            .println("=======================================================");
        System.out.println("Connection details:");
        // System.out.println( "  Driver class: "
        // + lDatabase.getDriverClassName( ) );
        System.out.println("  JDBC URL:     " + lDatabase.getExternalJdbcUrl());
        System.out.println("  username:     '" + lDatabase.getUsername() + "'");
        System.out.println("  password:     '" + lDatabase.getPassword() + "'");
        System.out.println("Interrupt the program to stop the database.");
        System.out
            .println("=======================================================");
        System.out
            .println("You must now populate the database with a schema. Use 'ant help' for information.");
        for (;;) {
            Thread.sleep(1000);
        }
    }
}
