/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.test.transactions;

import java.util.ArrayList;
import java.util.List;

import javax.transaction.UserTransaction;

import org.wamblee.general.ThreadSpecificProxyFactory;
import org.wamblee.general.ThreadSpecificProxyFactory.CreationCallback;
import org.wamblee.test.persistence.JpaBuilder;

/**
 * 
 * <p>
 * Simple transaction manager provides a simple mechanism to manage transactions
 * in test code through the {@link UserTransaction} object. To construct the
 * transaction manager use:
 * </p>
 * 
 * <pre>
 * SimpleTransactionManager manager = new SimpleTransactionManager(
 *     new DefaultUserTransactionFactory());
 * </pre>
 * 
 * <p>
 * Next, add resources to manage using {@link #addResource(TransactionResource)}:
 * 
 * <pre>
 * manager.addResource(jpaTester.getJpaBuilder());
 * </pre>
 * <p>
 * As you can see from the above, {@link JpaBuilder} is a resource that can be
 * used so this mechanism can be used with JPA testing.
 * </p>
 * 
 * <p>
 * The next step is to manage transactions using the standard
 * <code>UserTransaction</code> APIs:
 * </p>
 * 
 * <pre>
 *      UserTransaction transaction = manager.getTransaction();
 *      transaction.begin();
 *     
 *      ... do work...
 *  
 *      transaction.commit();
 * </pre>
 * 
 * @author Erik Brakkee
 * 
 */
public class SimpleTransactionManager {

    private UserTransactionFactory factory;
    private List<TransactionResource> resources;
    private UserTransactionCallback transactionFInishedCallback;
    private ThreadSpecificProxyFactory<UserTransaction> transaction;

    /**
     * Constructs the transaction manager.
     * 
     * @param aFactory
     *            Factory to create transactions with.
     */
    public SimpleTransactionManager(UserTransactionFactory aFactory) {
        factory = aFactory;

        resources = new ArrayList<TransactionResource>();

        transactionFInishedCallback = new UserTransactionCallback() {
            @Override
            public void transactionFinished() {
                transaction.set(factory.create(this, resources));
            }
        };
        transaction = new ThreadSpecificProxyFactory<UserTransaction>(
            UserTransaction.class, new CreationCallback<UserTransaction>() {
                @Override
                public UserTransaction create() {
                    return factory.create(transactionFInishedCallback,
                        resources);
                }
            });
    }

    /**
     * Adds a resource to manage. Adding resources is no longer allowed after
     * the first transaction has started.
     * 
     * @param aResource
     *            Resource.
     */
    public void addResource(TransactionResource aResource) {
        resources.add(aResource);
    }

    /**
     * Gets the user transaction. This is a contextual reference, meaning that
     * it will delegate to the appropriate thread-specific user transaction. It
     * is also safe to store in a JNDI tree and for caching by applications.
     * 
     * @return User transaction.
     */
    public UserTransaction getTransaction() {
        return transaction.getProxy();
    }

    /**
     * Gets the thread-specific transaction object.
     * 
     * @return Transaction object.
     */
    UserTransaction getThreadSpecificTransaction() {
        getTransaction(); // create tx if needed
        return transaction.get();
    }
}
