/*
 * Copyright 2005-2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wamblee.wicket.inject;

import org.apache.wicket.Component;
import org.apache.wicket.application.IComponentInstantiationListener;
import org.apache.wicket.protocol.http.WebApplication;
import org.wamblee.inject.InjectorBuilder;
import org.wamblee.inject.SimpleInjector;

/**
 * <p>
 * Component instantiation listener that injects references into a component
 * using the injector mini-framework. Together with an implementation of the
 * mini-framework, Contexts and Dependency Injection support can be provided or
 * for standard Java EE injection into components. Or, other injection
 * frameworks can be used.
 * </p>
 * <p/>
 * <p>
 * To use this injector override {@link WebApplication#init()} and add the
 * listener:
 * </p>
 * <p/>
 * <pre>
 * &#064;Override
 * protected void init() {
 *     super.init();
 *     addComponentInstantiationListener(new ComponentInstantiationInjector());
 * }
 * </pre>
 *
 * @author Erik Brakkee
 */
public class ComponentInstantiationInjector implements IComponentInstantiationListener {

    private SimpleInjector injector;

    /**
     * Constructs the component instantiation listener.
     */
    public ComponentInstantiationInjector() {
        injector = new SimpleInjector(InjectorBuilder.getInjectorFactory());
    }

    @Override
    public void onInstantiation(Component aComponent) {
        injector.inject(aComponent);
        aComponent.add(new InjectionBehavior(aComponent));
    }
}
