(function(factory) {

	if (typeof define === "function" && define.amd) {

		define([], factory);
	}
	else if (typeof module === "object" && module.exports) {

		module.exports = factory();
	}
	else {

		var result = factory();

		for (var key in result) {

			this[key] = result[key];
		}
	}
})(function() {

	var context = {};

	context = function() { return context.Datum.apply(this, arguments); };

	var array_method_Pop = (function () {

		function Pop(model, elements, properties) {

			this.model = model;

			this.originalPop = model.pop;

			var self = this;

			model.pop = function() {

				var popped = self.originalPop.apply(this, arguments);

				pop(model, elements, properties);

				return popped;
			};
		}

		function pop(model, elements, properties) {

			var property = properties.pop();

			if (property) {

				property.removeBinding();
			}

			for (var i = 0; i < elements.length; i++) {

				elements[i].removeLast();
			}

			model.subscribableLength = model.length;
			model.indexOf();
		}

		Pop.prototype.unbind = function() {

			this.model.pop = this.originalPop;
		};

		return Pop;
	})();

	var property_TransientProperty = (function () {

		function TransientProperty(property, propertyType) {

			this.property = property;

			this.propertyType = propertyType;

			if (property && this.isBinding(property)) {

				this.binding = property;
			}
			else if (property instanceof Array) {

				this.binding = propertyType.createArrayBinding(property);
			}
			else if (property && typeof(property) === "object") {

				this.binding = propertyType.createViewModel(property);
			}
		}

		TransientProperty.prototype.isBinding = function(object) {

			return object && object.applyBinding && object.removeBinding;
		};

		TransientProperty.prototype.applyBinding = function(element, model, key) {

			if (this.binding) {

				this.binding.applyBinding(element, model, key);
			}
		};

		TransientProperty.prototype.removeBinding = function() {

			if (this.binding) {

				this.binding.removeBinding();
			}
		};

		TransientProperty.prototype.isOlderThan = function(other) {

			if (typeof this.property === "object" || typeof other === "object") {

				return other && this.property !== other;
			}
			else {

				return false;
			}
		};

		return TransientProperty;
	})();

	var array_method_Push = (function (TransientProperty) {

		function Push(model, elements, properties, propertyType) {

			this.model = model;

			this.originalPush = model.push;

			var self = this;

			model.push = function() {

				self.originalPush.apply(this, arguments);

				for (var i = 0; i < arguments.length; i++) {

					var property = new TransientProperty(arguments[i], propertyType);

					push(model, properties, elements, property);
				}

				model.subscribableLength = model.length;
				model.indexOf();
			};
		}

		function push(model, properties, elements, property) {

			properties.push(property);

			for (var j = 0; j < elements.length; j++) {

				var element = elements[j];
				var child = element.append();

				property.applyBinding(child, model);
			}
		}

		Push.prototype.unbind = function() {

			this.model.push = this.originalPush;
		};

		return Push;
	})(property_TransientProperty);

	var array_method_Reverse = (function () {

		function Reverse(model, elements, properties) {

			this.model = model;

			this.originalReverse = model.reverse;

			var self = this;

			model.reverse = function() {

				self.originalReverse.apply(this, arguments);

				reverse(elements, properties);

				model.indexOf();
			};
		}

		function reverse(elements, properties) {

			for (var i = 0; i < elements.length; i++) {

				var element = elements[i];
				var children = element.removeChildren();

				element.appendChildren(children.reverse());
			}

			properties.reverse();
		}

		Reverse.prototype.unbind = function() {

			this.model.reverse = this.originalReverse;
		};

		return Reverse;
	})();

	var array_method_Shift = (function () {

		function Shift(model, elements, properties) {

			this.model = model;

			this.originalShift = model.shift;

			var self = this;

			model.shift = function() {

				var shifted = self.originalShift.apply(this, arguments);

				shift(model, elements, properties);

				return shifted;
			};
		}

		function shift(model, elements, properties) {

			var property = properties.shift();

			if (property) {

				property.removeBinding();
			}

			for (var i = 0; i < elements.length; i++) {

				elements[i].removeFirst();
			}

			model.subscribableLength = model.length;
			model.indexOf();
		}

		Shift.prototype.unbind = function() {

			this.model.shift = this.originalShift;
		};

		return Shift;
	})();

	var array_method_Sort = (function () {

		function Sort(model, elements, properties) {

			this.model = model;

			this.originalSort = model.sort;

			model.sort = function(comparison) {

				if (comparison) {

					sort(model, elements, properties, comparison);
				}
				else {

					sort(model, elements, properties, defaultComparison);
				}

				model.indexOf();
			};
		}

		function defaultComparison(a, b) {

			if (a > b) {

				return 1;
			}
			else if (b > a) {

				return -1;
			}
			else {

				return 0;
			}
		}

		function sort(model, elements, properties, comparison) {

			var modelChildrenProperties =
				getModelChildrenProperties(model, elements, properties);

			modelChildrenProperties.sort(function(a, b) {

				return comparison(a.model, b.model);
			});

			replaceSortedObjects(model, elements, properties, modelChildrenProperties);
		}

		function getModelChildrenProperties(model, elements, properties) {

			var allChildren = getAllChildren(elements);
			var modelChildrenProperties = [];

			for (var i = 0; i < model.length; i++) {

				modelChildrenProperties.push({

					model: model[i],
					property: properties[i],
					children: allChildren[i]
				});
			}

			return modelChildrenProperties;
		}

		function getAllChildren(elements) {

			var allChildren = [];

			for (var i = 0; i < elements.length; i++) {

				var children = elements[i].removeChildren();

				for (var j = 0; j < children.length; j++) {

					if (!allChildren[j]) {

						allChildren[j] = [];
					}

					allChildren[j][i] = children[j];
				}
			}

			return allChildren;
		}

		function replaceSortedObjects(model, elements, properties, modelChildrenProperties) {

			for (var i = 0; i < model.length; i++) {

				model[i] = modelChildrenProperties[i].model;
				properties[i] = modelChildrenProperties[i].property;

				for (var j = 0; j < elements.length; j++) {

					elements[j].appendChild(modelChildrenProperties[i].children[j]);
				}
			}
		}

		Sort.prototype.unbind = function() {

			this.model.sort = this.originalSort;
		};

		return Sort;
	})();

	var array_method_Splice = (function (TransientProperty) {

		function Splice(model, elements, properties, propertyType) {

			this.model = model;

			this.originalSplice = model.splice;

			var self = this;

			model.splice = function(start, deleteCount) {

				start = normaliseStart(model.length, start);
				var newObjects = [].slice.call(arguments, 2);

				removeObjects(model, elements, properties, start, deleteCount);
				insertObjects(model, elements, properties, propertyType, start, newObjects);

				var spliced = self.originalSplice.apply(this, arguments);
				model.subscribableLength = model.length;
				model.indexOf();

				return spliced;
			};
		}

		function normaliseStart(length, start) {

			if (start < 0) {

				start = length + start;
			}

			start = Math.min(length, start);
			start = Math.max(0, start);

			return start;
		}

		function removeObjects(model, elements, properties, start, deleteCount) {

			for (var i = 0; i < elements.length; i++) {

				var element = elements[i];
				var end = Math.min(start + deleteCount, model.length) - 1;

				for (var j = end; j >= start; j--) {

					element.removeAtIndex(j);
				}

				properties.splice(start, deleteCount);
			}
		}

		function insertObjects(model, elements, properties, propertyType, start, newObjects) {

			for (var i = newObjects.length - 1; i >= 0; i--) {

				var property = new TransientProperty(newObjects[i], propertyType);

				properties.splice(start, 0, property);

				for (var j = 0; j < elements.length; j++) {

					var element = elements[j];
					var child = element.insertAtIndex(start);

					property.applyBinding(child, model);
				}
			}
		}

		Splice.prototype.unbind = function() {

			this.model.splice = this.originalSplice;
		};

		return Splice;
	})(property_TransientProperty);

	var array_method_Unshift = (function (TransientProperty) {

		function Unshift(model, elements, properties, propertyType) {

			this.model = model;

			this.originalUnshift = model.unshift;

			var self = this;

			model.unshift = function() {

				self.originalUnshift.apply(this, arguments);

				for (var i = arguments.length - 1; i >= 0; i--) {

					var property = new TransientProperty(arguments[i], propertyType);

					unshift(model, elements, properties, property);
				}

				model.subscribableLength = model.length;
				model.indexOf();
			};
		}

		function unshift(model, elements, properties, property) {

			properties.unshift(property);

			for (var j = 0; j < elements.length; j++) {

				var element = elements[j];
				var child = element.prepend();

				property.applyBinding(child, model);
			}
		}

		Unshift.prototype.unbind = function() {

			this.model.unshift = this.originalUnshift;
		};

		return Unshift;
	})(property_TransientProperty);

	var element_Elements = (function () {

		function Elements() {

			this.elements = [];
		}

		Elements.prototype.add = function(element) {

			this.elements.push(element);
		};

		Elements.prototype.removeOld = function() {

			var removed = [];

			for (var i = this.elements.length - 1; i >= 0; i--) {

				var element = this.elements[i];

				if (element.removedFromDocument()) {

					this.elements.splice(i, 1);
					removed.push(element);
				}
			}

			return removed;
		};

		Elements.prototype.getElementEqualTo = function(element) {

			for (var i = 0; i < this.elements.length; i++) {

				if (this.elements[i].equals(element)) {

					return this.elements[i];
				}
			}
		};

		Elements.prototype.contains = function(element) {

			for (var i = 0; i < this.elements.length; i++) {

				if (this.elements[i].equals(element)) {

					return true;
				}
			}

			return false;
		};

		Elements.prototype.empty = function() {

			var elements = this.elements;

			this.elements = [];

			return elements;
		};

		Elements.prototype.get = function() {

			return this.elements;
		};

		return Elements;
	})();

	var tracking_Registry = (function () {

		var registry = [];
		var registering = false;

		function Registry() {}

		// The registry, the registering flag
		// and the following three methods
		// mediate the process of dependency tracking.
		// When a binding callback is supplied to a binding
		// the binding firstly requests registrations
		// then executes the callback.
		// Any datum evaluated during the callback execution
		// will check that updater assigners are being registered
		// and if so will register an updater assigner in the registry.
		// After the callback has been executed the binding will
		// instruct the registry to assign the updater
		// which it supplies to the registry.
		// The updater assigners will then be called to assign
		// the updater to any datum which supplied an assigner.
		// If in the future the value of a datum which has been
		// assigned the updater changes the datum will call the
		// updater which will update the elements which are bound
		// to the binding.
		Registry.prototype.registerUpdaterAssigner = function(assigner) {

			if (registering) {

				registry.push(assigner);
			}
		};

		Registry.prototype.requestRegistrations = function() {

			registry = [];
			registering = true;
		};

		Registry.prototype.assignUpdater = function(dependant) {

			for (var i = 0; i < registry.length; i++) {

				registry[i](dependant);
			}

			registering = false;
		};

		return Registry;
	})();

	context.Datum = (function (
		Elements,
		Registry) {

		function Datum(datum) {

			var properties = {

				datum: datum,
				dependants: new Elements(),
				registry: new Registry()
			};

			function provider(value) {

				if (isNotValue(value)) {

					return get.call(properties);
				}
				else {

					set.call(properties, value);
				}
			}

			return provider;
		}

		function isNotValue(value) {

			return typeof value === "undefined" || value instanceof Node;
		}

		function get() {

			var self = this;

			this.registry.registerUpdaterAssigner(function(dependant) {

				self.dependants.add(dependant);
			});

			return this.datum;
		}

		function set(value) {

			this.datum = value;

			this.dependants.removeOld();
			updateDependants.call(this, value);
		}

		function updateDependants(value) {

			var dependantArray = this.dependants.get();

			for (var i = 0; i < dependantArray.length; i++) {

				dependantArray[i].call(value);
			}
		}

		return Datum;
	})(element_Elements, tracking_Registry);

	var array_ArrayBinding = (function (
		Elements,
		TransientProperty,
		Datum,
		Push,
		Pop,
		Shift,
		Unshift,
		Reverse,
		Sort,
		Splice) {

		function ArrayBinding(model, propertyType) {

			this.properties = new Array(model.length);

			this.model = model;

			this.propertyType = propertyType;

			this.boundElements = new Elements();

			this.methods = [];

			(function createProperties(self) {

				for (var i = 0; i < model.length; i++) {

					self.properties[i] = new TransientProperty(model[i], propertyType);
				}
			})(this);

			(function createSubscribableLength() {

				if (typeof model.subscribableLength !== "undefined") {

					return;
				}

				var length = new Datum(model.length);

				Object.defineProperty(model, "subscribableLength", {

					get: function() {

						return length();
					},
					set: function(value) {

						length(value);
					}
				});
			})();

			(function createIndexOf() {

				var flag = new Datum();
				var original = model.indexOf;

				model.indexOf = function() {

					if (arguments.length) {

						flag();
					}
					else {

						flag(true);
					}

					return original.apply(model, arguments);
				};
			})();
		}

		ArrayBinding.prototype.applyBinding = function(element, parentModel, name) {

			var removed = this.boundElements.removeOld();
			this.resetElements(removed);

			if (!this.methods.length) {

				this.methods = this.createArrayMethods();
			}

			if (element.get()) {

				this.bindElements(element, parentModel, name);
			}
		};

		ArrayBinding.prototype.createArrayMethods = function() {

			var elements = this.boundElements.get();

			return [
				new Push(this.model, elements, this.properties, this.propertyType),
				new Pop(this.model, elements, this.properties),
				new Shift(this.model, elements, this.properties),
				new Unshift(this.model, elements, this.properties, this.propertyType),
				new Reverse(this.model, elements, this.properties),
				new Sort(this.model, elements, this.properties),
				new Splice(this.model, elements, this.properties, this.propertyType)
			];
		};

		ArrayBinding.prototype.bindElements = function(element, parentModel, name) {

			if (!this.boundElements.contains(element)) {

				this.boundElements.add(element.toArrayElement(this.model.length));
			}

			var arrayElement = this.boundElements.getElementEqualTo(element);
			var value = parentModel[name];

			for (var i = 0; i < this.properties.length; i++) {

				this.properties[i].applyBinding(arrayElement.getChildAtIndex(i), value);
			}
		};

		ArrayBinding.prototype.removeBinding = function() {

			for (var i = 0; i < this.methods.length; i++) {

				this.methods[i].unbind();
			}

			this.methods = [];

			var elements = this.boundElements.empty();

			this.resetElements(elements);
		};

		ArrayBinding.prototype.resetElements = function(elements) {

			for (var i = 0; i < elements.length; i++) {

				elements[i].reset();
			}
		};

		return ArrayBinding;
	})(element_Elements, property_TransientProperty, context.Datum, array_method_Push, array_method_Pop, array_method_Shift, array_method_Unshift, array_method_Reverse, array_method_Sort, array_method_Splice);

	var array_ArrayItemElement = (function () {

		function ArrayItemElement(element) {

			this.element = element;

			this.index = 0;
		}

		ArrayItemElement.prototype.clone = function() {

			var clone = this.element.cloneNode(true);

			this.number(clone, this.index++);

			return clone;
		};

		ArrayItemElement.prototype.number = function(element, index) {

			if (element.id) {

				element.id = element.id + "_" + index;
			}

			if (element.hasAttribute("name")) {

				var name = element.getAttribute("name") + "_" + index;

				element.setAttribute("name", name);
			}

			for (var i = 0; i < element.children.length; i++) {

				this.number(element.children[i], index);
			}
		};

		ArrayItemElement.prototype.get = function() {

			return this.element;
		};

		return ArrayItemElement;
	})();

	var array_ArrayElement = (function (ArrayItemElement) {

		function ArrayElement(domElement, initialLength) {

			this.domElement = domElement;

			this.element = domElement.get();

			// There is a performance advantage
			// to keeping the children in an array
			// instead of retrieving children from
			// the element's children property
			// which is not an array
			this.children = new Array(initialLength);

			(function checkElementHasOnlyOneChild(self) {

				if (self.element.children.length !== 1) {

					var message =
						"An array must be bound to an element with exactly one child.";
					throw new Error(message);
				}
			})(this);

			(function createRebinder() {

				domElement.createRebinder(function() {});
			})();

			(function getChild(self) {

				var childElement = self.element.children[0];

				self.element.removeChild(childElement);

				self.child = new ArrayItemElement(childElement);
			})(this);

			(function copyElement(self) {

				for (var i = 0; i < initialLength; i++) {

					var child = self.child.clone();

					self.children[i] = child;

					self.element.appendChild(child);
				}
			})(this);
		}

		ArrayElement.prototype.append = function() {

			var child = this.child.clone();

			this.children.push(child);

			this.element.appendChild(child);

			return this.domElement.createElement(child);
		};

		ArrayElement.prototype.prepend = function() {

			var child = this.child.clone();

			this.children.unshift(child);

			this.element.insertBefore(child, this.element.firstChild);

			return this.domElement.createElement(child);
		};

		ArrayElement.prototype.insertAtIndex = function(index) {

			var child = this.child.clone();

			this.element.insertBefore(child, this.children[index]);

			this.children.splice(index, 0, child);

			return this.domElement.createElement(child);
		};

		ArrayElement.prototype.removeFirst = function() {

			if (this.element.firstElementChild) {

				this.children.shift();

				this.element.removeChild(this.element.firstElementChild);
			}
		};

		ArrayElement.prototype.removeLast = function() {

			if (this.element.lastElementChild) {

				this.children.pop();

				this.element.removeChild(this.element.lastElementChild);
			}
		};

		ArrayElement.prototype.removeAtIndex = function(index) {

			var removed = this.children.splice(index, 1);

			this.element.removeChild(removed[0]);
		};

		ArrayElement.prototype.removeChildren = function() {

			var children = this.children;

			this.children = [];

			while (this.element.lastChild) {

				this.element.removeChild(this.element.lastChild);
			}

			return children;
		};

		ArrayElement.prototype.appendChildren = function(children) {

			for (var i = 0; i < children.length; i++) {

				var child = children[i];

				this.children.push(child);

				this.element.appendChild(child);
			}
		};

		ArrayElement.prototype.appendChild = function(child) {

			this.children.push(child);

			this.element.appendChild(child);
		};

		ArrayElement.prototype.reset = function() {

			this.children = [];

			while (this.element.lastChild) {

				this.element.removeChild(this.element.lastChild);
			}

			this.element.appendChild(this.child.get());
		};

		ArrayElement.prototype.getChildAtIndex = function(i) {

			return this.domElement.createElement(this.children[i]);
		};

		ArrayElement.prototype.removedFromDocument = function() {

			return this.domElement.removedFromDocument();
		};

		ArrayElement.prototype.equals = function(other) {

			return other.hasEqual(this.element);
		};

		ArrayElement.prototype.hasEqual = function(otherElement) {

			return this.element === otherElement;
		};

		ArrayElement.prototype.get = function() {

			return this.domElement;
		};

		return ArrayElement;
	})(array_ArrayItemElement);

	var tracking_Dependant = (function () {

		function Dependant(callback, binding, element) {

			this.callback = callback;

			this.binding = binding;

			this.element = element;
		}

		Dependant.prototype.call = function(value) {

			this.callback(value);
		};

		Dependant.prototype.removedFromDocument = function() {

			return this.element.removedFromDocument();
		};

		Dependant.prototype.equals = function(other) {

			return other.hasEqual(this.binding, this.element);
		};

		Dependant.prototype.hasEqual = function(otherBinding, otherElement) {

			return this.binding === otherBinding && this.element.equals(otherElement);
		};

		return Dependant;
	})();

	var callback_CallbackBinder = (function (
		Elements,
		Dependant,
		Registry) {

		function CallbackBinder(binding) {

			var boundElements = new Elements();

			function provider() {

				return binding.test(this);
			}

			provider.applyBinding = function(element, model) {

				removeOldBindings(binding, boundElements);
				bindElements(element, model, binding, boundElements);
			};

			provider.removeBinding = function() {

				remove(binding, boundElements);
			};

			return provider;
		}

		function removeOldBindings(binding, boundElements) {

			var removed = boundElements.removeOld();

			for (var i = 0; i < removed.length; i++) {

				var element = removed[i].get();

				if (element) {

					binding.resetElement(element);
				}
			}
		}

		function bindElements(element, model, binding, boundElements) {

			if (element.get() && !boundElements.contains(element)) {

				binding.setUpElement(model, element.get());
				new Registry().requestRegistrations();
				binding.updateElement(model, element.get());
				createCallback(model, element, binding);
				boundElements.add(element);
			}
		}

		function createCallback(model, element, binding) {

			var running = false;

			function callback(value) {

				if (!running) {

					running = true;
					binding.updateElement(model, element.get(), value);
					running = false;
				}
			}

			new Registry().assignUpdater(new Dependant(callback, binding, element));
		}

		function remove(binding, boundElements) {

			var elements = boundElements.empty();

			for (var i = 0; i < elements.length; i++) {

				var element = elements[i].get();

				if (element) {

					binding.resetElement(element);
				}
			}
		}

		return CallbackBinder;
	})(element_Elements, tracking_Dependant, tracking_Registry);

	var callback_binding_ClassesBinding = (function () {

		function ClassesBinding(callbacks) {

			this.callbacks = callbacks;
		}

		ClassesBinding.prototype.setUpElement = function() {};

		ClassesBinding.prototype.updateElement = function(parentModel, element) {

			var classes = this.splitClasses(element);

			for (var key in this.callbacks) {

				var index = classes.indexOf(key);

				if (this.callbacks[key].call(parentModel, element)) {

					if (index < 0) {

						classes.push(key);
					}
				}
				else if (index + 1) {

					classes.splice(index, 1);
				}
			}

			element.className = classes.join(" ");
		};

		ClassesBinding.prototype.splitClasses = function(element) {

			var classes;

			if (element.className) {

				classes = element.className.split(" ");
			}
			else {

				classes = [];
			}

			return classes;
		};

		ClassesBinding.prototype.resetElement = function() {};

		ClassesBinding.prototype.test = function(parentModel) {

			var classes = {

				classes: {}
			};

			for (var key in this.callbacks) {

				classes.classes[key] = this.callbacks[key].bind(parentModel);
			}

			return classes;
		};

		return ClassesBinding;
	})();

	var callback_binding_DestroyBinding = (function () {

		function DestroyBinding(destroy) {

			this.destroy = destroy;
		}

		DestroyBinding.prototype.setUpElement = function(model) {

			this.parentModel = model;
		};

		DestroyBinding.prototype.updateElement = function() {};

		DestroyBinding.prototype.resetElement = function(element) {

			this.destroy.call(this.parentModel, element);
		};

		DestroyBinding.prototype.test = function(parentModel) {

			return {

				destroy: this.destroy.bind(parentModel)
			};
		};

		return DestroyBinding;
	})();

	var callback_binding_EventsBinding = (function () {

		function EventsBinding(callbacks) {

			this.callbacks = callbacks;

			this.listeners = [];
		}

		EventsBinding.prototype.setUpElement = function(model, element) {

			for (var key in this.callbacks) {

				var callback = this.callbacks[key];

				var listener = this.createListener(model, element, callback);

				this.listeners.push({ key: key, element: element, listener: listener });
				element.addEventListener(key, listener);
			}
		};

		EventsBinding.prototype.createListener = function(model, element, callback) {

			return function(event) {

				callback.call(model, event, element);
			};
		};

		EventsBinding.prototype.updateElement = function() {};

		EventsBinding.prototype.resetElement = function(element) {

			for (var i = 0; i < this.listeners.length; i++) {

				var listener = this.listeners[i];

				if (listener.element === element) {

					element.removeEventListener(listener.key, listener.listener);
					break;
				}
			}

			this.listeners.splice(i, 1);
		};

		EventsBinding.prototype.test = function(parentModel) {

			var events = {

				events: {}
			};

			for (var key in this.callbacks) {

				events.events[key] = this.callbacks[key].bind(parentModel);
			}

			if (this.callbacks.click) {

				events.click = this.callbacks.click.bind(parentModel);
			}

			return events;
		};

		return EventsBinding;
	})();

	var callback_binding_InitBinding = (function () {

		function InitBinding(init) {

			this.init = init;
		}

		InitBinding.prototype.setUpElement = function(model, element) {

			this.init.call(model, element);
		};

		InitBinding.prototype.updateElement = function() {};

		InitBinding.prototype.resetElement = function() {};

		InitBinding.prototype.test = function(parentModel) {

			return {

				init: this.init.bind(parentModel)
			};
		};

		return InitBinding;
	})();

	var callback_binding_TextBinding = (function () {

		function TextBinding(text) {

			this.text = text;
		}

		TextBinding.prototype.setUpElement = function() {};

		TextBinding.prototype.updateElement = function(parentModel, element) {

			element.textContent = this.text.call(parentModel, element);
		};

		TextBinding.prototype.resetElement = function(element) {

			element.textContent = "";
		};

		TextBinding.prototype.test = function(parentModel) {

			return {

				text: this.text.bind(parentModel)
			};
		};

		return TextBinding;
	})();

	var callback_binding_UpdateBinding = (function () {

		function UpdateBinding(update) {

			this.update = update;
		}

		UpdateBinding.prototype.setUpElement = function() {};

		UpdateBinding.prototype.updateElement = function(parentModel, element) {

			this.update.call(parentModel, element);
		};

		UpdateBinding.prototype.resetElement = function() {};

		UpdateBinding.prototype.test = function(parentModel) {

			return {

				update: this.update.bind(parentModel)
			};
		};

		return UpdateBinding;
	})();

	var callback_binding_ValueBinding = (function () {

		function ValueBinding(value) {

			this.value = value;

			this.listeners = [];
		}

		ValueBinding.prototype.setUpElement = function(parentModel, element) {

			var self = this;

			function listener(event) {

				self.value.call(parentModel, event.target.value, element);
			}

			this.listeners.push({ element: element, listener: listener });
			element.addEventListener("change", listener);
		};

		ValueBinding.prototype.updateElement = function(parentModel, element) {

			element.value = this.value.call(parentModel, undefined, element);
		};

		ValueBinding.prototype.resetElement = function(element) {

			for (var i = 0; i < this.listeners.length; i++) {

				var listener = this.listeners[i];

				if (listener.element === element) {

					element.removeEventListener("change", listener.listener);
					element.value = "";
					break;
				}
			}

			this.listeners.splice(i, 1);
		};

		ValueBinding.prototype.test = function(parentModel) {

			return {

				value: this.value.bind(parentModel)
			};
		};

		return ValueBinding;
	})();

	var callback_binding_VisibleBinding = (function () {

		function VisibleBinding(visible) {

			this.visible = visible;
		}

		VisibleBinding.prototype.setUpElement = function() {};

		VisibleBinding.prototype.updateElement = function(parentModel, element) {

			if (this.visible.call(parentModel, element)) {

				element.style.display = null;
			}
			else {

				element.style.display = "none";
			}
		};

		VisibleBinding.prototype.resetElement = function(element) {

			element.style.display = null;
		};

		VisibleBinding.prototype.test = function(parentModel) {

			return {

				visible: this.visible.bind(parentModel)
			};
		};

		return VisibleBinding;
	})();

	context.Binding = (function (
		CallbackBinder,
		TextBinding,
		ValueBinding,
		EventsBinding,
		InitBinding,
		UpdateBinding,
		VisibleBinding,
		DestroyBinding,
		ClassesBinding) {

		function Binding(callbacks) {

			var bindings = [];

			if (callbacks.text) {

				bindings.push(new CallbackBinder(new TextBinding(callbacks.text)));
			}

			if (callbacks.value) {

				bindings.push(new CallbackBinder(new ValueBinding(callbacks.value)));
			}

			if (callbacks.click) {

				bindings.push(
					new CallbackBinder(new EventsBinding({ click: callbacks.click })));
			}

			if (callbacks.events) {

				bindings.push(new CallbackBinder(new EventsBinding(callbacks.events)));
			}

			if (callbacks.init) {

				bindings.push(new CallbackBinder(new InitBinding(callbacks.init)));
			}

			if (callbacks.update) {

				bindings.push(new CallbackBinder(new UpdateBinding(callbacks.update)));
			}

			if (callbacks.destroy) {

				bindings.push(new CallbackBinder(new DestroyBinding(callbacks.destroy)));
			}

			if (callbacks.visible) {

				bindings.push(new CallbackBinder(new VisibleBinding(callbacks.visible)));
			}

			if (callbacks.classes) {

				bindings.push(new CallbackBinder(new ClassesBinding(callbacks.classes)));
			}

			function provider() {

				return mergeCallbacks(bindings, this);
			}

			provider.applyBinding = function(element, model) {

				for (var i = 0; i < bindings.length; i++) {

					bindings[i].applyBinding(element, model);
				}
			};

			provider.removeBinding = function() {

				for (var i = 0; i < bindings.length; i++) {

					bindings[i].removeBinding();
				}
			};

			return provider;
		}

		function mergeCallbacks(bindings, parentModel) {

			var handle = {};

			for (var i = 0; i < bindings.length; i++) {

				var callbacks = bindings[i].call(parentModel);

				for (var key in callbacks) {

					handle[key] = callbacks[key];
				}
			}

			return handle;
		}

		return Binding;
	})(callback_CallbackBinder, callback_binding_TextBinding, callback_binding_ValueBinding, callback_binding_EventsBinding, callback_binding_InitBinding, callback_binding_UpdateBinding, callback_binding_VisibleBinding, callback_binding_DestroyBinding, callback_binding_ClassesBinding);

	context.Classes = (function (
		ClassesBinding,
		CallbackBinder) {

		function Classes(callbacks) {

			return new CallbackBinder(new ClassesBinding(callbacks));
		}

		return Classes;
	})(callback_binding_ClassesBinding, callback_CallbackBinder);

	context.Click = (function (
		EventsBinding,
		CallbackBinder) {

		function Click(click) {

			return new CallbackBinder(new EventsBinding({ click: click }));
		}

		return Click;
	})(callback_binding_EventsBinding, callback_CallbackBinder);

	context.Destroy = (function (
		DestroyBinding,
		CallbackBinder) {

		function Destroy(destroy) {

			return new CallbackBinder(new DestroyBinding(destroy));
		}

		return Destroy;
	})(callback_binding_DestroyBinding, callback_CallbackBinder);

	context.Events = (function (
		EventsBinding,
		CallbackBinder) {

		function Events(callbacks) {

			return new CallbackBinder(new EventsBinding(callbacks));
		}

		return Events;
	})(callback_binding_EventsBinding, callback_CallbackBinder);

	context.Init = (function (
		InitBinding,
		CallbackBinder) {

		function Init(init) {

			return new CallbackBinder(new InitBinding(init));
		}

		return Init;
	})(callback_binding_InitBinding, callback_CallbackBinder);

	context.Text = (function (
		TextBinding,
		CallbackBinder) {

		function Text(text) {

			return new CallbackBinder(new TextBinding(text));
		}

		return Text;
	})(callback_binding_TextBinding, callback_CallbackBinder);

	context.Update = (function (
		UpdateBinding,
		CallbackBinder) {

		function Update(update) {

			return new CallbackBinder(new UpdateBinding(update));
		}

		return Update;
	})(callback_binding_UpdateBinding, callback_CallbackBinder);

	context.Value = (function (
		ValueBinding,
		CallbackBinder) {

		function Value(value) {

			return new CallbackBinder(new ValueBinding(value));
		}

		return Value;
	})(callback_binding_ValueBinding, callback_CallbackBinder);

	context.Visible = (function (
		VisibleBinding,
		CallbackBinder) {

		function Visible(visible) {

			return new CallbackBinder(new VisibleBinding(visible));
		}

		return Visible;
	})(callback_binding_VisibleBinding, callback_CallbackBinder);

	var object_ObjectElement = (function () {

		function ObjectElement(element) {

			this.element = element;

			this.children = [].slice.call(element.childNodes);
		}

		ObjectElement.prototype.removeChildren = function() {

			var children = this.element.childNodes;

			for (var i = children.length - 1; i >= 0; i--) {

				this.element.removeChild(children[i]);
			}
		};

		ObjectElement.prototype.replaceChildren = function() {

			for (var i = 0; i < this.children.length; i++) {

				this.element.appendChild(this.children[i]);
			}
		};

		ObjectElement.prototype.removedFromDocument = function() {

			return !document.contains(this.element);
		};

		ObjectElement.prototype.equals = function(other) {

			return other.hasEqual(this.element);
		};

		ObjectElement.prototype.hasEqual = function(otherElement) {

			return this.element === otherElement;
		};

		return ObjectElement;
	})();

	var element_DOMElement = (function (
		ObjectElement,
		ArrayElement) {

		function DOMElement(element) {

			this.element = element;
		}

		DOMElement.prototype.isInScope = function(scope) {

			return isInScope(this.element, scope);
		};

		function isInScope(element, scope) {

			var currentElement = element.parentElement;

			if (!currentElement) {

				return false;
			}
			else if (currentElement === scope) {

				return true;
			}
			else if (currentElement.__DATUM__REBIND) {

				return false;
			}
			else {

				return isInScope(currentElement, scope);
			}
		}

		DOMElement.prototype.removedFromDocument = function() {

			return !document.body.contains(this.element);
		};

		DOMElement.prototype.getMatchingElements = function(key) {

			var elements = this.element.querySelectorAll("[data-bind=" + key + "]");
			var elementsArray = [];

			for (var i = 0; i < elements.length; i++) {

				var newElement = new DOMElement(elements[i]);

				if (newElement.isInScope(this.element)) {

					elementsArray.push(newElement);
				}
			}

			if (this.element.dataset.bind === key) {

				elementsArray.push(new DOMElement(this.element));
			}

			return elementsArray;
		};

		DOMElement.prototype.createRebinder = function(rebinder) {

			this.element.__DATUM__REBIND = rebinder;
		};

		DOMElement.prototype.rebind = function() {

			this.element.__DATUM__REBIND();
		};

		DOMElement.prototype.callBindingCallback = function(model) {

			if (model.onBind) {

				model.onBind(this.element);
			}
		};

		DOMElement.prototype.equals = function(other) {

			return other.hasEqual(this.element);
		};

		DOMElement.prototype.hasEqual = function(otherElement) {

			return this.element === otherElement;
		};

		DOMElement.prototype.toObjectElement = function() {

			return new ObjectElement(this.element);
		};

		DOMElement.prototype.toArrayElement = function(initialLength) {

			return new ArrayElement(this, initialLength);
		};

		DOMElement.prototype.get = function() {

			return this.element;
		};

		DOMElement.prototype.createElement = function(element) {

			return new DOMElement(element);
		};

		return DOMElement;
	})(object_ObjectElement, array_ArrayElement);

	var element_NullDOMElement = (function () {

		function NullDOMElement() {}

		NullDOMElement.prototype.isInScope = function() {

			return true;
		};

		NullDOMElement.prototype.removedFromDocument = function() {

			return true;
		};

		NullDOMElement.prototype.getMatchingElements = function() {

			return [];
		};

		NullDOMElement.prototype.createRebinder = function() {};

		NullDOMElement.prototype.rebind = function() {};

		NullDOMElement.prototype.callBindingCallback = function() {};

		NullDOMElement.prototype.equals = function(other) {

			return other instanceof NullDOMElement;
		};

		NullDOMElement.prototype.hasEqual = function() {

			return false;
		};

		NullDOMElement.prototype.get = function() {

			return null;
		};

		return NullDOMElement;
	})();

	var object_ObjectBinding = (function (
		Elements,
		Registry,
		Dependant) {

		function ObjectBinding(scope) {

			this.scope = scope;

			this.removed = false;

			this.boundElements = new Elements();
		}

		ObjectBinding.prototype.applyBinding = function(element, model, name) {

			var removed = this.boundElements.removeOld();

			for (var i = 0; i < removed.length; i++) {

				removed[i].replaceChildren();
			}

			if (element.get()) {

				this.bindElements(element, model, name);
			}
		};

		ObjectBinding.prototype.bindElements = function(element, model, name) {

			if (this.boundElements.contains(element)) {

				this.updateElement(element, model && model[name]);
			}
			else {

				this.boundElements.add(element.toObjectElement());
				new Registry().requestRegistrations();
				this.updateElement(element, model && model[name]);
				this.createCallback(this.scope, element);
			}
		};

		ObjectBinding.prototype.updateElement = function(element, model) {

			var objectElement = this.boundElements.getElementEqualTo(element);

			if (model) {

				if (this.removed) {

					this.removed = false;
					objectElement.replaceChildren();
				}
			}
			else {

				this.removed = true;
				objectElement.removeChildren();
			}
		};

		ObjectBinding.prototype.createCallback = function(scope, element) {

			var running = false;

			function callback() {

				if (!running) {

					running = true;
					scope.rebind();
					running = false;
				}
			}

			new Registry().assignUpdater(new Dependant(callback, this, element));
		};

		ObjectBinding.prototype.removeBinding = function() {

			var elements = this.boundElements.empty();

			for (var i = 0; i < elements.length; i++) {

				elements[i].replaceChildren();
			}
		};

		return ObjectBinding;
	})(element_Elements, tracking_Registry, tracking_Dependant);

	var object_Serialisable = (function () {

		function Serialisable(model) {

			if (!model.toJSON) {

				model.toJSON = function() { return toJSON(model); };
			}
		}

		function toJSON(model) {

			var json = {};

			if (model instanceof Array) {

				json = [];
			}

			for (var key in model) {

				var property = model[key];

				if (property &&
					property.toJSON &&
					typeof(property) === "object" &&
					(!property.applyBinding || property instanceof Array)) {

					json[key] = property.toJSON();
				}

				if (!property ||
					(typeof(property) !== "object" &&
						typeof(property) !== "function")) {

					json[key] = property;
				}
			}

			return json;
		}

		return Serialisable;
	})();

	var property_PermanentProperty = (function () {

		function PermanentProperty(property, propertyType, scope) {

			this.scope = scope;

			this.property = property;

			this.propertyType = propertyType;

			this.propertyInjected = false;

			if (typeof(property) === "object" && !this.isBinding(property)) {

				this.objectBinding = propertyType.createObjectBinding(scope);
			}
		}

		PermanentProperty.prototype.applyBinding = function(element, model, key) {

			if (typeof(this.property) !== "function" &&
				!this.isBinding(this.property) &&
				!this.propertyInjected) {

				this.propertyType.injectProperty(this.property, model, key);
				this.propertyInjected = true;
			}

			if (this.objectBinding) {

				this.objectBinding.applyBinding(element, model, key);
			}
		};

		PermanentProperty.prototype.isBinding = function(object) {

			return object && object.applyBinding && object.removeBinding;
		};

		PermanentProperty.prototype.removeBinding = function() {

			if (this.objectBinding) {

				this.objectBinding.removeBinding();
			}
		};

		PermanentProperty.prototype.hasScope = function(element) {

			return this.scope.equals(element);
		};

		return PermanentProperty;
	})();

	var property_PropertyType = (function (
		Datum,
		ObjectBinding,
		ArrayBinding) {

		function PropertyType(createViewModel) {

			this.createViewModelCallback = createViewModel;
		}

		PropertyType.prototype.injectProperty = function(property, model, key) {

			var datum = new Datum(property);

			Object.defineProperty(model, key, {

				get: function() {

					return datum();
				},
				set: function(value) {

					datum(value);
				}
			});
		};

		PropertyType.prototype.createViewModel = function(model) {

			return this.createViewModelCallback(model);
		};

		PropertyType.prototype.createObjectBinding = function(scope) {

			return new ObjectBinding(scope);
		};

		PropertyType.prototype.createArrayBinding = function(model) {

			return new ArrayBinding(model, this);
		};

		return PropertyType;
	})(context.Datum, object_ObjectBinding, array_ArrayBinding);

	var object_ViewModel = (function (
		Elements,
		Serialisable,
		TransientProperty,
		PermanentProperty,
		PropertyType,
		NullDOMElement) {

		function ViewModel(model) {

			this.model = model;

			this.boundElements = new Elements();

			this.transientProperties = {};

			this.permanentProperties = {};

			new Serialisable(model);
		}

		ViewModel.prototype.applyBinding = function(element) {

			this.boundElements.removeOld();
			this.unbindOldProperties();

			if (!this.boundElements.contains(element)) {

				this.boundElements.add(element);
				this.createRebinder(element);
				element.callBindingCallback(this.model);
			}

			this.createPermanentProperties(element);
			this.createTransientProperties();
			this.bindProperties(element);
		};

		ViewModel.prototype.createRebinder = function(element) {

			var self = this;

			element.createRebinder(function() {

				ViewModel.prototype.applyBinding.call(self, element);
			});
		};

		ViewModel.prototype.unbindOldProperties = function() {

			for (var key in this.transientProperties) {

				if (!this.model[key]) {

					this.transientProperties[key].removeBinding();
					delete this.transientProperties[key];
				}
			}
		};

		ViewModel.prototype.createPermanentProperties = function(element) {

			for (var key in this.model) {

				if (!this.permanentProperties[key] ||
					!this.permanentProperties[key].hasScope(element)) {

					this.permanentProperties[key] =
						new PermanentProperty(
							this.model[key],
							this.createPropertyType(),
							element);
				}
			}
		};

		ViewModel.prototype.createTransientProperties = function() {

			for (var key in this.model) {

				if (this.isNew(key)) {

					if (this.transientProperties[key]) {

						this.transientProperties[key].removeBinding();
					}

					this.transientProperties[key] =
						new TransientProperty(this.model[key], this.createPropertyType());
				}
			}
		};

		ViewModel.prototype.isNew = function(key) {

			var property = this.transientProperties[key];

			return !property || property.isOlderThan(this.model[key]);
		};

		ViewModel.prototype.createPropertyType = function() {

			return new PropertyType(function(model) { return new ViewModel(model); });
		};

		ViewModel.prototype.bindProperties = function(scope) {

			for (var key in this.permanentProperties) {

				var elements = this.getElements(scope, key);

				for (var i = 0; i < elements.length; i++) {

					var element = elements[i];

					this.permanentProperties[key].applyBinding(element, this.model, key);
					this.transientProperties[key].applyBinding(element, this.model, key);
				}
			}
		};

		ViewModel.prototype.getElements = function(scope, name) {

			var elements = scope.getMatchingElements(name);

			return elements.length ? elements : [new NullDOMElement()];
		};

		ViewModel.prototype.removeBinding = function() {

			for (var key in this.permanentProperties) {

				this.permanentProperties[key].removeBinding();
				this.transientProperties[key].removeBinding();
			}
		};

		return ViewModel;
	})(element_Elements, object_Serialisable, property_TransientProperty, property_PermanentProperty, property_PropertyType, element_NullDOMElement);

	var root_DOMWatcher = (function () {

		function DOMWatcher(scope) {

			var self = this;

			this.observer = new MutationObserver(function() {

				return callback.apply(self, arguments);
			});

			this.observer.observe(scope, { childList: true, subtree: true });
		}

		function callback(mutations) {

			var mutation = mutations[0];
			var notTextMutation = mutation.target.children.length;

			if (notTextMutation) {

				var element = mutation.target;

				this.rebindElement(element);
			}
		}

		DOMWatcher.prototype.rebindElement = function(element) {

			if (element && element.__DATUM__REBIND) {

				element.__DATUM__REBIND();
			}
			else if (element) {

				this.rebindElement(element.parentElement);
			}
		};

		DOMWatcher.prototype.disconnect = function() {

			this.observer.disconnect();
		};

		return DOMWatcher;
	})();

	context.BindingRoot = (function (
		ViewModel,
		DOMWatcher,
		DOMElement) {

		var flag = false;

		function BindingRoot(model) {

			(function checkModelType() {

				if (typeof model !== "object") {

					throw new Error("The binding root must be an object.");
				}

				if (model instanceof Array) {

					throw new Error("The binding root cannot be an array.");
				}
			})();

			(function assertUniqueness() {

				if (flag) {

					throw new Error(
						"The binding root is unique and " +
						"cannot be instantiated multiple times.");
				}
				else {

					flag = true;
				}
			})();

			var rootViewModel = new ViewModel(model);
			rootViewModel.applyBinding(new DOMElement(document.body));

			var domWatcher = new DOMWatcher(document.body);

			function disconnect() {

				rootViewModel.removeBinding();
				domWatcher.disconnect();
				flag = false;
			}

			return {

				disconnect: disconnect
			};
		}

		return BindingRoot;
	})(object_ViewModel, root_DOMWatcher, element_DOMElement);

	return context;
});
