/**
 * @license Copyright 2013 Andy Earnshaw, MIT License
 *
 * Implements the ECMAScript Internationalization API in ES5-compatible environments,
 * following the ECMA-402 specification as closely as possible
 *
 * ECMA-402: http://ecma-international.org/ecma-402/1.0/
 *
 * CLDR format locale data should be provided using IntlPolyfill.__addLocaleData().
 */
/*jshint proto:true, eqnull:true, boss:true, laxbreak:true, newcap:false, shadow:true, funcscope:true */
/*globals global, define, exports, module, window*/

(function (global, factory) {
    var IntlPolyfill = factory();

    // register in -all- the module systems (at once)
    if (typeof define === 'function' && define.amd)
        define(IntlPolyfill);

    if (typeof exports === 'object')
        module.exports = IntlPolyfill;

    if (!global.Intl) {
        global.Intl = IntlPolyfill;
        IntlPolyfill.__applyLocaleSensitivePrototypes();
    }

    global.IntlPolyfill = IntlPolyfill;

})(typeof global !== 'undefined' ? global : this, function() {
"use strict";
var
    Intl = {},

    realDefineProp = (function () {
        try { return !!Object.defineProperty({}, 'a', {}); }
        catch (e) { return false; }
    })(),

    // Need a workaround for getters in ES3
    es3  = !realDefineProp && !Object.prototype.__defineGetter__,

    // We use this a lot (and need it for proto-less objects)
    hop = Object.prototype.hasOwnProperty,

    // Naive defineProperty for compatibility
    defineProperty = realDefineProp ? Object.defineProperty : function (obj, name, desc) {
        if ('get' in desc && obj.__defineGetter__)
            obj.__defineGetter__(name, desc.get);

        else if (!hop.call(obj, name) || 'value' in desc)
            obj[name] = desc.value;
    },

    // Array.prototype.indexOf, as good as we need it to be
    arrIndexOf = Array.prototype.indexOf || function (search) {
        /*jshint validthis:true */
        var t = this;
        if (!t.length)
            return -1;

        for (var i = arguments[1] || 0, max = t.length; i < max; i++) {
            if (t[i] === search)
                return i;
        }

        return -1;
    },

    // Create an object with the specified prototype (2nd arg required for Record)
    objCreate = Object.create || function (proto, props) {
        var obj;

        function F() {}
        F.prototype = proto;
        obj = new F();

        for (var k in props) {
            if (hop.call(props, k))
                defineProperty(obj, k, props[k]);
        }

        return obj;
    },

    // Snapshot some (hopefully still) native built-ins
    arrSlice  = Array.prototype.slice,
    arrConcat = Array.prototype.concat,
    arrPush   = Array.prototype.push,
    arrJoin   = Array.prototype.join,
    arrShift  = Array.prototype.shift,
    arrUnshift= Array.prototype.unshift,

    // Naive Function.prototype.bind for compatibility
    fnBind = Function.prototype.bind || function (thisObj) {
        var fn = this,
            args = arrSlice.call(arguments, 1);

        // All our (presently) bound functions have either 1 or 0 arguments. By returning
        // different function signatures, we can pass some tests in ES3 environments
        if (fn.length === 1) {
            return function (a) {
                return fn.apply(thisObj, arrConcat.call(args, arrSlice.call(arguments)));
            };
        }
        else {
            return function () {
                return fn.apply(thisObj, arrConcat.call(args, arrSlice.call(arguments)));
            };
        }
    },

    // Default locale is the first-added locale data for us
    defaultLocale,

    // Object housing internal properties for constructors
    internals = objCreate(null),

    // Keep internal properties internal
    secret = Math.random(),

    // An object map of date component keys, saves using a regex later
    dateWidths = objCreate(null, { narrow:{}, short:{}, long:{} }),

    // Each constructor prototype should be an instance of the constructor itself, but we
    // can't initialise them as such until some locale data has been added, so this is how
    // we keep track
    numberFormatProtoInitialised = false,
    dateTimeFormatProtoInitialised = false,

    // Some regular expressions we're using
    expCurrencyCode = /^[A-Z]{3}$/,
    expUnicodeExSeq = /-u(?:-[0-9a-z]{2,8})+/gi, // See `extension` below

    expBCP47Syntax,
    expExtSequences,
    expVariantDupes,
    expSingletonDupes,

    // IANA Subtag Registry redundant tag and subtag maps
    redundantTags = {
        tags: {
            "art-lojban":   "jbo",       "i-ami":        "ami",       "i-bnn":       "bnn",  "i-hak":      "hak",
            "i-klingon":    "tlh",       "i-lux":        "lb",        "i-navajo":    "nv",   "i-pwn":      "pwn",
            "i-tao":        "tao",       "i-tay":        "tay",       "i-tsu":       "tsu",  "no-bok":     "nb",
            "no-nyn":       "nn",        "sgn-BE-FR":    "sfb",       "sgn-BE-NL":   "vgt",  "sgn-CH-DE":  "sgg",
            "zh-guoyu":     "cmn",       "zh-hakka":     "hak",       "zh-min-nan":  "nan",  "zh-xiang":   "hsn",
            "sgn-BR":       "bzs",       "sgn-CO":       "csn",       "sgn-DE":      "gsg",  "sgn-DK":     "dsl",
            "sgn-ES":       "ssp",       "sgn-FR":       "fsl",       "sgn-GB":      "bfi",  "sgn-GR":     "gss",
            "sgn-IE":       "isg",       "sgn-IT":       "ise",       "sgn-JP":      "jsl",  "sgn-MX":     "mfs",
            "sgn-NI":       "ncs",       "sgn-NL":       "dse",       "sgn-NO":      "nsl",  "sgn-PT":     "psr",
            "sgn-SE":       "swl",       "sgn-US":       "ase",       "sgn-ZA":      "sfs",  "zh-cmn":     "cmn",
            "zh-cmn-Hans":  "cmn-Hans",  "zh-cmn-Hant":  "cmn-Hant",  "zh-gan":      "gan",  "zh-wuu":     "wuu",
            "zh-yue":       "yue"
        },
        subtags: {
              BU: "MM",   DD: "DE",   FX: "FR",   TP: "TL",   YD: "YE",   ZR: "CD",  heploc: "alalc97",
            'in': "id",   iw: "he",   ji:  "yi",  jw: "jv",   mo: "ro",  ayx: "nun", bjd: "drl",
             ccq: "rki", cjr: "mom", cka: "cmr", cmk: "xch", drh: "khk", drw: "prs", gav: "dev",
             hrr: "jal", ibi: "opa", kgh: "kml", lcq: "ppr", mst: "mry", myt: "mry", sca: "hle",
             tie: "ras", tkk: "twm", tlw: "weo", tnf: "prs", ybd: "rki", yma: "lrr"
        },
        extLang: {
            aao: [ "aao", "ar"  ], abh: [ "abh", "ar"  ], abv: [ "abv", "ar"  ], acm: [ "acm", "ar"  ],
            acq: [ "acq", "ar"  ], acw: [ "acw", "ar"  ], acx: [ "acx", "ar"  ], acy: [ "acy", "ar"  ],
            adf: [ "adf", "ar"  ], ads: [ "ads", "sgn" ], aeb: [ "aeb", "ar"  ], aec: [ "aec", "ar"  ],
            aed: [ "aed", "sgn" ], aen: [ "aen", "sgn" ], afb: [ "afb", "ar"  ], afg: [ "afg", "sgn" ],
            ajp: [ "ajp", "ar"  ], apc: [ "apc", "ar"  ], apd: [ "apd", "ar"  ], arb: [ "arb", "ar"  ],
            arq: [ "arq", "ar"  ], ars: [ "ars", "ar"  ], ary: [ "ary", "ar"  ], arz: [ "arz", "ar"  ],
            ase: [ "ase", "sgn" ], asf: [ "asf", "sgn" ], asp: [ "asp", "sgn" ], asq: [ "asq", "sgn" ],
            asw: [ "asw", "sgn" ], auz: [ "auz", "ar"  ], avl: [ "avl", "ar"  ], ayh: [ "ayh", "ar"  ],
            ayl: [ "ayl", "ar"  ], ayn: [ "ayn", "ar"  ], ayp: [ "ayp", "ar"  ], bbz: [ "bbz", "ar"  ],
            bfi: [ "bfi", "sgn" ], bfk: [ "bfk", "sgn" ], bjn: [ "bjn", "ms"  ], bog: [ "bog", "sgn" ],
            bqn: [ "bqn", "sgn" ], bqy: [ "bqy", "sgn" ], btj: [ "btj", "ms"  ], bve: [ "bve", "ms"  ],
            bvl: [ "bvl", "sgn" ], bvu: [ "bvu", "ms"  ], bzs: [ "bzs", "sgn" ], cdo: [ "cdo", "zh"  ],
            cds: [ "cds", "sgn" ], cjy: [ "cjy", "zh"  ], cmn: [ "cmn", "zh"  ], coa: [ "coa", "ms"  ],
            cpx: [ "cpx", "zh"  ], csc: [ "csc", "sgn" ], csd: [ "csd", "sgn" ], cse: [ "cse", "sgn" ],
            csf: [ "csf", "sgn" ], csg: [ "csg", "sgn" ], csl: [ "csl", "sgn" ], csn: [ "csn", "sgn" ],
            csq: [ "csq", "sgn" ], csr: [ "csr", "sgn" ], czh: [ "czh", "zh"  ], czo: [ "czo", "zh"  ],
            doq: [ "doq", "sgn" ], dse: [ "dse", "sgn" ], dsl: [ "dsl", "sgn" ], dup: [ "dup", "ms"  ],
            ecs: [ "ecs", "sgn" ], esl: [ "esl", "sgn" ], esn: [ "esn", "sgn" ], eso: [ "eso", "sgn" ],
            eth: [ "eth", "sgn" ], fcs: [ "fcs", "sgn" ], fse: [ "fse", "sgn" ], fsl: [ "fsl", "sgn" ],
            fss: [ "fss", "sgn" ], gan: [ "gan", "zh"  ], gds: [ "gds", "sgn" ], gom: [ "gom", "kok" ],
            gse: [ "gse", "sgn" ], gsg: [ "gsg", "sgn" ], gsm: [ "gsm", "sgn" ], gss: [ "gss", "sgn" ],
            gus: [ "gus", "sgn" ], hab: [ "hab", "sgn" ], haf: [ "haf", "sgn" ], hak: [ "hak", "zh"  ],
            hds: [ "hds", "sgn" ], hji: [ "hji", "ms"  ], hks: [ "hks", "sgn" ], hos: [ "hos", "sgn" ],
            hps: [ "hps", "sgn" ], hsh: [ "hsh", "sgn" ], hsl: [ "hsl", "sgn" ], hsn: [ "hsn", "zh"  ],
            icl: [ "icl", "sgn" ], ils: [ "ils", "sgn" ], inl: [ "inl", "sgn" ], ins: [ "ins", "sgn" ],
            ise: [ "ise", "sgn" ], isg: [ "isg", "sgn" ], isr: [ "isr", "sgn" ], jak: [ "jak", "ms"  ],
            jax: [ "jax", "ms"  ], jcs: [ "jcs", "sgn" ], jhs: [ "jhs", "sgn" ], jls: [ "jls", "sgn" ],
            jos: [ "jos", "sgn" ], jsl: [ "jsl", "sgn" ], jus: [ "jus", "sgn" ], kgi: [ "kgi", "sgn" ],
            knn: [ "knn", "kok" ], kvb: [ "kvb", "ms"  ], kvk: [ "kvk", "sgn" ], kvr: [ "kvr", "ms"  ],
            kxd: [ "kxd", "ms"  ], lbs: [ "lbs", "sgn" ], lce: [ "lce", "ms"  ], lcf: [ "lcf", "ms"  ],
            liw: [ "liw", "ms"  ], lls: [ "lls", "sgn" ], lsg: [ "lsg", "sgn" ], lsl: [ "lsl", "sgn" ],
            lso: [ "lso", "sgn" ], lsp: [ "lsp", "sgn" ], lst: [ "lst", "sgn" ], lsy: [ "lsy", "sgn" ],
            ltg: [ "ltg", "lv"  ], lvs: [ "lvs", "lv"  ], lzh: [ "lzh", "zh"  ], max: [ "max", "ms"  ],
            mdl: [ "mdl", "sgn" ], meo: [ "meo", "ms"  ], mfa: [ "mfa", "ms"  ], mfb: [ "mfb", "ms"  ],
            mfs: [ "mfs", "sgn" ], min: [ "min", "ms"  ], mnp: [ "mnp", "zh"  ], mqg: [ "mqg", "ms"  ],
            mre: [ "mre", "sgn" ], msd: [ "msd", "sgn" ], msi: [ "msi", "ms"  ], msr: [ "msr", "sgn" ],
            mui: [ "mui", "ms"  ], mzc: [ "mzc", "sgn" ], mzg: [ "mzg", "sgn" ], mzy: [ "mzy", "sgn" ],
            nan: [ "nan", "zh"  ], nbs: [ "nbs", "sgn" ], ncs: [ "ncs", "sgn" ], nsi: [ "nsi", "sgn" ],
            nsl: [ "nsl", "sgn" ], nsp: [ "nsp", "sgn" ], nsr: [ "nsr", "sgn" ], nzs: [ "nzs", "sgn" ],
            okl: [ "okl", "sgn" ], orn: [ "orn", "ms"  ], ors: [ "ors", "ms"  ], pel: [ "pel", "ms"  ],
            pga: [ "pga", "ar"  ], pks: [ "pks", "sgn" ], prl: [ "prl", "sgn" ], prz: [ "prz", "sgn" ],
            psc: [ "psc", "sgn" ], psd: [ "psd", "sgn" ], pse: [ "pse", "ms"  ], psg: [ "psg", "sgn" ],
            psl: [ "psl", "sgn" ], pso: [ "pso", "sgn" ], psp: [ "psp", "sgn" ], psr: [ "psr", "sgn" ],
            pys: [ "pys", "sgn" ], rms: [ "rms", "sgn" ], rsi: [ "rsi", "sgn" ], rsl: [ "rsl", "sgn" ],
            sdl: [ "sdl", "sgn" ], sfb: [ "sfb", "sgn" ], sfs: [ "sfs", "sgn" ], sgg: [ "sgg", "sgn" ],
            sgx: [ "sgx", "sgn" ], shu: [ "shu", "ar"  ], slf: [ "slf", "sgn" ], sls: [ "sls", "sgn" ],
            sqk: [ "sqk", "sgn" ], sqs: [ "sqs", "sgn" ], ssh: [ "ssh", "ar"  ], ssp: [ "ssp", "sgn" ],
            ssr: [ "ssr", "sgn" ], svk: [ "svk", "sgn" ], swc: [ "swc", "sw"  ], swh: [ "swh", "sw"  ],
            swl: [ "swl", "sgn" ], syy: [ "syy", "sgn" ], tmw: [ "tmw", "ms"  ], tse: [ "tse", "sgn" ],
            tsm: [ "tsm", "sgn" ], tsq: [ "tsq", "sgn" ], tss: [ "tss", "sgn" ], tsy: [ "tsy", "sgn" ],
            tza: [ "tza", "sgn" ], ugn: [ "ugn", "sgn" ], ugy: [ "ugy", "sgn" ], ukl: [ "ukl", "sgn" ],
            uks: [ "uks", "sgn" ], urk: [ "urk", "ms"  ], uzn: [ "uzn", "uz"  ], uzs: [ "uzs", "uz"  ],
            vgt: [ "vgt", "sgn" ], vkk: [ "vkk", "ms"  ], vkt: [ "vkt", "ms"  ], vsi: [ "vsi", "sgn" ],
            vsl: [ "vsl", "sgn" ], vsv: [ "vsv", "sgn" ], wuu: [ "wuu", "zh"  ], xki: [ "xki", "sgn" ],
            xml: [ "xml", "sgn" ], xmm: [ "xmm", "ms"  ], xms: [ "xms", "sgn" ], yds: [ "yds", "sgn" ],
            ysl: [ "ysl", "sgn" ], yue: [ "yue", "zh"  ], zib: [ "zib", "sgn" ], zlm: [ "zlm", "ms"  ],
            zmi: [ "zmi", "ms"  ], zsl: [ "zsl", "sgn" ], zsm: [ "zsm", "ms"  ]
        }
    },

    // Currency minor units output from tools/getISO4217data.js, formatted
    currencyMinorUnits = {
        BHD: 3, BYR: 0, XOF: 0, BIF: 0, XAF: 0, CLF: 0, CLP: 0, KMF: 0, DJF: 0,
        XPF: 0, GNF: 0, ISK: 0, IQD: 3, JPY: 0, JOD: 3, KRW: 0, KWD: 3, LYD: 3,
        OMR: 3, PYG: 0, RWF: 0, TND: 3, UGX: 0, UYI: 0, VUV: 0, VND: 0
    };

/**
 * Defines regular expressions for various operations related to the BCP 47 syntax,
 * as defined at http://tools.ietf.org/html/bcp47#section-2.1
 */
(function () {
    var
        // extlang       = 3ALPHA              ; selected ISO 639 codes
        //                 *2("-" 3ALPHA)      ; permanently reserved
        extlang = '[a-z]{3}(?:-[a-z]{3}){0,2}',

        // language      = 2*3ALPHA            ; shortest ISO 639 code
        //                 ["-" extlang]       ; sometimes followed by
        //                                     ; extended language subtags
        //               / 4ALPHA              ; or reserved for future use
        //               / 5*8ALPHA            ; or registered language subtag
        language = '(?:[a-z]{2,3}(?:-' + extlang + ')?|[a-z]{4}|[a-z]{5,8})',

        // script        = 4ALPHA              ; ISO 15924 code
        script = '[a-z]{4}',

        // region        = 2ALPHA              ; ISO 3166-1 code
        //               / 3DIGIT              ; UN M.49 code
        region = '(?:[a-z]{2}|\\d{3})',

        // variant       = 5*8alphanum         ; registered variants
        //               / (DIGIT 3alphanum)
        variant = '(?:[a-z0-9]{5,8}|\\d[a-z0-9]{3})',

        //                                     ; Single alphanumerics
        //                                     ; "x" reserved for private use
        // singleton     = DIGIT               ; 0 - 9
        //               / %x41-57             ; A - W
        //               / %x59-5A             ; Y - Z
        //               / %x61-77             ; a - w
        //               / %x79-7A             ; y - z
        singleton = '[0-9a-wy-z]',

        // extension     = singleton 1*("-" (2*8alphanum))
        extension = singleton + '(?:-[a-z0-9]{2,8})+',

        // privateuse    = "x" 1*("-" (1*8alphanum))
        privateuse = 'x(?:-[a-z0-9]{1,8})+',

        // irregular     = "en-GB-oed"         ; irregular tags do not match
        //               / "i-ami"             ; the 'langtag' production and
        //               / "i-bnn"             ; would not otherwise be
        //               / "i-default"         ; considered 'well-formed'
        //               / "i-enochian"        ; These tags are all valid,
        //               / "i-hak"             ; but most are deprecated
        //               / "i-klingon"         ; in favor of more modern
        //               / "i-lux"             ; subtags or subtag
        //               / "i-mingo"           ; combination
        //               / "i-navajo"
        //               / "i-pwn"
        //               / "i-tao"
        //               / "i-tay"
        //               / "i-tsu"
        //               / "sgn-BE-FR"
        //               / "sgn-BE-NL"
        //               / "sgn-CH-DE"
        irregular = '(?:en-GB-oed'
                  + '|i-(?:ami|bnn|default|enochian|hak|klingon|lux|mingo|navajo|pwn|tao|tay|tsu)'
                  + '|sgn-(?:BE-FR|BE-NL|CH-DE))',

        // regular       = "art-lojban"        ; these tags match the 'langtag'
        //               / "cel-gaulish"       ; production, but their subtags
        //               / "no-bok"            ; are not extended language
        //               / "no-nyn"            ; or variant subtags: their meaning
        //               / "zh-guoyu"          ; is defined by their registration
        //               / "zh-hakka"          ; and all of these are deprecated
        //               / "zh-min"            ; in favor of a more modern
        //               / "zh-min-nan"        ; subtag or sequence of subtags
        //               / "zh-xiang"
        regular = '(?:art-lojban|cel-gaulish|no-bok|no-nyn'
                + '|zh-(?:guoyu|hakka|min|min-nan|xiang))',

        // grandfathered = irregular           ; non-redundant tags registered
        //               / regular             ; during the RFC 3066 era
        grandfathered = '(?:' + irregular + '|' + regular + ')',

        // langtag       = language
        //                 ["-" script]
        //                 ["-" region]
        //                 *("-" variant)
        //                 *("-" extension)
        //                 ["-" privateuse]
        langtag = language + '(?:-' + script + ')?(?:-' + region + ')?(?:-'
                + variant + ')*(?:-' + extension + ')*(?:-' + privateuse + ')?';

    // Language-Tag  = langtag             ; normal language tags
    //               / privateuse          ; private use tag
    //               / grandfathered       ; grandfathered tags
    expBCP47Syntax = RegExp('^(?:'+langtag+'|'+privateuse+'|'+grandfathered+')$', 'i');

    // Match duplicate variants in a language tag
    expVariantDupes = RegExp('^(?!x).*?-('+variant+')-(?:\\w{4,8}-(?!x-))*\\1\\b', 'i');

    // Match duplicate singletons in a language tag (except in private use)
    expSingletonDupes = RegExp('^(?!x).*?-('+singleton+')-(?:\\w+-(?!x-))*\\1\\b', 'i');

    // Match all extension sequences
    expExtSequences = RegExp('-'+extension, 'ig');
})();

// Sect 6.2 Language Tags
// ======================

/**
 * The IsStructurallyValidLanguageTag abstract operation verifies that the locale
 * argument (which must be a String value)
 *
 * - represents a well-formed BCP 47 language tag as specified in RFC 5646 section
 *   2.1, or successor,
 * - does not include duplicate variant subtags, and
 * - does not include duplicate singleton subtags.
 *
 * The abstract operation returns true if locale can be generated from the ABNF
 * grammar in section 2.1 of the RFC, starting with Language-Tag, and does not
 * contain duplicate variant or singleton subtags (other than as a private use
 * subtag). It returns false otherwise. Terminal value characters in the grammar are
 * interpreted as the Unicode equivalents of the ASCII octet values given.
 */
function /* 6.2.2 */IsStructurallyValidLanguageTag(locale) {
    // represents a well-formed BCP 47 language tag as specified in RFC 5646
    if (!expBCP47Syntax.test(locale))
        return false;

    // does not include duplicate variant subtags, and
    if (expVariantDupes.test(locale))
        return false;

    // does not include duplicate singleton subtags.
    if (expSingletonDupes.test(locale))
        return false;

    return true;
}

/**
 * The CanonicalizeLanguageTag abstract operation returns the canonical and case-
 * regularized form of the locale argument (which must be a String value that is
 * a structurally valid BCP 47 language tag as verified by the
 * IsStructurallyValidLanguageTag abstract operation). It takes the steps
 * specified in RFC 5646 section 4.5, or successor, to bring the language tag
 * into canonical form, and to regularize the case of the subtags, but does not
 * take the steps to bring a language tag into “extlang form” and to reorder
 * variant subtags.

 * The specifications for extensions to BCP 47 language tags, such as RFC 6067,
 * may include canonicalization rules for the extension subtag sequences they
 * define that go beyond the canonicalization rules of RFC 5646 section 4.5.
 * Implementations are allowed, but not required, to apply these additional rules.
 */
function /* 6.2.3 */CanonicalizeLanguageTag (locale) {
    var match, parts;

    // A language tag is in 'canonical form' when the tag is well-formed
    // according to the rules in Sections 2.1 and 2.2

    // Section 2.1 says all subtags use lowercase...
    locale = locale.toLowerCase();

    // ...with 2 exceptions: 'two-letter and four-letter subtags that neither
    // appear at the start of the tag nor occur after singletons.  Such two-letter
    // subtags are all uppercase (as in the tags "en-CA-x-ca" or "sgn-BE-FR") and
    // four-letter subtags are titlecase (as in the tag "az-Latn-x-latn").
    parts = locale.split('-');
    for (var i = 1, max = parts.length; i < max; i++) {
        // Two-letter subtags are all uppercase
        if (parts[i].length === 2)
            parts[i] = parts[i].toUpperCase();

        // Four-letter subtags are titlecase
        else if (parts[i].length === 4)
            parts[i] = parts[i].charAt(0).toUpperCase() + parts[i].slice(1);

        // Is it a singleton?
        else if (parts[i].length === 1 && parts[i] != 'x')
            break;
    }
    locale = arrJoin.call(parts, '-');

    // The steps laid out in RFC 5646 section 4.5 are as follows:

    // 1.  Extension sequences are ordered into case-insensitive ASCII order
    //     by singleton subtag.
    if ((match = locale.match(expExtSequences)) && match.length > 1) {
        // The built-in sort() sorts by ASCII order, so use that
        match.sort();

        // Replace all extensions with the joined, sorted array
        locale = locale.replace(
            RegExp('(?:' + expExtSequences.source + ')+', 'i'),
            arrJoin.call(match, '')
        );
    }

    // 2.  Redundant or grandfathered tags are replaced by their 'Preferred-
    //     Value', if there is one.
    if (hop.call(redundantTags.tags, locale))
        locale = redundantTags.tags[locale];

    // 3.  Subtags are replaced by their 'Preferred-Value', if there is one.
    //     For extlangs, the original primary language subtag is also
    //     replaced if there is a primary language subtag in the 'Preferred-
    //     Value'.
    parts = locale.split('-');

    for (var i = 1, max = parts.length; i < max; i++) {
        if (hop.call(redundantTags.subtags, parts[i]))
            parts[i] = redundantTags.subtags[parts[i]];

        else if (hop.call(redundantTags.extLang, parts[i])) {
            parts[i] = redundantTags.extLang[parts[i]][0];

            // For extlang tags, the prefix needs to be removed if it is redundant
            if (i === 1 && redundantTags.extLang[parts[1]][1] === parts[0]) {
                parts = arrSlice.call(parts, i++);
                max -= 1;
            }
        }
    }

    return arrJoin.call(parts, '-');
}

/**
 * The DefaultLocale abstract operation returns a String value representing the
 * structurally valid (6.2.2) and canonicalized (6.2.3) BCP 47 language tag for the
 * host environment’s current locale.
 */
function /* 6.2.4 */DefaultLocale () {
    return defaultLocale;
}

// Sect 6.3 Currency Codes
// =======================

/**
 * The IsWellFormedCurrencyCode abstract operation verifies that the currency argument
 * (after conversion to a String value) represents a well-formed 3-letter ISO currency
 * code. The following steps are taken:
 */
function /* 6.3.1 */IsWellFormedCurrencyCode(currency) {
    var
        // 1. Let `c` be ToString(currency)
        c = String(currency),

        // 2. Let `normalized` be the result of mapping c to upper case as described
        //    in 6.1.
        normalized = toLatinUpperCase(c);

    // 3. If the string length of normalized is not 3, return false.
    // 4. If normalized contains any character that is not in the range "A" to "Z"
    //    (U+0041 to U+005A), return false.
    if (expCurrencyCode.test(normalized) === false)
        return false;

    // 5. Return true
    return true;
}

// Sect 9.2 Abstract Operations
// ============================
function /* 9.2.1 */CanonicalizeLocaleList (locales) {
// The abstract operation CanonicalizeLocaleList takes the following steps:

    // 1. If locales is undefined, then a. Return a new empty List
    if (locales === undefined)
        return new List();

    var
        // 2. Let seen be a new empty List.
        seen = new List(),

        // 3. If locales is a String value, then
        //    a. Let locales be a new array created as if by the expression new
        //    Array(locales) where Array is the standard built-in constructor with
        //    that name and locales is the value of locales.
        locales = typeof locales === 'string' ? [ locales ] : locales,

        // 4. Let O be ToObject(locales).
        O = toObject(locales),

        // 5. Let lenValue be the result of calling the [[Get]] internal method of
        //    O with the argument "length".
        // 6. Let len be ToUint32(lenValue).
        len = O.length,

        // 7. Let k be 0.
        k = 0;

    // 8. Repeat, while k < len
    while (k < len) {
        var
            // a. Let Pk be ToString(k).
            Pk = String(k),

            // b. Let kPresent be the result of calling the [[HasProperty]] internal
            //    method of O with argument Pk.
            kPresent = Pk in O;

        // c. If kPresent is true, then
        if (kPresent) {
            var
                // i. Let kValue be the result of calling the [[Get]] internal
                //     method of O with argument Pk.
                kValue = O[Pk];

            // ii. If the type of kValue is not String or Object, then throw a
            //     TypeError exception.
            if (kValue == null || (typeof kValue !== 'string' && typeof kValue !== 'object'))
                throw new TypeError('String or Object type expected');

            var
                // iii. Let tag be ToString(kValue).
                tag = String(kValue);

            // iv. If the result of calling the abstract operation
            //     IsStructurallyValidLanguageTag (defined in 6.2.2), passing tag as
            //     the argument, is false, then throw a RangeError exception.
            if (!IsStructurallyValidLanguageTag(tag))
                throw new RangeError("'" + tag + "' is not a structurally valid language tag");

            // v. Let tag be the result of calling the abstract operation
            //    CanonicalizeLanguageTag (defined in 6.2.3), passing tag as the
            //    argument.
            tag = CanonicalizeLanguageTag(tag);

            // vi. If tag is not an element of seen, then append tag as the last
            //     element of seen.
            if (arrIndexOf.call(seen, tag) === -1)
                arrPush.call(seen, tag);
        }

        // d. Increase k by 1.
        k++;
    }

    // 9. Return seen.
    return seen;
}

/**
 * The BestAvailableLocale abstract operation compares the provided argument
 * locale, which must be a String value with a structurally valid and
 * canonicalized BCP 47 language tag, against the locales in availableLocales and
 * returns either the longest non-empty prefix of locale that is an element of
 * availableLocales, or undefined if there is no such element. It uses the
 * fallback mechanism of RFC 4647, section 3.4. The following steps are taken:
 */
function /* 9.2.2 */BestAvailableLocale (availableLocales, locale) {
    var
       // 1. Let candidate be locale
       candidate = locale;

    // 2. Repeat
    while (true) {
        // a. If availableLocales contains an element equal to candidate, then return
        // candidate.
        if (arrIndexOf.call(availableLocales, candidate) > -1)
            return candidate;

        var
            // b. Let pos be the character index of the last occurrence of "-"
            // (U+002D) within candidate. If that character does not occur, return
            // undefined.
            pos = candidate.lastIndexOf('-');

        if (pos < 0)
            return;

        // c. If pos ≥ 2 and the character "-" occurs at index pos-2 of candidate,
        //    then decrease pos by 2.
        if (pos >= 2 && candidate.charAt(pos - 2) == '-')
            pos -= 2;

        // d. Let candidate be the substring of candidate from position 0, inclusive,
        //    to position pos, exclusive.
        candidate = candidate.substring(0, pos);
    }
}

/**
 * The LookupMatcher abstract operation compares requestedLocales, which must be
 * a List as returned by CanonicalizeLocaleList, against the locales in
 * availableLocales and determines the best available language to meet the
 * request. The following steps are taken:
 */
function /* 9.2.3 */LookupMatcher (availableLocales, requestedLocales) {
    var
        // 1. Let i be 0.
        i = 0,

        // 2. Let len be the number of elements in requestedLocales.
        len = requestedLocales.length,

        // 3. Let availableLocale be undefined.
        availableLocale;

    // 4. Repeat while i < len and availableLocale is undefined:
    while (i < len && !availableLocale) {
        var
            // a. Let locale be the element of requestedLocales at 0-origined list
            //    position i.
            locale = requestedLocales[i],

            // b. Let noExtensionsLocale be the String value that is locale with all
            //    Unicode locale extension sequences removed.
            noExtensionsLocale = String(locale).replace(expUnicodeExSeq, ''),

            // c. Let availableLocale be the result of calling the
            //    BestAvailableLocale abstract operation (defined in 9.2.2) with
            //    arguments availableLocales and noExtensionsLocale.
            availableLocale = BestAvailableLocale(availableLocales, noExtensionsLocale);

        // d. Increase i by 1.
        i++;
    }

    var
        // 5. Let result be a new Record.
        result = new Record();

    // 6. If availableLocale is not undefined, then
    if (availableLocale !== undefined) {
        // a. Set result.[[locale]] to availableLocale.
        result['[[locale]]'] = availableLocale;

        // b. If locale and noExtensionsLocale are not the same String value, then
        if (String(locale) !== String(noExtensionsLocale)) {
            var
                // i. Let extension be the String value consisting of the first
                //    substring of locale that is a Unicode locale extension sequence.
                extension = locale.match(expUnicodeExSeq)[0],

                // ii. Let extensionIndex be the character position of the initial
                //     "-" of the first Unicode locale extension sequence within locale.
                extensionIndex = locale.indexOf('-u-');

            // iii. Set result.[[extension]] to extension.
            result['[[extension]]'] = extension;

            // iv. Set result.[[extensionIndex]] to extensionIndex.
            result['[[extensionIndex]]'] = extensionIndex;
        }
    }
    // 7. Else
    else
        // a. Set result.[[locale]] to the value returned by the DefaultLocale abstract
        //    operation (defined in 6.2.4).
        result['[[locale]]'] = DefaultLocale();

    // 8. Return result
    return result;
}

/**
 * The BestFitMatcher abstract operation compares requestedLocales, which must be
 * a List as returned by CanonicalizeLocaleList, against the locales in
 * availableLocales and determines the best available language to meet the
 * request. The algorithm is implementation dependent, but should produce results
 * that a typical user of the requested locales would perceive as at least as
 * good as those produced by the LookupMatcher abstract operation. Options
 * specified through Unicode locale extension sequences must be ignored by the
 * algorithm. Information about such subsequences is returned separately.
 * The abstract operation returns a record with a [[locale]] field, whose value
 * is the language tag of the selected locale, which must be an element of
 * availableLocales. If the language tag of the request locale that led to the
 * selected locale contained a Unicode locale extension sequence, then the
 * returned record also contains an [[extension]] field whose value is the first
 * Unicode locale extension sequence, and an [[extensionIndex]] field whose value
 * is the index of the first Unicode locale extension sequence within the request
 * locale language tag.
 */
function /* 9.2.4 */BestFitMatcher (availableLocales, requestedLocales) {
    return LookupMatcher(availableLocales, requestedLocales);
}

/**
 * The ResolveLocale abstract operation compares a BCP 47 language priority list
 * requestedLocales against the locales in availableLocales and determines the
 * best available language to meet the request. availableLocales and
 * requestedLocales must be provided as List values, options as a Record.
 */
function /* 9.2.5 */ResolveLocale (availableLocales, requestedLocales, options, relevantExtensionKeys, localeData) {
    if (availableLocales.length === 0) {
        throw new ReferenceError('No locale data has been provided for this object yet.');
    }

    // The following steps are taken:
    var
        // 1. Let matcher be the value of options.[[localeMatcher]].
        matcher = options['[[localeMatcher]]'];

    // 2. If matcher is "lookup", then
    if (matcher === 'lookup')
        var
            // a. Let r be the result of calling the LookupMatcher abstract operation
            //    (defined in 9.2.3) with arguments availableLocales and
            //    requestedLocales.
            r = LookupMatcher(availableLocales, requestedLocales);

    // 3. Else
    else
        var
            // a. Let r be the result of calling the BestFitMatcher abstract
            //    operation (defined in 9.2.4) with arguments availableLocales and
            //    requestedLocales.
            r = BestFitMatcher(availableLocales, requestedLocales);

    var
        // 4. Let foundLocale be the value of r.[[locale]].
        foundLocale = r['[[locale]]'];

    // 5. If r has an [[extension]] field, then
    if (hop.call(r, '[[extension]]'))
        var
            // a. Let extension be the value of r.[[extension]].
            extension = r['[[extension]]'],
            // b. Let extensionIndex be the value of r.[[extensionIndex]].
            extensionIndex = r['[[extensionIndex]]'],
            // c. Let split be the standard built-in function object defined in ES5,
            //    15.5.4.14.
            split = String.prototype.split,
            // d. Let extensionSubtags be the result of calling the [[Call]] internal
            //    method of split with extension as the this value and an argument
            //    list containing the single item "-".
            extensionSubtags = split.call(extension, '-'),
            // e. Let extensionSubtagsLength be the result of calling the [[Get]]
            //    internal method of extensionSubtags with argument "length".
            extensionSubtagsLength = extensionSubtags.length;

    var
        // 6. Let result be a new Record.
        result = new Record();

    // 7. Set result.[[dataLocale]] to foundLocale.
    result['[[dataLocale]]'] = foundLocale;

    var
        // 8. Let supportedExtension be "-u".
        supportedExtension = '-u',
        // 9. Let i be 0.
        i = 0,
        // 10. Let len be the result of calling the [[Get]] internal method of
        //     relevantExtensionKeys with argument "length".
        len = relevantExtensionKeys.length;

    // 11 Repeat while i < len:
    while (i < len) {
        var
            // a. Let key be the result of calling the [[Get]] internal method of
            //    relevantExtensionKeys with argument ToString(i).
            key = relevantExtensionKeys[i],
            // b. Let foundLocaleData be the result of calling the [[Get]] internal
            //    method of localeData with the argument foundLocale.
            foundLocaleData = localeData[foundLocale],
            // c. Let keyLocaleData be the result of calling the [[Get]] internal
            //    method of foundLocaleData with the argument key.
            keyLocaleData = foundLocaleData[key],
            // d. Let value be the result of calling the [[Get]] internal method of
            //    keyLocaleData with argument "0".
            value = keyLocaleData['0'],
            // e. Let supportedExtensionAddition be "".
            supportedExtensionAddition = '',
            // f. Let indexOf be the standard built-in function object defined in
            //    ES5, 15.4.4.14.
            indexOf = arrIndexOf;

        // g. If extensionSubtags is not undefined, then
        if (extensionSubtags !== undefined) {
            var
                // i. Let keyPos be the result of calling the [[Call]] internal
                //    method of indexOf with extensionSubtags as the this value and
                // an argument list containing the single item key.
                keyPos = indexOf.call(extensionSubtags, key);

            // ii. If keyPos ≠ -1, then
            if (keyPos !== -1) {
                // 1. If keyPos + 1 < extensionSubtagsLength and the length of the
                //    result of calling the [[Get]] internal method of
                //    extensionSubtags with argument ToString(keyPos +1) is greater
                //    than 2, then
                if (keyPos + 1 < extensionSubtagsLength
                        && extensionSubtags[keyPos + 1].length > 2) {
                    var
                        // a. Let requestedValue be the result of calling the [[Get]]
                        //    internal method of extensionSubtags with argument
                        //    ToString(keyPos + 1).
                        requestedValue = extensionSubtags[keyPos + 1],
                        // b. Let valuePos be the result of calling the [[Call]]
                        //    internal method of indexOf with keyLocaleData as the
                        //    this value and an argument list containing the single
                        //    item requestedValue.
                        valuePos = indexOf.call(keyLocaleData, requestedValue);

                    // c. If valuePos ≠ -1, then
                    if (valuePos !== -1)
                        var
                            // i. Let value be requestedValue.
                            value = requestedValue,
                            // ii. Let supportedExtensionAddition be the
                            //     concatenation of "-", key, "-", and value.
                            supportedExtensionAddition = '-' + key + '-' + value;
                }
                // 2. Else
                else {
                    var
                        // a. Let valuePos be the result of calling the [[Call]]
                        // internal method of indexOf with keyLocaleData as the this
                        // value and an argument list containing the single item
                        // "true".
                        valuePos = indexOf(keyLocaleData, 'true');

                    // b. If valuePos ≠ -1, then
                    if (valuePos !== -1)
                        var
                            // i. Let value be "true".
                            value = 'true';
                }
            }
        }
        // h. If options has a field [[<key>]], then
        if (hop.call(options, '[[' + key + ']]')) {
            var
                // i. Let optionsValue be the value of options.[[<key>]].
                optionsValue = options['[[' + key + ']]'];

            // ii. If the result of calling the [[Call]] internal method of indexOf
            //     with keyLocaleData as the this value and an argument list
            //     containing the single item optionsValue is not -1, then
            if (indexOf.call(keyLocaleData, optionsValue) !== -1) {
                // 1. If optionsValue is not equal to value, then
                if (optionsValue !== value) {
                    // a. Let value be optionsValue.
                    value = optionsValue;
                    // b. Let supportedExtensionAddition be "".
                    supportedExtensionAddition = '';
                }
            }
        }
        // i. Set result.[[<key>]] to value.
        result['[[' + key + ']]'] = value;

        // j. Append supportedExtensionAddition to supportedExtension.
        supportedExtension += supportedExtensionAddition;

        // k. Increase i by 1.
        i++;
    }
    // 12. If the length of supportedExtension is greater than 2, then
    if (supportedExtension.length > 2) {
        var
            // a. Let preExtension be the substring of foundLocale from position 0,
            //    inclusive, to position extensionIndex, exclusive.
            preExtension = foundLocale.substring(0, extensionIndex),
            // b. Let postExtension be the substring of foundLocale from position
            //    extensionIndex to the end of the string.
            postExtension = foundLocale.substring(extensionIndex),
            // c. Let foundLocale be the concatenation of preExtension,
            //    supportedExtension, and postExtension.
            foundLocale = preExtension + supportedExtension + postExtension;
    }
    // 13. Set result.[[locale]] to foundLocale.
    result['[[locale]]'] = foundLocale;

    // 14. Return result.
    return result;
}

/**
 * The LookupSupportedLocales abstract operation returns the subset of the
 * provided BCP 47 language priority list requestedLocales for which
 * availableLocales has a matching locale when using the BCP 47 Lookup algorithm.
 * Locales appear in the same order in the returned list as in requestedLocales.
 * The following steps are taken:
 */
function /* 9.2.6 */LookupSupportedLocales (availableLocales, requestedLocales) {
    var
        // 1. Let len be the number of elements in requestedLocales.
        len = requestedLocales.length,
        // 2. Let subset be a new empty List.
        subset = new List(),
        // 3. Let k be 0.
        k = 0;

    // 4. Repeat while k < len
    while (k < len) {
        var
            // a. Let locale be the element of requestedLocales at 0-origined list
            //    position k.
            locale = requestedLocales[k],
            // b. Let noExtensionsLocale be the String value that is locale with all
            //    Unicode locale extension sequences removed.
            noExtensionsLocale = String(locale).replace(expUnicodeExSeq, ''),
            // c. Let availableLocale be the result of calling the
            //    BestAvailableLocale abstract operation (defined in 9.2.2) with
            //    arguments availableLocales and noExtensionsLocale.
            availableLocale = BestAvailableLocale(availableLocales, noExtensionsLocale);

        // d. If availableLocale is not undefined, then append locale to the end of
        //    subset.
        if (availableLocale !== undefined)
            arrPush.call(subset, locale);

        // e. Increment k by 1.
        k++;
    }

    var
        // 5. Let subsetArray be a new Array object whose elements are the same
        //    values in the same order as the elements of subset.
        subsetArray = arrSlice.call(subset);

    // 6. Return subsetArray.
    return subsetArray;
}

/**
 * The BestFitSupportedLocales abstract operation returns the subset of the
 * provided BCP 47 language priority list requestedLocales for which
 * availableLocales has a matching locale when using the Best Fit Matcher
 * algorithm. Locales appear in the same order in the returned list as in
 * requestedLocales. The steps taken are implementation dependent.
 */
function /*9.2.7 */BestFitSupportedLocales (availableLocales, requestedLocales) {
    // ###TODO: implement this function as described by the specification###
    return LookupSupportedLocales(availableLocales, requestedLocales);
}

/**
 * The SupportedLocales abstract operation returns the subset of the provided BCP
 * 47 language priority list requestedLocales for which availableLocales has a
 * matching locale. Two algorithms are available to match the locales: the Lookup
 * algorithm described in RFC 4647 section 3.4, and an implementation dependent
 * best-fit algorithm. Locales appear in the same order in the returned list as
 * in requestedLocales. The following steps are taken:
 */
function /*9.2.8 */SupportedLocales (availableLocales, requestedLocales, options) {
    // 1. If options is not undefined, then
    if (options !== undefined) {
        var
            // a. Let options be ToObject(options).
            options = new Record(toObject(options)),
            // b. Let matcher be the result of calling the [[Get]] internal method of
            //    options with argument "localeMatcher".
            matcher = options.localeMatcher;

        // c. If matcher is not undefined, then
        if (matcher !== undefined) {
            // i. Let matcher be ToString(matcher).
            matcher = String(matcher);

            // ii. If matcher is not "lookup" or "best fit", then throw a RangeError
            //     exception.
            if (matcher !== 'lookup' && matcher !== 'best fit')
                throw new RangeError('matcher should be "lookup" or "best fit"');
        }
    }
    // 2. If matcher is undefined or "best fit", then
    if (matcher === undefined || matcher === 'best fit')
        var
            // a. Let subset be the result of calling the BestFitSupportedLocales
            //    abstract operation (defined in 9.2.7) with arguments
            //    availableLocales and requestedLocales.
            subset = BestFitSupportedLocales(availableLocales, requestedLocales);
    // 3. Else
    else
        var
            // a. Let subset be the result of calling the LookupSupportedLocales
            //    abstract operation (defined in 9.2.6) with arguments
            //    availableLocales and requestedLocales.
            subset = LookupSupportedLocales(availableLocales, requestedLocales);

    // 4. For each named own property name P of subset,
    for (var P in subset) {
        if (!hop.call(subset, P))
            continue;

        // a. Let desc be the result of calling the [[GetOwnProperty]] internal
        //    method of subset with P.
        // b. Set desc.[[Writable]] to false.
        // c. Set desc.[[Configurable]] to false.
        // d. Call the [[DefineOwnProperty]] internal method of subset with P, desc,
        //    and true as arguments.
        defineProperty(subset, P, {
            writable: false, configurable: false, value: subset[P]
        });
    }
    // "Freeze" the array so no new elements can be added
    defineProperty(subset, 'length', { writable: false });

    // 5. Return subset
    return subset;
}

/**
 * The GetOption abstract operation extracts the value of the property named
 * property from the provided options object, converts it to the required type,
 * checks whether it is one of a List of allowed values, and fills in a fallback
 * value if necessary.
 */
function /*9.2.9 */GetOption (options, property, type, values, fallback) {
    var
        // 1. Let value be the result of calling the [[Get]] internal method of
        //    options with argument property.
        value = options[property];

    // 2. If value is not undefined, then
    if (value !== undefined) {
        // a. Assert: type is "boolean" or "string".
        // b. If type is "boolean", then let value be ToBoolean(value).
        // c. If type is "string", then let value be ToString(value).
        value = type === 'boolean' ? Boolean(value)
                  : (type === 'string' ? String(value) : value);

        // d. If values is not undefined, then
        if (values !== undefined) {
            // i. If values does not contain an element equal to value, then throw a
            //    RangeError exception.
            if (arrIndexOf.call(values, value) === -1)
                throw new RangeError("'" + value + "' is not an allowed value for `" + property +'`');
        }

        // e. Return value.
        return value;
    }
    // Else return fallback.
    return fallback;
}

/**
 * The GetNumberOption abstract operation extracts a property value from the
 * provided options object, converts it to a Number value, checks whether it is
 * in the allowed range, and fills in a fallback value if necessary.
 */
function /* 9.2.10 */GetNumberOption (options, property, minimum, maximum, fallback) {
    var
        // 1. Let value be the result of calling the [[Get]] internal method of
        //    options with argument property.
        value = options[property];

    // 2. If value is not undefined, then
    if (value !== undefined) {
        // a. Let value be ToNumber(value).
        value = Number(value);

        // b. If value is NaN or less than minimum or greater than maximum, throw a
        //    RangeError exception.
        if (isNaN(value) || value < minimum || value > maximum)
            throw new RangeError('Value is not a number or outside accepted range');

        // c. Return floor(value).
        return Math.floor(value);
    }
    // 3. Else return fallback.
    return fallback;
}

// 11.1 The Intl.NumberFormat constructor
// ======================================

// Define the NumberFormat constructor internally so it cannot be tainted
function NumberFormatConstructor () {
    var locales = arguments[0];
    var options = arguments[1];

    if (!this || this === Intl) {
        return new Intl.NumberFormat(locales, options);
    }

    return InitializeNumberFormat(toObject(this), locales, options);
}

defineProperty(Intl, 'NumberFormat', {
    configurable: true,
    writable: true,
    value: NumberFormatConstructor
});

// Must explicitly set prototypes as unwritable
defineProperty(Intl.NumberFormat, 'prototype', {
    writable: false
});

/**
 * The abstract operation InitializeNumberFormat accepts the arguments
 * numberFormat (which must be an object), locales, and options. It initializes
 * numberFormat as a NumberFormat object.
 */
function /*11.1.1.1 */InitializeNumberFormat (numberFormat, locales, options) {
    var
    // This will be a internal properties object if we're not already initialized
        internal = getInternalProperties(numberFormat),

    // Create an object whose props can be used to restore the values of RegExp props
        regexpState = createRegExpRestore();

    // 1. If numberFormat has an [[initializedIntlObject]] internal property with
    // value true, throw a TypeError exception.
    if (internal['[[initializedIntlObject]]'] === true)
        throw new TypeError('`this` object has already been initialized as an Intl object');

    // Need this to access the `internal` object
    defineProperty(numberFormat, '__getInternalProperties', {
        value: function () {
            // NOTE: Non-standard, for internal use only
            if (arguments[0] === secret)
                return internal;
        }
    });

    // 2. Set the [[initializedIntlObject]] internal property of numberFormat to true.
    internal['[[initializedIntlObject]]'] = true;

    var
    // 3. Let requestedLocales be the result of calling the CanonicalizeLocaleList
    //    abstract operation (defined in 9.2.1) with argument locales.
        requestedLocales = CanonicalizeLocaleList(locales);

    // 4. If options is undefined, then
    if (options === undefined)
        // a. Let options be the result of creating a new object as if by the
        // expression new Object() where Object is the standard built-in constructor
        // with that name.
        options = {};

    // 5. Else
    else
        // a. Let options be ToObject(options).
        options = toObject(options);

    var
    // 6. Let opt be a new Record.
        opt = new Record(),

    // 7. Let matcher be the result of calling the GetOption abstract operation
    //    (defined in 9.2.9) with the arguments options, "localeMatcher", "string",
    //    a List containing the two String values "lookup" and "best fit", and
    //    "best fit".
        matcher =  GetOption(options, 'localeMatcher', 'string', new List('lookup', 'best fit'), 'best fit');

    // 8. Set opt.[[localeMatcher]] to matcher.
    opt['[[localeMatcher]]'] = matcher;

    var
    // 9. Let NumberFormat be the standard built-in object that is the initial value
    //    of Intl.NumberFormat.
    // 10. Let localeData be the value of the [[localeData]] internal property of
    //     NumberFormat.
        localeData = internals.NumberFormat['[[localeData]]'],

    // 11. Let r be the result of calling the ResolveLocale abstract operation
    //     (defined in 9.2.5) with the [[availableLocales]] internal property of
    //     NumberFormat, requestedLocales, opt, the [[relevantExtensionKeys]]
    //     internal property of NumberFormat, and localeData.
        r = ResolveLocale(
                internals.NumberFormat['[[availableLocales]]'], requestedLocales,
                opt, internals.NumberFormat['[[relevantExtensionKeys]]'], localeData
            );

    // 12. Set the [[locale]] internal property of numberFormat to the value of
    //     r.[[locale]].
    internal['[[locale]]'] = r['[[locale]]'];

    // 13. Set the [[numberingSystem]] internal property of numberFormat to the value
    //     of r.[[nu]].
    internal['[[numberingSystem]]'] = r['[[nu]]'];

    // The specification doesn't tell us to do this, but it's helpful later on
    internal['[[dataLocale]]'] = r['[[dataLocale]]'];

    var
    // 14. Let dataLocale be the value of r.[[dataLocale]].
        dataLocale = r['[[dataLocale]]'],

    // 15. Let s be the result of calling the GetOption abstract operation with the
    //     arguments options, "style", "string", a List containing the three String
    //     values "decimal", "percent", and "currency", and "decimal".
        s = GetOption(options, 'style', 'string', new List('decimal', 'percent', 'currency'), 'decimal');

    // 16. Set the [[style]] internal property of numberFormat to s.
    internal['[[style]]'] = s;

    var
    // 17. Let c be the result of calling the GetOption abstract operation with the
    //     arguments options, "currency", "string", undefined, and undefined.
        c = GetOption(options, 'currency', 'string');

    // 18. If c is not undefined and the result of calling the
    //     IsWellFormedCurrencyCode abstract operation (defined in 6.3.1) with
    //     argument c is false, then throw a RangeError exception.
    if (c !== undefined && !IsWellFormedCurrencyCode(c))
        throw new RangeError("'" + c + "' is not a valid currency code");

    // 19. If s is "currency" and c is undefined, throw a TypeError exception.
    if (s === 'currency' && c === undefined)
        throw new TypeError('Currency code is required when style is currency');

    // 20. If s is "currency", then
    if (s === 'currency') {
        // a. Let c be the result of converting c to upper case as specified in 6.1.
        c = c.toUpperCase();

        // b. Set the [[currency]] internal property of numberFormat to c.
        internal['[[currency]]'] = c;

        var
        // c. Let cDigits be the result of calling the CurrencyDigits abstract
        //    operation (defined below) with argument c.
            cDigits = CurrencyDigits(c);
    }

    var
    // 21. Let cd be the result of calling the GetOption abstract operation with the
    //     arguments options, "currencyDisplay", "string", a List containing the
    //     three String values "code", "symbol", and "name", and "symbol".
        cd = GetOption(options, 'currencyDisplay', 'string', new List('code', 'symbol', 'name'), 'symbol');

    // 22. If s is "currency", then set the [[currencyDisplay]] internal property of
    //     numberFormat to cd.
    if (s === 'currency')
        internal['[[currencyDisplay]]'] = cd;

    var
    // 23. Let mnid be the result of calling the GetNumberOption abstract operation
    //     (defined in 9.2.10) with arguments options, "minimumIntegerDigits", 1, 21,
    //     and 1.
        mnid = GetNumberOption(options, 'minimumIntegerDigits', 1, 21, 1);

    // 24. Set the [[minimumIntegerDigits]] internal property of numberFormat to mnid.
    internal['[[minimumIntegerDigits]]'] = mnid;

    var
    // 25. If s is "currency", then let mnfdDefault be cDigits; else let mnfdDefault
    //     be 0.
        mnfdDefault = s === 'currency' ? cDigits : 0,

    // 26. Let mnfd be the result of calling the GetNumberOption abstract operation
    //     with arguments options, "minimumFractionDigits", 0, 20, and mnfdDefault.
        mnfd = GetNumberOption(options, 'minimumFractionDigits', 0, 20, mnfdDefault);

    // 27. Set the [[minimumFractionDigits]] internal property of numberFormat to mnfd.
    internal['[[minimumFractionDigits]]'] = mnfd;

    var
    // 28. If s is "currency", then let mxfdDefault be max(mnfd, cDigits); else if s
    //     is "percent", then let mxfdDefault be max(mnfd, 0); else let mxfdDefault
    //     be max(mnfd, 3).
        mxfdDefault = s === 'currency' ? Math.max(mnfd, cDigits)
                    : (s === 'percent' ? Math.max(mnfd, 0) : Math.max(mnfd, 3)),

    // 29. Let mxfd be the result of calling the GetNumberOption abstract operation
    //     with arguments options, "maximumFractionDigits", mnfd, 20, and mxfdDefault.
        mxfd = GetNumberOption(options, 'maximumFractionDigits', mnfd, 20, mxfdDefault);

    // 30. Set the [[maximumFractionDigits]] internal property of numberFormat to mxfd.
    internal['[[maximumFractionDigits]]'] = mxfd;

    var
    // 31. Let mnsd be the result of calling the [[Get]] internal method of options
    //     with argument "minimumSignificantDigits".
        mnsd = options.minimumSignificantDigits,

    // 32. Let mxsd be the result of calling the [[Get]] internal method of options
    //     with argument "maximumSignificantDigits".
        mxsd = options.maximumSignificantDigits;

    // 33. If mnsd is not undefined or mxsd is not undefined, then:
    if (mnsd !== undefined || mxsd !== undefined) {
        // a. Let mnsd be the result of calling the GetNumberOption abstract
        //    operation with arguments options, "minimumSignificantDigits", 1, 21,
        //    and 1.
        mnsd = GetNumberOption(options, 'minimumSignificantDigits', 1, 21, 1);

        // b. Let mxsd be the result of calling the GetNumberOption abstract
        //     operation with arguments options, "maximumSignificantDigits", mnsd,
        //     21, and 21.
        mxsd = GetNumberOption(options, 'maximumSignificantDigits', mnsd, 21, 21);

        // c. Set the [[minimumSignificantDigits]] internal property of numberFormat
        //    to mnsd, and the [[maximumSignificantDigits]] internal property of
        //    numberFormat to mxsd.
        internal['[[minimumSignificantDigits]]'] = mnsd;
        internal['[[maximumSignificantDigits]]'] = mxsd;
    }
    var
    // 34. Let g be the result of calling the GetOption abstract operation with the
    //     arguments options, "useGrouping", "boolean", undefined, and true.
        g = GetOption(options, 'useGrouping', 'boolean', undefined, true);

    // 35. Set the [[useGrouping]] internal property of numberFormat to g.
    internal['[[useGrouping]]'] = g;

    var
    // 36. Let dataLocaleData be the result of calling the [[Get]] internal method of
    //     localeData with argument dataLocale.
        dataLocaleData = localeData[dataLocale],

    // 37. Let patterns be the result of calling the [[Get]] internal method of
    //     dataLocaleData with argument "patterns".
        patterns = dataLocaleData.patterns;

    // 38. Assert: patterns is an object (see 11.2.3)

    var
    // 39. Let stylePatterns be the result of calling the [[Get]] internal method of
    //     patterns with argument s.
        stylePatterns = patterns[s];

    // 40. Set the [[positivePattern]] internal property of numberFormat to the
    //     result of calling the [[Get]] internal method of stylePatterns with the
    //     argument "positivePattern".
    internal['[[positivePattern]]'] = stylePatterns.positivePattern;

    // 41. Set the [[negativePattern]] internal property of numberFormat to the
    //     result of calling the [[Get]] internal method of stylePatterns with the
    //     argument "negativePattern".
    internal['[[negativePattern]]'] = stylePatterns.negativePattern;

    // 42. Set the [[boundFormat]] internal property of numberFormat to undefined.
    internal['[[boundFormat]]'] = undefined;

    // 43. Set the [[initializedNumberFormat]] internal property of numberFormat to
    //     true.
    internal['[[initializedNumberFormat]]'] = true;

    // In ES3, we need to pre-bind the format() function
    if (es3)
        numberFormat.format = GetFormatNumber.call(numberFormat);

    // Restore the RegExp properties
    regexpState.exp.test(regexpState.input);

    // Return the newly initialised object
    return numberFormat;
}

function CurrencyDigits(currency) {
    // When the CurrencyDigits abstract operation is called with an argument currency
    // (which must be an upper case String value), the following steps are taken:

    // 1. If the ISO 4217 currency and funds code list contains currency as an
    // alphabetic code, then return the minor unit value corresponding to the
    // currency from the list; else return 2.
    return currencyMinorUnits[currency] !== undefined
                ? currencyMinorUnits[currency]
                : 2;
}

/* 11.2.3 */internals.NumberFormat = {
    '[[availableLocales]]': [],
    '[[relevantExtensionKeys]]': ['nu'],
    '[[localeData]]': {}
};

/**
 * When the supportedLocalesOf method of Intl.NumberFormat is called, the
 * following steps are taken:
 */
/* 11.2.2 */defineProperty(Intl.NumberFormat, 'supportedLocalesOf', {
    configurable: true,
    writable: true,
    value: fnBind.call(supportedLocalesOf, internals.NumberFormat)
});

/**
 * This named accessor property returns a function that formats a number
 * according to the effective locale and the formatting options of this
 * NumberFormat object.
 */
/* 11.3.2 */defineProperty(Intl.NumberFormat.prototype, 'format', {
    configurable: true,
    get: GetFormatNumber
});

function GetFormatNumber() {
        var internal = this != null && typeof this === 'object' && getInternalProperties(this);

        // Satisfy test 11.3_b
        if (!internal || !internal['[[initializedNumberFormat]]'])
            throw new TypeError('`this` value for format() is not an initialized Intl.NumberFormat object.');

        // The value of the [[Get]] attribute is a function that takes the following
        // steps:

        // 1. If the [[boundFormat]] internal property of this NumberFormat object
        //    is undefined, then:
        if (internal['[[boundFormat]]'] === undefined) {
            var
            // a. Let F be a Function object, with internal properties set as
            //    specified for built-in functions in ES5, 15, or successor, and the
            //    length property set to 1, that takes the argument value and
            //    performs the following steps:
                F = function (value) {
                    // i. If value is not provided, then let value be undefined.
                    // ii. Let x be ToNumber(value).
                    // iii. Return the result of calling the FormatNumber abstract
                    //      operation (defined below) with arguments this and x.
                    return FormatNumber(this, /* x = */Number(value));
                },

            // b. Let bind be the standard built-in function object defined in ES5,
            //    15.3.4.5.
            // c. Let bf be the result of calling the [[Call]] internal method of
            //    bind with F as the this value and an argument list containing
            //    the single item this.
                bf = fnBind.call(F, this);

            // d. Set the [[boundFormat]] internal property of this NumberFormat
            //    object to bf.
            internal['[[boundFormat]]'] = bf;
        }
        // Return the value of the [[boundFormat]] internal property of this
        // NumberFormat object.
        return internal['[[boundFormat]]'];
    }

/**
 * When the FormatNumber abstract operation is called with arguments numberFormat
 * (which must be an object initialized as a NumberFormat) and x (which must be a
 * Number value), it returns a String value representing x according to the
 * effective locale and the formatting options of numberFormat.
 */
function FormatNumber (numberFormat, x) {
    var n,

    // Create an object whose props can be used to restore the values of RegExp props
        regexpState = createRegExpRestore(),

        internal = getInternalProperties(numberFormat),
        locale = internal['[[dataLocale]]'],
        nums   = internal['[[numberingSystem]]'],
        data   = internals.NumberFormat['[[localeData]]'][locale],
        ild    = data.symbols[nums] || data.symbols.latn,

    // 1. Let negative be false.
        negative = false;

    // 2. If the result of isFinite(x) is false, then
    if (isFinite(x) === false) {
        // a. If x is NaN, then let n be an ILD String value indicating the NaN value.
        if (isNaN(x))
            n = ild.nan;

        // b. Else
        else {
            // a. Let n be an ILD String value indicating infinity.
            n = ild.infinity;
            // b. If x < 0, then let negative be true.
            if (x < 0)
                negative = true;
        }
    }
    // 3. Else
    else {
        // a. If x < 0, then
        if (x < 0) {
            // i. Let negative be true.
            negative = true;
            // ii. Let x be -x.
            x = -x;
        }

        // b. If the value of the [[style]] internal property of numberFormat is
        //    "percent", let x be 100 × x.
        if (internal['[[style]]'] === 'percent')
            x *= 100;

        // c. If the [[minimumSignificantDigits]] and [[maximumSignificantDigits]]
        //    internal properties of numberFormat are present, then
        if (hop.call(internal, '[[minimumSignificantDigits]]') &&
                hop.call(internal, '[[maximumSignificantDigits]]'))
            // i. Let n be the result of calling the ToRawPrecision abstract operation
            //    (defined below), passing as arguments x and the values of the
            //    [[minimumSignificantDigits]] and [[maximumSignificantDigits]]
            //    internal properties of numberFormat.
            n = ToRawPrecision(x,
                  internal['[[minimumSignificantDigits]]'],
                  internal['[[maximumSignificantDigits]]']);
        // d. Else
        else
            // i. Let n be the result of calling the ToRawFixed abstract operation
            //    (defined below), passing as arguments x and the values of the
            //    [[minimumIntegerDigits]], [[minimumFractionDigits]], and
            //    [[maximumFractionDigits]] internal properties of numberFormat.
            n = ToRawFixed(x,
                  internal['[[minimumIntegerDigits]]'],
                  internal['[[minimumFractionDigits]]'],
                  internal['[[maximumFractionDigits]]']);

        // e. If the value of the [[numberingSystem]] internal property of
        //    numberFormat matches one of the values in the “Numbering System” column
        //    of Table 2 below, then
        if (numSys[nums]) {
            // i. Let digits be an array whose 10 String valued elements are the
            //    UTF-16 string representations of the 10 digits specified in the
            //    “Digits” column of Table 2 in the row containing the value of the
            //    [[numberingSystem]] internal property.
            var digits = numSys[internal['[[numberingSystem]]']];
            // ii. Replace each digit in n with the value of digits[digit].
            n = String(n).replace(/\d/g, function (digit) {
                return digits[digit];
            });
        }
        // f. Else use an implementation dependent algorithm to map n to the
        //    appropriate representation of n in the given numbering system.
        else
            n = String(n); // ###TODO###

        // g. If n contains the character ".", then replace it with an ILND String
        //    representing the decimal separator.
        n = n.replace(/\./g, ild.decimal);

        // h. If the value of the [[useGrouping]] internal property of numberFormat
        //    is true, then insert an ILND String representing a grouping separator
        //    into an ILND set of locations within the integer part of n.
        if (internal['[[useGrouping]]'] === true) {
            var
                parts  = n.split(ild.decimal),
                igr    = parts[0],

                // Primary group represents the group closest to the decimal
                pgSize = data.patterns.primaryGroupSize || 3,

                // Secondary group is every other group
                sgSize = data.patterns.secondaryGroupSize || pgSize;

            // Group only if necessary
            if (igr.length > pgSize) {
                var
                    groups = new List(),

                    // Index of the primary grouping separator
                    end    = igr.length - pgSize,

                    // Starting index for our loop
                    idx    = end % sgSize,

                    start  = igr.slice(0, idx);

                if (start.length)
                    arrPush.call(groups, start);

                // Loop to separate into secondary grouping digits
                while (idx < end) {
                    arrPush.call(groups, igr.slice(idx, idx + sgSize));
                    idx += sgSize;
                }

                // Add the primary grouping digits
                arrPush.call(groups, igr.slice(end));

                parts[0] = arrJoin.call(groups, ild.group);
            }

            n = arrJoin.call(parts, ild.decimal);
        }
    }

    var
    // 4. If negative is true, then let result be the value of the [[negativePattern]]
    //    internal property of numberFormat; else let result be the value of the
    //    [[positivePattern]] internal property of numberFormat.
        result = internal[negative === true ? '[[negativePattern]]' : '[[positivePattern]]'];

    // 5. Replace the substring "{number}" within result with n.
    result = result.replace('{number}', n);

    // 6. If the value of the [[style]] internal property of numberFormat is
    //    "currency", then:
    if (internal['[[style]]'] === 'currency') {
        var cd,
        // a. Let currency be the value of the [[currency]] internal property of
        //    numberFormat.
            currency = internal['[[currency]]'],

        // Shorthand for the currency data
            cData = data.currencies[currency];

        // b. If the value of the [[currencyDisplay]] internal property of
        //    numberFormat is "code", then let cd be currency.
        // c. Else if the value of the [[currencyDisplay]] internal property of
        //    numberFormat is "symbol", then let cd be an ILD string representing
        //    currency in short form. If the implementation does not have such a
        //    representation of currency, then use currency itself.
        // d. Else if the value of the [[currencyDisplay]] internal property of
        //    numberFormat is "name", then let cd be an ILD string representing
        //    currency in long form. If the implementation does not have such a
        //    representation of currency, then use currency itself.
        switch (internal['[[currencyDisplay]]']) {
            case 'symbol':
                cd = cData || currency;
                break;

            default:
            case 'code':
            case 'name':
                cd = currency;
        }

        // e. Replace the substring "{currency}" within result with cd.
        result = result.replace('{currency}', cd);
    }

    // Restore the RegExp properties
    regexpState.exp.test(regexpState.input);

    // 7. Return result.
    return result;
}

/**
 * When the ToRawPrecision abstract operation is called with arguments x (which
 * must be a finite non-negative number), minPrecision, and maxPrecision (both
 * must be integers between 1 and 21) the following steps are taken:
 */
function ToRawPrecision (x, minPrecision, maxPrecision) {
    var
    // 1. Let p be maxPrecision.
        p = maxPrecision;

    // 2. If x = 0, then
    if (x === 0) {
        var
        // a. Let m be the String consisting of p occurrences of the character "0".
            m = arrJoin.call(Array (p + 1), '0'),
        // b. Let e be 0.
            e = 0;
    }
    // 3. Else
    else {
        // a. Let e and n be integers such that 10ᵖ⁻¹ ≤ n < 10ᵖ and for which the
        //    exact mathematical value of n × 10ᵉ⁻ᵖ⁺¹ – x is as close to zero as
        //    possible. If there are two such sets of e and n, pick the e and n for
        //    which n × 10ᵉ⁻ᵖ⁺¹ is larger.
        var
            e = log10Floor(Math.abs(x)),

            // Easier to get to m from here
            f = Math.round(Math.exp((Math.abs(e - p + 1)) * Math.LN10)),

        // b. Let m be the String consisting of the digits of the decimal
        //    representation of n (in order, with no leading zeroes)
            m = String(Math.round(e - p + 1 < 0 ? x * f : x / f));
    }

    // 4. If e ≥ p, then
    if (e >= p)
        // a. Return the concatenation of m and e-p+1 occurrences of the character "0".
        return m + arrJoin.call(Array(e-p+1 + 1), '0');

    // 5. If e = p-1, then
    else if (e === p - 1)
        // a. Return m.
        return m;

    // 6. If e ≥ 0, then
    else if (e >= 0)
        // a. Let m be the concatenation of the first e+1 characters of m, the character
        //    ".", and the remaining p–(e+1) characters of m.
        m = m.slice(0, e + 1) + '.' + m.slice(e + 1);

    // 7. If e < 0, then
    else if (e < 0)
        // a. Let m be the concatenation of the String "0.", –(e+1) occurrences of the
        //    character "0", and the string m.
        m = '0.' + arrJoin.call(Array (-(e+1) + 1), '0') + m;

    // 8. If m contains the character ".", and maxPrecision > minPrecision, then
    if (m.indexOf(".") >= 0 && maxPrecision > minPrecision) {
        var
        // a. Let cut be maxPrecision – minPrecision.
            cut = maxPrecision - minPrecision;

        // b. Repeat while cut > 0 and the last character of m is "0":
        while (cut > 0 && m.charAt(m.length-1) === '0') {
            //  i. Remove the last character from m.
            m = m.slice(0, -1);

            //  ii. Decrease cut by 1.
            cut--;
        }

        // c. If the last character of m is ".", then
        if (m.charAt(m.length-1) === '.')
            //    i. Remove the last character from m.
            m = m.slice(0, -1);
    }
    // 9. Return m.
    return m;
}

/**
 * When the ToRawFixed abstract operation is called with arguments x (which must
 * be a finite non-negative number), minInteger (which must be an integer between
 * 1 and 21), minFraction, and maxFraction (which must be integers between 0 and
 * 20) the following steps are taken:
 */
function ToRawFixed (x, minInteger, minFraction, maxFraction) {
    // (or not because Number.toPrototype.toFixed does a lot of it for us)
    var idx,

        // We can pick up after the fixed formatted string (m) is created
        m   = Number.prototype.toFixed.call(x, maxFraction),

        // 4. If [maxFraction] ≠ 0, then
        //    ...
        //    e. Let int be the number of characters in a.
        //
        // 5. Else let int be the number of characters in m.
        igr = m.split(".")[0].length,  // int is a reserved word

        // 6. Let cut be maxFraction – minFraction.
        cut = maxFraction - minFraction,

        exp = (idx = m.indexOf('e')) > -1 ? m.slice(idx + 1) : 0;

    if (exp) {
        m = m.slice(0, idx).replace('.', '');
        m += arrJoin.call(Array(exp - (m.length - 1) + 1), '0')
          + '.' + arrJoin.call(Array(maxFraction + 1), '0');

        igr = m.length;
    }

    // 7. Repeat while cut > 0 and the last character of m is "0":
    while (cut > 0 && m.slice(-1) === "0") {
        // a. Remove the last character from m.
        m = m.slice(0, -1);

        // b. Decrease cut by 1.
        cut--;
    }

    // 8. If the last character of m is ".", then
    if (m.slice(-1) === ".")
        // a. Remove the last character from m.
        m = m.slice(0, -1);

    // 9. If int < minInteger, then
    if (igr < minInteger)
        // a. Let z be the String consisting of minInteger–int occurrences of the
        //    character "0".
        var z = arrJoin.call(Array(minInteger - igr + 1), '0');

    // 10. Let m be the concatenation of Strings z and m.
    // 11. Return m.
    return (z ? z : '') + m;
}

// Sect 11.3.2 Table 2, Numbering systems
// ======================================
var numSys = {
    arab:    [ '\u0660', '\u0661', '\u0662', '\u0663', '\u0664', '\u0665', '\u0666', '\u0667', '\u0668', '\u0669' ],
    arabext: [ '\u06F0', '\u06F1', '\u06F2', '\u06F3', '\u06F4', '\u06F5', '\u06F6', '\u06F7', '\u06F8', '\u06F9' ],
    bali:    [ '\u1B50', '\u1B51', '\u1B52', '\u1B53', '\u1B54', '\u1B55', '\u1B56', '\u1B57', '\u1B58', '\u1B59' ],
    beng:    [ '\u09E6', '\u09E7', '\u09E8', '\u09E9', '\u09EA', '\u09EB', '\u09EC', '\u09ED', '\u09EE', '\u09EF' ],
    deva:    [ '\u0966', '\u0967', '\u0968', '\u0969', '\u096A', '\u096B', '\u096C', '\u096D', '\u096E', '\u096F' ],
    fullwide:[ '\uFF10', '\uFF11', '\uFF12', '\uFF13', '\uFF14', '\uFF15', '\uFF16', '\uFF17', '\uFF18', '\uFF19' ],
    gujr:    [ '\u0AE6', '\u0AE7', '\u0AE8', '\u0AE9', '\u0AEA', '\u0AEB', '\u0AEC', '\u0AED', '\u0AEE', '\u0AEF' ],
    guru:    [ '\u0A66', '\u0A67', '\u0A68', '\u0A69', '\u0A6A', '\u0A6B', '\u0A6C', '\u0A6D', '\u0A6E', '\u0A6F' ],
    hanidec: [ '\u3007', '\u4E00', '\u4E8C', '\u4E09', '\u56DB', '\u4E94', '\u516D', '\u4E03', '\u516B', '\u4E5D' ],
    khmr:    [ '\u17E0', '\u17E1', '\u17E2', '\u17E3', '\u17E4', '\u17E5', '\u17E6', '\u17E7', '\u17E8', '\u17E9' ],
    knda:    [ '\u0CE6', '\u0CE7', '\u0CE8', '\u0CE9', '\u0CEA', '\u0CEB', '\u0CEC', '\u0CED', '\u0CEE', '\u0CEF' ],
    laoo:    [ '\u0ED0', '\u0ED1', '\u0ED2', '\u0ED3', '\u0ED4', '\u0ED5', '\u0ED6', '\u0ED7', '\u0ED8', '\u0ED9' ],
    latn:    [ '\u0030', '\u0031', '\u0032', '\u0033', '\u0034', '\u0035', '\u0036', '\u0037', '\u0038', '\u0039' ],
    limb:    [ '\u1946', '\u1947', '\u1948', '\u1949', '\u194A', '\u194B', '\u194C', '\u194D', '\u194E', '\u194F' ],
    mlym:    [ '\u0D66', '\u0D67', '\u0D68', '\u0D69', '\u0D6A', '\u0D6B', '\u0D6C', '\u0D6D', '\u0D6E', '\u0D6F' ],
    mong:    [ '\u1810', '\u1811', '\u1812', '\u1813', '\u1814', '\u1815', '\u1816', '\u1817', '\u1818', '\u1819' ],
    mymr:    [ '\u1040', '\u1041', '\u1042', '\u1043', '\u1044', '\u1045', '\u1046', '\u1047', '\u1048', '\u1049' ],
    orya:    [ '\u0B66', '\u0B67', '\u0B68', '\u0B69', '\u0B6A', '\u0B6B', '\u0B6C', '\u0B6D', '\u0B6E', '\u0B6F' ],
    tamldec: [ '\u0BE6', '\u0BE7', '\u0BE8', '\u0BE9', '\u0BEA', '\u0BEB', '\u0BEC', '\u0BED', '\u0BEE', '\u0BEF' ],
    telu:    [ '\u0C66', '\u0C67', '\u0C68', '\u0C69', '\u0C6A', '\u0C6B', '\u0C6C', '\u0C6D', '\u0C6E', '\u0C6F' ],
    thai:    [ '\u0E50', '\u0E51', '\u0E52', '\u0E53', '\u0E54', '\u0E55', '\u0E56', '\u0E57', '\u0E58', '\u0E59' ],
    tibt:    [ '\u0F20', '\u0F21', '\u0F22', '\u0F23', '\u0F24', '\u0F25', '\u0F26', '\u0F27', '\u0F28', '\u0F29' ]
};

/**
 * This function provides access to the locale and formatting options computed
 * during initialization of the object.
 *
 * The function returns a new object whose properties and attributes are set as
 * if constructed by an object literal assigning to each of the following
 * properties the value of the corresponding internal property of this
 * NumberFormat object (see 11.4): locale, numberingSystem, style, currency,
 * currencyDisplay, minimumIntegerDigits, minimumFractionDigits,
 * maximumFractionDigits, minimumSignificantDigits, maximumSignificantDigits, and
 * useGrouping. Properties whose corresponding internal properties are not present
 * are not assigned.
 */
/* 11.3.3 */defineProperty(Intl.NumberFormat.prototype, 'resolvedOptions', {
    configurable: true,
    writable: true,
    value: function () {
        var prop,
            descs = new Record(),
            props = [
                'locale', 'numberingSystem', 'style', 'currency', 'currencyDisplay',
                'minimumIntegerDigits', 'minimumFractionDigits', 'maximumFractionDigits',
                'minimumSignificantDigits', 'maximumSignificantDigits', 'useGrouping'
            ],
            internal = this != null && typeof this === 'object' && getInternalProperties(this);

        // Satisfy test 11.3_b
        if (!internal || !internal['[[initializedNumberFormat]]'])
            throw new TypeError('`this` value for resolvedOptions() is not an initialized Intl.NumberFormat object.');

        for (var i = 0, max = props.length; i < max; i++) {
            if (hop.call(internal, prop = '[['+ props[i] +']]'))
                descs[props[i]] = { value: internal[prop], writable: true, configurable: true, enumerable: true };
        }

        return objCreate({}, descs);
    }
});

// 12.1 The Intl.DateTimeFormat constructor
// ==================================

// Define the DateTimeFormat constructor internally so it cannot be tainted
function DateTimeFormatConstructor () {
    var locales = arguments[0];
    var options = arguments[1];

    if (!this || this === Intl) {
        return new Intl.DateTimeFormat(locales, options);
    }
    return InitializeDateTimeFormat(toObject(this), locales, options);
}

defineProperty(Intl, 'DateTimeFormat', {
    configurable: true,
    writable: true,
    value: DateTimeFormatConstructor
});

// Must explicitly set prototypes as unwritable
defineProperty(DateTimeFormatConstructor, 'prototype', {
    writable: false
});

/**
 * The abstract operation InitializeDateTimeFormat accepts the arguments dateTimeFormat
 * (which must be an object), locales, and options. It initializes dateTimeFormat as a
 * DateTimeFormat object.
 */
function/* 12.1.1.1 */InitializeDateTimeFormat (dateTimeFormat, locales, options) {
    var
    // This will be a internal properties object if we're not already initialized
        internal = getInternalProperties(dateTimeFormat),

    // Create an object whose props can be used to restore the values of RegExp props
        regexpState = createRegExpRestore();

    // 1. If dateTimeFormat has an [[initializedIntlObject]] internal property with
    //    value true, throw a TypeError exception.
    if (internal['[[initializedIntlObject]]'] === true)
        throw new TypeError('`this` object has already been initialized as an Intl object');

    // Need this to access the `internal` object
    defineProperty(dateTimeFormat, '__getInternalProperties', {
        value: function () {
            // NOTE: Non-standard, for internal use only
            if (arguments[0] === secret)
                return internal;
        }
    });

    // 2. Set the [[initializedIntlObject]] internal property of numberFormat to true.
    internal['[[initializedIntlObject]]'] = true;

    var
    // 3. Let requestedLocales be the result of calling the CanonicalizeLocaleList
    //    abstract operation (defined in 9.2.1) with argument locales.
        requestedLocales = CanonicalizeLocaleList(locales),

    // 4. Let options be the result of calling the ToDateTimeOptions abstract
    //    operation (defined below) with arguments options, "any", and "date".
        options = ToDateTimeOptions(options, 'any', 'date'),

    // 5. Let opt be a new Record.
        opt = new Record();

    // 6. Let matcher be the result of calling the GetOption abstract operation
    //    (defined in 9.2.9) with arguments options, "localeMatcher", "string", a List
    //    containing the two String values "lookup" and "best fit", and "best fit".
        matcher = GetOption(options, 'localeMatcher', 'string', new List('lookup', 'best fit'), 'best fit');

    // 7. Set opt.[[localeMatcher]] to matcher.
    opt['[[localeMatcher]]'] = matcher;

    var
    // 8. Let DateTimeFormat be the standard built-in object that is the initial
    //    value of Intl.DateTimeFormat.
        DateTimeFormat = internals.DateTimeFormat, // This is what we *really* need

    // 9. Let localeData be the value of the [[localeData]] internal property of
    //    DateTimeFormat.
        localeData = DateTimeFormat['[[localeData]]'],

    // 10. Let r be the result of calling the ResolveLocale abstract operation
    //     (defined in 9.2.5) with the [[availableLocales]] internal property of
    //      DateTimeFormat, requestedLocales, opt, the [[relevantExtensionKeys]]
    //      internal property of DateTimeFormat, and localeData.
        r = ResolveLocale(DateTimeFormat['[[availableLocales]]'], requestedLocales,
                opt, DateTimeFormat['[[relevantExtensionKeys]]'], localeData);

    // 11. Set the [[locale]] internal property of dateTimeFormat to the value of
    //     r.[[locale]].
    internal['[[locale]]'] = r['[[locale]]'];

    // 12. Set the [[calendar]] internal property of dateTimeFormat to the value of
    //     r.[[ca]].
    internal['[[calendar]]'] = r['[[ca]]'];

    // 13. Set the [[numberingSystem]] internal property of dateTimeFormat to the value of
    //     r.[[nu]].
    internal['[[numberingSystem]]'] = r['[[nu]]'];

    // The specification doesn't tell us to do this, but it's helpful later on
    internal['[[dataLocale]]'] = r['[[dataLocale]]'];

    var
    // 14. Let dataLocale be the value of r.[[dataLocale]].
        dataLocale = r['[[dataLocale]]'],

    // 15. Let tz be the result of calling the [[Get]] internal method of options with
    //     argument "timeZone".
        tz = options.timeZone;

    // 16. If tz is not undefined, then
    if (tz !== undefined) {
        // a. Let tz be ToString(tz).
        // b. Convert tz to upper case as described in 6.1.
        //    NOTE: If an implementation accepts additional time zone values, as permitted
        //          under certain conditions by the Conformance clause, different casing
        //          rules apply.
        tz = toLatinUpperCase(tz);

        // c. If tz is not "UTC", then throw a RangeError exception.
        // ###TODO: accept more time zones###
        if (tz !== 'UTC')
            throw new RangeError('timeZone is not supported.');
    }

    // 17. Set the [[timeZone]] internal property of dateTimeFormat to tz.
    internal['[[timeZone]]'] = tz;

    // 18. Let opt be a new Record.
    opt = new Record();

    // 19. For each row of Table 3, except the header row, do:
    for (var prop in dateTimeComponents) {
        if (!hop.call(dateTimeComponents, prop))
            continue;

        var
        // 20. Let prop be the name given in the Property column of the row.
        // 21. Let value be the result of calling the GetOption abstract operation,
        //     passing as argument options, the name given in the Property column of the
        //     row, "string", a List containing the strings given in the Values column of
        //     the row, and undefined.
            value = GetOption(options, prop, 'string', dateTimeComponents[prop]);

        // 22. Set opt.[[<prop>]] to value.
        opt['[['+prop+']]'] = value;
    }

    var
        // Assigned a value below
        bestFormat,

        // 23. Let dataLocaleData be the result of calling the [[Get]] internal method of
        //     localeData with argument dataLocale.
        dataLocaleData = localeData[dataLocale],

        // 24. Let formats be the result of calling the [[Get]] internal method of
        //     dataLocaleData with argument "formats".
        formats = dataLocaleData.formats,
        // 25. Let matcher be the result of calling the GetOption abstract operation with
        //     arguments options, "formatMatcher", "string", a List containing the two String
        //     values "basic" and "best fit", and "best fit".
        matcher = GetOption(options, 'formatMatcher', 'string', new List('basic', 'best fit'), 'best fit');

    // 26. If matcher is "basic", then
    if (matcher === 'basic')
        // 27. Let bestFormat be the result of calling the BasicFormatMatcher abstract
        //     operation (defined below) with opt and formats.
        bestFormat = BasicFormatMatcher(opt, formats);

    // 28. Else
    else
        // 29. Let bestFormat be the result of calling the BestFitFormatMatcher
        //     abstract operation (defined below) with opt and formats.
        bestFormat = BestFitFormatMatcher(opt, formats);

    // 30. For each row in Table 3, except the header row, do
    for (var prop in dateTimeComponents) {
        if (!hop.call(dateTimeComponents, prop))
            continue;

        // a. Let prop be the name given in the Property column of the row.
        // b. Let pDesc be the result of calling the [[GetOwnProperty]] internal method of
        //    bestFormat with argument prop.
        // c. If pDesc is not undefined, then
        if (hop.call(bestFormat, prop)) {
            var
            // i. Let p be the result of calling the [[Get]] internal method of bestFormat
            //    with argument prop.
                p = bestFormat[prop];

            // ii. Set the [[<prop>]] internal property of dateTimeFormat to p.
            internal['[['+prop+']]'] = p;
        }
    }

    var
        // Assigned a value below
        pattern,

    // 31. Let hr12 be the result of calling the GetOption abstract operation with
    //     arguments options, "hour12", "boolean", undefined, and undefined.
        hr12 = GetOption(options, 'hour12', 'boolean'/*, undefined, undefined*/);

    // 32. If dateTimeFormat has an internal property [[hour]], then
    if (internal['[[hour]]']) {
        // a. If hr12 is undefined, then let hr12 be the result of calling the [[Get]]
        //    internal method of dataLocaleData with argument "hour12".
        hr12 = hr12 === undefined ? dataLocaleData.hour12 : hr12;

        // b. Set the [[hour12]] internal property of dateTimeFormat to hr12.
        internal['[[hour12]]'] = hr12;

        // c. If hr12 is true, then
        if (hr12 === true) {
            var
            // i. Let hourNo0 be the result of calling the [[Get]] internal method of
            //    dataLocaleData with argument "hourNo0".
                hourNo0 = dataLocaleData.hourNo0;

            // ii. Set the [[hourNo0]] internal property of dateTimeFormat to hourNo0.
            internal['[[hourNo0]]'] = hourNo0;

            // iii. Let pattern be the result of calling the [[Get]] internal method of
            //      bestFormat with argument "pattern12".
            pattern = bestFormat.pattern12;
        }

        // d. Else
        else
            // i. Let pattern be the result of calling the [[Get]] internal method of
            //    bestFormat with argument "pattern".
            pattern = bestFormat.pattern;
    }

    // 33. Else
    else
        // a. Let pattern be the result of calling the [[Get]] internal method of
        //    bestFormat with argument "pattern".
        pattern = bestFormat.pattern;

    // 34. Set the [[pattern]] internal property of dateTimeFormat to pattern.
    internal['[[pattern]]'] = pattern;

    // 35. Set the [[boundFormat]] internal property of dateTimeFormat to undefined.
    internal['[[boundFormat]]'] = undefined;

    // 36. Set the [[initializedDateTimeFormat]] internal property of dateTimeFormat to
    //     true.
    internal['[[initializedDateTimeFormat]]'] = true;

    // In ES3, we need to pre-bind the format() function
    if (es3)
        dateTimeFormat.format = GetFormatDateTime.call(dateTimeFormat);

    // Restore the RegExp properties
    regexpState.exp.test(regexpState.input);

    // Return the newly initialised object
    return dateTimeFormat;
}

/**
 * Several DateTimeFormat algorithms use values from the following table, which provides
 * property names and allowable values for the components of date and time formats:
 */
var dateTimeComponents = {
         weekday: [ "narrow", "short", "long" ],
             era: [ "narrow", "short", "long" ],
            year: [ "2-digit", "numeric" ],
           month: [ "2-digit", "numeric", "narrow", "short", "long" ],
             day: [ "2-digit", "numeric" ],
            hour: [ "2-digit", "numeric" ],
          minute: [ "2-digit", "numeric" ],
          second: [ "2-digit", "numeric" ],
    timeZoneName: [ "short", "long" ]
};

/**
 * When the ToDateTimeOptions abstract operation is called with arguments options,
 * required, and defaults, the following steps are taken:
 */
function ToDateTimeOptions (options, required, defaults) {
    // 1. If options is undefined, then let options be null, else let options be
    //    ToObject(options).
    if (options === undefined)
        options = null;

    else {
        // (#12) options needs to be a Record, but it also needs to inherit properties
        var opt2 = toObject(options);
        options = new Record();

        for (var k in opt2)
            options[k] = opt2[k];
    }

    var
    // 2. Let create be the standard built-in function object defined in ES5, 15.2.3.5.
        create = objCreate,

    // 3. Let options be the result of calling the [[Call]] internal method of create with
    //    undefined as the this value and an argument list containing the single item
    //    options.
        options = create(options),

    // 4. Let needDefaults be true.
        needDefaults = true;

    // 5. If required is "date" or "any", then
    if (required === 'date' || required === 'any') {
        // a. For each of the property names "weekday", "year", "month", "day":
            // i. If the result of calling the [[Get]] internal method of options with the
            //    property name is not undefined, then let needDefaults be false.
        if (options.weekday !== undefined || options.year !== undefined
                || options.month !== undefined || options.day !== undefined)
            needDefaults = false;
    }

    // 6. If required is "time" or "any", then
    if (required === 'time' || required === 'any') {
        // a. For each of the property names "hour", "minute", "second":
            // i. If the result of calling the [[Get]] internal method of options with the
            //    property name is not undefined, then let needDefaults be false.
        if (options.hour !== undefined || options.minute !== undefined || options.second !== undefined)
                needDefaults = false;
    }

    // 7. If needDefaults is true and defaults is either "date" or "all", then
    if (needDefaults && (defaults === 'date' || defaults === 'all'))
        // a. For each of the property names "year", "month", "day":
            // i. Call the [[DefineOwnProperty]] internal method of options with the
            //    property name, Property Descriptor {[[Value]]: "numeric", [[Writable]]:
            //    true, [[Enumerable]]: true, [[Configurable]]: true}, and false.
        options.year = options.month = options.day = 'numeric';

    // 8. If needDefaults is true and defaults is either "time" or "all", then
    if (needDefaults && (defaults === 'time' || defaults === 'all'))
        // a. For each of the property names "hour", "minute", "second":
            // i. Call the [[DefineOwnProperty]] internal method of options with the
            //    property name, Property Descriptor {[[Value]]: "numeric", [[Writable]]:
            //    true, [[Enumerable]]: true, [[Configurable]]: true}, and false.
        options.hour = options.minute = options.second = 'numeric';

    // 9. Return options.
    return options;
}

/**
 * When the BasicFormatMatcher abstract operation is called with two arguments options and
 * formats, the following steps are taken:
 */
function BasicFormatMatcher (options, formats) {
    return calculateScore(options, formats);
}

/**
 * Calculates score for BestFitFormatMatcher and BasicFormatMatcher.
 * Abstracted from BasicFormatMatcher section.
 */
function calculateScore (options, formats, bestFit) {
    var
    // Additional penalty type when bestFit === true
       diffDataTypePenalty = 8,

    // 1. Let removalPenalty be 120.
        removalPenalty = 120,

    // 2. Let additionPenalty be 20.
        additionPenalty = 20,

    // 3. Let longLessPenalty be 8.
        longLessPenalty = 8,

    // 4. Let longMorePenalty be 6.
        longMorePenalty = 6,

    // 5. Let shortLessPenalty be 6.
        shortLessPenalty = 6,

    // 6. Let shortMorePenalty be 3.
        shortMorePenalty = 3,

    // 7. Let bestScore be -Infinity.
        bestScore = -Infinity,

    // 8. Let bestFormat be undefined.
        bestFormat,

    // 9. Let i be 0.
        i = 0,

    // 10. Let len be the result of calling the [[Get]] internal method of formats with argument "length".
        len = formats.length;

    // 11. Repeat while i < len:
    while (i < len) {
        var
        // a. Let format be the result of calling the [[Get]] internal method of formats with argument ToString(i).
            format = formats[i],

        // b. Let score be 0.
            score = 0;

        // c. For each property shown in Table 3:
        for (var property in dateTimeComponents) {
            if (!hop.call(dateTimeComponents, property))
                continue;

            var
            // i. Let optionsProp be options.[[<property>]].
                optionsProp = options['[['+ property +']]'],

            // ii. Let formatPropDesc be the result of calling the [[GetOwnProperty]] internal method of format
            //     with argument property.
            // iii. If formatPropDesc is not undefined, then
            //     1. Let formatProp be the result of calling the [[Get]] internal method of format with argument property.
                formatProp = hop.call(format, property) ? format[property] : undefined;

            // iv. If optionsProp is undefined and formatProp is not undefined, then decrease score by
            //     additionPenalty.
            if (optionsProp === undefined && formatProp !== undefined)
                score -= additionPenalty;

            // v. Else if optionsProp is not undefined and formatProp is undefined, then decrease score by
            //    removalPenalty.
            else if (optionsProp !== undefined && formatProp === undefined)
                score -= removalPenalty;

            // vi. Else
            else {
                var
                // 1. Let values be the array ["2-digit", "numeric", "narrow", "short",
                //    "long"].
                    values = [ '2-digit', 'numeric', 'narrow', 'short', 'long' ],

                // 2. Let optionsPropIndex be the index of optionsProp within values.
                    optionsPropIndex = arrIndexOf.call(values, optionsProp),

                // 3. Let formatPropIndex be the index of formatProp within values.
                    formatPropIndex = arrIndexOf.call(values, formatProp),

                // 4. Let delta be max(min(formatPropIndex - optionsPropIndex, 2), -2).
                    delta = Math.max(Math.min(formatPropIndex - optionsPropIndex, 2), -2);

                // When the bestFit argument is true, subtract additional penalty where data types are not the same
                if (bestFit && (
                    ((optionsProp === 'numeric' || optionsProp === '2-digit') && (formatProp !== 'numeric' && formatProp !== '2-digit'))
                 || ((optionsProp !== 'numeric' && optionsProp !== '2-digit') && (formatProp === '2-digit' || formatProp === 'numeric'))
                ))
                    score -= diffDataTypePenalty;

                // 5. If delta = 2, decrease score by longMorePenalty.
                if (delta === 2)
                    score -= longMorePenalty;

                // 6. Else if delta = 1, decrease score by shortMorePenalty.
                else if (delta === 1)
                    score -= shortMorePenalty;

                // 7. Else if delta = -1, decrease score by shortLessPenalty.
                else if (delta === -1)
                    score -= shortLessPenalty;

                // 8. Else if delta = -2, decrease score by longLessPenalty.
                else if (delta === -2)
                    score -= longLessPenalty;
            }
        }

        // d. If score > bestScore, then
        if (score > bestScore) {
            // i. Let bestScore be score.
            bestScore = score;

            // ii. Let bestFormat be format.
            bestFormat = format;
        }

        // e. Increase i by 1.
        i++;
    }

    // 12. Return bestFormat.
    return bestFormat;
}

/**
 * When the BestFitFormatMatcher abstract operation is called with two arguments options
 * and formats, it performs implementation dependent steps, which should return a set of
 * component representations that a typical user of the selected locale would perceive as
 * at least as good as the one returned by BasicFormatMatcher.
 *
 * This polyfill defines the algorithm to be the same as BasicFormatMatcher,
 * with the addition of bonus points awarded where the requested format is of
 * the same data type as the potentially matching format.
 *
 * For example,
 *
 *     { month: 'numeric', day: 'numeric' }
 *
 * should match
 *
 *     { month: '2-digit', day: '2-digit' }
 *
 * rather than
 *
 *     { month: 'short', day: 'numeric' }
 *
 * This makes sense because a user requesting a formatted date with numeric parts would
 * not expect to see the returned format containing narrow, short or long part names
 */
function BestFitFormatMatcher (options, formats) {
    return calculateScore(options, formats, true);
}

/* 12.2.3 */internals.DateTimeFormat = {
    '[[availableLocales]]': [],
    '[[relevantExtensionKeys]]': ['ca', 'nu'],
    '[[localeData]]': {}
};

/**
 * When the supportedLocalesOf method of Intl.DateTimeFormat is called, the
 * following steps are taken:
 */
/* 12.2.2 */defineProperty(Intl.DateTimeFormat, 'supportedLocalesOf', {
    configurable: true,
    writable: true,
    value: fnBind.call(supportedLocalesOf, internals.DateTimeFormat)
});

/**
 * This named accessor property returns a function that formats a number
 * according to the effective locale and the formatting options of this
 * DateTimeFormat object.
 */
/* 12.3.2 */defineProperty(Intl.DateTimeFormat.prototype, 'format', {
    configurable: true,
    get: GetFormatDateTime
});

function GetFormatDateTime() {
    var internal = this != null && typeof this === 'object' && getInternalProperties(this);

    // Satisfy test 12.3_b
    if (!internal || !internal['[[initializedDateTimeFormat]]'])
        throw new TypeError('`this` value for format() is not an initialized Intl.DateTimeFormat object.');

    // The value of the [[Get]] attribute is a function that takes the following
    // steps:

    // 1. If the [[boundFormat]] internal property of this DateTimeFormat object
    //    is undefined, then:
    if (internal['[[boundFormat]]'] === undefined) {
        var
        // a. Let F be a Function object, with internal properties set as
        //    specified for built-in functions in ES5, 15, or successor, and the
        //    length property set to 0, that takes the argument date and
        //    performs the following steps:
            F = function () {
                //   i. If date is not provided or is undefined, then let x be the
                //      result as if by the expression Date.now() where Date.now is
                //      the standard built-in function defined in ES5, 15.9.4.4.
                //  ii. Else let x be ToNumber(date).
                // iii. Return the result of calling the FormatDateTime abstract
                //      operation (defined below) with arguments this and x.
                var x = Number(arguments.length === 0 ? Date.now() : arguments[0]);
                return FormatDateTime(this, x);
            },
        // b. Let bind be the standard built-in function object defined in ES5,
        //    15.3.4.5.
        // c. Let bf be the result of calling the [[Call]] internal method of
        //    bind with F as the this value and an argument list containing
        //    the single item this.
            bf = fnBind.call(F, this);
        // d. Set the [[boundFormat]] internal property of this NumberFormat
        //    object to bf.
        internal['[[boundFormat]]'] = bf;
    }
    // Return the value of the [[boundFormat]] internal property of this
    // NumberFormat object.
    return internal['[[boundFormat]]'];
}

/**
 * When the FormatDateTime abstract operation is called with arguments dateTimeFormat
 * (which must be an object initialized as a DateTimeFormat) and x (which must be a Number
 * value), it returns a String value representing x (interpreted as a time value as
 * specified in ES5, 15.9.1.1) according to the effective locale and the formatting
 * options of dateTimeFormat.
 */
function FormatDateTime(dateTimeFormat, x) {
    // 1. If x is not a finite Number, then throw a RangeError exception.
    if (!isFinite(x))
        throw new RangeError('Invalid valid date passed to format');

    var
        internal = dateTimeFormat.__getInternalProperties(secret),

    // Creating restore point for properties on the RegExp object... please wait
        regexpState = createRegExpRestore(),

    // 2. Let locale be the value of the [[locale]] internal property of dateTimeFormat.
        locale = internal['[[locale]]'],

    // 3. Let nf be the result of creating a new NumberFormat object as if by the
    // expression new Intl.NumberFormat([locale], {useGrouping: false}) where
    // Intl.NumberFormat is the standard built-in constructor defined in 11.1.3.
        nf = new Intl.NumberFormat([locale], {useGrouping: false}),

    // 4. Let nf2 be the result of creating a new NumberFormat object as if by the
    // expression new Intl.NumberFormat([locale], {minimumIntegerDigits: 2, useGrouping:
    // false}) where Intl.NumberFormat is the standard built-in constructor defined in
    // 11.1.3.
        nf2 = new Intl.NumberFormat([locale], {minimumIntegerDigits: 2, useGrouping: false}),

    // 5. Let tm be the result of calling the ToLocalTime abstract operation (defined
    // below) with x, the value of the [[calendar]] internal property of dateTimeFormat,
    // and the value of the [[timeZone]] internal property of dateTimeFormat.
        tm = ToLocalTime(x, internal['[[calendar]]'], internal['[[timeZone]]']),

    // 6. Let result be the value of the [[pattern]] internal property of dateTimeFormat.
        result = internal['[[pattern]]'],

    // Need the locale minus any extensions
        dataLocale = internal['[[dataLocale]]'],

    // Need the calendar data from CLDR
        localeData = internals.DateTimeFormat['[[localeData]]'][dataLocale].calendars,
        ca = internal['[[calendar]]'];

    // 7. For each row of Table 3, except the header row, do:
    for (var p in dateTimeComponents) {
        // a. If dateTimeFormat has an internal property with the name given in the
        //    Property column of the row, then:
        if (hop.call(internal, '[['+ p +']]')) {
            var
            // Assigned values below
                pm, fv,

            //   i. Let p be the name given in the Property column of the row.
            //  ii. Let f be the value of the [[<p>]] internal property of dateTimeFormat.
                f = internal['[['+ p +']]'],

            // iii. Let v be the value of tm.[[<p>]].
                v = tm['[['+ p +']]'];

            //  iv. If p is "year" and v ≤ 0, then let v be 1 - v.
            if (p === 'year' && v <= 0)
                v = 1 - v;

            //   v. If p is "month", then increase v by 1.
            else if (p === 'month')
                v++;

            //  vi. If p is "hour" and the value of the [[hour12]] internal property of
            //      dateTimeFormat is true, then
            else if (p === 'hour' && internal['[[hour12]]'] === true) {
                // 1. Let v be v modulo 12.
                v = v % 12;

                // 2. If v is equal to the value of tm.[[<p>]], then let pm be false; else
                //    let pm be true.
                pm = v !== tm['[['+ p +']]'];

                // 3. If v is 0 and the value of the [[hourNo0]] internal property of
                //    dateTimeFormat is true, then let v be 12.
                if (v === 0 && internal['[[hourNo0]]'] === true)
                    v = 12;
            }

            // vii. If f is "numeric", then
            if (f === 'numeric')
                // 1. Let fv be the result of calling the FormatNumber abstract operation
                //    (defined in 11.3.2) with arguments nf and v.
                fv = FormatNumber(nf, v);

            // viii. Else if f is "2-digit", then
            else if (f === '2-digit') {
                // 1. Let fv be the result of calling the FormatNumber abstract operation
                //    with arguments nf2 and v.
                fv = FormatNumber(nf2, v);

                // 2. If the length of fv is greater than 2, let fv be the substring of fv
                //    containing the last two characters.
                if (fv.length > 2)
                    fv = fv.slice(-2);
            }

            // ix. Else if f is "narrow", "short", or "long", then let fv be a String
            //     value representing f in the desired form; the String value depends upon
            //     the implementation and the effective locale and calendar of
            //     dateTimeFormat. If p is "month", then the String value may also depend
            //     on whether dateTimeFormat has a [[day]] internal property. If p is
            //     "timeZoneName", then the String value may also depend on the value of
            //     the [[inDST]] field of tm.
            else if (f in dateWidths) {
                switch (p) {
                    case 'month':
                        fv = resolveDateString(localeData, ca, 'months', f, tm['[['+ p +']]']);
                        break;

                    case 'weekday':
                        try {
                            fv = resolveDateString(localeData, ca, 'days', f, tm['[['+ p +']]']);
                            // fv = resolveDateString(ca.days, f)[tm['[['+ p +']]']];
                        } catch (e) {
                            throw new Error('Could not find weekday data for locale '+locale);
                        }
                        break;

                    case 'timeZoneName':
                        fv = ''; // TODO
                        break;

                    // TODO: Era
                    default:
                        fv = tm['[['+ p +']]'];
                }
            }

            // x. Replace the substring of result that consists of "{", p, and "}", with
            //    fv.
            result = result.replace('{'+ p +'}', fv);
        }
    }
    // 8. If dateTimeFormat has an internal property [[hour12]] whose value is true, then
    if (internal['[[hour12]]'] === true) {
        // a. If pm is true, then let fv be an implementation and locale dependent String
        //    value representing “post meridiem”; else let fv be an implementation and
        //    locale dependent String value representing “ante meridiem”.
        fv = resolveDateString(localeData, ca, 'dayPeriods', pm ? 'pm' : 'am');

        // b. Replace the substring of result that consists of "{ampm}", with fv.
        result = result.replace('{ampm}', fv);
    }

    // Restore properties of the RegExp object
    regexpState.exp.test(regexpState.input);

    // 9. Return result.
    return result;
}

/**
 * When the ToLocalTime abstract operation is called with arguments date, calendar, and
 * timeZone, the following steps are taken:
 */
function ToLocalTime(date, calendar, timeZone) {
    // 1. Apply calendrical calculations on date for the given calendar and time zone to
    //    produce weekday, era, year, month, day, hour, minute, second, and inDST values.
    //    The calculations should use best available information about the specified
    //    calendar and time zone. If the calendar is "gregory", then the calculations must
    //    match the algorithms specified in ES5, 15.9.1, except that calculations are not
    //    bound by the restrictions on the use of best available information on time zones
    //    for local time zone adjustment and daylight saving time adjustment imposed by
    //    ES5, 15.9.1.7 and 15.9.1.8.
    // ###TODO###
    var d = new Date(date),
        m = 'get' + (timeZone || '');

    // 2. Return a Record with fields [[weekday]], [[era]], [[year]], [[month]], [[day]],
    //    [[hour]], [[minute]], [[second]], and [[inDST]], each with the corresponding
    //    calculated value.
    return new Record({
        '[[weekday]]': d[m + 'Day'](),
        '[[era]]'    : +(d[m + 'FullYear']() >= 0),
        '[[year]]'   : d[m + 'FullYear'](),
        '[[month]]'  : d[m + 'Month'](),
        '[[day]]'    : d[m + 'Date'](),
        '[[hour]]'   : d[m + 'Hours'](),
        '[[minute]]' : d[m + 'Minutes'](),
        '[[second]]' : d[m + 'Seconds'](),
        '[[inDST]]'  : false // ###TODO###
    });
}

/**
 * The function returns a new object whose properties and attributes are set as if
 * constructed by an object literal assigning to each of the following properties the
 * value of the corresponding internal property of this DateTimeFormat object (see 12.4):
 * locale, calendar, numberingSystem, timeZone, hour12, weekday, era, year, month, day,
 * hour, minute, second, and timeZoneName. Properties whose corresponding internal
 * properties are not present are not assigned.
 */
/* 12.3.3 */defineProperty(Intl.DateTimeFormat.prototype, 'resolvedOptions', {
    writable: true,
    configurable: true,
    value: function () {
        var prop,
            descs = new Record(),
            props = [
                'locale', 'calendar', 'numberingSystem', 'timeZone', 'hour12', 'weekday',
                'era', 'year', 'month', 'day', 'hour', 'minute', 'second', 'timeZoneName'
            ],
            internal = this != null && typeof this === 'object' && getInternalProperties(this);

        // Satisfy test 12.3_b
        if (!internal || !internal['[[initializedDateTimeFormat]]'])
            throw new TypeError('`this` value for resolvedOptions() is not an initialized Intl.DateTimeFormat object.');

        for (var i = 0, max = props.length; i < max; i++) {
            if (hop.call(internal, prop = '[[' + props[i] + ']]'))
                descs[props[i]] = { value: internal[prop], writable: true, configurable: true, enumerable: true };
        }

        return objCreate({}, descs);
    }
});

// Sect 13 Locale Sensitive Functions of the ECMAScript Language Specification
// ===========================================================================

var ls = Intl.__localeSensitiveProtos = {
    Number: {},
    Date:   {}
};

/**
 * When the toLocaleString method is called with optional arguments locales and options,
 * the following steps are taken:
 */
/* 13.2.1 */ls.Number.toLocaleString = function () {
    // Satisfy test 13.2.1_1
    if (Object.prototype.toString.call(this) !== '[object Number]')
        throw new TypeError('`this` value must be a number for Number.prototype.toLocaleString()');

    // 1. Let x be this Number value (as defined in ES5, 15.7.4).
    // 2. If locales is not provided, then let locales be undefined.
    // 3. If options is not provided, then let options be undefined.
    // 4. Let numberFormat be the result of creating a new object as if by the
    //    expression new Intl.NumberFormat(locales, options) where
    //    Intl.NumberFormat is the standard built-in constructor defined in 11.1.3.
    // 5. Return the result of calling the FormatNumber abstract operation
    //    (defined in 11.3.2) with arguments numberFormat and x.
    return FormatNumber(new NumberFormatConstructor(arguments[0], arguments[1]), this);
};

/**
 * When the toLocaleString method is called with optional arguments locales and options,
 * the following steps are taken:
 */
/* 13.3.1 */ls.Date.toLocaleString = function () {
    // Satisfy test 13.3.0_1
    if (Object.prototype.toString.call(this) !== '[object Date]')
        throw new TypeError('`this` value must be a Date instance for Date.prototype.toLocaleString()');

    var
    // 1. Let x be this time value (as defined in ES5, 15.9.5).
        x = +this;

    // 2. If x is NaN, then return "Invalid Date".
    if (isNaN(x))
        return 'Invalid Date';

    var
    // 3. If locales is not provided, then let locales be undefined.
        locales = arguments[0],

    // 4. If options is not provided, then let options be undefined.
        options = arguments[1],

    // 5. Let options be the result of calling the ToDateTimeOptions abstract
    //    operation (defined in 12.1.1) with arguments options, "any", and "all".
        options = ToDateTimeOptions(options, 'any', 'all'),

    // 6. Let dateTimeFormat be the result of creating a new object as if by the
    //    expression new Intl.DateTimeFormat(locales, options) where
    //    Intl.DateTimeFormat is the standard built-in constructor defined in 12.1.3.
        dateTimeFormat = new DateTimeFormatConstructor(locales, options);

    // 7. Return the result of calling the FormatDateTime abstract operation (defined
    //    in 12.3.2) with arguments dateTimeFormat and x.
    return FormatDateTime(dateTimeFormat, x);
};

/**
 * When the toLocaleDateString method is called with optional arguments locales and
 * options, the following steps are taken:
 */
/* 13.3.2 */ls.Date.toLocaleDateString = function () {
    // Satisfy test 13.3.0_1
    if (Object.prototype.toString.call(this) !== '[object Date]')
        throw new TypeError('`this` value must be a Date instance for Date.prototype.toLocaleDateString()');

    var
    // 1. Let x be this time value (as defined in ES5, 15.9.5).
        x = +this;

    // 2. If x is NaN, then return "Invalid Date".
    if (isNaN(x))
        return 'Invalid Date';

    var
    // 3. If locales is not provided, then let locales be undefined.
        locales = arguments[0],

    // 4. If options is not provided, then let options be undefined.
        options = arguments[1],

    // 5. Let options be the result of calling the ToDateTimeOptions abstract
    //    operation (defined in 12.1.1) with arguments options, "date", and "date".
        options = ToDateTimeOptions(options, 'date', 'date'),

    // 6. Let dateTimeFormat be the result of creating a new object as if by the
    //    expression new Intl.DateTimeFormat(locales, options) where
    //    Intl.DateTimeFormat is the standard built-in constructor defined in 12.1.3.
        dateTimeFormat = new DateTimeFormatConstructor(locales, options);

    // 7. Return the result of calling the FormatDateTime abstract operation (defined
    //    in 12.3.2) with arguments dateTimeFormat and x.
    return FormatDateTime(dateTimeFormat, x);
};

/**
 * When the toLocaleTimeString method is called with optional arguments locales and
 * options, the following steps are taken:
 */
/* 13.3.3 */ls.Date.toLocaleTimeString = function () {
    // Satisfy test 13.3.0_1
    if (Object.prototype.toString.call(this) !== '[object Date]')
        throw new TypeError('`this` value must be a Date instance for Date.prototype.toLocaleTimeString()');

    var
    // 1. Let x be this time value (as defined in ES5, 15.9.5).
        x = +this;

    // 2. If x is NaN, then return "Invalid Date".
    if (isNaN(x))
        return 'Invalid Date';

    var
    // 3. If locales is not provided, then let locales be undefined.
        locales = arguments[0],

    // 4. If options is not provided, then let options be undefined.
        options = arguments[1],

    // 5. Let options be the result of calling the ToDateTimeOptions abstract
    //    operation (defined in 12.1.1) with arguments options, "time", and "time".
        options = ToDateTimeOptions(options, 'time', 'time'),

    // 6. Let dateTimeFormat be the result of creating a new object as if by the
    //    expression new Intl.DateTimeFormat(locales, options) where
    //    Intl.DateTimeFormat is the standard built-in constructor defined in 12.1.3.
        dateTimeFormat = new DateTimeFormatConstructor(locales, options);

    // 7. Return the result of calling the FormatDateTime abstract operation (defined
    //    in 12.3.2) with arguments dateTimeFormat and x.
    return FormatDateTime(dateTimeFormat, x);
};

defineProperty(Intl, '__applyLocaleSensitivePrototypes', {
    writable: true,
    configurable: true,
    value: function () {
        defineProperty(Number.prototype, 'toLocaleString', { writable: true, configurable: true, value: ls.Number.toLocaleString });

        for (var k in ls.Date) {
            if (hop.call(ls.Date, k))
                defineProperty(Date.prototype, k, { writable: true, configurable: true, value: ls.Date[k] });
        }
    }
});

/**
 * Can't really ship a single script with data for hundreds of locales, so we provide
 * this __addLocaleData method as a means for the developer to add the data on an
 * as-needed basis
 */
defineProperty(Intl, '__addLocaleData', {
    value: function (data) {
        if (!IsStructurallyValidLanguageTag(data.locale))
            throw new Error("Object passed doesn't identify itself with a valid language tag");

        addLocaleData(data, data.locale);
    }
});

function addLocaleData (data, tag) {
    // Both NumberFormat and DateTimeFormat require number data, so throw if it isn't present
    if (!data.number)
        throw new Error("Object passed doesn't contain locale data for Intl.NumberFormat");

    var locale,
        locales = [ tag ],
        parts   = tag.split('-');

    // Create fallbacks for locale data with scripts, e.g. Latn, Hans, Vaii, etc
    if (parts.length > 2 && parts[1].length == 4)
        arrPush.call(locales, parts[0] + '-' + parts[2]);

    while (locale = arrShift.call(locales)) {
        // Add to NumberFormat internal properties as per 11.2.3
        arrPush.call(internals.NumberFormat['[[availableLocales]]'], locale);
        internals.NumberFormat['[[localeData]]'][locale] = data.number;

        // ...and DateTimeFormat internal properties as per 12.2.3
        if (data.date) {
            data.date.nu = data.number.nu;
            arrPush.call(internals.DateTimeFormat['[[availableLocales]]'], locale);
            internals.DateTimeFormat['[[localeData]]'][locale] = data.date;
        }
    }

    // If this is the first set of locale data added, make it the default
    if (defaultLocale === undefined)
        defaultLocale = tag;

    // 11.3 (the NumberFormat prototype object is an Intl.NumberFormat instance)
    if (!numberFormatProtoInitialised) {
        InitializeNumberFormat(Intl.NumberFormat.prototype);
        numberFormatProtoInitialised = true;
    }

    // 11.3 (the NumberFormat prototype object is an Intl.NumberFormat instance)
    if (data.date && !dateTimeFormatProtoInitialised) {
        InitializeDateTimeFormat(Intl.DateTimeFormat.prototype);
        dateTimeFormatProtoInitialised = true;
    }
}

// Helper functions
// ================

/**
 * A function to deal with the inaccuracy of calculating log10 in pre-ES6
 * JavaScript environments. Math.log(num) / Math.LN10 was responsible for
 * causing issue #62.
 */
function log10Floor (n) {
    // ES6 provides the more accurate Math.log10
    if (typeof Math.log10 === 'function')
        return Math.floor(Math.log10(n));

    var x = Math.round(Math.log(n) * Math.LOG10E);
    return x - (Number('1e' + x) > n);
}

/**
 * A merge of the Intl.{Constructor}.supportedLocalesOf functions
 * To make life easier, the function should be bound to the constructor's internal
 * properties object.
 */
function supportedLocalesOf(locales) {
    /*jshint validthis:true */

    // Bound functions only have the `this` value altered if being used as a constructor,
    // this lets us imitate a native function that has no constructor
    if (!hop.call(this, '[[availableLocales]]'))
        throw new TypeError('supportedLocalesOf() is not a constructor');

    var
    // Create an object whose props can be used to restore the values of RegExp props
        regexpState = createRegExpRestore(),

    // 1. If options is not provided, then let options be undefined.
        options = arguments[1],

    // 2. Let availableLocales be the value of the [[availableLocales]] internal
    //    property of the standard built-in object that is the initial value of
    //    Intl.NumberFormat.

        availableLocales = this['[[availableLocales]]'],

    // 3. Let requestedLocales be the result of calling the CanonicalizeLocaleList
    //    abstract operation (defined in 9.2.1) with argument locales.
        requestedLocales = CanonicalizeLocaleList(locales);

    // Restore the RegExp properties
    regexpState.exp.test(regexpState.input);

    // 4. Return the result of calling the SupportedLocales abstract operation
    //    (defined in 9.2.8) with arguments availableLocales, requestedLocales,
    //    and options.
    return SupportedLocales(availableLocales, requestedLocales, options);
}

/**
 * Returns a string for a date component, resolved using multiple inheritance as specified
 * as specified in the Unicode Technical Standard 35.
 */
function resolveDateString(data, ca, component, width, key) {
    // From http://www.unicode.org/reports/tr35/tr35.html#Multiple_Inheritance:
    // 'In clearly specified instances, resources may inherit from within the same locale.
    //  For example, ... the Buddhist calendar inherits from the Gregorian calendar.'
    var obj = data[ca] && data[ca][component]
                ? data[ca][component]
                : data.gregory[component],

        // "sideways" inheritance resolves strings when a key doesn't exist
        alts = {
            narrow: ['short', 'long'],
            short:  ['long', 'narrow'],
            long:   ['short', 'narrow']
        },

        //
        resolved = hop.call(obj, width)
                  ? obj[width]
                  : hop.call(obj, alts[width][0])
                      ? obj[alts[width][0]]
                      : obj[alts[width][1]];

    // `key` wouldn't be specified for components 'dayPeriods'
    return key != null ? resolved[key] : resolved;
}

/**
 * A map that doesn't contain Object in its prototype chain
 */
Record.prototype = objCreate(null);
function Record (obj) {
    // Copy only own properties over unless this object is already a Record instance
    for (var k in obj) {
        if (obj instanceof Record || hop.call(obj, k))
            defineProperty(this, k, { value: obj[k], enumerable: true, writable: true, configurable: true });
    }
}

/**
 * An ordered list
 */
List.prototype = objCreate(null);
function List() {
    defineProperty(this, 'length', { writable:true, value: 0 });

    if (arguments.length)
        arrPush.apply(this, arrSlice.call(arguments));
}

/**
 * Constructs a regular expression to restore tainted RegExp properties
 */
function createRegExpRestore () {
    var esc = /[.?*+^$[\]\\(){}|-]/g,
        lm  = RegExp.lastMatch,
        ml  = RegExp.multiline ? 'm' : '',
        ret = { input: RegExp.input },
        reg = new List(),
        has = false,
        cap = {};

    // Create a snapshot of all the 'captured' properties
    for (var i = 1; i <= 9; i++)
        has = (cap['$'+i] = RegExp['$'+i]) || has;

    // Now we've snapshotted some properties, escape the lastMatch string
    lm = lm.replace(esc, '\\$&');

    // If any of the captured strings were non-empty, iterate over them all
    if (has) {
        for (var i = 1; i <= 9; i++) {
            var m = cap['$'+i];

            // If it's empty, add an empty capturing group
            if (!m)
                lm = '()' + lm;

            // Else find the string in lm and escape & wrap it to capture it
            else {
                m = m.replace(esc, '\\$&');
                lm = lm.replace(m, '(' + m + ')');
            }

            // Push it to the reg and chop lm to make sure further groups come after
            arrPush.call(reg, lm.slice(0, lm.indexOf('(') + 1));
            lm = lm.slice(lm.indexOf('(') + 1);
        }
    }

    // Create the regular expression that will reconstruct the RegExp properties
    ret.exp = new RegExp(arrJoin.call(reg, '') + lm, ml);

    return ret;
}

/**
 * Convert only a-z to uppercase as per section 6.1 of the spec
 */
function toLatinUpperCase (str) {
    var i = str.length;

    while (i--) {
        var ch = str.charAt(i);

        if (ch >= "a" && ch <= "z")
            str = str.slice(0, i) + ch.toUpperCase() + str.slice(i+1);
    }

    return str;
}

/**
 * Mimics ES5's abstract ToObject() function
 */
function toObject (arg) {
    if (arg == null)
        throw new TypeError('Cannot convert null or undefined to object');

    return Object(arg);
}

/**
 * Returns "internal" properties for an object
 */
function getInternalProperties (obj) {
    if (hop.call(obj, '__getInternalProperties'))
        return obj.__getInternalProperties(secret);
    else
        return objCreate(null);
}

(function () {var a=["gregory","buddhist","chinese","coptic","ethioaa","ethiopic","generic","hebrew","indian","islamic","japanese","persian","roc","numeric","long","2-digit","{year} {month} {day}, {weekday} {hour}:{minute}:{second}","{year} {month} {day}, {weekday} {hour}:{minute}:{second} {ampm}","{year} {month} {day}, {weekday}","{year} {month} {day}","{year}-{month}-{day}","{year}-{month}","{year} {month}","{month} {day}","{month}-{day}","{hour}:{minute}:{second}","{hour}:{minute}:{second} {ampm}","{hour}:{minute}","{hour}:{minute} {ampm}","BE","M01","M02","M03","M04","M05","M06","M07","M08","M09","M10","M11","M12","Tout","Baba","Hator","Kiahk","Toba","Amshir","Baramhat","Baramouda","Bashans","Paona","Epep","Mesra","Nasie","ERA0","ERA1","Meskerem","Tekemt","Hedar","Tahsas","Ter","Yekatit","Megabit","Miazia","Genbot","Sene","Hamle","Nehasse","Pagumen","Qun","Nah","Cig","Agd","Cax","Qas","Qad","Leq","Way","Dit","Xim","Kax","Qunxa Garablu","Kudo","Ciggilta Kudo","Agda Baxis","Caxah Alsa","Qasa Dirri","Qado Dirri","Leqeeni","Waysu","Diteli","Ximoli","Kaxxa Garablu","Aca","Etl","Tal","Arb","Kam","Gum","Sab","Acaada","Etleeni","Talaata","Arbaqa","Kamiisi","Gumqata","Sabti","Yaasuusuk Duma","Yaasuusuk Wadir","saaku","carra","Tishri","Heshvan","Kislev","Tevet","Shevat","Adar I","Adar","Nisan","Iyar","Sivan","Tamuz","Av","Elul","Adar II","AM","Chaitra","Vaisakha","Jyaistha","Asadha","Sravana","Bhadra","Asvina","Kartika","Agrahayana","Pausa","Magha","Phalguna","Saka","Muh.","Saf.","Rab. I","Rab. II","Jum. I","Jum. II","Raj.","Sha.","Ram.","Shaw.","Dhuʻl-Q.","Dhuʻl-H.","Muharram","Safar","Rabiʻ I","Rabiʻ II","Jumada I","Jumada II","Rajab","Shaʻban","Ramadan","Shawwal","Dhuʻl-Qiʻdah","Dhuʻl-Hijjah","AH","Taika (645-650)","Hakuchi (650-671)","Hakuhō (672-686)","Shuchō (686-701)","Taihō (701-704)","Keiun (704-708)","Wadō (708-715)","Reiki (715-717)","Yōrō (717-724)","Jinki (724-729)","Tempyō (729-749)","Tempyō-kampō (749-749)","Tempyō-shōhō (749-757)","Tempyō-hōji (757-765)","Temphō-jingo (765-767)","Jingo-keiun (767-770)","Hōki (770-780)","Ten-ō (781-782)","Enryaku (782-806)","Daidō (806-810)","Kōnin (810-824)","Tenchō (824-834)","Jōwa (834-848)","Kajō (848-851)","Ninju (851-854)","Saiko (854-857)","Tennan (857-859)","Jōgan (859-877)","Genkei (877-885)","Ninna (885-889)","Kampyō (889-898)","Shōtai (898-901)","Engi (901-923)","Enchō (923-931)","Shōhei (931-938)","Tengyō (938-947)","Tenryaku (947-957)","Tentoku (957-961)","Ōwa (961-964)","Kōhō (964-968)","Anna (968-970)","Tenroku (970-973)","Ten-en (973-976)","Jōgen (976-978)","Tengen (978-983)","Eikan (983-985)","Kanna (985-987)","Ei-en (987-989)","Eiso (989-990)","Shōryaku (990-995)","Chōtoku (995-999)","Chōhō (999-1004)","Kankō (1004-1012)","Chōwa (1012-1017)","Kannin (1017-1021)","Jian (1021-1024)","Manju (1024-1028)","Chōgen (1028-1037)","Chōryaku (1037-1040)","Chōkyū (1040-1044)","Kantoku (1044-1046)","Eishō (1046-1053)","Tengi (1053-1058)","Kōhei (1058-1065)","Jiryaku (1065-1069)","Enkyū (1069-1074)","Shōho (1074-1077)","Shōryaku (1077-1081)","Eiho (1081-1084)","Ōtoku (1084-1087)","Kanji (1087-1094)","Kaho (1094-1096)","Eichō (1096-1097)","Shōtoku (1097-1099)","Kōwa (1099-1104)","Chōji (1104-1106)","Kashō (1106-1108)","Tennin (1108-1110)","Ten-ei (1110-1113)","Eikyū (1113-1118)","Gen-ei (1118-1120)","Hoan (1120-1124)","Tenji (1124-1126)","Daiji (1126-1131)","Tenshō (1131-1132)","Chōshō (1132-1135)","Hoen (1135-1141)","Eiji (1141-1142)","Kōji (1142-1144)","Tenyō (1144-1145)","Kyūan (1145-1151)","Ninpei (1151-1154)","Kyūju (1154-1156)","Hogen (1156-1159)","Heiji (1159-1160)","Eiryaku (1160-1161)","Ōho (1161-1163)","Chōkan (1163-1165)","Eiman (1165-1166)","Nin-an (1166-1169)","Kaō (1169-1171)","Shōan (1171-1175)","Angen (1175-1177)","Jishō (1177-1181)","Yōwa (1181-1182)","Juei (1182-1184)","Genryuku (1184-1185)","Bunji (1185-1190)","Kenkyū (1190-1199)","Shōji (1199-1201)","Kennin (1201-1204)","Genkyū (1204-1206)","Ken-ei (1206-1207)","Shōgen (1207-1211)","Kenryaku (1211-1213)","Kenpō (1213-1219)","Shōkyū (1219-1222)","Jōō (1222-1224)","Gennin (1224-1225)","Karoku (1225-1227)","Antei (1227-1229)","Kanki (1229-1232)","Jōei (1232-1233)","Tempuku (1233-1234)","Bunryaku (1234-1235)","Katei (1235-1238)","Ryakunin (1238-1239)","En-ō (1239-1240)","Ninji (1240-1243)","Kangen (1243-1247)","Hōji (1247-1249)","Kenchō (1249-1256)","Kōgen (1256-1257)","Shōka (1257-1259)","Shōgen (1259-1260)","Bun-ō (1260-1261)","Kōchō (1261-1264)","Bun-ei (1264-1275)","Kenji (1275-1278)","Kōan (1278-1288)","Shōō (1288-1293)","Einin (1293-1299)","Shōan (1299-1302)","Kengen (1302-1303)","Kagen (1303-1306)","Tokuji (1306-1308)","Enkei (1308-1311)","Ōchō (1311-1312)","Shōwa (1312-1317)","Bunpō (1317-1319)","Genō (1319-1321)","Genkyō (1321-1324)","Shōchū (1324-1326)","Kareki (1326-1329)","Gentoku (1329-1331)","Genkō (1331-1334)","Kemmu (1334-1336)","Engen (1336-1340)","Kōkoku (1340-1346)","Shōhei (1346-1370)","Kentoku (1370-1372)","Bunchũ (1372-1375)","Tenju (1375-1379)","Kōryaku (1379-1381)","Kōwa (1381-1384)","Genchũ (1384-1392)","Meitoku (1384-1387)","Kakei (1387-1389)","Kōō (1389-1390)","Meitoku (1390-1394)","Ōei (1394-1428)","Shōchō (1428-1429)","Eikyō (1429-1441)","Kakitsu (1441-1444)","Bun-an (1444-1449)","Hōtoku (1449-1452)","Kyōtoku (1452-1455)","Kōshō (1455-1457)","Chōroku (1457-1460)","Kanshō (1460-1466)","Bunshō (1466-1467)","Ōnin (1467-1469)","Bunmei (1469-1487)","Chōkyō (1487-1489)","Entoku (1489-1492)","Meiō (1492-1501)","Bunki (1501-1504)","Eishō (1504-1521)","Taiei (1521-1528)","Kyōroku (1528-1532)","Tenmon (1532-1555)","Kōji (1555-1558)","Eiroku (1558-1570)","Genki (1570-1573)","Tenshō (1573-1592)","Bunroku (1592-1596)","Keichō (1596-1615)","Genwa (1615-1624)","Kan-ei (1624-1644)","Shōho (1644-1648)","Keian (1648-1652)","Shōō (1652-1655)","Meiryaku (1655-1658)","Manji (1658-1661)","Kanbun (1661-1673)","Enpō (1673-1681)","Tenwa (1681-1684)","Jōkyō (1684-1688)","Genroku (1688-1704)","Hōei (1704-1711)","Shōtoku (1711-1716)","Kyōhō (1716-1736)","Genbun (1736-1741)","Kanpō (1741-1744)","Enkyō (1744-1748)","Kan-en (1748-1751)","Hōryaku (1751-1764)","Meiwa (1764-1772)","An-ei (1772-1781)","Tenmei (1781-1789)","Kansei (1789-1801)","Kyōwa (1801-1804)","Bunka (1804-1818)","Bunsei (1818-1830)","Tenpō (1830-1844)","Kōka (1844-1848)","Kaei (1848-1854)","Ansei (1854-1860)","Man-en (1860-1861)","Bunkyū (1861-1864)","Genji (1864-1865)","Keiō (1865-1868)","M","T","S","H","Bunchū (1372-1375)","Genchū (1384-1392)","Meiji","Taishō","Shōwa","Heisei","Farvardin","Ordibehesht","Khordad","Tir","Mordad","Shahrivar","Mehr","Aban","Azar","Dey","Bahman","Esfand","AP","Before R.O.C.","R.O.C.","latn","{number}","-{number}","{currency}{number}","-{currency}{number}","{number}%","-{number}%",".",",","NaN","%","∞","A$","R$","CA$","CN¥","Fdj","Br","€","£","HK$","₪","₹","JP¥","₩","MX$","NZ$","฿","NT$","US$","₫","FCFA","EC$","CFA","CFPF","Liiqen","Nfk","{weekday}, {day} {month} {year} {hour}:{minute}:{second}","{weekday}, {day} {month} {year} {hour}:{minute}:{second} {ampm}","{weekday}, {day} {month} {year}","{day} {month} {year}","{month}/{day}/{year}","{month}/{year}","{month} {year}","{month}/{day}","J","F","A","O","N","D","Jan","Feb","Mar","Apr","Mei","Jun","Jul","Aug","Sep","Okt","Nov","Des","Januarie","Februarie","Maart","April","Junie","Julie","Augustus","September","Oktober","November","Desember","So.","Ma.","Di.","Wo.","Do.","Vr.","Sa.","So","Ma","Di","Wo","Do","Vr","Sa","Sondag","Maandag","Dinsdag","Woensdag","Donderdag","Vrydag","Saterdag","v.C.","n.C.","vgj","vg","v.g.j.","g.j.","voor Christus","na Christus","vm.","nm.","{currency} {number}","-{currency} {number}"," ","$","R","{weekday} {day} {month} {year} {hour}:{minute}:{second}","{weekday} {day} {month} {year} {hour}:{minute}:{second} {ampm}","{weekday} {day} {month} {year}","{day}/{month}/{year}","{day} {month}","{day}/{month}","nùm","kɨz","tɨd","taa","see","nzu","dum","fɔe","dzu","lɔm","kaa","fwo","ndzɔ̀ŋɔ̀nùm","ndzɔ̀ŋɔ̀kƗ̀zùʔ","ndzɔ̀ŋɔ̀tƗ̀dʉ̀ghà","ndzɔ̀ŋɔ̀tǎafʉ̄ghā","ndzɔ̀ŋèsèe","ndzɔ̀ŋɔ̀nzùghò","ndzɔ̀ŋɔ̀dùmlo","ndzɔ̀ŋɔ̀kwîfɔ̀e","ndzɔ̀ŋɔ̀tƗ̀fʉ̀ghàdzughù","ndzɔ̀ŋɔ̀ghǔuwelɔ̀m","ndzɔ̀ŋɔ̀chwaʔàkaa wo","ndzɔ̀ŋèfwòo","nts","kpa","ghɔ","tɔm","ume","ghɨ","dzk","tsuʔntsɨ","tsuʔukpà","tsuʔughɔe","tsuʔutɔ̀mlò","tsuʔumè","tsuʔughɨ̂m","tsuʔndzɨkɔʔɔ","SK","BK","Sěe Kɨ̀lesto","Bǎa Kɨ̀lesto","a.g","a.k","{number}{currency}","-{number}{currency}","{weekday}, {month} {day}, {year} {hour}:{minute}:{second}","{weekday}, {month} {day}, {year} {hour}:{minute}:{second} {ampm}","{weekday}, {month} {day}, {year}","{year}/{month}/{day}","S-Ɔ","K-Ɔ","E-Ɔ","E-O","E-K","O-A","A-K","D-Ɔ","F-Ɛ","Ɔ-A","Ɔ-O","M-Ɔ","Sanda-Ɔpɛpɔn","Kwakwar-Ɔgyefuo","Ebɔw-Ɔbenem","Ebɔbira-Oforisuo","Esusow Aketseaba-Kɔtɔnimba","Obirade-Ayɛwohomumu","Ayɛwoho-Kitawonsa","Difuu-Ɔsandaa","Fankwa-Ɛbɔ","Ɔbɛsɛ-Ahinime","Ɔberɛfɛw-Obubuo","Mumu-Ɔpɛnimba","Kwe","Dwo","Ben","Wuk","Yaw","Fia","Mem","Kwesida","Dwowda","Benada","Wukuda","Yawda","Fida","Memeneda","AK","KE","Ansa Kristo","Kristo Ekyiri","AN","EW","GH₵","{weekday}፣ {month} {day} {year} {hour}:{minute}:{second}","{weekday}፣ {month} {day} {year} {hour}:{minute}:{second} {ampm}","{weekday}፣ {month} {day} {year}","ዓ/ዓ","ዓ/ም","1","2","3","4","5","6","7","8","9","10","11","12","13","መስከ","ጥቅም","ኅዳር","ታኅሣ","ጥር","የካቲ","መጋቢ","ሚያዝ","ግንቦ","ሰኔ","ሐምሌ","ነሐሴ","ጳጉሜ","መስከረም","ጥቅምት","ታኅሣሥ","የካቲት","መጋቢት","ሚያዝያ","ግንቦት","ጳጉሜን","ጃ","ፌ","ማ","ኤ","ሜ","ጁ","ኦ","ሴ","ኖ","ዲ","ጃንዩ","ፌብሩ","ማርች","ኤፕሪ","ሜይ","ጁን","ጁላይ","ኦገስ","ሴፕቴ","ኦክተ","ኖቬም","ዲሴም","ጃንዩወሪ","ፌብሩወሪ","ኤፕሪል","ኦገስት","ሴፕቴምበር","ኦክተውበር","ኖቬምበር","ዲሴምበር","እ","ሰ","ረ","ሐ","ዓ","ቅ","እሑድ","ሰኞ","ማክሰ","ረቡዕ","ሐሙስ","ዓርብ","ቅዳሜ","ማክሰኞ","ዓመተ ዓለም","ዓመተ ምሕረት","ጥዋት","ከሰዓት","ሙሀረም","ሳፈር","ረቢዑል አወል","ረቢዑል አኺር","ጀማደል አወል","ጀማደል አኺር","ረጀብ","ሻእባን","ረመዳን","ሸዋል","ዙልቂዳህ","ዙልሂጃህ","AU$","ብር","{weekday}، {day} {month}، {year} {hour}:{minute}:{second}","{weekday}، {day} {month}، {year} {hour}:{minute}:{second} {ampm}","{weekday}، {day} {month}، {year}","{day} {month}، {year}","{day}‏/{month}‏/{year}","{month}‏/{year}","{day}/‏{month}","التقويم البوذي","توت","بابه","هاتور","كيهك","طوبة","أمشير","برمهات","برمودة","بشنس","بؤونة","أبيب","مسرى","نسيئ","مسكريم","تكمت","هدار","تهساس","تر","يكتت","مجابيت","ميازيا","جنبت","سين","هامل","نهاس","باجمن","ي","ف","م","أ","و","ن","ل","غ","س","ك","ب","د","يناير","فبراير","مارس","أبريل","مايو","يونيو","يوليو","أغسطس","سبتمبر","أكتوبر","نوفمبر","ديسمبر","الأحد","الاثنين","الثلاثاء","الأربعاء","الخميس","الجمعة","السبت","ق.م","قبل الميلاد","ميلادي","ص","تشري","مرحشوان","كيسلو","طيفت","شباط","آذار الأول","آذار","نيسان","أيار","سيفان","تموز","آب","أيلول","آذار الثاني","١","٢","٣","٤","٥","٦","٧","٨","٩","١٠","١١","١٢","محرم","صفر","ربيع الأول","ربيع الآخر","جمادى الأولى","جمادى الآخرة","رجب","شعبان","رمضان","شوال","ذو القعدة","ذو الحجة","هـ","تيكا","هاكتشي","هاكهو","شتشو","تيهو","كيين","وادو","رييكي","يورو","جينكي","تمبيو","تمبيو-كامبو","تمبيو-شوهو","تمبيو-هوجي","تمفو-جينجو","جينجو-كيين","هوكي","تن-أو","إنرياكو","ديدو","كونين","تنتشو","شووا (٨٣٤–٨٤٨)‏","كاجو","نينجو","سيكو","تنان","جوجان","جينكيي","نينا","كامبيو","شوتاي","انجي","انتشو","شوهيي","تنجيو","تنرياكو","تنتوكو","أووا","كوهو","آنا","تينروكو","تن-نن","جوجن","تنجن","إيكان","كانا","اي-ان","ايسو","شورياكو (٩٩٠–٩٩٥)‏","تشوتوكو","تشوهو","كانكو","تشووا","كانين","جاين","مانجو","تشوجين","تشورياكو","تشوكيو (١٠٤٠–١٠٤٤)‏","كانتوكو","ايشو (١٠٤٦–١٠٥٣)‏","تينجي","كوهيي","جيرياكو","انكيو (١٠٦٩–١٠٧٤)‏","شوهو (١٠٧٤–١٠٧٧)‏","شورياكو (١٠٧٧–١٠٨١)‏","ايهو","أوتوكو","كانجي","كاهو","ايتشو","شوتوكو","كووا (١٠٩٩–١١٠٤)‏","تشوجي","كاشو","تنين","تن-اي","ايكيو (١١١٣–١١١٨)‏","جن-اي","هوان","تنجي","ديجي","تنشو (١١٣١–١١٣٢)‏","تشوشو","هوين","ايجي","كوجي (١١٤٢–١١٤٤)‏","تنيو","كيوان","نينبيي","كيوجو","هجين","هيجي","ايرياكو","أوهو","تشوكان","ايمان","نين-ان","كاو","شون","أنجين","جيشو","يووا","جيي","جنريوكو","بنجي","كنكيو","شوجي","كنين","جنكيو (١٢٠٤–١٢٠٦)‏","كن-اي","شوجن (١٢٠٧–١٢١١)‏","كنرياكو","كنبو (١٢١٣–١٢١٩)‏","شوكيو","جو","جيننين","كروكو","أنتيي","كنكي","جويي","تمبكو","بنرياكو","كاتيي","رياكنين","ان-أو","نينجي","كنجين","هوجي","كنتشو","كوجن","شوكا","شوجن (١٢٥٩–١٢٦٠)‏","بن-أو","كوتشو","بن-اي","كنجي","كوان","شوو (١٢٨٨–١٢٩٣)‏","اينين","شوان","كنجن","كجن","توكجي","انكي","أوتشو","شووا (١٣١٢–١٣١٧)‏","بنبو","جنو","جنكيو (١٣٢١–١٣٢٤)‏","شوتشو (١٣٢٤–١٣٢٦)‏","كريكي","جنتكو","جنكو","كمو","إنجن","كوككو","شوهي","كنتكو","بنتشو","تنجو","كورياكو","كووا (١٣٨١–١٣٨٤)‏","جنتشو","مييتكو (١٣٨٤–١٣٨٧)‏","كاكي","كو","مييتكو (١٣٩٠–١٣٩٤)‏","أويي","شوتشو (١٤٢٨–١٤٢٩)‏","ايكيو (١٤٢٩–١٤٤١)‏","ككيتسو","بن-أن","هوتكو","كيوتكو","كوشو","تشوركو","كنشو","بنشو","أونين","بنمي","تشوكيو (١٤٨٧–١٤٨٩)‏","انتكو","ميو","بنكي","ايشو (١٥٠٤–١٥٢١)‏","تييي","كيوركو","تنمن","كوجي (١٥٥٥–١٥٥٨)‏","ايركو","جنكي","تنشو (١٥٧٣–١٥٩٢)‏","بنركو","كيتشو","جنوا","كان-اي","شوهو (١٦٤٤–١٦٤٨)‏","كيان","شوو (١٦٥٢–١٦٥٥)‏","ميرياكو","منجي","كنبن","انبو","تنوا","جوكيو","جنركو","هويي","شوتكو","كيوهو","جنبن","كنبو (١٧٤١–١٧٤٤)‏","انكيو (١٧٤٤–١٧٤٨)‏","كان-ان","هورياكو","مييوا","ان-اي","تنمي","كنسي","كيووا","بنكا","بنسي","تنبو","كوكا","كاي","أنسي","من-ان","بنكيو","جنجي","كيو","ميجي","تيشو","شووا","هيسي","فرفردن","أذربيهشت","خرداد","تار","مرداد","شهرفار","مهر","آيان","آذر","دي","بهمن","اسفندار","ه‍.ش","جمهورية الصي","arab","٫","٬","ليس رقم","٪","د.إ.‏","د.ب.‏","ي.ص","د.ج.‏","ج.م.‏","ر.ه.‏","د.ع.‏","د.أ.‏","ف.ج.ق.‏","د.ك.‏","ل.ل.‏","د.ل.‏","د.م.‏","أ.م.‏","ر.ع.‏","ر.ق.‏","ر.س.‏","د.س.‏","ج.س.","ج.ج.س.","ل.س.‏","د.ت.‏","***","ر.ي.‏","{year}/{month}","ج","جانفي","فيفري","أفريل","ماي","جوان","جويلية","أوت","ش","آ","ح","ت","كانون الثاني","حزيران","تشرین الأول","تشرين الثاني","كانون الأول","تشرين الأول","يوليوز","غشت","شتنبر","نونبر","دجنبر","إ","إبريل","أغشت","شتمبر","دجمبر","GB£","জানু","ফেব্ৰু","মাৰ্চ","এপ্ৰিল","মে","জুন","জুলাই","আগ","সেপ্ট","অক্টো","নভে","ডিসে","জানুৱাৰী","ফেব্ৰুৱাৰী","আগষ্ট","ছেপ্তেম্বৰ","অক্টোবৰ","নৱেম্বৰ","ডিচেম্বৰ","ৰবি","সোম","মঙ্গল","বুধ","বৃহষ্পতি","শুক্ৰ","শনি","দেওবাৰ","সোমবাৰ","মঙ্গলবাৰ","বুধবাৰ","বৃহষ্পতিবাৰ","শুক্ৰবাৰ","শনিবাৰ","BCE","CE","পূৰ্বাহ্ণ","অপৰাহ্ণ","beng","Mac","Ago","Dec","Januari","Februari","Machi","Aprili","Juni","Julai","Agosti","Septemba","Oktoba","Novemba","Desemba","Jpi","Jtt","Jnn","Jtn","Alh","Ijm","Jmo","Jumapili","Jumatatu","Jumanne","Jumatano","Alhamisi","Ijumaa","Jumamosi","KM","BM","Kabla yakwe Yethu","Baada yakwe Yethu","icheheavo","ichamthi","{number} {currency}","-{number} {currency}","TSh","X","P","xin","feb","mar","abr","may","xun","xnt","ago","set","och","pay","avi","de xineru","de febreru","de marzu","d'abril","de mayu","de xunu","de xunetu","d'agostu","de setiembre","d'ochobre","de payares","d'avientu","do","ll","ma","mi","xu","vi","sa","dom","llu","mie","xue","vie","sab","domingu","llunes","martes","miércoles","xueves","vienres","sábadu","a.C.","d.C.","PM","¥","{day} {month} {year}, {weekday} {hour}:{minute}:{second}","{day} {month} {year}, {weekday} {hour}:{minute}:{second} {ampm}","{day} {month} {year}, {weekday}","{day}.{month}.{year}","{month}.{year}","{day}.{month}","yan","fev","apr","iyn","iyl","avq","sen","okt","noy","dek","yanvar","fevral","mart","aprel","iyun","iyul","avqust","sentyabr","oktyabr","noyabr","dekabr","B.","B.E.","Ç.A.","Ç.","C.A.","C","Ş.","bazar","bazar ertəsi","çərşənbə axşamı","çərşənbə","cümə axşamı","cümə","şənbə","e.ə.","b.e.","eramızdan əvvəl","bizim eramızın","man.","{weekday}, {day}, {month}, {year} {hour}:{minute}:{second}","{weekday}, {day}, {month}, {year} {hour}:{minute}:{second} {ampm}","{weekday}, {day}, {month}, {year}","{day} {month}, {year}","{month}, {year}","јанвар","феврал","март","апрел","май","ијун","ијул","август","сентјабр","октјабр","нојабр","декабр","базар","базар ертәси","чәршәнбә ахшамы","чәршәнбә","ҹүмә ахшамы","ҹүмә","шәнбә","ман.","kɔn","mac","mat","mto","mpu","hil","nje","hik","dip","bio","liɓ","Kɔndɔŋ","Màcɛ̂l","Màtùmb","Màtop","M̀puyɛ","Hìlòndɛ̀","Njèbà","Hìkaŋ","Dìpɔ̀s","Bìòôm","Màyɛsèp","Lìbuy li ńyèe","nɔy","nja","uum","ŋge","mbɔ","kɔɔ","jon","ŋgwà nɔ̂y","ŋgwà njaŋgumba","ŋgwà ûm","ŋgwà ŋgê","ŋgwà mbɔk","ŋgwà kɔɔ","ŋgwà jôn","b.Y.K","m.Y.K","bisū bi Yesù Krǐstò","i mbūs Yesù Krǐstò","I bikɛ̂glà","I ɓugajɔp","{number} %","-{number} %","{weekday}, {day} {month} {year} {hour}.{minute}.{second}","{weekday}, {day} {month} {year} {hour}.{minute}.{second} {ampm}","{hour}.{minute}.{second}","{hour}.{minute}.{second} {ampm}","{hour}.{minute}","{hour}.{minute} {ampm}","с","л","к","м","ч","ж","в","сту","лют","сак","кра","мая","чэр","ліп","жні","вер","кас","ліс","сне","студзеня","лютага","сакавіка","красавіка","чэрвеня","ліпеня","жніўня","верасня","кастрычніка","лістапада","снежня","нд","пн","аў","ср","чц","пт","сб","нядзеля","панядзелак","аўторак","серада","чацвер","пятніца","субота","да н.э.","н.э.","да палудня","пасля палудня","р.","рас. руб.","Epr","Oga","Dis","Epreo","Ogasti","Disemba","Pa Mulungu","Palichimo","Palichibuli","Palichitatu","Palichine","Palichisano","Pachibelushi","BC","AD","Before Yesu","After Yesu","uluchelo","akasuba","K","Hut","Vil","Dat","Tai","Han","Sit","Nan","Tis","Kum","Kmj","Kmb","pa mwedzi gwa hutala","pa mwedzi gwa wuvili","pa mwedzi gwa wudatu","pa mwedzi gwa wutai","pa mwedzi gwa wuhanu","pa mwedzi gwa sita","pa mwedzi gwa saba","pa mwedzi gwa nane","pa mwedzi gwa tisa","pa mwedzi gwa kumi","pa mwedzi gwa kumi na moja","pa mwedzi gwa kumi na mbili","Mul","Hiv","Hid","Hit","Hih","Lem","pa mulungu","pa shahuviluha","pa hivili","pa hidatu","pa hitayi","pa hihanu","pa shahulembela","Kabla ya Mtwaa","Baada ya Mtwaa","pamilau","pamunyi","{weekday}, {day} {month} {year} г., {hour}:{minute}:{second}","{weekday}, {day} {month} {year} г., {hour}:{minute}:{second} {ampm}","я","ф","а","ю","о","н","д","ян.","февр.","апр.","юни","юли","авг.","септ.","окт.","ноем.","дек.","януари","февруари","април","септември","октомври","ноември","декември","вт","чт","неделя","понеделник","вторник","сряда","четвъртък","петък","събота","сл.н.е.","пр.Хр.","сл.Хр.","пр.н.е.","от н.е.","пр.об.","сл.об.","тишри","хешван","кислев","тебет","шебат","адар I","адар","нисан","иар","сиван","тамуз","ав","елул","адар II","чайтра","вайсакха","джаинтха","асадха","сравана","бхада","азвина","картика","аграхайана","пауза","магха","пхалгуна","мухарам","сафар","раби-1","раби-2","джумада-1","джумада-2","раджаб","шабан","рамазан","Шавал","Дхул-Каада","Дхул-хиджа","лв.","руб.","щ.д.","zan","awi","mɛ","zuw","zul","uti","sɛt","ɔku","now","des","zanwuye","feburuye","marisi","awirili","zuwɛn","zuluye","sɛtanburu","ɔkutɔburu","nowanburu","desanburu","kar","ntɛ","tar","ara","ala","jum","sib","kari","ntɛnɛ","tarata","araba","alamisa","juma","sibiri","J.-C. ɲɛ","ni J.-C.","jezu krisiti ɲɛ","jezu krisiti minkɛ","{weekday}, {day} {month}, {year} {hour}:{minute}:{second}","{weekday}, {day} {month}, {year} {hour}:{minute}:{second} {ampm}","{weekday}, {day} {month}, {year}","জা","ফে","মা","এ","জু","আ","সে","অ","ন","ডি","জানুয়ারী","ফেব্রুয়ারী","মার্চ","এপ্রিল","আগস্ট","সেপ্টেম্বর","অক্টোবর","নভেম্বর","ডিসেম্বর","রঃ","সোঃ","মঃ","বুঃ","বৃঃ","শুঃ","শোঃ","রবি","বৃহস্পতি","শুক্র","রবিবার","সোমবার","মঙ্গলবার","বুধবার","বৃহষ্পতিবার","শুক্রবার","শনিবার","খ্রিস্টপূর্ব","খৃষ্টাব্দ","পূর্বাহ্ণ","অপরাহ্ণ","চৈত্র","বৈশাখ","জৈষ্ঠ্য","আষাঢ়","শ্রাবণ","ভাদ্র","আশ্বিন","কার্তিক","অগ্রহায়ণ","পৌষ","মাঘ","ফাল্গুন","সাল","মহররম","সফর","রবিউল আউয়াল","রবিউস সানি","জমাদিউল আউয়াল","জমাদিউস সানি","রজব","শা'বান","রমজান","শাওয়াল","জ্বিলকদ","জ্বিলহজ্জ","যুগ","({number}{currency})","সংখ্যা না","৳","{year} ལོ་འི་{month}ཙེས་{day}","ཟླ་༡","ཟླ་༢","ཟླ་༣","ཟླ་༤","ཟླ་༥","ཟླ་༦","ཟླ་༧","ཟླ་༨","ཟླ་༩","ཟླ་༡༠","ཟླ་༡༡","ཟླ་༡༢","ཟླ་བ་དང་པོ་","ཟླ་བ་གཉིས་པ་","ཟླ་བ་སུམ་པ་","ཟླ་བ་བཞི་པ་","ཟླ་བ་ལྔ་པ་","ཟླ་བ་དྲུག་པ་","ཟླ་བ་བདུན་པ་","ཟླ་བ་བརྒྱད་པ་","ཟླ་བ་དགུ་པ་","ཟླ་བ་བཅུ་པ་","ཟླ་བ་བཅུ་གཅིག་པ་","ཟླ་བ་བཅུ་གཉིས་པ་","ཉི་མ་","ཟླ་བ་","མིག་དམར་","ཧླག་པ་","ཕུར་བུ་","སངས་","སྤེན་པ་","གཟའ་ཉི་མ་","གཟའ་ཟླ་བ་","གཟའ་མིག་དམར་","གཟའ་ཧླག་པ་","གཟའ་ཕུར་བུ་","གཟའ་སངས་","གཟའ་སྤེན་པ་","སྤྱི་ལོ་སྔོན།","སྤྱི་ལོ།","སྔ་དྲོ་","ཕྱི་དྲོ་","ཨང་མེན་","Gen","Cʼhwe","Meur","Ebr","Mae","Mezh","Goue","Eost","Gwen","Here","Du","Ker","Genver","Cʼhwevrer","Meurzh","Ebrel","Mezheven","Gouere","Gwengolo","Kerzu","sul","lun","meu.","mer.","yaou","gwe.","sad.","Sul","Lun","Mercʼher","Yaou","Gwener","Sadorn","{day}-{month}","जानुवारी","फेब्रुवारी","मार्स","एफ्रिल","मे","जुन","जुलाइ","आगस्थ","सेबथेज्ब़र","अखथबर","नबेज्ब़र","दिसेज्ब़र","रबि","सम","मंगल","बुद","बिसथि","सुखुर","सुनि","रबिबार","समबार","मंगलबार","बुदबार","बिसथिबार","सुखुरबार","सुनिबार","ईसा.पूर्व","सन","फुं","बेलासे","बैसागो/बैसाग","जेथो","आसार","सावुन","भाद्र","आसिन","खाथि","आगाह्न","फुस","मागो","फागुन","सैथो","{weekday}, {day}. {month} {year}. {hour}:{minute}:{second}","{weekday}, {day}. {month} {year}. {hour}:{minute}:{second} {ampm}","{weekday}, {day}. {month} {year}","{day}. {month} {year}","{day}. {month}","jan","maj","jun","jul","avg","sep","nov","dec","januar","februar","april","juni","juli","avgust","septembar","oktobar","novembar","decembar","ned","pon","uto","sri","čet","pet","sub","nedjelja","ponedjeljak","utorak","srijeda","četvrtak","petak","subota","p. n. e.","n. e","Pre nove ere","Nove ere","pre podne","popodne","muh.","saf.","rab. i","rab. ii","džum. i","džum. ii","redž.","ša.","ram.","še.","zul-k.","zul-h.","muharem","safer","rabiʻ i","rabiʻ ii","džumade i","džumade ii","redžeb","šaʻban","ramazan","ševal","zul-kade","zul-hidže","Kč","zl","din.","БЕ","Таут","Баба","Хатор","Киахк","Тоба","Амшир","Барамхат","Барамуда","Башанс","Паона","Епеп","Месра","Наси","Мескерем","Текемт","Хедар","Тахсас","Тер","Јекатит","Мегабит","Миазиа","Генбот","Сене","Хамле","Нехасе","Пагумен","јан","феб","мар","апр","мај","јун","јул","авг","сеп","окт","нов","дец","јануар","фебруар","јуни","јули","септембар","октобар","новембар","децембар","нед","пон","уто","сри","чет","пет","суб","недеља","понедељак","уторак","сриједа","четвртак","петак","п.н.е.","н.е.","п. н. е.","н. е.","Пре нове ере","Нове ере","пре подне","поподне","Тишри","Хешван","Кислев","Тевет","Шеват","Адар I","Адар","Нисан","Ијар","Сиван","Тамуз","Ав","Елул","Адар II","Чаитра","Ваисака","Јиаиста","Асада","Сравана","Бадра","Асвина","Картика","Аргајана","Пауза","Мага","Фалгуна","САКА","Мух.","Саф.","Реб. 1","Реб 2","Џум. 1","Џум. 2","Реџ.","Ша.","Рам.","Ше.","Зул-к.","Зул-х.","Мурахам","Сафар","Рабиʻ I","Рабиʻ II","Јумада I","Јумада II","Рађаб","Шаʻбан","Рамадан","Дуʻл-Киʻда","Дуʻл-хиђа","АХ","Таика (645-650)","Хакучи (650-671)","Хакухо (672-686)","Шучо (686-701)","Таихо (701-704)","Кеиун (704-708)","Вадо (708-715)","Реики (715-717)","Јоро (717-724)","Јинки (724-729)","Темпио (729-749)","Темпио-кампо (749-749)","Темпио-шохо (749-757)","Темпио-хођи (757-765)","Темпо-ђинго (765-767)","Ђинго-кеиун (767-770)","Хоки (770-780)","Тен-о (781-782)","Енрјаку (782-806)","Даидо (806-810)","Конин (810-824)","Тенчо (824-834)","Шова (834-848)","Кајо (848-851)","Нињу (851-854)","Саико (854-857)","Тенан (857-859)","Јоган (859-877)","Генкеи (877-885)","Ниња (885-889)","Кампјо (889-898)","Шотаи (898-901)","Енђи (901-923)","Енчо (923-931)","Шохеи (931-938)","Тенгјо (938-947)","Тенриаку (947-957)","Тентоку (957-961)","Ова (961-964)","Кохо (964-968)","Ана (968-970)","Тенроку (970-973)","Тен-ен (973-976)","Јоген (976-978)","Тенген (978-983)","Еикан (983-985)","Кана (985-987)","Еи-ен (987-989)","Еисо (989-990)","Шорјаку (990-995)","Чотоку (995-999)","Чохо (999-1004)","Канко (1004-1012)","Чова (1012-1017)","Канин (1017-1021)","Ђиан (1021-1024)","Мању (1024-1028)","Чоген (1028-1037)","Чорјаку (1037-1040)","Чокју (1040-1044)","Кантоку (1044-1046)","Еишо (1046-1053)","Тенђи (1053-1058)","Кохеи (1058-1065)","Ђирјаку (1065-1069)","Енкју (1069-1074)","Шохо (1074-1077)","Шорјаку (1077-1081)","Еишо (1081-1084)","Отоку (1084-1087)","Канђи (1087-1094)","Кахо (1094-1096)","Еичо (1096-1097)","Шотоку (1097-1099)","Кова (1099-1104)","Чођи (1104-1106)","Кашо (1106-1108)","Тенин (1108-1110)","Тен-еи (1110-1113)","Еикју (1113-1118)","Ђен-еи (1118-1120)","Хоан (1120-1124)","Тенђи (1124-1126)","Даиђи (1126-1131)","Теншо (1131-1132)","Чошао (1132-1135)","Хоен (1135-1141)","Еиђи (1141-1142)","Кођи (1142-1144)","Тењо (1144-1145)","Кјуан (1145-1151)","Нинпеи (1151-1154)","Кјују (1154-1156)","Хоген (1156-1159)","Хеиђи (1159-1160)","Еирјаку (1160-1161)","Охо (1161-1163)","Чокан (1163-1165)","Еиман (1165-1166)","Нин-ан (1166-1169)","Као (1169-1171)","Шоан (1171-1175)","Анген (1175-1177)","Ђишо (1177-1181)","Јова (1181-1182)","Ђуеи (1182-1184)","Генрјуку (1184-1185)","Бунђи (1185-1190)","Кенкју (1190-1199)","Шођи (1199-1201)","Кенин (1201-1204)","Генкју (1204-1206)","Кен-еи (1206-1207)","Шоген (1207-1211)","Кенрјаку (1211-1213)","Кенпо (1213-1219)","Шокју (1219-1222)","Ђу (1222-1224)","Ђенин (1224-1225)","Кароку (1225-1227)","Антеи (1227-1229)","Канки (1229-1232)","Ђоеи (1232-1233)","Темпуку (1233-1234)","Бунрјаку (1234-1235)","Катеи (1235-1238)","Рјакунин (1238-1239)","Ен-о (1239-1240)","Нињи (1240-1243)","Канген (1243-1247)","Хођи (1247-1249)","Кенчо (1249-1256)","Коген (1256-1257)","Шока (1257-1259)","Шоген (1259-1260)","Бун-о (1260-1261)","Кочо (1261-1264)","Бун-еи (1264-1275)","Кенђи (1275-1278)","Коан (1278-1288)","Шу (1288-1293)","Еинин (1293-1299)","Шоан (1299-1302)","Кенген (1302-1303)","Каген (1303-1306)","Токуђи (1306-1308)","Енкеи (1308-1311)","Очо (1311-1312)","Шова (1312-1317)","Бунпо (1317-1319)","Ђено (1319-1321)","Ђенкјо (1321-1324)","Шочу (1324-1326)","Кареки (1326-1329)","Гентоку (1329-1331)","Генко (1331-1334)","Кему (1334-1336)","Енген (1336-1340)","Кококу (1340-1346)","Шохеи (1346-1370)","Кентоку (1370-1372)","Бучу (1372-1375)","Тењу (1375-1379)","Корјаку (1379-1381)","Кова (1381-1384)","Генчу (1384-1392)","Меитоку (1384-1387)","Какеи (1387-1389)","Ку (1389-1390)","Меитоку (1390-1394)","Оеи (1394-1428)","Шочо (1428-1429)","Еикјо (1429-1441)","Какитсу (1441-1444)","Бун-ан (1444-1449)","Хотоку (1449-1452)","Кјотоку (1452-1455)","Кошо (1455-1457)","Чороку (1457-1460)","Каншо (1460-1466)","Буншо (1466-1467)","Онин (1467-1469)","Бунмеи (1469-1487)","Чокјо (1487-1489)","Ентоку (1489-1492)","Меио (1492-1501)","Бунки (1501-1504)","Еишо (1504-1521)","Таиеи (1521-1528)","Кјороку (1528-1532)","Тенмон (1532-1555)","Кођи (1555-1558)","Еироку (1558-1570)","Генки (1570-1573)","Теншо (1573-1592)","Бунроку (1592-1596)","Кеичо (1596-1615)","Генва (1615-1624)","Кан-еи (1624-1644)","Шохо (1644-1648)","Кеиан (1648-1652)","Шу (1652-1655)","Меирјаку (1655-1658)","Мањи (1658-1661)","Канбун (1661-1673)","Енпо (1673-1681)","Тенва (1681-1684)","Јокјо (1684-1688)","Генроку (1688-1704)","Хоеи (1704-1711)","Шотоку (1711-1716)","Кјохо (1716-1736)","Генбун (1736-1741)","Канпо (1741-1744)","Енкјо (1744-1748)","Кан-ен (1748-1751)","Хорјаку (1751-1764)","Меива (1764-1772)","Ан-еи (1772-1781)","Тенмеи (1781-1789)","Кансеи (1789-1801)","Кјова (1801-1804)","Бунка (1804-1818)","Бунсеи (1818-1830)","Тенпо (1830-1844)","Кока (1844-1848)","Каеи (1848-1854)","Ансеи (1854-1860)","Ман-ен (1860-1861)","Бункју (1861-1864)","Генђи (1864-1865)","Кеико (1865-1868)","Меиђи","Таишо","Шова","Хаисеи","Фаравадин","Ордибехешт","Кордад","Тир","Мордад","Шахривар","Мехр","Абан","Азар","Деј","Бахман","Есфанд","АП","Пре РК","РК","КМ","Кч","зл","дин.","Тл","ልደት","ካብኽ","ክብላ","ፋጅኺ","ክቢቅ","ም/ት","ኰር","ማርያ","ያኸኒ","መተሉ","ም/ም","ተሕሳ","ልደትሪ","ካብኽብቲ","ፋጅኺሪ","ክቢቅሪ","ምኪኤል ትጟኒሪ","ኰርኩ","ማርያም ትሪ","ያኸኒ መሳቅለሪ","ምኪኤል መሽወሪ","ተሕሳስሪ","ሰ/ቅ","ሰኑ","ሰሊጝ","ለጓ","ኣምድ","ኣርብ","ሰ/ሽ","ሰንበር ቅዳዅ","ለጓ ወሪ ለብዋ","ሰንበር ሽጓዅ","ይጅ","ኣድ","ፋዱስ ጃብ","ፋዱስ ደምቢ","eB","GN","FB","MÇ","AB","MG","JN","JL","AG","ST","OC","NV","DS","gen.","feb.","març","abr.","maig","juny","jul.","ag.","set.","oct.","nov.","des.","gener","febrer","abril","juliol","agost","setembre","octubre","novembre","desembre","dg.","dl.","dt.","dc.","dj.","dv.","ds.","diumenge","dilluns","dimarts","dimecres","dijous","divendres","dissabte","aC","dC","a. de la n. e.","de la n.e.","abans de Crist","després de Crist","de la n. e.","a. m.","p. m.","₧","KBZ","KBR","KST","KKN","KTN","KMK","KMS","KMN","KMW","KKM","KNK","KNB","Okwokubanza","Okwakabiri","Okwakashatu","Okwakana","Okwakataana","Okwamukaaga","Okwamushanju","Okwamunaana","Okwamwenda","Okwaikumi","Okwaikumi na kumwe","Okwaikumi na ibiri","SAN","ORK","OKB","OKS","OKN","OKT","OMK","Sande","Orwokubanza","Orwakabiri","Orwakashatu","Orwakana","Orwakataano","Orwamukaaga","Kurisito Atakaijire","Kurisito Yaijire","USh","{month} {day}, {year}","ᎤᏃ","ᎧᎦ","ᎠᏅ","ᎧᏬ","ᎠᏂ","ᏕᎭ","ᎫᏰ","ᎦᎶ","ᏚᎵ","ᏚᏂ","ᏅᏓ","ᎥᏍ","ᎤᏃᎸᏔᏅ","ᎧᎦᎵ","ᎠᏅᏱ","ᎧᏬᏂ","ᎠᏂᏍᎬᏘ","ᏕᎭᎷᏱ","ᎫᏰᏉᏂ","ᎦᎶᏂ","ᏚᎵᏍᏗ","ᏚᏂᏅᏗ","ᏅᏓᏕᏆ","ᎥᏍᎩᏱ","ᏆᏍᎬ","ᏉᏅᎯ","ᏔᎵᏁ","ᏦᎢᏁ","ᏅᎩᏁ","ᏧᎾᎩ","ᏈᏕᎾ","ᎤᎾᏙᏓᏆᏍᎬ","ᎤᎾᏙᏓᏉᏅᎯ","ᏔᎵᏁᎢᎦ","ᏦᎢᏁᎢᎦ","ᏅᎩᏁᎢᎦ","ᏧᎾᎩᎶᏍᏗ","ᎤᎾᏙᏓᏈᏕᎾ","ᎤᏓᎷᎸ","ᎤᎶᏐᏅ","Ꮟ ᏥᏌ ᎾᏕᎲᏍᎬᎾ","ᎠᎩᏃᎮᎵᏓᏍᏗᏱ ᎠᏕᏘᏱᏍᎬ ᏱᎰᏩ ᏧᏓᏂᎸᎢᏍᏗ","ᏌᎾᎴ","ᏒᎯᏱᎢᏗᏢ","{weekday} {day}. {month}. {year} {hour}:{minute}:{second}","{weekday} {day}. {month}. {year} {hour}:{minute}:{second} {ampm}","{weekday} {day}. {month}. {year}","{day}. {month}. {year}","led","úno","bře","dub","kvě","čvn","čvc","srp","zář","říj","lis","pro","ledna","února","března","dubna","května","června","července","srpna","září","října","listopadu","prosince","ne","po","út","st","čt","pá","so","neděle","pondělí","úterý","středa","čtvrtek","pátek","sobota","př.n.l.","n.l.","př. n. l.","n. l.","Před R. O. C.","Kčs","EMU","ECU","I","Ch","E","G","Rh","Ion","Chwef","Mawrth","Ebrill","Mai","Meh","Gorff","Awst","Medi","Hyd","Tach","Rhag","Ionawr","Chwefror","Mehefin","Gorffennaf","Hydref","Tachwedd","Rhagfyr","Su","Ll","Me","Ia","Gw","Llun","Maw","Mer","Iau","Sad","Dydd Sul","Dydd Llun","Dydd Mawrth","Dydd Mercher","Dydd Iau","Dydd Gwener","Dydd Sadwrn","CC","Cyn Crist","Oed Crist","{weekday} {day}. {month} {year} {hour}.{minute}.{second}","{weekday} {day}. {month} {year} {hour}.{minute}.{second} {ampm}","{weekday} {day}. {month} {year}","jan.","mar.","apr.","jun.","aug.","sep.","okt.","dec.","marts","august","september","oktober","november","december","sø","ti","on","to","fr","lø","søn.","tir.","ons.","tor.","fre.","lør.","søndag","mandag","tirsdag","onsdag","torsdag","fredag","lørdag","fKr","eKr","fvt","vt","f.Kr.","e.Kr.","f.v.t.","v.t.","før vesterlandsk tidsregning","vesterlandsk tidsregning","Minguo","kr","Imb","Kaw","Kad","Kan","Kas","Kar","Mfu","Wun","Ike","Iku","Imw","Iwi","Mori ghwa imbiri","Mori ghwa kawi","Mori ghwa kadadu","Mori ghwa kana","Mori ghwa kasanu","Mori ghwa karandadu","Mori ghwa mfungade","Mori ghwa wunyanya","Mori ghwa ikenda","Mori ghwa ikumi","Mori ghwa ikumi na imweri","Mori ghwa ikumi na iwi","Jum","Jim","Ngu","Ituku ja jumwa","Kuramuka jimweri","Kuramuka kawi","Kuramuka kadadu","Kuramuka kana","Kuramuka kasanu","Kifula nguwo","KK","Kabla ya Kristo","Baada ya Kristo","Luma lwa K","luma lwa p","Ksh","{weekday}, {day}. {month} {year} {hour}:{minute}:{second}","{weekday}, {day}. {month} {year} {hour}:{minute}:{second} {ampm}","Jän.","Feb.","März","Apr.","Juli","Aug.","Sep.","Okt.","Nov.","Dez.","Jänner","Februar","August","Dezember","Mo.","Mi.","Fr.","Sonntag","Montag","Dienstag","Mittwoch","Donnerstag","Freitag","Samstag","v. Chr.","n. Chr.","vdZ","dZ","v. u. Z.","u. Z.","vor der gewöhnlichen Zeitrechnung","der gewöhnlichen Zeitrechnung","vorm.","nachm.","öS","Jan.","Januar","{currency}-{number}","'","Žan","Fee","Awi","Žuw","Žuy","Ut","Sek","Noo","Dee","Žanwiye","Feewiriye","Marsi","Awiril","Žuweŋ","Žuyye","Sektanbur","Oktoobur","Noowanbur","Deesanbur","Ati","Ata","Ala","Alm","Alz","Asi","Alhadi","Atinni","Atalaata","Alarba","Alzuma","Asibti","IJ","IZ","Isaa jine","Isaa zamanoo","Subbaahi","Zaarikay b","di","ŋgɔn","sɔŋ","diɓ","emi","esɔ","mad","diŋ","nyɛt","tin","elá","dimɔ́di","ŋgɔndɛ","sɔŋɛ","diɓáɓá","emiasele","esɔpɛsɔpɛ","madiɓɛ́díɓɛ́","diŋgindi","nyɛtɛki","mayésɛ́","tiníní","eláŋgɛ́","ét","mɔ́s","kwa","muk","ŋgi","ɗón","esa","éti","mɔ́sú","kwasú","mukɔ́sú","ŋgisú","ɗónɛsú","esaɓasú","ɓ.Ys","mb.Ys","ɓoso ɓwá yáɓe lá","mbúsa kwédi a Yés","idiɓa","ebyámu","Fe","Ab","Sú","Se","Ok","No","De","Sanvie","Fébirie","Mars","Aburil","Mee","Sueŋ","Súuyee","Settembar","Oktobar","Novembar","Disambar","Dim","Ten","Ara","Arj","Sib","Dimas","Teneŋ","Talata","Alarbay","Aramisay","Arjuma","Sibiti","ArY","AtY","Ariŋuu Yeesu","Atooŋe Yeesu","{weekday}, ལོ{year} ཟླ་{month} ཚེ་{day} {hour}:{minute}:{second}","{weekday}, ལོ{year} ཟླ་{month} ཚེ་{day} {hour}:{minute}:{second} {ampm}","{weekday}, ལོ{year} ཟླ་{month} ཚེ་{day}","{year} སྤྱི་ཟླ་{month}","{month} ཚེ་{day}","༡","༢","༣","༥","༦","༧","༨","༡༠","༡༡","༡༢","༤","༩","ཟླ་དངཔ་","ཟླ་གཉིས་པ་","ཟླ་གསུམ་པ་","ཟླ་བཞི་པ་","ཟླ་ལྔ་པ་","ཟླ་དྲུག་པ","ཟླ་བདུན་པ་","ཟླ་བརྒྱད་པ་","ཟླ་དགུ་པ་","ཟླ་བཅུ་པ་","ཟླ་བཅུ་གཅིག་པ་","ཟླ་བཅུ་གཉིས་པ་","ཟླ་","མིར་","ལྷག་","ཕུར་","སྤེན་","ཉི་","གཟའ་ལྷག་པ་","གཟའ་པ་སངས་","སྔ་ཆ་","ཕྱི་ཆ་","tibt","ཨང་མད","གྲངས་མེད","Nu.","KR₩","TH฿","Mbe","Kai","Kat","Gat","Gan","Mug","Knn","Ken","Igi","Mweri wa mbere","Mweri wa kaĩri","Mweri wa kathatũ","Mweri wa kana","Mweri wa gatano","Mweri wa gatantatũ","Mweri wa mũgwanja","Mweri wa kanana","Mweri wa kenda","Mweri wa ikũmi","Mweri wa ikũmi na ũmwe","Mweri wa ikũmi na Kaĩrĩ","Kma","Tat","Ine","Tan","Arm","Maa","NMM","Kiumia","Njumatatu","Njumaine","Njumatano","Aramithi","Njumaa","NJumamothii","MK","TK","Mbere ya Kristo","Thutha wa Kristo","KI","UT","{month} {day} lia, {year}","MMMM d 'li","MMMM d 'li{ampm}","dzv","dzd","ted","afɔ","dam","mas","sia","dea","any","kel","ade","dzm","dzove","dzodze","tedoxe","afɔfiẽ","damɛ","masa","siamlɔm","deasiamime","anyɔnyɔ","kele","adeɛmekpɔxe","dzome","foa","ƒoave","d","t","a","m","s","k","afɔfĩe","dama","kɔs","dzo","bla","kuɖ","yaw","fiɖ","mem","kɔsiɖa","dzoɖa","blaɖa","kuɖa","yawoɖa","fiɖa","memleɖa","hY","Yŋ","Hafi Yesu Va Do ŋgɔ","Yesu Ŋɔli","ŋdi","ɣetrɔ","dasiamime","hafi R.O.C.","mnn","{weekday}, {day} {month} {year} - {hour}:{minute}:{second}","{weekday}, {day} {month} {year} - {hour}:{minute}:{second} {ampm}","Ι","Φ","Μ","Α","Σ","Ο","Ν","Δ","Ιαν","Φεβ","Μαρ","Απρ","Μαΐ","Ιουν","Ιουλ","Αυγ","Σεπ","Οκτ","Νοε","Δεκ","Ιανουαρίου","Φεβρουαρίου","Μαρτίου","Απριλίου","Μαΐου","Ιουνίου","Ιουλίου","Αυγούστου","Σεπτεμβρίου","Οκτωβρίου","Νοεμβρίου","Δεκεμβρίου","Κυ","Δε","Τρ","Τε","Πέ","Πα","Σά","Κυρ","Δευ","Τρί","Τετ","Πέμ","Παρ","Σάβ","Κυριακή","Δευτέρα","Τρίτη","Τετάρτη","Πέμπτη","Παρασκευή","Σάββατο","π.Χ.","μ.Χ.","π.Κ.Χ.","ΚΧ","π.μ.","μ.μ.","Πριν R.O.C.","Δρχ","{weekday}, {month} {day}, {year}, {hour}:{minute}:{second}","{weekday}, {month} {day}, {year}, {hour}:{minute}:{second} {ampm}","Mo1","Mo2","Mo3","Mo4","Mo5","Mo6","Mo7","Mo8","Mo9","Mo10","Mo11","Mo12","Month1","Month2","Month3","Month4","Month5","Month6","Month7","Month8","Month9","Month10","Month11","Month12","May","Oct","January","February","March","June","July","October","December","Mo","Tu","We","Th","Fr","Sun","Mon","Tue","Wed","Thu","Fri","Sat","Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday","B","Before Christ","Anno Domini","Before Common Era","Common Era","{weekday} {day} {month} {year}, {hour}:{minute}:{second}","{weekday} {day} {month} {year}, {hour}:{minute}:{second} {ampm}","𐐖","𐐙","𐐣","𐐁","𐐂","𐐝","𐐉","𐐤","𐐔","𐐖𐐰𐑌","𐐙𐐯𐐺","𐐣𐐪𐑉","𐐁𐐹𐑉","𐐣𐐩","𐐖𐐭𐑌","𐐖𐐭𐑊","𐐂𐑀","𐐝𐐯𐐹","𐐉𐐿𐐻","𐐤𐐬𐑂","𐐔𐐨𐑅","𐐖𐐰𐑌𐐷𐐭𐐯𐑉𐐨","𐐙𐐯𐐺𐑉𐐭𐐯𐑉𐐨","𐐣𐐪𐑉𐐽","𐐁𐐹𐑉𐐮𐑊","𐐖𐐭𐑊𐐴","𐐂𐑀𐐲𐑅𐐻","𐐝𐐯𐐹𐐻𐐯𐑋𐐺𐐲𐑉","𐐉𐐿𐐻𐐬𐐺𐐲𐑉","𐐤𐐬𐑂𐐯𐑋𐐺𐐲𐑉","𐐔𐐨𐑅𐐯𐑋𐐺𐐲𐑉","𐐝𐐲𐑌","𐐣𐐲𐑌","𐐓𐐭𐑆","𐐎𐐯𐑌","𐐛𐐲𐑉","𐐙𐑉𐐴","𐐝𐐰𐐻","𐐝𐐲𐑌𐐼𐐩","𐐣𐐲𐑌𐐼𐐩","𐐓𐐭𐑆𐐼𐐩","𐐎𐐯𐑌𐑆𐐼𐐩","𐐛𐐲𐑉𐑆𐐼𐐩","𐐙𐑉𐐴𐐼𐐩","𐐝𐐰𐐻𐐲𐑉𐐼𐐩","𐐒","𐐈","𐐒𐐗","𐐈𐐔","𐐒𐐲𐑁𐐬𐑉 𐐗𐑉𐐴𐑅𐐻","𐐈𐑌𐐬 𐐔𐐱𐑋𐐮𐑌𐐨","𐐈𐐣","𐐑𐐣","am","pm","SAKA","{weekday}, {day} {month}, {year}, {hour}:{minute}:{second}","{weekday}, {day} {month}, {year}, {hour}:{minute}:{second} {ampm}","Bunchū","Genchū","a.m.","p.m.","{weekday} {day} {month}, {year}, {hour}:{minute}:{second}","{weekday} {day} {month}, {year}, {hour}:{minute}:{second} {ampm}","{weekday} {day} {month}, {year}","Ar","MOP$","{weekday}, {day} {month} {year}, {hour}:{minute}:{second}","{weekday}, {day} {month} {year}, {hour}:{minute}:{second} {ampm}","Rs","₦","₱","RF","SR","Le","NAf.","T$","VT","WS$","aŭg","januaro","februaro","marto","aprilo","majo","junio","julio","aŭgusto","septembro","oktobro","novembro","decembro","lu","me","ĵa","ve","dimanĉo","lundo","mardo","merkredo","ĵaŭdo","vendredo","sabato","aK","pK","atm","ptm","{weekday}, {day} de {month} de {year} {hour}:{minute}:{second}","{weekday}, {day} de {month} de {year} {hour}:{minute}:{second} {ampm}","{weekday}, {day} de {month} de {year}","{day} de {month} de {year}","{month} de {year}","{day} de {month}","ene.","may.","ago.","sept.","dic.","enero","febrero","marzo","mayo","agosto","septiembre","noviembre","diciembre","DO","LU","MA","MI","JU","VI","SA","dom.","lun.","mié.","jue.","vie.","sáb.","domingo","lunes","jueves","viernes","sábado","a. C.","d. C.","a. e. c.","e. c.","antes de Cristo","anno Dómini","antes de R.O.C.","AR$","{month}-{year}","Bs","{day}-{month}-{year}","₡","Q","L","My","Jn","Jl","Ag","febr.","mzo.","my.","do.","lu.","ma.","mi.","ju.","vi.","sá.","miér.","vier.","sáb","Naf","Kz","Afl.","ZMK","C$","B/.","S/.","{currency} -{number}","₲","Bs.","{weekday}, {day}. {month} {year} {hour}:{minute}.{second}","{weekday}, {day}. {month} {year} {hour}:{minute}.{second} {ampm}","{hour}:{minute}.{second}","{hour}:{minute}.{second} {ampm}","V","jaan","veebr","märts","mai","juuni","juuli","aug","sept","dets","jaanuar","veebruar","aprill","oktoober","detsember","pühapäev","esmaspäev","teisipäev","kolmapäev","neljapäev","reede","laupäev","e.m.a.","m.a.j.","ema","enne meie aega","meie aja järgi","BG","U","urt.","ots.","api.","mai.","eka.","uzt.","abu.","ira.","urr.","aza.","abe.","urtarrilak","otsailak","martxoak","apirilak","maiatzak","ekainak","uztailak","abuztuak","irailak","urriak","azaroak","abenduak","ig.","al.","ar.","az.","og.","or.","lr.","igandea","astelehena","asteartea","asteazkena","osteguna","ostirala","larunbata","K.a.","K.o.","R.O.C. aurretik","% {number}","-% {number}","ngo","ngb","ngl","ngn","ngt","ngs","ngz","ngm","nge","nga","ngad","ngab","ngɔn osú","ngɔn bɛ̌","ngɔn lála","ngɔn nyina","ngɔn tána","ngɔn saməna","ngɔn zamgbála","ngɔn mwom","ngɔn ebulú","ngɔn awóm","ngɔn awóm ai dziá","ngɔn awóm ai bɛ̌","sɔ́n","mɔ́n","smb","sml","smn","fúl","sér","sɔ́ndɔ","mɔ́ndi","sɔ́ndɔ məlú mə́bɛ̌","sɔ́ndɔ məlú mə́lɛ́","sɔ́ndɔ məlú mə́nyi","fúladé","séradé","oyk","ayk","osúsúa Yésus kiri","ámvus Yésus Kirís","kíkíríg","ngəgógəle","{weekday} {day} {month} {year}،‏ {hour}:{minute}:{second}","{weekday} {day} {month} {year}،‏ {hour}:{minute}:{second} {ampm}","ژ","ا","فوریهٔ","آوریل","مـی","ژوئن","جول","اوت","سپتامبر","اکتبر","نوامبر","دسم","جنوری","فبروری","مارچ","اپریل","می","جون","جولای","اگست","سپتمبر","اکتوبر","نومبر","دسمبر","۱ش","۲ش","۳ش","۴ش","۵ش","یکشنبه","دوشنبه","سه‌شنبه","چهارشنبه","پنجشنبه","جمعه","شنبه","ق","ق.م.","م.","قبل از میلاد","میلادی","قبل‌ازظهر","بعدازظهر","ک","ط","تشری","حشوان","کسلو","طوت","واذار","نیسان","ایار","سیوان","ایلول","واذار الثانی","ر","ذ","ربیع الاول","ربیع الثانی","جمادی الاول","جمادی الثانی","ذیقعدهٔ","ذیحجهٔ","ه‍.ق.","هجری قمری","خ","فروردین","اردیبهشت","تیر","شهریور","آبان","دی","اسفند","حمل","ثور","جوزا","سرطان","اسد","سنبلهٔ","میزان","عقرب","قوس","جدی","دلو","حوت","ه‍. ش.","هجری شمسی","arabext","‎{currency}{number}","-‎{currency}{number}","ناعدد","؋","$CA","¥CN","$HK","ریال","$MX","$NZ","$EC","ژانویهٔ","مهٔ","ژوئیهٔ","دسامبر","ه‍.ش.","sii","col","mbo","duu","kor","mor","juk","slt","yar","jol","bow","siilo","colte","mbooy","seeɗto","duujal","korse","morso","juko","siilto","yarkomaa","jolal","bowte","dew","aaɓ","maw","naa","mwd","hbi","dewo","aaɓnde","mawbaare","njeslaare","naasaande","mawnde","hoore-biir","H-I","C-I","Hade Iisa","Caggal Iisa","subaka","kikiiɗe","FG","UM","tammikuuta","helmikuuta","maaliskuuta","huhtikuuta","toukokuuta","kesäkuuta","heinäkuuta","elokuuta","syyskuuta","lokakuuta","marraskuuta","joulukuuta","su","ke","pe","la","sunnuntaina","maanantaina","tiistaina","keskiviikkona","torstaina","perjantaina","lauantaina","eK","jK","eaa","jaa","eKr.","jKr.","eaa.","jaa.","ennen Kristuksen syntymää","jälkeen Kristuksen syntymän","ennen ajanlaskun alkua","jälkeen ajanlaskun alun","ap.","ip.","tišríkuuta","hešvánkuuta","kislévkuuta","tevétkuuta","ševátkuuta","adárkuuta I","adárkuuta","nisánkuuta","ijjárkuuta","sivánkuuta","tammúzkuuta","abkuuta","elúlkuuta","adárkuuta II","muharram","safar","rabi’ al-awwal","rabi’ al-akhir","džumada-l-ula","džumada-l-akhira","radžab","ša’ban","ramadan","šawwal","dhu-l-qa’da","dhu-l-hiddža","epäluku","Ene","Peb","Abr","Hun","Hul","Set","Nob","Enero","Pebrero","Marso","Abril","Mayo","Hunyo","Hulyo","Agosto","Setyembre","Oktubre","Nobyembre","Disyembre","Li","Lu","Mi","Hu","Bi","Lin","Miy","Huw","Biy","Linggo","Lunes","Martes","Miyerkules","Huwebes","Biyernes","Sabado","{weekday}, {year} {month} {day} {hour}:{minute}:{second}","{weekday}, {year} {month} {day} {hour}:{minute}:{second} {ampm}","{weekday}, {year} {month} {day}","mars","apríl","desember","má","tý","hó","le","sun","mán","týs","mik","hós","frí","ley","sunnudagur","mánadagur","týsdagur","mikudagur","hósdagur","fríggjadagur","leygardagur","flt","lt","f.l.t.","l.t.","fyrir Krist","eftir Krist","fyrir látlaus tímatal","látlaus tímatal","um fyrrapartur","um seinnapartur","¤¤¤","dangi","E.B.","ère b.","ère bouddhiste","1yuè","2yuè","3yuè","4yuè","5yuè","6yuè","7yuè","8yuè","9yuè","10yuè","11yuè","12yuè","zhēngyuè","èryuè","sānyuè","sìyuè","wǔyuè","liùyuè","qīyuè","bāyuè","jiǔyuè","shíyuè","shíyīyuè","shí’èryuè","tout","bâb.","hât.","kya.","toub.","amsh.","barma.","barmo.","bash.","ba’o.","abî.","mis.","al-n.","bâbâ","hâtour","kyakh","toubah","amshîr","barmahât","barmoudah","bashans","ba’ounah","abîb","misra","al-nasi","av. D.","ap. D.","avant Dioclétien","après Dioclétien","mäs.","teq.","hed.","tah.","ter","yäk.","mäg.","miy.","gue.","sän.","ham.","näh.","pag.","mäskäräm","teqemt","hedar","tahesas","yäkatit","mägabit","miyazya","guenbot","säné","hamlé","nähasé","pagumén","av. Inc.","ap. Inc.","avant l’Incarnation","après l’Incarnation","janv.","févr.","avr.","juin","juil.","août","déc.","janvier","février","avril","juillet","septembre","octobre","décembre","je","dim.","jeu.","ven.","sam.","dimanche","lundi","mardi","mercredi","jeudi","vendredi","samedi","av. J.-C.","ap. J.-C.","AEC","EC","avant Jésus-Christ","après Jésus-Christ","tis.","hes.","kis.","téb.","sché.","ad.I","adar","nis.","iyar","siv.","tam.","ab","ell.","ad.II","Tisseri","Hesvan","Tébeth","Schébat","Nissan","Tamouz","Elloul","Anno Mundi","Ā","chai.","vai.","jyai.","āsha.","shrā.","bhā.","āshw.","kār.","mār.","pau.","māgh","phāl.","chaitra","vaishākh","jyaishtha","āshādha","shrāvana","bhādrapad","āshwin","kārtik","mārgashīrsha","paush","phālgun","mouh.","rab. aw.","rab. th.","joum. oul.","joum. tha.","raj.","chaa.","chaw.","dhou. q.","dhou. h.","mouharram","rabia al awal","rabia ath-thani","joumada al oula","joumada ath-thania","rajab","chaabane","chawwal","dhou al qi`da","dhou al-hijja","avant RdC","RdC","$AR","$AU","$BM","$BN","$BS","$BZ","$CL","$CO","$CV","£CY","£EG","$FJ","£FK","£GB","£GI","£IE","£IL","₤IT","¥JP","£LB","$LR","£MT","$NA","$RH","$SB","£SD","$SG","£SH","$SR","£SS","$TT","$TW","$US","$UY","FCFP","FBu","FC","DA","CF","LS","DT","Z","Zen","Fev","Avr","Jug","Lui","Avo","Otu","Dic","Zenâr","Fevrâr","Març","Avrîl","Jugn","Avost","Setembar","Otubar","Dicembar","joi","vin","domenie","lunis","martars","miercus","joibe","vinars","sabide","pdC","ddC","a.","p.","Ean","Feabh","Márta","Aib","Beal","Meith","Iúil","Lún","MFómh","DFómh","Samh","Noll","Eanáir","Feabhra","Aibreán","Bealtaine","Meitheamh","Lúnasa","Meán Fómhair","Deireadh Fómhair","Samhain","Nollaig","Domh","Luan","Máirt","Céad","Déar","Aoine","Sath","Dé Domhnaigh","Dé Luain","Dé Máirt","Dé Céadaoin","Déardaoin","Dé hAoine","Dé Sathairn","RC","Roimh Chríost","AÍ f.","Faoi","Gearr","Màrt","Gibl","Cèit","Ògmh","Iuch","Lùna","Sult","Dàmh","Dùbh","Am Faoilleach","An Gearran","Am Màrt","An Giblean","An Cèitean","An t-Ògmhios","An t-Iuchar","An Lùnastal","An t-Sultain","An Dàmhair","An t-Samhain","An Dùbhlachd","DiD","DiL","DiM","DiC","Dia","Dih","DiS","DiDòmhnaich","DiLuain","DiMàirt","DiCiadain","Diardaoin","DihAoine","DiSathairne","Ro Chrìost","f","xan","xuñ","xul","out","xaneiro","febreiro","maio","xuño","xullo","setembro","outubro","luns","mt","mc","xv","mér","xov","ven","mércores","xoves","venres","despois de Cristo","Mär","Dez","Auguscht","Septämber","Oktoober","Novämber","Dezämber","Su.","Mä.","Zi.","Du.","Sunntig","Määntig","Ziischtig","Mittwuch","Dunschtig","Friitig","Samschtig","v. d. Z.","d. Z.","nam.","’","જા","ફે","મા","એ","મે","જૂ","જુ","ઑ","સ","ન","ડિ","જાન્યુ","ફેબ્રુ","માર્ચ","એપ્રિલ","જૂન","જુલાઈ","ઑગસ્ટ","સપ્ટે","ઑક્ટો","નવે","ડિસે","જાન્યુઆરી","ફેબ્રુઆરી","સપ્ટેમ્બર","ઑક્ટોબર","નવેમ્બર","ડિસેમ્બર","ર","સો","મં","બુ","ગુ","શુ","શ","રવિ","સોમ","મંગળ","બુધ","ગુરુ","શુક્ર","શનિ","રવિવાર","સોમવાર","મંગળવાર","બુધવાર","ગુરુવાર","શુક્રવાર","શનિવાર","ઇ સ પુ","ઇસ","ઈસુના જન્મ પહેલા","ઇસવીસન","સામાન્ય યુગ પહેલા","સામાન્ય યુગ","ઈસવીસન પૂર્વે","Can","Cul","Agt","Chanuari","Feburari","Apiriri","Chulai","Okitoba","Nobemba","Cpr","Ctt","Cmn","Cmt","Ars","Icm","Est","Chumapiri","Chumatato","Chumaine","Chumatano","Aramisi","Ichuma","Esabato","YA","YK","Yeso ataiborwa","Yeso kaiboirwe","Ma/Mo","Mambia/Mog","J-guer","T-arree","Mayrnt","Avrril","Boaldyn","M-souree","J-souree","Luanistyn","M-fouyir","J-fouyir","M.Houney","M.Nollick","Jerrey-geuree","Toshiaght-arree","Averil","Mean-souree","Jerrey-souree","Mean-fouyir","Jerrey-fouyir","Mee Houney","Mee ny Nollick","Jed","Jel","Jem","Jerc","Jerd","Jeh","Jes","Jedoonee","Jelhein","Jemayrt","Jercean","Jerdein","Jeheiney","Jesarn","Fab","Afi","Yun","Yul","Agu","Nuw","Janairu","Faburairu","Maris","Afirilu","Mayu","Yuni","Yuli","Agusta","Satumba","Nuwamba","Disamba","Lh","Ta","Lr","Al","Ju","As","Lahadi","Litinin","Laraba","Alhamis","Jumma'a","Asabar","KHAI","BHAI","Kafin haihuwar annab","Bayan haihuwar annab","Ian.","Pep.","Mal.","ʻAp.","Iun.","Iul.","ʻAu.","Kep.","ʻOk.","Now.","Kek.","Ianuali","Pepeluali","Malaki","ʻApelila","Iune","Iulai","ʻAukake","Kepakemapa","ʻOkakopa","Nowemapa","Kekemapa","LP","P1","P2","P3","P4","P5","P6","Lāpule","Poʻakahi","Poʻalua","Poʻakolu","Poʻahā","Poʻalima","Poʻaono","{weekday}, {day} ב{month} {year}, {hour}:{minute}:{second}","{weekday}, {day} ב{month} {year}, {hour}:{minute}:{second} {ampm}","{weekday}, {day} ב{month} {year}","{day} ב{month} {year}","{day} ב{month}","ינו","פבר","מרץ","אפר","מאי","יונ","יול","אוג","ספט","אוק","נוב","דצמ","ינואר","פברואר","אפריל","יוני","יולי","אוגוסט","ספטמבר","אוקטובר","נובמבר","דצמבר","א׳","ב׳","ג׳","ד׳","ה׳","ו׳","ש׳","יום א׳","יום ב׳","יום ג׳","יום ד׳","יום ה׳","יום ו׳","שבת","יום ראשון","יום שני","יום שלישי","יום רביעי","יום חמישי","יום שישי","יום שבת","לפנה״ס","לסה״נ","לפני הספירה","לספירה","לפנה״צ","אחה״צ","תש","חש","כס","טב","שב","אא","אד","ני","אי","סי","תמ","אב","אל","א2","תשרי","חשון","כסלו","טבת","שבט","אדר א׳","אדר","ניסן","אייר","סיון","תמוז","אלול","אדר ב׳","לבה״ע","מוחרם","צפר","רביע א׳","רביע ב׳","ג׳ומאדא א׳","ג׳ומאדא ב׳","רג׳ב","שעבאן","רמדאן","שוואל","ד׳ו אל-קעדה","ד׳ו אל-חיג׳ה","רביע אל-אוול","רביע א-ת׳אני","ג׳ומאדא אל-אולא","ג׳ומאדא א-ת׳אניה","שנת היג׳רה","טאיקה","נינג׳ו","שוטוקו","ל״י","मस्केरेम","टेकेम्ट","हेदर","तहसास","टर","येकाटिट","मेगाबिट","मियाज़िया","गनबोट","सेन","हम्ले","नेहासे","पागूमन","ज","फ़","मा","अ","म","जू","जु","सि","न","दि","जन","फ़र","मार्च","अप्रै","मई","जून","जुला","अग","सितं","अक्टू","नवं","दिसं","जनवरी","फ़रवरी","अप्रैल","जुलाई","अगस्त","सितंबर","अक्टूबर","नवंबर","दिसंबर","र","सो","मं","बु","गु","शु","श","रवि","सोम","बुध","गुरु","शुक्र","शनि","रविवार","सोमवार","मंगलवार","बुधवार","गुरुवार","शुक्रवार","शनिवार","ईसा-पूर्व","ईस्वी","पूर्वाह्न","अपराह्न","चैत्र","वैशाख","ज्येष्ठ","आषाढ़","श्रावण","अश्विन","कार्तिक","अग्रहायण","पौष","माघ","फाल्गुन","शक","मुहर्रम","सफर","राबी प्रथम","राबी द्वितीय","जुम्डा प्रथम","जुम्डा द्वितीय","रजब","शावन","रमजान","शव्व्ल","जिल-क्दाह","जिल्-हिज्जाह","ताएका (645–650)","हाकूची (650–671)","हाकूहो (672–686)","शूचो (686–701)","ताहिओ (701–704)","केउन (704–708)","वाडू (708–715)","रैकी (715–717)","योरो (717–724)","जिंकी (724–729)","टेम्प्यो (729–749)","टेम्प्यो-काम्पो (749–749)","टेम्प्यो-शोहो (749–757)","टेम्प्यो-होजी (757–765)","टेम्प्यो-जिंगो (765–767)","टेम्प्यो-किउन (767–770)","होकी (770–780)","टेनो (781–782)","इंर्याकू (782–806)","डाईडू (806–810)","क़ोनिन (810–824)","टेंचो (824–834)","शोवा (834–848)","काज्यो (848–851)","निंजू (851–854)","शाईकू (854–857)","टेनन (857–859)","जोगन् (859–877)","गेंकेई (877–885)","निन्ना (885–889)","केम्प्यो (889–898)","शूताई (898–901)","ईंगी (901–923)","ईंचो (923–931)","शोहेई (931–938)","टेंग्यो (938–947)","टेंर्याकू (947–957)","टेंटूकू (957–961)","ओवा (961–964)","कोहो (964–968)","अन्ना (968–970)","टेंरोकू (970–973)","टेन-एन (973–976)","जोगन् (976–978)","टेंगेन (978–983)","ईकान (983–985)","कन्ना (985–987)","ई-एन (987–989)","एइसो (989–990)","शोर्याकू (990–995)","चोटूकु (995–999)","चोहो (999–1004)","कंको (1004–1012)","च्योवा (1012–1017)","कन्निन (1017–1021)","ज़ियान (1021–1024)","मंजू (1024–1028)","चोगन (1028–1037)","चोर्याकू (1037–1040)","चोक्यु (1040–1044)","कांटूको (1044–1046)","ईशो (1046–1053)","टेंगी (1053–1058)","कोहैइ (1058–1065)","जिर्याकू (1065–1069)","ईंक्यू (1069–1074)","सोहो (1074–1077)","शोर्याकू (1077–1081)","ईहो (1081–1084)","ओटूको (1084–1087)","कांजि (1087–1094)","कोहो (1094–1096)","ईचो (1096–1097)","शोटूको (1097–1099)","कोवा (1099–1104)","चोजी (1104–1106)","काशो (1106–1108)","टेन्निन (1108–1110)","टेन-ई (1110–1113)","ईक्यू (1113–1118)","जेन-ई (1118–1120)","होआन (1120–1124)","तेंजी (1124–1126)","दाईजी (1126–1131)","टेंशो (1131–1132)","चोशो (1132–1135)","होएन (1135–1141)","ईजी (1141–1142)","कोजी (1142–1144)","टेन्यो (1144–1145)","क्यूआन (1145–1151)","निंपैई (1151–1154)","क्योजो (1154–1156)","होगेन (1156–1159)","हैजी (1159–1160)","ईर्याकू (1160–1161)","ओहो (1161–1163)","चोकान (1163–1165)","ईमान (1165–1166)","निन-आन (1166–1169)","काओ (1169–1171)","शोअन (1171–1175)","अंजन (1175–1177)","जिशो (1177–1181)","योवा (1181–1182)","जूऐई (1182–1184)","जेंर्याकू (1184–1185)","बूंजी (1185–1190)","केंक्यू (1190–1199)","शोजी (1199–1201)","केन्निन (1201–1204)","जेंक्यू (1204–1206)","केन-ई (1206–1207)","शोगेन (1207–1211)","केंर्याकू (1211–1213)","केंपो (1213–1219)","शोक्यू (1219–1222)","जू (1222–1224)","जेन्निन (1224–1225)","कोरोकू (1225–1227)","अंटैइ (1227–1229)","कांकी (1229–1232)","जोएई (1232–1233)","टेम्पूकू (1233–1234)","बुंर्याकू (1234–1235)","काटेई (1235–1238)","र्याकूनिन (1238–1239)","ईन-ओ (1239–1240)","निंजी (1240–1243)","कांजेन (1243–1247)","होजी (1247–1249)","केंचो (1249–1256)","कोगेन (1256–1257)","शोका (1257–1259)","शोगेन (1259–1260)","बुन-ओ (1260–1261)","कोचो (1261–1264)","बुन-ई (1264–1275)","केंजी (1275–1278)","कोअन (1278–1288)","शो (1288–1293)","ईनिन (1293–1299)","शोअन (1299–1302)","केंजेन (1302–1303)","काजेन (1303–1306)","टोकूजी (1306–1308)","ईंकेई (1308–1311)","ओचो (1311–1312)","शोवा (1312–1317)","बुंपो (1317–1319)","जेनो (1319–1321)","जेंक्यो (1321–1324)","शोचू (1324–1326)","कारेकी (1326–1329)","जेंटोकू (1329–1331)","गेंको (1331–1334)","केम्मू (1334–1336)","ईंजेन (1336–1340)","कोकोकू (1340–1346)","शोहेई (1346–1370)","केंटोकू (1370–1372)","बूंचो (1372–1375)","टेंजो (1375–1379)","कोर्याकू (1379–1381)","कोवा (1381–1384)","जेंचू (1384–1392)","मेटोकू (1384–1387)","काकेई (1387–1389)","कू (1389–1390)","मेटोकू (1390–1394)","ओई (1394–1428)","शोचो (1428–1429)","ईक्यो (1429–1441)","काकीत्सू (1441–1444)","बुन-अन (1444–1449)","होटोकू (1449–1452)","क्योटोकू (1452–1455)","कोशो (1455–1457)","चोरोकू (1457–1460)","कांशो (1460–1466)","बुंशो (1466–1467)","ओनिन (1467–1469)","बुन्मेई (1469–1487)","चोक्यो (1487–1489)","ईंटोकू (1489–1492)","मेईओ (1492–1501)","बुंकी (1501–1504)","ईशो (1504–1521)","ताईएई (1521–1528)","क्योरोकू (1528–1532)","टेन्मन (1532–1555)","कोजी (1555–1558)","ईरोकू (1558–1570)","जेंकी (1570–1573)","टेंशो (1573–1592)","बुंरोकू (1592–1596)","केईचो (1596–1615)","जेनवा (1615–1624)","कान-एई (1624–1644)","शोहो (1644–1648)","केईआन (1648–1652)","शो (1652–1655)","मेईर्याकू (1655–1658)","मानजी (1658–1661)","कनबुन (1661–1673)","ईंपो (1673–1681)","टेंवा (1681–1684)","जोक्यो (1684–1688)","जेंरोकू (1688–1704)","होएई (1704–1711)","शोटूको (1711–1716)","क्योहो (1716–1736)","जेंबुन (1736–1741)","कांपो (1741–1744)","इंक्यो (1744–1748)","कान-एन (1748–1751)","होर्याकू (1751–1764)","मेईवा (1764–1772)","अन-एई (1772–1781)","टेनमेई (1781–1789)","कांसेई (1789–1801)","क्योवा (1801–1804)","बुंका (1804–1818)","बुंसेई (1818–1830)","टेंपो (1830–1844)","कोका (1844–1848)","काईए (1848–1854)","अंसेई (1854–1860)","मान-ईन (1860–1861)","बुंक्यौ (1861–1864)","जेंजी (1864–1865)","केईओ (1865–1868)","मेजी","ताईशो","शोवा","हेईसेई","फर्वादिन","ओर्दिवेहेस्ट","खोरर्दाद","टिर","मोरदाद","शाहरीवर्","मेहर","अवन","अज़र","डे","बहमन","ईस्फन्द्","{month}. {year}","1.","2.","3.","4.","5.","6.","7.","8.","9.","10.","11.","12.","sij","velj","ožu","tra","svi","lip","kol","ruj","stu","siječnja","veljače","ožujka","travnja","svibnja","lipnja","srpnja","kolovoza","rujna","listopada","studenoga","prosinca","pr.n.e.","pr. Kr.","p. Kr.","pr. n. e.","n.e.","Prije Krista","Poslije Krista","prije R.O.C.","kn","{year}. {month} {day}., {weekday}  {hour}:{minute}:{second}","{year}. {month} {day}., {weekday} {ampm} {hour}:{minute}:{second}","{year}. {month} {day}., {weekday}","{year}. {month} {day}","{year}. {month}. {day}","{year}. {month}","{month}. {day}","{ampm} {hour}:{minute}:{second}","{ampm} {hour}:{minute}","Á","Sz","márc.","ápr.","máj.","jún.","júl.","szept.","január","február","március","április","május","június","július","augusztus","szeptember","október","Sze","Cs","Szo","vasárnap","hétfő","kedd","szerda","csütörtök","péntek","szombat","ie.","isz.","i. e.","i. sz.","Kr. e.","Kr. u.","időszámításunk előtt","időszámításunk szerint","de.","du.","Tisri","Hesván","Kiszlév","Tévész","Svát","Ádár I","Ádár","Niszán","Ijár","Sziván","Áv","Ádár II","TÉ","Moh.","Rébi I","Rébi II","Dsem. I","Dsem. II","Red.","Sab.","Sev.","Dsül k.","Dsül h.","Moharrem","Rébi el avvel","Rébi el accher","Dsemádi el avvel","Dsemádi el accher","Redseb","Sabán","Ramadán","Sevvál","Dsül kade","Dsül hedse","MF","R.O.C. előtt","Ft","{year}թ. {month} {day}, {weekday}, {hour}:{minute}:{second}","{year}թ. {month} {day}, {weekday}, {hour}:{minute}:{second} {ampm}","{year}թ. {month} {day}, {weekday}","{year}թ. {month}","Հ","Փ","Մ","Ա","Օ","Ս","Ն","Դ","հնվ","փտվ","մրտ","ապր","մյս","հնս","հլս","օգս","սպտ","հկտ","նյմ","դկտ","հունվարի","փետրվարի","մարտի","ապրիլի","մայիսի","հունիսի","հուլիսի","օգոստոսի","սեպտեմբերի","հոկտեմբերի","նոյեմբերի","դեկտեմբերի","կիր","երկ","երք","չրք","հնգ","ուր","շբթ","կիրակի","երկուշաբթի","երեքշաբթի","չորեքշաբթի","հինգշաբթի","ուրբաթ","շաբաթ","մ.թ.ա.","մ.թ.","կեսօրից առաջ","կեսօրից հետո","դր.","oct","januario","februario","martio","augusto","decembre","mer","jov","dominica","lunedi","martedi","mercuridi","jovedi","venerdi","sabbato","a.Chr.","p.Chr.","ante Christo","post Christo","Maret","Agustus","Min","Sen","Sel","Rab","Minggu","Senin","Selasa","Rabu","Kamis","Jumat","Sabtu","SM","SEU","EU","Syaw.","Sya'ban","Ramadhan","Syawal","Sebelum R.O.C.","Rp","Jen","Juu","Ọgọ","Ọkt","Jenụwarị","Febrụwarị","Maachị","Eprel","Juun","Julaị","Ọgọọst","Ọktoba","Ụka","Mọn","Tiu","Wen","Tọọ","Fraị","Mbọsị Ụka","Mọnde","Tiuzdee","Wenezdee","Tọọzdee","Fraịdee","Satọdee","T.K.","A.K.","Tupu Kristi","Afọ Kristi","A.M.","P.M.","ꋍꆪ","ꑍꆪ","ꌕꆪ","ꇖꆪ","ꉬꆪ","ꃘꆪ","ꏃꆪ","ꉆꆪ","ꈬꆪ","ꊰꆪ","ꊰꊪꆪ","ꊰꑋꆪ","ꑭꆏ","ꆏꋍ","ꆏꑍ","ꆏꌕ","ꆏꇖ","ꆏꉬ","ꆏꃘ","ꑭꆏꑍ","ꆏꊂꋍ","ꆏꊂꑍ","ꆏꊂꌕ","ꆏꊂꇖ","ꆏꊂꉬ","ꆏꊂꃘ","ꃅꋊꂿ","ꃅꋊꊂ","ꎸꄑ","ꁯꋒ","{weekday}, {day}. {month} {year}, {hour}:{minute}:{second}","{weekday}, {day}. {month} {year}, {hour}:{minute}:{second} {ampm}","maí","ágú.","nóv.","janúar","febrúar","júní","júlí","ágúst","nóvember","su.","má.","þr.","fi.","fö.","la.","sun.","mán.","þri.","mið.","fim.","fös.","lau.","mánudagur","þriðjudagur","miðvikudagur","fimmtudagur","föstudagur","laugardagur","f.k.","e.k.","f.h.","e.h.","EB","gen","mag","giu","lug","ott","dic","gennaio","febbraio","aprile","maggio","giugno","luglio","settembre","ottobre","dicembre","gio","domenica","lunedì","martedì","mercoledì","giovedì","venerdì","Prima della R.O.C.","{year}年{month}月{day}日({weekday}) {hour}:{minute}:{second}","{year}年{month}月{day}日({weekday}) {ampm}{hour}:{minute}:{second}","{year}年{month}月{day}日({weekday}","{year}年{month}月{day}","{year}年{month}","{month}月{day}","{ampm}{hour}:{minute}:{second}","{ampm}{hour}:{minute}","仏暦","正","二","三","四","五","六","七","八","九","十","十一","十二","正月","二月","三月","四月","五月","六月","七月","八月","九月","十月","十一月","十二月","トウト","ババ","ハトール","キアック","トーバ","アムシール","バラムハート","バラモウダ","バシャンス","パオーナ","エペープ","メスラ","ナシエ","メスケレム","テケムト","ヘダル","ターサス","テル","イェカティト","メガビト","ミアジア","ゲンボト","セネ","ハムレ","ネハッセ","パグメン","1月","2月","3月","4月","5月","6月","7月","8月","9月","10月","11月","12月","日","月","火","水","木","金","土","日曜日","月曜日","火曜日","水曜日","木曜日","金曜日","土曜日","紀元前","西暦","西暦紀元前","午前","午後","ティスレ","へシボン","キスレブ","テベット","シバット","アダル I","アダル","ニサン","イヤル","シバン","タムズ","アヴ","エルル","アダル II","カイトラ","ヴァイサカ","ジャイスタ","アーサダ","スラバナ","バードラ","アスビナ","カルディカ","アヴラハヤナ","パウサ","マーガ","パルグナ","サカ","ムハッラム","サフアル","ラビー・ウル・アウワル","ラビー・ウッ・サーニー","ジュマーダル・アウワル","ジュマーダッサーニー","ラジャブ","シャアバーン","ラマダーン","シャウワール","ズル・カイダ","ズル・ヒッジャ","大化","白雉","白鳯","朱鳥","大宝","慶雲","和銅","霊亀","養老","神亀","天平","天平感宝","天平勝宝","天平宝字","天平神護","神護景雲","宝亀","天応","延暦","大同","弘仁","天長","承和","嘉祥","仁寿","斉衡","天安","貞観","元慶","仁和","寛平","昌泰","延喜","延長","承平","天慶","天暦","天徳","応和","康保","安和","天禄","天延","貞元","天元","永観","寛和","永延","永祚","正暦","長徳","長保","寛弘","長和","寛仁","治安","万寿","長元","長暦","長久","寛徳","永承","天喜","康平","治暦","延久","承保","承暦","永保","応徳","寛治","嘉保","永長","承徳","康和","長治","嘉承","天仁","天永","永久","元永","保安","天治","大治","天承","長承","保延","永治","康治","天養","久安","仁平","久寿","保元","平治","永暦","応保","長寛","永万","仁安","嘉応","承安","安元","治承","養和","寿永","元暦","文治","建久","正治","建仁","元久","建永","承元","建暦","建保","承久","貞応","元仁","嘉禄","安貞","寛喜","貞永","天福","文暦","嘉禎","暦仁","延応","仁治","寛元","宝治","建長","康元","正嘉","正元","文応","弘長","文永","建治","弘安","正応","永仁","正安","乾元","嘉元","徳治","延慶","応長","正和","文保","元応","元亨","正中","嘉暦","元徳","元弘","建武","延元","興国","正平","建徳","文中","天授","康暦","弘和","元中","至徳","嘉慶","康応","明徳","応永","正長","永享","嘉吉","文安","宝徳","享徳","康正","長禄","寛正","文正","応仁","文明","長享","延徳","明応","文亀","永正","大永","享禄","天文","弘治","永禄","元亀","天正","文禄","慶長","元和","寛永","正保","慶安","承応","明暦","万治","寛文","延宝","天和","貞享","元禄","宝永","正徳","享保","元文","寛保","延享","寛延","宝暦","明和","安永","天明","寛政","享和","文化","文政","天保","弘化","嘉永","安政","万延","文久","元治","慶応","明治","大正","昭和","平成","ファルヴァルディーン","オルディーベヘシュト","ホルダード","ティール","モルダード","シャハリーヴァル","メフル","アーバーン","アーザル","デイ","バフマン","エスファンド","民国前","民国","元","￥","￦","{month}.{day}.{year}","Nduŋmbi Saŋ","Pɛsaŋ Pɛ́pá","Pɛsaŋ Pɛ́tát","Pɛsaŋ Pɛ́nɛ́kwa","Pɛsaŋ Pataa","Pɛsaŋ Pɛ́nɛ́ntúkú","Pɛsaŋ Saambá","Pɛsaŋ Pɛ́nɛ́fɔm","Pɛsaŋ Pɛ́nɛ́pfúꞋú","Pɛsaŋ Nɛgɛ́m","Pɛsaŋ Ntsɔ̌pmɔ́","Pɛsaŋ Ntsɔ̌ppá","Sɔ́ndi","Mɔ́ndi","Ápta Mɔ́ndi","Wɛ́nɛsɛdɛ","Tɔ́sɛdɛ","Fɛlâyɛdɛ","Sásidɛ","tsɛttsɛt mɛŋgu’ mi ɛ́ lɛɛnɛ Kɛlísɛtɔ gɔ ńɔ́","tsɛttsɛt mɛŋgu’ mi ɛ́ fúnɛ Kɛlísɛtɔ tɔ́ mɔ́","mba’mba’","ŋka mbɔ́t nji","Aprilyi","Junyi","Julyai","Agusti","Iju","Jumapilyi","Jumatatuu","Jumatanu","Kabla ya Kristu","Baada ya Kristu","utuko","kyiukonyi","ი","თ","მ","ა","ს","ო","ნ","დ","იან","თებ","მარ","აპრ","მაი","ივნ","ივლ","აგვ","სექ","ოქტ","ნოე","დეკ","იანვარი","თებერვალი","მარტი","აპრილი","მაისი","ივნისი","ივლისი","აგვისტო","სექტემბერი","ოქტომბერი","ნოემბერი","დეკემბერი","კვ","ორ","სმ","ოთ","ხთ","პრ","შბ","კვი","ორშ","სამ","ოთხ","ხუთ","პარ","შაბ","კვირა","ორშაბათი","სამშაბათი","ოთხშაბათი","ხუთშაბათი","პარასკევი","შაბათი","ძვ. წ.","ახ. წ.","ძველი წელთაღრიცხვით","ახალი წელთაღრიცხვით","დილის","საღამოს","თიშრეი","ხეშვანი","ქისლევი","ტევეთი","შვატი","ადარი I","ადარი","ნისანი","იარი","სივანი","თამუზი","ავი","ელული","ადარი II","მუჰ.","საფ.","რაბ. I","რაბ. II","ჯუმ. I","ჯუმ. II","რაჯ.","შაბ.","რამ.","შავ.","ზულ-კ.","ზულ-ჰ.","მუჰარამი","საფარი","რაბი ულ-ავალი","რაბი ულ-ახირი","ჯუმადა ულ-ავალი","ჯუმადა ულ-ახირი","რაჯაბი","შაბანი","რამადანი","შავალი","ზულ-კაადა","ზულ-ჰიჯა","ფარვარდინი","ორდიბეჰეშთი","ხორდადი","თირი","მორდადი","შაჰრივარი","მეჰრი","აბანი","აზარი","დეი","ბაჰმანი","ესფანდი","არ არის რიცხვი","Yen","Fur","Meɣ","Yeb","Ɣuc","Cte","Tub","Nun","Duǧ","Yennayer","Fuṛar","Meɣres","Yebrir","Mayyu","Yunyu","Yulyu","Ɣuct","Ctembeṛ","Tubeṛ","Nunembeṛ","Duǧembeṛ","Yan","San","Kraḍ","Kuẓ","Sam","Sḍis","Say","Yanass","Sanass","Kraḍass","Kuẓass","Samass","Sḍisass","Sayass","snd. T.Ɛ","sld. T.Ɛ","send talalit n Ɛisa","seld talalit n Ɛisa","n tufat","n tmeddit","Kel","Ktũ","Ktn","Tha","Moo","Nya","Knd","Ĩku","Ĩkm","Ĩkl","Mwai wa mbee","Mwai wa kelĩ","Mwai wa katatũ","Mwai wa kana","Mwai wa katano","Mwai wa thanthatũ","Mwai wa muonza","Mwai wa nyaanya","Mwai wa kenda","Mwai wa ĩkumi","Mwai wa ĩkumi na ĩmwe","Mwai wa ĩkumi na ilĩ","Wky","Wkw","Wkl","Wtũ","Wkn","Wtn","Wth","Wa kyumwa","Wa kwambĩlĩlya","Wa kelĩ","Wa katatũ","Wa kana","Wa katano","Wa thanthatũ","MY","IY","Mbee wa Yesũ","Ĩtina wa Yesũ","Ĩyakwakya","Ĩyawĩoo","Mwedi Ntandi","Mwedi wa Pili","Mwedi wa Tatu","Mwedi wa Nchechi","Mwedi wa Nnyano","Mwedi wa Nnyano na Umo","Mwedi wa Nnyano na Mivili","Mwedi wa Nnyano na Mitatu","Mwedi wa Nnyano na Nchechi","Mwedi wa Nnyano na Nnyano","Mwedi wa Nnyano na Nnyano na U","Mwedi wa Nnyano na Nnyano na M","Ll2","Ll3","Ll4","Ll5","Ll6","Ll7","Ll1","Liduva lyapili","Liduva lyatatu","Liduva lyanchechi","Liduva lyannyano","Liduva lyannyano na linji","Liduva lyannyano na mavili","Liduva litandi","AY","NY","Akanapawa Yesu","Nankuida Yesu","Muhi","Chilo","{weekday}, {day} di {month} di {year} {hour}:{minute}:{second}","{weekday}, {day} di {month} di {year} {hour}:{minute}:{second} {ampm}","{weekday}, {day} di {month} di {year}","{month} di {year}","Nuv","Diz","Janeru","Fevereru","Marsu","Maiu","Junhu","Julhu","Agostu","Setenbru","Otubru","Nuvenbru","Dizenbru","du","si","te","ku","ki","se","sig","kua","kin","ses","dumingu","sigunda-fera","tersa-fera","kuarta-fera","kinta-fera","sesta-fera","sabadu","DK","Antis di Kristu","Dispos di Kristu","Alj","Ass","Atini","Atalata","Alhamiisa","Aljuma","Assabdu","Isaa jamanoo","Adduha","Aluula","JEN","WKR","WGT","WKN","WTN","WTD","WMJ","WNN","WKD","WIK","WMW","DIT","Njenuarĩ","Mwere wa kerĩ","Mwere wa gatatũ","Mwere wa kana","Mwere wa gatano","Mwere wa gatandatũ","Mwere wa mũgwanja","Mwere wa kanana","Mwere wa kenda","Mwere wa ikũmi","Mwere wa ikũmi na ũmwe","Ndithemba","KMA","NTT","NMN","NMT","ART","NMA","Njumatatũ","Njumatana","Njumamothi","Kiroko","Hwaĩ-inĩ","қаң.","ақп.","нау.","сәу.","мам.","мау.","шіл.","там.","қыр.","қаз.","қар.","желт.","қаңтар","ақпан","наурыз","сәуір","мамыр","маусым","шілде","тамыз","қыркүйек","қазан","қараша","желтоқсан","жс.","дс.","сс.","ср.","бс.","жм.","сб.","жексенбі","дуйсенбі","сейсенбі","сәрсенбі","бейсенбі","жұма","сенбі","б.з.д.","б.з.","түске дейін","түстен кейін","₸","{day}/{month} {year}","pamba","wanja","mbiyɔ mɛndoŋgɔ","Nyɔlɔmbɔŋgɔ","Mɔnɔ ŋgbanja","Nyaŋgwɛ ŋgbanja","kuŋgwɛ","fɛ","njapi","nyukul","ɓulɓusɛ","sɔndi","mɛrkɛrɛdi","yedi","vaŋdɛrɛdi","mɔnɔ sɔndi","januari","februari","martsi","aprili","maji","augustusi","septemberi","oktoberi","novemberi","decemberi","ata","pin","sis","tal","arf","sabaat","ataasinngorneq","marlunngorneq","pingasunngorneq","sisamanngorneq","tallimanngorneq","arfininngorneq","Kr.s.","Kr.k.","Kr.in.si.","Kr.in.king.","Kristusip inunngornerata siornagut","Kristusip inunngornerata kingornagut","ulloqeqqata-tungaa","ulloqeqqata-kingorna","Nkr","Skr","Nga","Kip","Iwa","Nge","Wak","Rop","Kog","Bur","Epe","Aen","Mulgul","Ng'atyato","Kiptamo","Iwat kut","Ng'eiyet","Waki","Roptui","Kipkogaga","Buret","Epeso","Kipsunde netai","Kipsunde nebo aeng","Som","Ang","Mut","Loh","Betutab tisap","Betut netai","Betutab aeng'","Betutab somok","Betutab ang'wan","Betutab mut","Betutab lo","KO","Amait kesich Jesu","Kokakesich Jesu","Beet","Kemo","ព.ស.","មករា","កុម្ភៈ","មីនា","មេសា","ឧសភា","មិថុនា","កក្កដា","សីហា","កញ្ញា","តុលា","វិច្ឆិកា","ធ្នូ","អាទិត្យ","ចន្ទ","អង្គារ","ពុធ","ព្រហស្បតិ៍","សុក្រ","សៅរ៍","មុន គ.ស.","គ.ស.","ក្រោយ គ.ស.","មុន​គ្រិស្តសករាជ","គ្រិស្តសករាជ","ព្រឹក","ល្ងាច","៛","{day} {month}, {year} {weekday} {hour}:{minute}:{second}","{day} {month}, {year} {weekday} {hour}:{minute}:{second} {ampm}","{day} {month}, {year} {weekday}","{day}, {month}, {year}","ಜ","ಫೆ","ಮಾ","ಏ","ಮೇ","ಜೂ","ಜು","ಆ","ಸೆ","ಅ","ನ","ಡಿ","ಜನ.","ಫೆಬ್ರು.","ಏಪ್ರಿ.","ಜು.","ಆಗ.","ಸೆಪ್ಟೆಂ.","ಅಕ್ಟೋ.","ನವೆಂ.","ಡಿಸೆಂ.","ಜನವರಿ","ಫೆಬ್ರವರಿ","ಮಾರ್ಚ್","ಏಪ್ರಿಲ್","ಜೂನ್","ಜುಲೈ","ಆಗಸ್ಟ್","ಸಪ್ಟೆಂಬರ್","ಅಕ್ಟೋಬರ್","ನವೆಂಬರ್","ಡಿಸೆಂಬರ್","ರವಿ","ಸೋಮ","ಮಂಗಳ","ಬುಧ","ಗುರು","ಶುಕ್ರ","ಶನಿ","ರ.","ಸೋ.","ಮಂ.","ಬು.","ಗು.","ಶು.","ಶನಿ.","ರವಿವಾರ","ಸೋಮವಾರ","ಮಂಗಳವಾರ","ಬುಧವಾರ","ಗುರುವಾರ","ಶುಕ್ರವಾರ","ಶನಿವಾರ","ಕ್ರಿ.ಪೂ","ಜಾಹೀ","ಸಾಮಾನ್ಯ ಯುಗ","ಈಸಪೂವ೯.","ಕ್ರಿಸ್ತ ಶಕ","{year}년 {month} {day}일 ({weekday})  {hour}:{minute}:{second}","{year}년 {month} {day}일 ({weekday}) {ampm} {hour}:{minute}:{second}","{year}년 {month} {day}일 ({weekday}","{year}년 {month} {day}","{year}년 {month}","불기","투트","바바흐","하투르","키야흐크","투바흐","암쉬르","바라마트","바라문다흐","바샨스","바우나흐","아비브","미스라","나시","1월","2월","3월","4월","5월","6월","7월","8월","9월","10월","11월","12월","매스캐램","테켐트","헤다르","타흐사스","테르","얘카티트","매가비트","미야지야","겐보트","새네","함레","내하세","파구맨","일","월","화","수","목","금","토","일요일","월요일","화요일","수요일","목요일","금요일","토요일","기원전","서기","서력기원전","서력기원","오전","오후","디스리월","말케스월","기슬르월","데벳월","스밧월","아달월 1","아달월","닛산월","이야르월","시완월","담무르월","압월","엘룰월","아달월 2","다이카 (645 ~ 650)","하쿠치 (650 ~ 671)","하쿠호 (672 ~ 686)","슈초 (686 ~ 701)","다이호 (701 ~ 704)","게이운 (704 ~ 708)","와도 (708 ~ 715)","레이키 (715 ~ 717)","요로 (717 ~ 724)","진키 (724 ~ 729)","덴표 (729 ~ 749)","덴표칸포 (749 ~ 749)","덴표쇼호 (749 ~ 757)","덴표호지 (757 ~ 765)","덴표진고 (765 ~ 767)","진고케이운 (767 ~ 770)","호키 (770 ~ 780)","덴오 (781 ~ 782)","엔랴쿠 (782 ~ 806)","다이도 (806 ~ 810)","고닌 (810 ~ 824)","덴초 (824 ~ 834)","조와 (834 ~ 848)","가쇼 (848 ~ 851)","닌주 (851 ~ 854)","사이코 (854 ~ 857)","덴난 (857 ~ 859)","조간 (859 ~ 877)","간교 (877 ~ 885)","닌나 (885 ~ 889)","간표 (889 ~ 898)","쇼타이 (898 ~ 901)","엔기 (901 ~ 923)","엔초 (923 ~ 931)","조헤이 (931 ~ 938)","덴교 (938 ~ 947)","덴랴쿠 (947 ~ 957)","덴토쿠 (957 ~ 961)","오와 (961 ~ 964)","고호 (964 ~ 968)","안나 (968 ~ 970)","덴로쿠 (970 ~ 973)","덴엔 (973 ~ 976)","조겐 (976 ~ 978)","덴겐 (978 ~ 983)","에이간 (983 ~ 985)","간나 (985 ~ 987)","에이엔 (987 ~ 989)","에이소 (989 ~ 990)","쇼랴쿠 (990 ~ 995)","조토쿠 (995 ~ 999)","조호 (999 ~ 1004)","간코 (1004 ~ 1012)","조와 (1012 ~ 1017)","간닌 (1017 ~ 1021)","지안 (1021 ~ 1024)","만주 (1024 ~ 1028)","조겐 (1028 ~ 1037)","조랴쿠 (1037 ~ 1040)","조큐 (1040 ~ 1044)","간토쿠 (1044 ~ 1046)","에이쇼 (1046 ~ 1053)","덴기 (1053 ~ 1058)","고헤이 (1058 ~ 1065)","지랴쿠 (1065 ~ 1069)","엔큐 (1069 ~ 1074)","조호 (1074 ~ 1077)","쇼랴쿠 (1077 ~ 1081)","에이호 (1081 ~ 1084)","오토쿠 (1084 ~ 1087)","간지 (1087 ~ 1094)","가호 (1094 ~ 1096)","에이초 (1096 ~ 1097)","조토쿠 (1097 ~ 1099)","고와 (1099 ~ 1104)","조지 (1104 ~ 1106)","가쇼 (1106 ~ 1108)","덴닌 (1108 ~ 1110)","덴에이 (1110 ~ 1113)","에이큐 (1113 ~ 1118)","겐에이 (1118 ~ 1120)","호안 (1120 ~ 1124)","덴지 (1124 ~ 1126)","다이지 (1126 ~ 1131)","덴쇼 (1131 ~ 1132)","조쇼 (1132 ~ 1135)","호엔 (1135 ~ 1141)","에이지 (1141 ~ 1142)","고지 (1142 ~ 1144)","덴요 (1144 ~ 1145)","규안 (1145 ~ 1151)","닌페이 (1151 ~ 1154)","규주 (1154 ~ 1156)","호겐 (1156 ~ 1159)","헤이지 (1159 ~ 1160)","에이랴쿠 (1160 ~ 1161)","오호 (1161 ~ 1163)","조칸 (1163 ~ 1165)","에이만 (1165 ~ 1166)","닌난 (1166 ~ 1169)","가오 (1169 ~ 1171)","조안 (1171 ~ 1175)","안겐 (1175 ~ 1177)","지쇼 (1177 ~ 1181)","요와 (1181 ~ 1182)","주에이 (1182 ~ 1184)","겐랴쿠 (1184 ~ 1185)","분지 (1185 ~ 1190)","겐큐 (1190 ~ 1199)","쇼지 (1199 ~ 1201)","겐닌 (1201 ~ 1204)","겐큐 (1204 ~ 1206)","겐에이 (1206 ~ 1207)","조겐 (1207 ~ 1211)","겐랴쿠 (1211 ~ 1213)","겐포 (1213 ~ 1219)","조큐 (1219 ~ 1222)","조오 (1222 ~ 1224)","겐닌 (1224 ~ 1225)","가로쿠 (1225 ~ 1227)","안테이 (1227 ~ 1229)","간키 (1229 ~ 1232)","조에이 (1232 ~ 1233)","덴푸쿠 (1233 ~ 1234)","분랴쿠 (1234 ~ 1235)","가테이 (1235 ~ 1238)","랴쿠닌 (1238 ~ 1239)","엔오 (1239 ~ 1240)","닌지 (1240 ~ 1243)","간겐 (1243 ~ 1247)","호지 (1247 ~ 1249)","겐초 (1249 ~ 1256)","고겐 (1256 ~ 1257)","쇼카 (1257 ~ 1259)","쇼겐 (1259 ~ 1260)","분오 (1260 ~ 1261)","고초 (1261 ~ 1264)","분에이 (1264 ~ 1275)","겐지 (1275 ~ 1278)","고안 (1278 ~ 1288)","쇼오 (1288 ~ 1293)","에이닌 (1293 ~ 1299)","쇼안 (1299 ~ 1302)","겐겐 (1302 ~ 1303)","가겐 (1303 ~ 1306)","도쿠지 (1306 ~ 1308)","엔쿄 (1308 ~ 1311)","오초 (1311 ~ 1312)","쇼와 (1312 ~ 1317)","분포 (1317 ~ 1319)","겐오 (1319 ~ 1321)","겐코 (1321 ~ 1324)","쇼추 (1324 ~ 1326)","가랴쿠 (1326 ~ 1329)","겐토쿠 (1329 ~ 1331)","겐코 (1331 ~ 1334)","겐무 (1334 ~ 1336)","엔겐 (1336 ~ 1340)","고코쿠 (1340 ~ 1346)","쇼헤이 (1346 ~ 1370)","겐토쿠 (1370 ~ 1372)","분추 (1372 ~ 1375)","덴주 (1375 ~ 1379)","고랴쿠 (1379 ~ 1381)","고와 (1381 ~ 1384)","겐추 (1384 ~ 1392)","메이토쿠 (1384 ~ 1387)","가쿄 (1387 ~ 1389)","고오 (1389 ~ 1390)","메이토쿠 (1390 ~ 1394)","오에이 (1394 ~ 1428)","쇼초 (1428 ~ 1429)","에이쿄 (1429 ~ 1441)","가키쓰 (1441 ~ 1444)","분안 (1444 ~ 1449)","호토쿠 (1449 ~ 1452)","교토쿠 (1452 ~ 1455)","고쇼 (1455 ~ 1457)","조로쿠 (1457 ~ 1460)","간쇼 (1460 ~ 1466)","분쇼 (1466 ~ 1467)","오닌 (1467 ~ 1469)","분메이 (1469 ~ 1487)","조쿄 (1487 ~ 1489)<","엔토쿠 (1489 ~ 1492)","메이오 (1492 ~ 1501)","분키 (1501 ~ 1504)","에이쇼 (1504 ~ 1521)","다이에이 (1521 ~ 1528)","교로쿠 (1528 ~ 1532)","덴분 (1532 ~ 1555)","고지 (1555 ~ 1558)","에이로쿠 (1558 ~ 1570)","겐키 (1570 ~ 1573)","덴쇼 (1573 ~ 1592)","분로쿠 (1592 ~ 1596)","게이초 (1596 ~ 1615)","겐나 (1615 ~ 1624)","간에이 (1624 ~ 1644)","쇼호 (1644 ~ 1648)","게이안 (1648 ~ 1652)","조오 (1652 ~ 1655)","메이레키 (1655 ~ 1658)","만지 (1658 ~ 1661)","간분 (1661 ~ 1673)","엔포 (1673 ~ 1681)","덴나 (1681 ~ 1684)","조쿄 (1684 ~ 1688)","겐로쿠 (1688 ~ 1704)","호에이 (1704 ~ 1711)","쇼토쿠 (1711 ~ 1716)","교호 (1716 ~ 1736)","겐분 (1736 ~ 1741)","간포 (1741 ~ 1744)","엔쿄 (1744 ~ 1748)","간엔 (1748 ~ 1751)","호레키 (1751 ~ 1764)","메이와 (1764 ~ 1772)","안에이 (1772 ~ 1781)","덴메이 (1781 ~ 1789)","간세이 (1789 ~ 1801)","교와 (1801 ~ 1804)","분카 (1804 ~ 1818)","분세이 (1818 ~ 1830)","덴포 (1830 ~ 1844)","고카 (1844 ~ 1848)","가에이 (1848 ~ 1854)","안세이 (1854 ~ 1860)","만엔 (1860 ~ 1861)","분큐 (1861 ~ 1864)","겐지 (1864 ~ 1865)","게이오 (1865 ~ 1868)","메이지","다이쇼","쇼와","헤이세이","중화민국전","중화민국","({currency}{number})","जानेवारी","एप्रिल","जुलै","ओगस्ट","सेप्टेंबर","ओक्टोबर","नोव्हेंबर","डिसेंबर","मंगळ","आदित्यवार","मंगळार","क्रिस्तपूर्व","क्रिस्तशखा","म.पू.","म.नं.","جنؤری","فرؤری","مارٕچ","میٔ","جوٗن","جوٗلایی","ستمبر","اکتوٗبر","آتھوار","ژٔنٛدٕروار","بوٚموار","بودوار","برٛٮ۪سوار","جُمہ","بٹوار","اَتھوار","ژٔنٛدرٕروار","بی سی","اے ڈی","قبٕل مسیٖح","عیٖسوی سنہٕ","ربیٖع الاول","ربیٖع الثانی","ذِی القد","ذِی الحج","?","Januali","Febluali","Aplili","Jmn","Jumaapii","Jumaatatu","Jumaane","Jumaatano","Jumaamosi","Kabla ya Klisto","Baada ya Klisto","makeo","nyiaghuo","ŋ1","ŋ2","ŋ3","ŋ4","ŋ5","ŋ6","ŋ7","ŋ8","ŋ9","ŋ10","ŋ11","ŋ12","ŋwíí a ntɔ́ntɔ","ŋwíí akǝ bɛ́ɛ","ŋwíí akǝ ráá","ŋwíí akǝ nin","ŋwíí akǝ táan","ŋwíí akǝ táafɔk","ŋwíí akǝ táabɛɛ","ŋwíí akǝ táaraa","ŋwíí akǝ táanin","ŋwíí akǝ ntɛk","ŋwíí akǝ ntɛk di bɔ́k","ŋwíí akǝ ntɛk di bɛ́ɛ","lǝn","maa","mɛk","jǝǝ","júm","sam","sɔ́ndǝ","lǝndí","maadí","mɛkrɛdí","jǝǝdí","júmbá","samdí","d.Y.","k.Y.","di Yɛ́sus aká yálɛ","cámɛɛn kǝ kǝbɔpka Y","sárúwá","cɛɛ́nko","Fäb","Mäz","Mäi","Ouj","Säp","Jannewa","Fäbrowa","Määz","Aprell","Juuni","Juuli","Oujoß","Me.","Sunndaach","Moondaach","Dinnsdaach","Metwoch","Dunnersdaach","Friidaach","Samsdaach","vC","nC","vür Chrestus","noh Chrestus","vür der gewöhnlichen Zeitrechnung","Uhr vörmiddaachs","Uhr nommendaachs","Whe","Efn","Gor","Gwn","Hed","Kev","Mys Genver","Mys Whevrel","Mys Merth","Mys Ebrel","Mys Me","Mys Efan","Mys Gortheren","Mye Est","Mys Gwyngala","Mys Hedra","Mys Du","Mys Kevardhu","Mth","Mhr","Yow","Gwe","De Sul","De Lun","De Merth","De Merher","De Yow","De Gwener","De Sadorn","{year}-ж. {day}-{month}, {weekday}  {hour}:{minute}:{second}","{year}-ж. {day}-{month}, {weekday} {ampm} {hour}:{minute}:{second}","{year}-ж. {day}-{month}, {weekday}","{year}-ж. {day}-{month}","{year}-ж. {month}","Я","Ф","М","А","И","С","О","Н","Д","янв.","фев.","мар.","июн.","июл.","сен.","ноя.","январь","февраль","апрель","июнь","июль","сентябрь","октябрь","ноябрь","декабрь","Жк","Дш","Ше","Ша","Бш","Жм","Иш","Жек","Дүй","Шей","Шар","Бей","Жум","Ишм","б.з. ч.","б.з.ч.","б.з. чейин","түшкө чейинки","түштөн кийинки","сан эмес","сом","Fúngatɨ","Naanɨ","Keenda","Ikúmi","Inyambala","Idwaata","Mʉʉnchɨ","Vɨɨrɨ","Saatʉ","Inyi","Saano","Sasatʉ","Kʉfúngatɨ","Kʉnaanɨ","Kʉkeenda","Kwiikumi","Kwiinyambála","Kwiidwaata","Kʉmʉʉnchɨ","Kʉvɨɨrɨ","Kʉsaatʉ","Kwiinyi","Kʉsaano","Kʉsasatʉ","Píili","Táatu","Íne","Táano","Móosi","Jumapíiri","Jumatátu","Jumaíne","Jumatáano","Alamíisi","Ijumáa","Jumamóosi","KSA","KA","Kɨrɨsitʉ sɨ anavyaal","Kɨrɨsitʉ akavyaalwe","TOO","MUU","Apu","Seb","Oki","Janwaliyo","Febwaliyo","Marisi","Apuli","Maayi","Julaayi","Agusito","Sebuttemba","Okitobba","Bal","Lw2","Lw3","Lw4","Lw5","Lw6","Sabbiiti","Balaza","Lwakubiri","Lwakusatu","Lwakuna","Lwakutaano","Lwamukaaga","Kulisito nga tannaza","Bukya Kulisito Azaal","Wiótheȟika Wí","Thiyóȟeyuŋka Wí","Ištáwičhayazaŋ Wí","Pȟežítȟo Wí","Čhaŋwápetȟo Wí","Wípazukȟa-wašté Wí","Čhaŋpȟásapa Wí","Wasútȟuŋ Wí","Čhaŋwápeǧi Wí","Čhaŋwápe-kasná Wí","Waníyetu Wí","Tȟahékapšuŋ Wí","Aŋpétuwakȟaŋ","Aŋpétuwaŋži","Aŋpétunuŋpa","Aŋpétuyamni","Aŋpétutopa","Aŋpétuzaptaŋ","Owáŋgyužažapi","fbl","msi","apl","yun","yul","agt","stb","ɔtb","nvb","dsb","sánzá ya yambo","sánzá ya míbalé","sánzá ya mísáto","sánzá ya mínei","sánzá ya mítáno","sánzá ya motóbá","sánzá ya nsambo","sánzá ya mwambe","sánzá ya libwa","sánzá ya zómi","sánzá ya zómi na mɔ̌kɔ́","sánzá ya zómi na míbalé","eye","ybo","mbl","mst","min","mtn","mps","eyenga","mokɔlɔ mwa yambo","mokɔlɔ mwa míbalé","mokɔlɔ mwa mísáto","mokɔlɔ ya mínéi","mokɔlɔ ya mítáno","mpɔ́sɔ","libóso ya","nsima ya Y","Yambo ya Yézu Krís","Nsima ya Yézu Krís","ntɔ́ngɔ́","mpókwa","ພ.ສ.","ເທົາ","ບາບາ","ຮາໂຕ","ເຄຍ","ໂທບາ","ອຳເຊີ","ບາລຳຮາດ","ບາລາມູດາ","ບາສຮານ","ເປົານາ","ອີແປບ","ມາສລາ","ນາຊິວ","ອາເຊີ","ນາຊີວ","ແມສເຄີແຣມ","ເຕເກມ","ເຮດາ","ທາຊັສ","ເທີ","ເຍຄາທິດ","ເມກາບິດ","ເມຍເຊຍ","ເຈນບອດ","ເຊເນ","ຮຳເລ","ເນແຮສ໌","ພາກູເມນ","ມ.ກ.","ກ.ພ.","ມ.ນ.","ມ.ສ.","ພ.ພ.","ມິ.ຖ.","ກ.ລ.","ສ.ຫ.","ກ.ຍ.","ຕ.ລ.","ພ.ຈ.","ທ.ວ.","ມັງກອນ","ກຸມພາ","ມີນາ","ເມສາ","ພຶດສະພາ","ມິຖຸນາ","ກໍລະກົດ","ສິງຫາ","ກັນຍາ","ຕຸລາ","ພະຈິກ","ທັນວາ","ວັນອາທິດ","ວັນຈັນ","ວັນອັງຄານ","ວັນພຸດ","ວັນພະຫັດ","ວັນສຸກ","ວັນເສົາ","ກ່ອນ ຄ.ສ.","ຄ.ສ.","ກ່ອນທ່ຽງ","ຫຼັງທ່ຽງ","ທຣິດຣີ","ເຮວານ","ກິດເລບ","ເຕເວດ","ຊີວັດ","ອາດາ I","ອາດາ","ນິດຊານ","ອີຍາຣ","ສີວານ","ຕາມູ","ເອບ","ອີລູ","ອາດາ II","ອີຍາ","ອາດ II","ຈິຕຣາ","ວິສາຂະ","ເຊດຖາ","ອັດສາ","ສາຣາວານາ","ພະຕຣາ","ອັສວິຊາ","ການຕິກາ","ອັກຣາຮາຢານາ","ປຸສາ","ມາຄະ","ຜາລກຸນີ","ປຸສະຍາ","ມຸຮັດ","ເຄາະ","ຮອດບີ 1","ຮອກບີ 2","ນຸມາ 1","ນຸມາ 2","ເຮາະ","ຊະອ໌","ເຮາະມະ","ເຊົາ","ຊຸລກິອຸ","ຊຸລຫິຈ","ມຸຣະຮອມ","ຊາຟາຣ","ຮາບີ 1","ຮາບີ 2","ຈຸມາດາ 1","ຈຸມາດາ 2","ຮາຈັບ","ຊະບານ","ຮາມາດອນ","ເຊົາວັດ","ດຸອັດກິດະ","ດຸອັດກິຈະ","ທະອິກະ (645–650)","ຮາກູຊິ (650–671)","ຮາກູໂຮ (672–686)","ຊູໂຊ (686–701)","ທາອິໂຮ (701–704)","ເຄອຸງ (704–708)","ວະໂດ (708–715)","ເຣອິກິ (715–717)","ໂຢໂຣ (717–724)","ຈິງກິ (724–729)","ເທັມປຽວ (729–749)","ເທັມປຽວ-ຄໍາໂປ (749–749)","ເທັມປຽວ-ໂຊໂຮ (749–757)","ເທັມປຽວ-ໂຮຈິ (757–765)","ເທັມປຽວ-ຈິງໂງະ (765–767)","ຈິງໂງະ-ເຄອຸງ (767–770)","ໂຮກິ (770–780)","ເທັນ-ໂອ (781–782)","ເອັນຣຢາກຸ (782–806)","ດາອິໂດ (806–810)","ໂກນິນ (810–824)","ເທັນໂຊ (824–834)","ໂຊວະ (834–848)","ກະໂຈ (848–851)","ນິນຈູ (851–854)","ສະອິໂກະ (854–857)","ເທັນນານ (857–859)","ໂຈງານ (859–877)","ເກັນເກ (877–885)","ນິນນາ (885–889)","ກໍາປຽວ (889–898)","ໂຊຕາອິ (898–901)","ເອັນງິ (901–923)","ເອັນໂຊ (923–931)","ໂຊເຮ (931–938)","ເທັນງຽວ (938–947)","ເທັນຣຢາກູ (947–957)","ເທັນໂຕະກຸ (957–961)","ໂອວະ (961–964)","ໂກໂຮ (964–968)","ອານະ (968–970)","ເທັນໂຣະກຸ (970–973)","ເທັນ-ເອັນ (973–976)","ໂຈເງັນ (976–978)","ເທັນເງັນ (978–983)","ເອການ (983–985)","ການນະ (985–987)","ເອ-ເອັນ (987–989)","ເອໂຊ (989–990)","ໂຊຣຢະກຸ (990–995)","ໂຊໂຕະກຸ (995–999)","ໂຊໂຮ (999–1004)","ການໂກ (1004–1012)","ໂຊຫວະ (1012–1017)","ການນິງ (1017–1021)","ຈິອານ (1021–1024)","ມານຈຸ (1024–1028)","ໂຊເງັນ (1028–1037)","ໂຊເຣຢະກຸ (1037–1040)","ໂຊຄິວ (1040–1044)","ການໂຕະກຸ (1044–1046)","ເອະໂຊ (1046–1053)","ເທັນງິ (1053–1058)","ໂກເຮ (1058–1065)","ຈິເຣຢະກຸ (1065–1069)","ເອັນຄິວ (1069–1074)","ໂຊະໂຮ (1074–1077)","ໂຊະເຣຢະກຸ (1077–1081)","ເອໂຊະ (1081–1084)","ໂອໂຕະກຸ (1084–1087)","ການຈິ (1087–1094)","ກາໂຊ (1094–1096)","ເອະໂຊະ (1096–1097)","ໂຊະໂຕະກຸ (1097–1099)","ໂກະວະ (1099–1104)","ໂຊະຈິ (1104–1106)","ກາໂຊະ (1106–1108)","ເທັນນິນ (1108–1110)","ເທັນ-ອິ (1110–1113)","ເອກິວ (1113–1118)","ເຄັນ-ເອ (1118–1120)","ໂຮະອານ (1120–1124)","ເທັນຈິ (1124–1126)","ດາອິຈິ (1126–1131)","ເທັນໂຊະ (1131–1132)","ໂຊະໂຊະ (1132–1135)","ໂຮເອັນ (1135–1141)","ເອຈິ (1141–1142)","ໂກະຈິ (1142–1144)","ເທັນໂຢະ (1144–1145)","ຄິວອານ (1145–1151)","ນິນເປ (1151–1154)","ຄິວຈຸ (1154–1156)","ໂຮເຄັນ (1156–1159)","ເຮຈິ (1159–1160)","ເອເຣຢະກຸ (1160–1161)","ໂອໂຊ (1161–1163)","ໂຊະການ (1163–1165)","ເອມານ (1165–1166)","ນິນ-ອານ (1166–1169)","ກະໂອ (1169–1171)","ໂຊະອານ (1171–1175)","ອານເຄັນ (1175–1177)","ຈິໂຊະ (1177–1181)","ໂຢະວະ (1181–1182)","ຈຸເອະ (1182–1184)","ເຄັນເຣຢຸກິ (1184–1185)","ບັນຈິ (1185–1190)","ເກັນຄິວ (1190–1199)","ໂຊຈິ (1199–1201)","ເກັນນິນ (1201–1204)","ເຄັນກິວ (1204–1206)","ເກັນ-ເອະ (1206–1207)","ໂຊະເຄັນ (1207–1211)","ເກັນເຣຢະກຸ (1211–1213)","ເກັນໂປະ (1213–1219)","ໂຊະກິວ (1219–1222)","ໂຈະໂອະ (1222–1224)","ເຄັນນິນ (1224–1225)","ກາໂຮກຸ (1225–1227)","ອານເຕະ (1227–1229)","ການກິ (1229–1232)","ໂຈະເອະ (1232–1233)","ເທັມປຸກຸ (1233–1234)","ບັນເຣຢະກຸ (1234–1235)","ກາເຕະ (1235–1238)","ເຣຢະກຸນິນ (1238–1239)","ເອັນ-ໂອ (1239–1240)","ນິນຈີ (1240–1243)","ຄານເຈນ (1243–1247)","ໂຫຈີ (1247–1249)","ເຄນໂຊ (1249–1256)","ໂຄເຈນ (1256–1257)","ໂຊກາ (1257–1259)","ໂຊເກນ (1259–1260)","ບຸນ-ໂອ (1260–1261)","ໂຄໂຊ (1261–1264)","ບຸນ-ອີ (1264–1275)","ເຄນຈີ (1275–1278)","ເຄິນ (1278–1288)","ໂຊ (1288–1293)","ອິນນິນ (1293–1299)","ເຊີນ (1299–1302)","ເຄນເຈນ (1302–1303)","ຄາເຈນ (1303–1306)","ໂຕກູຈິ (1306–1308)","ອິນກິ (1308–1311)","ໂອໂຊ (1300–1312)","ໂຊວາ (1312–1317)","ບຸນໂປ (1317–1319)","ຈີໂນ (1319–1321)","ເຈນກຽວ (1321–1324)","ໂຊຊິ (1324–1326)","ຄາຣາກິ (1326–1329)","ເຈນໂຕກູ (1329–1331)","ເຈນໂກ (1331–1334)","ເກັມມຸ (1334–1336)","ເອັນເຈັນ (1336–1340)","ໂກໂກກຸ (1340–1346)","ໂຊຊິ (1346–1370)","ເຄນໂຕກຸ (1370–1372)","ບຸນຊຸ (1372–1375)","ເທັນຈຸ (1375–1379)","ຄໍຢາກຸ (1379–1381)","ໂກວາ (1381–1384)","ເຈັນຊຸ (1384–1392)","ມີໂຕກຸ (1384–1387)","ກາກິ (1387–1389)","ຄູ (1389–1390)","ມິໂຕກຸ (1390–1394)","ໂອອິ (1394–1428)","ໂຊໂຊ (1428–1429)","ອິກຽວ (1429–1441)","ກາກິຊຸ (1441–1444)","ບຸນ-ອານ (1444–1449)","ໂຫໂຕກຸ (1449–1452)","ກຽວໂຕກຸ (1452–1455)","ເກໂຊ (1455–1457)","ໂຊໂຣກຸ (1457–1460)","ຄານໂຊ (1460–1466)","ບຸນໂຊ (1466–1467)","ໂອນິນ (1467–1469)","ບຸນມິ (1469–1487)","ໂຊກຽວ (1487–1489)","ເອັນໂຕກຸ (1489–1492)","ມິໂອ (1492–1501)","ບຸນກິ (1501–1504)","ອິໂຊ (1504–1521)","ໄຕອິ (1521–1528)","ກຽວໂຣກຸ (1528–1532)","ເທັນມອນ (1532–1555)","ໂກຈິ (1555–1558)","ອິໂຣກຸ (1558–1570)","ເຈັນກິ (1570–1573)","ເທັນໂຊ (1573–1592)","ບຸນໂຣກຸ (1592–1596)","ຄິໂຊ (1596–1615)","ເກັນວາ (1615–1624)","ຄານ-ອິ (1624–1644)","ໂຊໂຊ (1644–1648)","ຄຽນ (1648–1652)","ຊຸ (1652–1655)","ເມຍຢາກຸ (1655–1658)","ແມນຈິ (1658–1661)","ການບຸນ (1661–1673)","ເອັນໂປ (1673–1681)","ເທັນວາ (1681–1684)","ໂຈກຽວ (1684–1688)","ເຈັນໂຣກຸ (1688–1704)","ໂຫອິ (1704–1711)","ຊຸຕຸກຸ (1711–1716)","ກຽວຫຸ (1716–1736)","ເຈັນບຸນ (1736–1741)","ຄານໂປ (1741–1744)","ເອັນກຽວ (1744–1748)","ຄານ-ອິນ (1748–1751)","ໂຫຢາກຸ (1751–1764)","ເມຍວາ (1764–1772)","ເອັນ-ອິ (1772–1781)","ເທັນມິ (1781–1789)","ຄານຊິ (1789–1801)","ກຽວວາ (1801–1804)","ບຸນກາ (1804–1818)","ບຸນຊິ (1818–1830)","ເທັນໂປ (1830–1844)","ກຸກາ (1844–1848)","ກາອິ (1848–1854)","ແອັນຊິ (1854–1860)","ແມັນ-ເອັນ (1860–1861)","ບຸນກຸ (1861–1864)","ເຈນຈີ (1864–1865)","ຄີໂອ (1865–1868)","ມີຈີ","ໄຕໂຊ","ໂຊວາ","ຮີຊີ","ຟາຣວາດິນ","ອໍຣດີບີເຫຣດ","ຄໍຣເດດ","ແຕຣ","ມໍຣເດດ","ຊາຣຫິວາ","ເມີ","ອາບານ","ອາຊາ","ດີຣ","ບຣາມານ","ເອສຟານ","ຟຣາວາດິນ","ມໍຣດາເ","ອາຊາຣ","ບຣາແມນ","ປີເປີເຊຍ","ກ່ອນ R.O.C.","ບໍ່ແມ່ນໂຕເລກ","₭","saus.","vas.","kov.","bal.","geg.","birž.","liep.","rugp.","rugs.","spal.","lapkr.","gruod.","sausis","vasaris","kovas","balandis","gegužė","birželis","liepa","rugpjūtis","rugsėjis","spalis","lapkritis","gruodis","Sk","Pr","An","Tr","Kt","Pn","Št","sk","pr","an","tr","kt","pn","št","sekmadienis","pirmadienis","antradienis","trečiadienis","ketvirtadienis","penktadienis","šeštadienis","po Kr.","prme","pome","pr. m. e.","po m. e.","prieš Kristų","po Kristaus","prieš mūsų erą","po mūsų erą","priešpiet","popiet","Lt","Cio","Lus","Muu","Lum","Luf","Kab","Lush","Lut","Cis","Ciongo","Lùishi","Lusòlo","Mùuyà","Lumùngùlù","Lufuimi","Kabàlàshìpù","Lùshìkà","Lutongolo","Lungùdi","Kaswèkèsè","Ciswà","Nko","Ndy","Ndg","Njw","Ngv","Lub","Lumingu","Nkodya","Ndàayà","Ndangù","Njòwa","Ngòvya","Lubingu","kmp. Y.K.","kny. Y. K.","Kumpala kwa Yezu Kli","Kunyima kwa Yezu Kli","Dinda","Dilolo","DAC","DAR","DAD","DAN","DAH","DAU","DAO","DAB","DOC","DAP","DGI","DAG","Dwe mar Achiel","Dwe mar Ariyo","Dwe mar Adek","Dwe mar Ang'wen","Dwe mar Abich","Dwe mar Auchiel","Dwe mar Abiriyo","Dwe mar Aboro","Dwe mar Ochiko","Dwe mar Apar","Dwe mar gi achiel","Dwe mar Apar gi ariyo","JMP","WUT","TAR","TAD","TAN","TAB","NGS","Jumapil","Wuok Tich","Tich Ariyo","Tich Adek","Tich Ang'wen","Tich Abich","Ngeso","Kapok Kristo obiro","Ka Kristo osebiro","OD","OT","J2","J3","J4","J5","Ij","J1","Jumapiri","Murwa wa Kanne","Murwa wa Katano","Imberi ya Kuuza Kwa","Muhiga Kuvita Kuuza","{currency}- {number}","maijs","jūn.","jūl.","janvāris","februāris","aprīlis","jūnijs","jūlijs","augusts","septembris","oktobris","novembris","decembris","Sv","Ot","Ce","Pk","svētdiena","pirmdiena","otrdiena","trešdiena","ceturtdiena","piektdiena","sestdiena","p.m.ē.","m.ē.","pmē","mē","pirms mūsu ēras","mūsu ērā","priekšpusdienā","pēcpusdienā","tišri","hešvans","kisļevs","tevets","ševats","1. adars","adars","nisans","ijars","sivans","tamuzs","avs","eluls","2. adars","muharams","safars","1. rabī","2. rabī","1. džumādā","2. džumādā","radžabs","šabans","ramadāns","šauvals","du al-kidā","du al-hidžā","nav skaitlis","Ls","Dal","Ará","Ɔɛn","Doy","Lép","Rok","Sás","Bɔ́r","Kús","Gís","Shʉ́","Ntʉ́","Oladalʉ́","Arát","Ɔɛnɨ́ɔɨŋɔk","Olodoyíóríê inkókúâ","Oloilépūnyīē inkókúâ","Kújúɔrɔk","Mórusásin","Ɔlɔ́ɨ́bɔ́rárɛ","Kúshîn","Olgísan","Pʉshʉ́ka","Ntʉ́ŋʉ́s","Jumapílí","Jumane","Jumatánɔ","Alaámisi","Jumáa","Jumamósi","EY","Meínō Yɛ́sʉ","Eínō Yɛ́sʉ","Ɛnkakɛnyá","Ɛndámâ","JAN","FEB","MAC","ĨPU","MĨĨ","NJU","NJR","AGA","SPT","NOV","DEC","Januarĩ","Feburuarĩ","Ĩpurũ","Mĩĩ","Njuni","Njuraĩ","Agasti","Oktũba","Dicemba","KIU","MRA","WAI","WET","WEN","JUM","Muramuko","Wairi","Wethatu","Wena","Wetano","NK","Mbere ya Kristũ","Nyuma ya Kristũ","RŨ","ŨG","avr","zin","zil","zanvie","fevriye","zilye","septam","oktob","novam","desam","dim","lin","ze","van","dimans","lindi","merkredi","zedi","vandredi","samdi","av. Z-K","ap. Z-K","avan Zezi-Krist","apre Zezi-Krist","Mey","Jon","Jol","Aog","Janoary","Febroary","Martsa","Aprily","Jona","Jolay","Aogositra","Septambra","Oktobra","Novambra","Desambra","Alah","Alats","Alar","Alak","Zom","Asab","Alahady","Alatsinainy","Alarobia","Alakamisy","Zoma","Asabotsy","Alohan'i JK","Aorian'i JK","Kwa","Una","Rar","Che","Moc","Moj","Yel","Mweri wo kwanza","Mweri wo unayeli","Mweri wo uneraru","Mweri wo unecheshe","Mweri wo unethanu","Mweri wo thanu na mocha","Mweri wo saba","Mweri wo nane","Mweri wo tisa","Mweri wo kumi","Mweri wo kumi na moja","Mweri wo kumi na yel'li","Sabato","Arahamisi","HY","YY","Hinapiya yesu","Yopia yesu","wichishu","mchochil'l","MTn","M1","A2","M3","N4","F5","I6","A7","I8","K9","mbegtug","imeg àbùbì","imeg mbəŋchubi","iməg ngwə̀t","iməg fog","iməg ichiibɔd","iməg àdùmbə̀ŋ","iməg ichika","iməg kud","iməg tèsiʼe","iməg zò","iməg krizmed","iməg mbegtug","Aneg 1","Aneg 2","Aneg 3","Aneg 4","Aneg 5","Aneg 6","Aneg 7","{weekday}, {day} {month} {year} г. {hour}:{minute}:{second}","{weekday}, {day} {month} {year} г. {hour}:{minute}:{second} {ampm}","ј","јан.","јун.","јул.","јануари","нед.","пон.","сре.","чет.","пет.","вт.","саб.","недела","среда","четврток","петок","сабота","претпладне","попладне","ден","ജനു","ഫെബ്രു","മാർ","ഏപ്രി","മേയ്","ജൂൺ","ജൂലൈ","ഓഗ","സെപ്റ്റം","ഒക്ടോ","നവം","ഡിസം","ജനുവരി","ഫെബ്രുവരി","മാർച്ച്","ഏപ്രിൽ","ആഗസ്റ്റ്","സെപ്റ്റംബർ","ഒക്‌ടോബർ","നവംബർ","ഡിസംബർ","ഞാ","തി","ചൊ","ബു","വ്യാ","വെ","ശ","ഞായർ","തിങ്കൾ","ചൊവ്വ","ബുധൻ","വ്യാഴം","വെള്ളി","ശനി","ഞായറാഴ്‌ച","തിങ്കളാഴ്‌ച","ചൊവ്വാഴ്ച","ബുധനാഴ്‌ച","വ്യാഴാഴ്‌ച","വെള്ളിയാഴ്‌ച","ശനിയാഴ്‌ച","ക്രി.മൂ","എഡി","ബിസിഇ","സിഇ","ക്രിസ്തുവിനു് മുമ്പ്‌","ക്രിസ്തുവിന് പിൻപ്","ചൈത്രം","വൈശാഖം","ജ്യേഷ്ഠം","ആഷാഢം","ശ്രാവണം","ഭാദ്രപാദം","ആശ്വിനം","കാർത്തികം","മാർഗശീർഷം","പൗഷം","മാഘം","ഫൽഗുനം","ശക","മുഹറം","സഫർ","റബീഹുൽ അവ്വൽ","റബീഹുൽ ആഖിർ","ജമാദുൽ അവ്വൽ","ജമാദുൽ ആഖിർ","റജബ്","ശഹബാൻ","റമളാൻ","ശവ്വാൽ","ദുൽ ഖഹദ്","ദുൽ ഹിജ്ജ","ഹിജറ","സംഖ്യയല്ല","1-р сар","2-р сар","3-р сар","4-р сар","5-р сар","6-р сар","7-р сар","8-р сар","9-р сар","10-р сар","11-р сар","12-р сар","Нэгдүгээр сар","Хоёрдугаар сар","Гуравдугаар сар","Дөрөвдүгээр сар","Тавдугаар сар","Зургадугаар сар","Долдугаар сар","Наймдугаар сар","Есдүгээр сар","Аравдугаар сар","Арван нэгдүгээр сар","Арван хоёрдугаар сар","Ня","Да","Мя","Лх","Пү","Ба","Бя","ням","даваа","мягмар","лхагва","пүрэв","баасан","бямба","м.э.ө","м.э.","МЭӨ","МЭ","НТӨ","НТ","манай эриний өмнөх","манай эриний","ҮӨ","ҮХ","₮","{weekday}, {day}, {month} {year}, {hour}:{minute}:{second}","{weekday}, {day}, {month} {year}, {hour}:{minute}:{second} {ampm}","{weekday}, {day}, {month} {year}","जा","फे","ए","ऑ","स","नो","डि","जाने","फेब्रु","एप्रि","ऑग","सप्टें","ऑक्टो","नोव्हें","डिसें","ऑगस्ट","सप्टेंबर","ऑक्टोबर","मंगळवार","ईसापूर्व","ईसापूर्व युग","ईसायुग","ईसवीसनपूर्व","ईसवीसन","[AM]","[PM]","deva","Fb","Mc","Ap","Og","Sp","Nv","Ds","Ogo","Ogos","Disember","Ah","Is","Ra","Kh","Ahd","Isn","Kha","Ahad","Isnin","Khamis","Jumaat","S.M.","TM","PG","PTG","RM","Fra","Mej","Ġun","Lul","Aww","Ott","Diċ","Jannar","Frar","Marzu","Mejju","Ġunju","Lulju","Awwissu","Settembru","Ottubru","Novembru","Diċembru","Ħad","Tne","Tli","Erb","Ħam","Ġim","Il-Ħadd","It-Tnejn","It-Tlieta","L-Erbgħa","Il-Ħamis","Il-Ġimgħa","Is-Sibt","QK","WK","QEK","EK","Qabel Kristu","Wara Kristu","QN","WN","FLO","CLA","CKI","FMF","MAD","MBI","MLI","MAM","FDE","FMU","FGW","FYU","Fĩi Loo","Cokcwaklaŋne","Cokcwaklii","Fĩi Marfoo","Madǝǝuutǝbijaŋ","Mamǝŋgwãafahbii","Mamǝŋgwãalii","Madǝmbii","Fĩi Dǝɓlii","Fĩi Mundaŋ","Fĩi Gwahlle","Fĩi Yuru","Cya","Cla","Czi","Cko","Cka","Cga","Cze","Com'yakke","Comlaaɗii","Comzyiiɗii","Comkolle","Comkaldǝɓlii","Comgaisuu","Comzyeɓsuu","PK","KǝPel Kristu","Pel Kristu","comme","lilli","ဇန်နဝါရီ","ဖေဖော်ဝါရီ","မတ်","ဧပြီ","မေ","ဇွန်","ဇူလိုင်","ဩဂုတ်","စက်တင်ဘာ","အောက်တိုဘာ","နိုဝင်ဘာ","ဒီဇင်ဘာ","တနင်္ဂနွေ","တနင်္လာ","အင်္ဂါ","ဗုဒ္ဓဟူး","ကြာသပတေး","သောကြာ","စနေ","ဘီစီ","အေဒီ","ဘီစီအီး","စီအီး","ခရစ်တော် မပေါ်မီကာလ","ခရစ်တော် ပေါ်ထွန်းပြီးကာလ","နံနက်","ညနေ","mymr","ဂဏန်းမဟုတ်သော","ǃKhanni","ǃKhanǀgôab","ǀKhuuǁkhâb","ǃHôaǂkhaib","ǃKhaitsâb","Gamaǀaeb","ǂKhoesaob","Aoǁkhuumûǁkhâb","Taraǀkhuumûǁkhâb","ǂNûǁnâiseb","ǀHooǂgaeb","Hôasoreǁkhâb","Son","Wu","Sontaxtsees","Mantaxtsees","Denstaxtsees","Wunstaxtsees","Dondertaxtsees","Fraitaxtsees","Satertaxtsees","Xristub aiǃâ","Xristub khaoǃgâ","ǁgoagas","ǃuias","{weekday} {day}. {month} {year}, {hour}.{minute}.{second}","{weekday} {day}. {month} {year}, {hour}.{minute}.{second} {ampm}","sø.","ti.","on.","to.","fr.","lø.","fvt.","vt.","før vår tidsregning","vår tidsregning","Zib","Nhlo","Mbi","Mab","Nkw","Nhla","Ntu","Ncw","Mpan","Lwe","Mpal","Zibandlela","Nhlolanja","Mbimbitho","Mabasa","Nkwenkwezi","Nhlangula","Ntulikazi","Ncwabakazi","Mpandula","Mfumfu","Lwezi","Mpalakazi","Mvu","Sin","Sih","Mgq","Sonto","Mvulo","Sibili","Sithathu","Sine","Sihlanu","Mgqibelo","UKristo angakabuyi","Ukristo ebuyile","{year} {month} {day}, {weekday}, {hour}:{minute}:{second}","{year} {month} {day}, {weekday}, {hour}:{minute}:{second} {ampm}","१","२","३","४","५","६","७","८","९","१०","११","१२","फेब्रुअरी","अप्रिल","अगस्ट","सेप्टेम्बर","अक्टोबर","नोभेम्बर","डिसेम्बर","फरवरी","अप्रेल","दिसम्बर","आइत","मङ्गल","बिही","आइतवार","मङ्गलवार","बिहीवार","ईसा पूर्व","सन्","जेठ","असार","साउन","भदौ","असोज","कात्तिक","मङसिर","पुस","चैत","नेरू","आइतबार","सोमबार","मङ्गलबार","बुधबार","बिहीबार","शुक्रबार","शनिबार","पूर्व मध्यान्ह","उत्तर मध्यान्ह","Tut","Babah","Hatur","Kiyahk","Tubah","Baramundah","Ba'unah","Abib","Misra","Nasi","Mäskäräm","Teqemt","T'er","Yäkatit","Mägabit","Miyazya","Säne","Nähase","Pagumän","mrt.","mei","maart","augustus","zo","wo","vr","za","zondag","maandag","dinsdag","woensdag","donderdag","vrijdag","zaterdag","gj","v.Chr.","n.Chr.","Voor Christus","vóór gewone jaartelling","gewone jaartelling","Tisjrie","Chesjwan","Sjevat","Adar A","Ijar","Tammoez","Elloel","Adar B","Vaishakha","Jyeshtha","Aashaadha","Shraavana","Bhaadrapada","Ashvina","Kaartika","Pausha","Maagha","Phaalguna","Moeh.","Joem. I","Joem. II","Sja.","Sjaw.","Doe al k.","Doe al h.","Moeharram","Rabiʻa al awal","Rabiʻa al thani","Joemadʻal awal","Joemadʻal thani","Sjaʻaban","Sjawal","Doe al kaʻaba","Doe al hizja","Saʻna Hizjria","{currency} {number}-","FJ$","SI$","ng1","ng2","ng3","ng4","ng5","ng6","ng7","ng8","ng9","ng10","ng11","kris","ngwɛn matáhra","ngwɛn ńmba","ngwɛn ńlal","ngwɛn ńna","ngwɛn ńtan","ngwɛn ńtuó","ngwɛn hɛmbuɛrí","ngwɛn lɔmbi","ngwɛn rɛbvuâ","ngwɛn wum","ngwɛn wum navŭr","krísimin","mbs","sas","mɔ́ndɔ","sɔ́ndɔ mafú mába","sɔ́ndɔ mafú málal","sɔ́ndɔ mafú mána","mabágá má sukul","sásadi","BL","PB","Bó Lahlɛ̄","Pfiɛ Burī","maná","kugú","{weekday} {day}. {month} {year}, {hour}:{minute}:{second}","{weekday} {day}. {month} {year}, {hour}:{minute}:{second} {ampm}","må.","ty.","måndag","tysdag","laurdag","før vesterlandsk tidsrekning","vesterlandsk tidsrekning","formiddag","ettermiddag","{weekday} , lyɛ̌ʼ {day} na {month}, {year} {hour}:{minute}:{second}","{weekday} , lyɛ̌ʼ {day} na {month}, {year} {hour}:{minute}:{second} {ampm}","{weekday} , lyɛ̌ʼ {day} na {month}, {year}","lyɛ̌ʼ {day} na' {month}, {year}","saŋ tsetsɛ̀ɛ lùm","saŋ kàg ngwóŋ","saŋ lepyè shúm","saŋ cÿó","saŋ tsɛ̀ɛ cÿó","saŋ njÿoláʼ","saŋ tyɛ̀b tyɛ̀b mbʉ̀","saŋ mbʉ̀ŋ","saŋ ngwɔ̀ʼ mbÿɛ","saŋ tàŋa tsetsáʼ","saŋ mejwoŋó","saŋ lùm","lyɛʼɛ́ sẅíŋtè","mvfò lyɛ̌ʼ","mbɔ́ɔntè mvfò lyɛ̌ʼ","tsètsɛ̀ɛ lyɛ̌ʼ","mbɔ́ɔntè tsetsɛ̀ɛ lyɛ̌ʼ","mvfò màga lyɛ̌ʼ","màga lyɛ̌ʼ","m.z.Y.","m.g.n.Y.","mé zyé Yěsô","mé gÿo ńzyé Yěsô","mbaʼámbaʼ","ncwònzém","Mat","Arh","Usi","Janabari","uFeberbari","uMatjhi","u-Apreli","Meyi","Julayi","Arhostosi","Usinyikhaba","Bil","Ne","Hla","Gqi","uSonto","uMvulo","uLesibili","Lesithathu","uLesine","ngoLesihlanu","umGqibelo","Apo","Nof","Janaware","Feberware","Matšhe","Aporele","Julae","Agostose","Setemere","Oktobore","Nofemere","Disemere","Mos","Bed","Mok","Sontaga","Mosupalogo","Labobedi","Laboraro","Labone","Labohlano","Mokibelo","{weekday}، {day} {month} {year} {hour}:{minute}:{second}","{weekday}، {day} {month} {year} {hour}:{minute}:{second} {ampm}","{weekday}، {day} {month} {year}","Tiop","Pɛt","Duɔ̱ɔ̱","Guak","Duä","Kor","Pay","Thoo","Tɛɛ","Laa","Kur","Tid","Tiop thar pɛt","Duɔ̱ɔ̱ŋ","Duät","Kornyoot","Pay yie̱tni","Tho̱o̱r","Tɛɛr","Laath","Tio̱p in di̱i̱t","Cäŋ","Jiec","Rɛw","Diɔ̱k","Ŋuaan","Dhieec","Bäkɛl","Cäŋ kuɔth","Jiec la̱t","Rɛw lätni","Diɔ̱k lätni","Ŋuaan lätni","Dhieec lätni","Bäkɛl lätni","ƐY","A ka̱n Yecu ni dap","Ɛ ca Yecu dap","RW","TŊ","Ama","Gur","Bit","Elb","Cam","Wax","Ado","Hag","Ful","Onk","Mud","Amajjii","Guraandhala","Bitooteessa","Elba","Caamsa","Waxabajjii","Adooleessa","Hagayya","Fuulbana","Onkololeessa","Sadaasa","Muddee","Dil","Wix","Qib","Rob","Dilbata","Wiixata","Qibxata","Roobii","Kamiisa","Jimaata","Sanbata","KD","KB","WD","WB","ଜାନୁଆରୀ","ଫେବ୍ରୁୟାରୀ","ମାର୍ଚ୍ଚ","ଅପ୍ରେଲ","ମେ","ଜୁନ","ଜୁଲାଇ","ଅଗଷ୍ଟ","ସେପ୍ଟେମ୍ବର","ଅକ୍ଟୋବର","ନଭେମ୍ବର","ଡିସେମ୍ବର","ରବି","ସୋମ","ମଙ୍ଗଳ","ବୁଧ","ଗୁରୁ","ଶୁକ୍ର","ଶନି","ରବିବାର","ସୋମବାର","ମଙ୍ଗଳବାର","ବୁଧବାର","ଗୁରୁବାର","ଶୁକ୍ରବାର","ଶନିବାର","июны","июлы","январы","февралы","мартъийы","апрелы","майы","августы","сентябры","октябры","ноябры","декабры","хцб","крс","дцг","ӕрт","цпр","мрб","сбт","хуыцаубон","къуырисӕр","дыццӕг","ӕртыццӕг","цыппӕрӕм","майрӕмбон","сабат","н.д.а.","н.д.","ӕмбисбоны размӕ","ӕмбисбоны фӕстӕ","НН","ਜ","ਫ਼","ਮਾ","ਅ","ਮ","ਜੂ","ਜੁ","ਸ","ਨ","ਦ","ਜਨਵਰੀ","ਫ਼ਰਵਰੀ","ਮਾਰਚ","ਅਪ੍ਰੈਲ","ਮਈ","ਜੂਨ","ਜੁਲਾਈ","ਅਗਸਤ","ਸਤੰਬਰ","ਅਕਤੂਬਰ","ਨਵੰਬਰ","ਦਸੰਬਰ","ਐਤ.","ਸੋਮ.","ਮੰਗਲ.","ਬੁਧ.","ਵੀਰ.","ਸ਼ੁੱਕਰ.","ਸ਼ਨੀ.","ਐਤਵਾਰ","ਸੋਮਵਾਰ","ਮੰਗਲਵਾਰ","ਬੁਧਵਾਰ","ਵੀਰਵਾਰ","ਸ਼ੁੱਕਰਵਾਰ","ਸ਼ਨੀਵਾਰ","ਈ. ਪੂ.","ਸੰਨ","ਈ. ਪੂ. ਯੁ.","ਈ.ਯੁ.","فروری","مئ","جولائی","اتوار","پیر","منگل","بُدھ","جمعرات","جمعہ","ہفتہ","ايساپورو","سں","l","c","w","p","g","sty","lut","kwi","cze","sie","wrz","paź","gru","stycznia","lutego","marca","kwietnia","maja","czerwca","lipca","sierpnia","września","października","grudnia","niedz.","pon.","wt.","śr.","czw.","pt.","sob.","niedziela","poniedziałek","wtorek","środa","czwartek","piątek","p.n.e.","Tiszri","Cheszwan","Kislew","Tewet","Szwat","Siwan","Aw","Dżu. I","Dżu. II","Ra.","Sza.","Szaw.","Zu al-k.","Zu al-h.","Dżumada I","Dżumada II","Radżab","Szaban","Szawwal","Zu al-kada","Zu al-hidżdża","Farwardin","Ordibeheszt","Chordād","Mordād","Szahriwar","Ābān","Āsar","Déi","Przed ROC","ROC","zł","جنوري","فبروري","اګست","غ.م.","غ.و.","وری","غویی","غبرګولی","چنګاښ","زمری","وږی","تله","لړم","لیندۍ","مرغومی","سلواغه","کب","Mês 1","Mês 2","Mês 3","Mês 4","Mês 5","Mês 6","Mês 7","Mês 8","Mês 9","Mês 10","Mês 11","Mês 12","dez","janeiro","fevereiro","março","junho","julho","dezembro","seg","qua","qui","sex","segunda-feira","terça-feira","quarta-feira","quinta-feira","sexta-feira","Antes de Cristo","Ano do Senhor","Antes de R.O.C.","Esc.","{weekday}, {day}/{month}/{year}, {hour}:{minute}:{second}","{weekday}, {day}/{month}/{year}, {hour}:{minute}:{second} {ampm}","{weekday}, {day}/{month}/{year}","M2","M4","M5","M6","M7","M8","M9","Out","Janeiro","Fevereiro","Março","Maio","Junho","Julho","Setembro","Outubro","Novembro","Dezembro","sg","qu","qi","sx","sb","a.E.C.","E.C.","da manhã","da tarde","Db","schan.","favr.","matg","zercl.","fan.","avust","sett.","schaner","favrer","avrigl","zercladur","fanadur","settember","october","gli","gie","dumengia","glindesdi","mesemna","gievgia","venderdi","sonda","av. Cr.","s. Cr.","avant Cristus","suenter Cristus","sm","Mut.","Gas.","Wer.","Mat.","Gic.","Kam.","Nya.","Kan.","Nze.","Ukw.","Ugu.","Uku.","Nzero","Ruhuhuma","Ntwarante","Ndamukiza","Rusama","Ruheshi","Mukakaro","Nyandagaro","Nyakanga","Gitugutu","Munyonyo","Kigarama","cu.","mbe.","kab.","gtu.","kan.","gnu.","gnd.","Ku w'indwi","Ku wa mbere","Ku wa kabiri","Ku wa gatatu","Ku wa kane","Ku wa gatanu","Ku wa gatandatu","Mb.Y.","Ny.Y","Mbere ya Yezu","Nyuma ya Yezu","Z.MU.","Z.MW.","e.b.","era budistă","Hathor","Mesori","El Nasii","meskerem","taqemt","tahsas","yekatit","megabit","miazia","genbot","sene","hamle","nehase","pagumen","ian.","iun.","iul.","ianuarie","februarie","martie","aprilie","iunie","iulie","septembrie","octombrie","noiembrie","decembrie","Jo","Vi","Sâ","Dum","Mie","Joi","Vin","Sâm","duminică","luni","marți","miercuri","vineri","sâmbătă","î.Hr.","d.Hr.","î.e.n","e.n.","înainte de Hristos","după Hristos","Tișrei","Heșvan","Șevat","Tammuz","Mweri wa kwanza","Mweri wa kaili","Mweri wa katatu","Mweri wa kaana","Mweri wa tanu","Mweri wa sita","Mweri wa saba","Mweri wa nane","Mweri wa tisa","Mweri wa ikumi","Mweri wa ikumi na moja","Mweri wa ikumi na mbili","Ijp","Ijt","Ijn","Ijtn","Ijumapili","Ijumatatu","Ijumanne","Ijumatano","Ijumamosi","Kabla ya Mayesu","Baada ya Mayesu","kang'ama","kingoto","Тот","Бабэ","Хатур","Кихак","Тубэ","Бармуда","Башнас","Бауна","Абиб","Мисра","Якатит","Магабит","Миазия","Сэнэ","Хамлэ","Нахасэ","Эпагомен","марта","июня","июля","сент.","нояб.","января","февраля","апреля","августа","сентября","октября","ноября","декабря","вс","воскресенье","понедельник","четверг","пятница","суббота","до н.э.","до н. э.","н. э.","Тишрей","Ияр","Таммуз","Элул","Чайтра","Ваисакха","Джанштха","Асадха","Бхадра","Азвина","Аграхайана","Магха","Пхалгуна","Мухаррам","Раби-уль-авваль","Раби-уль-ахир","Джумад-уль-авваль","Джумад-уль-ахир","Раджаб","Шаабан","Шавваль","Зуль-Каада","Зуль-Хиджжа","Эпоха Тайка (645-650)","Эпоха Хакути (650-671)","Эпоха Хакухо (672-686)","Эпоха Сючё (686-701)","Эпоха Тайхо (701-704)","Эпоха Кёюн (704-708)","Эпоха Вадо (708-715)","Эпоха Рэйки (715-717)","Эпоха Ёро (717-724)","Эпоха Дзинки (724-729)","Эпоха Темпьё (729-749)","Эпоха Темпьё (749-749)","Эпоха Темпьё-Сьохо (749-757)","Эпоха Темпьё-Ходзи (757-765)","Эпоха Темпьё-Ходзи (765-767)","Эпоха Джинго-Кёюн (767-770)","Эпоха Хоки (770-780)","Эпоха Теньё (781-782)","Эпоха Енряку (782-806)","Эпоха Дайдо (806-810)","Эпоха Конин (810-824)","Эпоха Тентьо (824-834)","Эпоха Шова (834-848)","Эпоха Кайо (848-851)","Эпоха Ниндзю (851-854)","Эпоха Сайко (854-857)","Эпоха Теннан (857-859)","Эпоха Йоган (859-877)","Эпоха Генкей (877-885)","Эпоха Нинна (885-889)","Эпоха Кампьё (889-898)","Эпоха Сьотай (898-901)","Эпоха Энги (901-923)","Эпоха Ентьо (923-931)","Эпоха Сьёхэй (931-938)","Эпоха Тенгьо (938-947)","Эпоха Тенрияку (947-957)","Эпоха Тентоку (957-961)","Эпоха Ова (961-964)","Эпоха Кохо (964-968)","Эпоха Анна (968-970)","Эпоха Тенроку (970-973)","Эпоха Теньен (973-976)","Эпоха Дзьоген (976-978)","Эпоха Тенген (978-983)","Эпоха Ейкан (983-985)","Эпоха Канна (985-987)","Эпоха Ейен (987-989)","Эпоха Ейсо (989-990)","Эпоха Сёряку (990-995)","Эпоха Тётоку (995-999)","Эпоха Тёхо (999-1004)","Эпоха Канко (1004-1012)","Эпоха Тёва (1012-1017)","Эпоха Каннин (1017-1021)","Эпоха Дзиан (1021-1024)","Эпоха Мандзю (1024-1028)","Эпоха Тёгэн (1028-1037)","Эпоха Тёряку (1037-1040)","Эпоха Тёкю (1040-1044)","Эпоха Катоку (1044-1046)","Эпоха Эйсо (1046-1053)","Эпоха Тэнги (1053-1058)","Эпоха Кохэй (1058-1065)","Эпоха Дзиряку (1065-1069)","Эпоха Энкю (1069-1074)","Эпоха Сёхо (1074-1077)","Эпоха Сёряку (1077-1081)","Эпоха Эйхо (1081-1084)","Эпоха Отоку (1084-1087)","Эпоха Кандзи (1087-1094)","Эпоха Кахо (1094-1096)","Эпоха Эйтё (1096-1097)","Эпоха Сётоку (1097-1099)","Эпоха Кова (1099-1104)","Эпоха Тёдзи (1104-1106)","Эпоха Касё (1106-1108)","Эпоха Тэннин (1108-1110)","Эпоха Тэнъэй (1110-1113)","Эпоха Эйкю (1113-1118)","Эпоха Гэнъэй (1118-1120)","Эпоха Хоан (1120-1124)","Эпоха Тэндзи (1124-1126)","Эпоха Дайдзи (1126-1131)","Эпоха Тэнсё (1131-1132)","Эпоха Тёсё (1132-1135)","Эпоха Хоэн (1135-1141)","Эпоха Эйдзи (1141-1142)","Эпоха Кодзи (1142-1144)","Эпоха Тэнё (1144-1145)","Эпоха Кюан (1145-1151)","Эпоха Нимпэй (1151-1154)","Эпоха Кюдзю (1154-1156)","Эпоха Хогэн (1156-1159)","Эпоха Хэйдзи (1159-1160)","Эпоха Эйряку (1160-1161)","Эпоха Охо (1161-1163)","Эпоха Тёкан (1163-1165)","Эпоха Эйман (1165-1166)","Эпоха Нинъан (1166-1169)","Эпоха Као (1169-1171)","Эпоха Сёан (1171-1175)","Эпоха Ангэн (1175-1177)","Эпоха Дзисё (1177-1181)","Эпоха Ёва (1181-1182)","Эпоха Дзюэй (1182-1184)","Эпоха Гэнрюку (1184-1185)","Эпоха Бундзи (1185-1190)","Эпоха Кэнкю (1190-1199)","Эпоха Сёдзи (1199-1201)","Эпоха Кэннин (1201-1204)","Эпоха Гэнкю (1204-1206)","Эпоха Кэнъэй (1206-1207)","Эпоха Сёгэн (1207-1211)","Эпоха Кэнряку (1211-1213)","Эпоха Кэмпо (1213-1219)","Эпоха Сёкю (1219-1222)","Эпоха Дзёо (1222-1224)","Эпоха Гэннин (1224-1225)","Эпоха Кароку (1225-1227)","Эпоха Антэй (1227-1229)","Эпоха Канки (1229-1232)","Эпоха Дзёэй (1232-1233)","Эпоха Тэмпуку (1233-1234)","Эпоха Бунряку (1234-1235)","Эпоха Катэй (1235-1238)","Эпоха Рякунин (1238-1239)","Эпоха Энъо (1239-1240)","Эпоха Ниндзи (1240-1243)","Эпоха Кангэн (1243-1247)","Эпоха Ходзи (1247-1249)","Эпоха Кэнтё (1249-1256)","Эпоха Когэн (1256-1257)","Эпоха Сёка (1257-1259)","Эпоха Сёгэн (1259-1260)","Эпоха Бунъо (1260-1261)","Эпоха Котё (1261-1264)","Эпоха Бунъэй (1264-1275)","Эпоха Кэндзи (1275-1278)","Эпоха Коан (1278-1288)","Эпоха Сёо (1288-1293)","Эпоха Эйнин (1293-1299)","Эпоха Сёан (1299-1302)","Эпоха Кэнгэн (1302-1303)","Эпоха Кагэн (1303-1306)","Эпоха Токудзи (1306-1308)","Эпоха Энкэй (1308-1311)","Эпоха Отё (1311-1312)","Эпоха Сёва (1312-1317)","Эпоха Бумпо (1317-1319)","Эпоха Гэно (1319-1321)","Эпоха Гэнкё (1321-1324)","Эпоха Сётю (1324-1326)","Эпоха Карэки (1326-1329)","Эпоха Гэнтоку (1329-1331)","Эпоха Гэнко (1331-1334)","Эпоха Кэмму (1334-1336)","Эпоха Энгэн (1336-1340)","Эпоха Кококу (1340-1346)","Эпоха Сёхэй (1346-1370)","Эпоха Кэнтоку (1370-1372)","Эпоха Бунтю (1372-1375)","Эпоха Иэндзю (1375-1379)","Эпоха Коряку (1379-1381)","Эпоха Кова (1381-1384)","Эпоха Гэнтю (1384-1392)","Эпоха Мэйтоку (1384-1387)","Эпоха Какэй (1387-1389)","Эпоха Коо (1389-1390)","Эпоха Мэйтоку (1390-1394)","Эпоха Оэй (1394-1428)","Эпоха Сётё (1428-1429)","Эпоха Эйкё (1429-1441)","Эпоха Какицу (1441-1444)","Эпоха Банъан (1444-1449)","Эпоха Хотоку (1449-1452)","Эпоха Кётоку (1452-1455)","Эпоха Косё (1455-1457)","Эпоха Тёроку (1457-1460)","Эпоха Кансё (1460-1466)","Эпоха Бунсё (1466-1467)","Эпоха Онин (1467-1469)","Эпоха Буммэй (1469-1487)","Эпоха Тёкё (1487-1489)","Эпоха Энтоку (1489-1492)","Эпоха Мэйо (1492-1501)","Эпоха Бунки (1501-1504)","Эпоха Эйсё (1504-1521)","Эпоха Тайэй (1521-1528)","Эпоха Кёроку (1528-1532)","Эпоха Тэммон (1532-1555)","Эпоха Кодзи (1555-1558)","Эпоха Эйроку (1558-1570)","Эпоха Гэнки (1570-1573)","Эпоха Тэнсё (1573-1592)","Эпоха Бунроку (1592-1596)","Эпоха Кэйтё (1596-1615)","Эпоха Гэнва (1615-1624)","Эпоха Канъэй (1624-1644)","Эпоха Сёхо (1644-1648)","Эпоха Кэйан (1648-1652)","Эпоха Сё (1652-1655)","Эпоха Мэйряку (1655-1658)","Эпоха Мандзи (1658-1661)","Эпоха Камбун (1661-1673)","Эпоха Эмпо (1673-1681)","Эпоха Тэнва (1681-1684)","Эпоха Дзёкё (1684-1688)","Эпоха Гэнроку (1688-1704)","Эпоха Хоэй (1704-1711)","Эпоха Сётоку (1711-1716)","Эпоха Кёхо (1716-1736)","Эпоха Гэмбун (1736-1741)","Эпоха Кампо (1741-1744)","Эпоха Энкё (1744-1748)","Эпоха Канъэн (1748-1751)","Эпоха Хоряку (1751-1764)","Эпоха Мэйва (1764-1772)","Эпоха Анъэй (1772-1781)","Эпоха Тэммэй (1781-1789)","Эпоха Кансэй (1789-1801)","Эпоха Кёва (1801-1804)","Эпоха Бунка (1804-1818)","Эпоха Бунсэй (1818-1830)","Эпоха Тэмпо (1830-1844)","Эпоха Кока (1844-1848)","Эпоха Каэй (1848-1854)","Эпоха Ансэй (1854-1860)","Эпоха Манъэн (1860-1861)","Эпоха Бункю (1861-1864)","Эпоха Гендзи (1864-1865)","Эпоха Кейо (1865-1868)","Эпоха Мэйдзи","Эпоха Тайсьо","Сьова","Эпоха Хэйсэй","Фарвардин","Хордад","Шахривер","Азер","Дей","Эсфанд","не число","ТМТ","₴","XXXX","mut.","gas.","wer.","mat.","gic.","kam.","nya.","nze.","ukw.","ugu.","uku.","Mutarama","Gashyantare","Werurwe","Mata","Gicuransi","Kamena","Kanama","Nzeli","Ukwakira","Ugushyingo","Ukuboza","cyu.","Ku cyumweru","Kuwa mbere","Kuwa kabiri","Kuwa gatatu","Kuwa kane","Kuwa gatanu","Kuwa gatandatu","Тохс","Олун","Клн_ттр","Мус_уст","Ыам_йн","Бэс_йн","От_йн","Атрдь_йн","Блҕн_йн","Алт","Сэт","Ахс","Тохсунньу","Олунньу","Кулун тутар","Муус устар","Ыам ыйын","Бэс ыйын","От ыйын","Атырдьых ыйын","Балаҕан ыйын","Алтынньы","Сэтинньи","Ахсынньы","Бс","Бн","Оп","Сэ","Чп","Бэ","Сб","Баскыһыанньа","Бэнидиэлинньик","Оптуорунньук","Сэрэдэ","Чэппиэр","Бээтиҥсэ","Субуота","б. э. и.","б. э","ЭИ","ЭК","Obo","Waa","Oku","Ong","Ime","Ile","Sap","Isi","Saa","Tom","Tob","Tow","Lapa le obo","Lapa le waare","Lapa le okuni","Lapa le ong'wan","Lapa le imet","Lapa le ile","Lapa le sapa","Lapa le isiet","Lapa le saal","Lapa le tomon","Lapa le tomon obo","Lapa le tomon waare","Are","Kun","Mderot ee are","Mderot ee kuni","Mderot ee ong'wan","Mderot ee inet","Mderot ee ile","Mderot ee sapa","Mderot ee kwe","Kabla ya Christo","Baada ya Christo","Tesiran","Teipa","{month} {day} {year}","Mup","Mwi","Msh","Mun","Mag","Muj","Msp","Mpg","Mye","Mus","Muh","Mupalangulwa","Mwitope","Mushende","Munyi","Mushende Magali","Mujimbi","Mushipepo","Mupuguto","Munyense","Mokhu","Musongandembwe","Muhaano","Mulungu","Alahamisi","Ashanali uKilisito","Pamwandi ya Kilisto","Lwamilawu","Pashamihe","ođđajage","guovva","njukča","cuoŋo","miesse","geasse","suoidne","borge","čakča","golggot","skábma","juovla","ođđajagemánnu","guovvamánnu","njukčamánnu","cuoŋománnu","miessemánnu","geassemánnu","suoidnemánnu","borgemánnu","čakčamánnu","golggotmánnu","skábmamánnu","juovlamánnu","sotn","vuos","maŋ","gask","duor","bear","láv","aejlege","måanta","däjsta","gaskevahkoe","dåarsta","bearjadahke","laavadahke","o.Kr.","m.Kr.","ovdal Kristtusa","maŋŋel Kristtusa","iđitbeaivet","eahketbeaivet","Dkr","ođđj","guov","njuk","cuo","mies","geas","suoi","borg","čakč","golg","skáb","juov","sotnabeaivi","vuossárga","maŋŋebárga","gaskavahkku","duorasdat","bearjadat","lávvardat","Fevreiro","Marco","Augusto","Otubro","Decembro","Pos","Pir","Nai","Sha","Dimingu","Chiposi","Chipiri","Chitatu","Chinai","Chishanu","Sabudu","AC","Nye","Mbä","Bêl","Fön","Len","Kük","Ngb","Nab","Kak","Nyenye","Fulundïgi","Mbängü","Ngubùe","Bêläwü","Föndo","Lengua","Kükürü","Mvuka","Ngberere","Nabändüru","Kakauka","Bk1","Bk2","Bk3","Bk4","Bk5","Lâp","Lây","Bikua-ôko","Bïkua-ûse","Bïkua-ptâ","Bïkua-usïö","Bïkua-okü","Lâpôsö","Lâyenga","KnK","NpK","Kôzo na Krîstu","Na pekô tî Krîstu","ND","LK","ⵉⵏⵏ","ⴱⵕⴰ","ⵎⴰⵕ","ⵉⴱⵔ","ⵎⴰⵢ","ⵢⵓⵏ","ⵢⵓⵍ","ⵖⵓⵛ","ⵛⵓⵜ","ⴽⵜⵓ","ⵏⵓⵡ","ⴷⵓⵊ","ⵉⵏⵏⴰⵢⵔ","ⴱⵕⴰⵢⵕ","ⵎⴰⵕⵚ","ⵉⴱⵔⵉⵔ","ⵎⴰⵢⵢⵓ","ⵢⵓⵏⵢⵓ","ⵢⵓⵍⵢⵓⵣ","ⵖⵓⵛⵜ","ⵛⵓⵜⴰⵏⴱⵉⵔ","ⴽⵜⵓⴱⵔ","ⵏⵓⵡⴰⵏⴱⵉⵔ","ⴷⵓⵊⴰⵏⴱⵉⵔ","ⴰⵙⴰ","ⴰⵢⵏ","ⴰⵙⵉ","ⴰⴽⵕ","ⴰⴽⵡ","ⴰⵙⵉⵎ","ⴰⵙⵉⴹ","ⴰⵙⴰⵎⴰⵙ","ⴰⵢⵏⴰⵙ","ⴰⵙⵉⵏⴰⵙ","ⴰⴽⵕⴰⵙ","ⴰⴽⵡⴰⵙ","ⵙⵉⵎⵡⴰⵙ","ⴰⵙⵉⴹⵢⴰⵙ","ⴷⴰⵄ","ⴷⴼⵄ","ⴷⴰⵜ ⵏ ⵄⵉⵙⴰ","ⴷⴼⴼⵉⵔ ⵏ ⵄⵉⵙⴰ","ⵜⵉⴼⴰⵡⵜ","ⵜⴰⴷⴳⴳⵯⴰⵜ","inn","bṛa","maṛ","ibr","ɣuc","cut","ktu","nuw","duj","innayr","bṛayṛ","maṛṣ","ibrir","mayyu","yunyu","yulyuz","ɣuct","cutanbir","ktubr","nuwanbir","dujanbir","asa","ayn","asi","akṛ","akw","asim","asiḍ","asamas","aynas","asinas","akṛas","akwas","asimwas","asiḍyas","daɛ","dfɛ","dat n ɛisa","dffir n ɛisa","tifawt","tadggʷat","{year} {month} {day}, {weekday}  {hour}.{minute}.{second}","{year} {month} {day}, {weekday} {ampm} {hour}.{minute}.{second}","{ampm} {hour}.{minute}.{second}","{ampm} {hour}.{minute}","ජ","පෙ","මා","අ","මැ","ජූ","සැ","ඔ","නෙ","දෙ","ජන","පෙබ","මාර්තු","අප්‍රේල්","මැයි","ජූනි","ජූලි","අගෝ","සැප්","ඔක්","නොවැ","දෙසැ","ජනවාරි","පෙබරවාරි","අගෝස්තු","සැප්තැම්බර්","ඔක්තෝබර්","නොවැම්බර්","දෙසැම්බර්","ඉරි","සඳු","අඟ","බදා","බ්‍රහ","සිකු","සෙන","ඉරිදා","සඳුදා","අඟහ","බදාදා","බ්‍රහස්","අඟහරුවාදා","බ්‍රහස්පතින්දා","සිකුරාදා","සෙනසුරාදා","ක්‍රි.පූ.","ක්‍රි.ව.","පො.යු.පෙ","පො.යු","ක්‍රිස්තු පූර්‍ව","ක්‍රිස්තු වර්‍ෂ","පෙ.ව.","ප.ව.","රු.","máj","jún","júl","januára","februára","apríla","mája","júna","júla","augusta","septembra","októbra","novembra","decembra","Po","St","Pi","ut","pi","nedeľa","pondelok","utorok","streda","štvrtok","piatok","pred n.l.","Mex$","j","o","n","avg.","marec","junij","julij","ned.","sre.","čet.","pet.","nedelja","ponedeljek","torek","sreda","četrtek","petek","pr. n. št.","po n. št.","pred našim štetjem","naše štetje","dop.","pop.","Ndi","Kuk","Kub","Chv","Chk","Chg","Gun","Mb","Zvi","Ndira","Kukadzi","Kurume","Kubvumbi","Chivabvu","Chikumi","Chikunguru","Nyamavhuvhu","Gunyana","Gumiguru","Mbudzi","Zvita","Svo","Muv","Chip","Chit","Chin","Chis","Svondo","Muvhuro","China","Mugovera","Kristo asati auya","Kristo ashaya","Kob","Lab","Afr","Lix","Tod","Sid","Sag","KIT","LIT","Bisha Koobaad","Bisha Labaad","Bisha Saddexaad","Bisha Afraad","Bisha Shanaad","Bisha Lixaad","Bisha Todobaad","Bisha Sideedaad","Bisha Sagaalaad","Bisha Tobnaad","Bisha Kow iyo Tobnaad","Bisha Laba iyo Tobnaad","Axd","Axad","Isniin","Talaado","Arbaco","Khamiis","Jimco","CK","CD","Ciise ka hor (CS)","Ciise ka dib (CS)","sn.","gn.","Shk","Pri","Maj","Qer","Gsh","Sht","Tet","Nën","Dhj","janar","shkurt","prill","qershor","korrik","gusht","shtator","tetor","nëntor","dhjetor","Die","Hën","Mër","Enj","Pre","e diel","e hënë","e martë","e mërkurë","e enjte","e premte","e shtunë","p.e.r.","e.r.","p.e.s.","e.s.","p.e.s","para erës së re","erës së re","para erës sonë","erës sonë","paradite","pasdite","Lekë","den","{weekday}, {day}. {month} {year}. {hour}.{minute}.{second}","{weekday}, {day}. {month} {year}. {hour}.{minute}.{second} {ampm}","сре","Taut","Amšir","Baramuda","Bašans","Jekatit","Nehase","sre","ponedeljak","n. e.","Tišri","Hešvan","Ševat","Čaitra","Vaisaka","Jiaista","Asada","Badra","Argajana","Pauza","Maga","Falguna","Reb. 1","Reb 2","Džum. 1","Džum. 2","Redž.","Ša.","Še.","Zul-k.","Zul-h.","Muraham","Rađab","Šaʻban","Šaval","Duʻl-Kiʻda","Duʻl-hiđa","Hakuči (650-671)","Hakuho (672-686)","Šučo (686-701)","Taiho (701-704)","Vado (708-715)","Joro (717-724)","Tempio (729-749)","Tempio-kampo (749-749)","Tempio-šoho (749-757)","Tempio-hođi (757-765)","Tempo-đingo (765-767)","Đingo-keiun (767-770)","Hoki (770-780)","Ten-o (781-782)","Enrjaku (782-806)","Daido (806-810)","Konin (810-824)","Tenčo (824-834)","Šova (834-848)","Kajo (848-851)","Tenan (857-859)","Jogan (859-877)","Ninja (885-889)","Kampjo (889-898)","Šotai (898-901)","Enđi (901-923)","Enčo (923-931)","Šohei (931-938)","Tengjo (938-947)","Tenriaku (947-957)","Ova (961-964)","Koho (964-968)","Ana (968-970)","Jogen (976-978)","Kana (985-987)","Šorjaku (990-995)","Čotoku (995-999)","Čoho (999-1004)","Kanko (1004-1012)","Čova (1012-1017)","Kanin (1017-1021)","Đian (1021-1024)","Čogen (1028-1037)","Čorjaku (1037-1040)","Čokju (1040-1044)","Eišo (1046-1053)","Tenđi (1053-1058)","Kohei (1058-1065)","Đirjaku (1065-1069)","Enkju (1069-1074)","Šoho (1074-1077)","Šorjaku (1077-1081)","Eišo (1081-1084)","Otoku (1084-1087)","Kanđi (1087-1094)","Eičo (1096-1097)","Šotoku (1097-1099)","Kova (1099-1104)","Čođi (1104-1106)","Kašo (1106-1108)","Tenin (1108-1110)","Eikju (1113-1118)","Đen-ei (1118-1120)","Tenđi (1124-1126)","Daiđi (1126-1131)","Tenšo (1131-1132)","Čošao (1132-1135)","Eiđi (1141-1142)","Kođi (1142-1144)","Tenjo (1144-1145)","Kjuan (1145-1151)","Kjuju (1154-1156)","Heiđi (1159-1160)","Eirjaku (1160-1161)","Oho (1161-1163)","Čokan (1163-1165)","Kao (1169-1171)","Šoan (1171-1175)","Đišo (1177-1181)","Jova (1181-1182)","Đuei (1182-1184)","Genrjuku (1184-1185)","Bunđi (1185-1190)","Kenkju (1190-1199)","Šođi (1199-1201)","Kenin (1201-1204)","Genkju (1204-1206)","Šogen (1207-1211)","Kenrjaku (1211-1213)","Kenpo (1213-1219)","Šokju (1219-1222)","Đu (1222-1224)","Đenin (1224-1225)","Đoei (1232-1233)","Bunrjaku (1234-1235)","Rjakunin (1238-1239)","En-o (1239-1240)","Hođi (1247-1249)","Kenčo (1249-1256)","Kogen (1256-1257)","Šoka (1257-1259)","Šogen (1259-1260)","Bun-o (1260-1261)","Kočo (1261-1264)","Kenđi (1275-1278)","Koan (1278-1288)","Šu (1288-1293)","Šoan (1299-1302)","Tokuđi (1306-1308)","Očo (1311-1312)","Šova (1312-1317)","Bunpo (1317-1319)","Đeno (1319-1321)","Đenkjo (1321-1324)","Šoču (1324-1326)","Genko (1331-1334)","Kemu (1334-1336)","Kokoku (1340-1346)","Šohei (1346-1370)","Buču (1372-1375)","Korjaku (1379-1381)","Kova (1381-1384)","Genču (1384-1392)","Ku (1389-1390)","Oei (1394-1428)","Šočo (1428-1429)","Eikjo (1429-1441)","Hotoku (1449-1452)","Kjotoku (1452-1455)","Košo (1455-1457)","Čoroku (1457-1460)","Kanšo (1460-1466)","Bunšo (1466-1467)","Onin (1467-1469)","Čokjo (1487-1489)","Meio (1492-1501)","Eišo (1504-1521)","Kjoroku (1528-1532)","Kođi (1555-1558)","Tenšo (1573-1592)","Keičo (1596-1615)","Genva (1615-1624)","Šoho (1644-1648)","Šu (1652-1655)","Meirjaku (1655-1658)","Enpo (1673-1681)","Tenva (1681-1684)","Jokjo (1684-1688)","Hoei (1704-1711)","Šotoku (1711-1716)","Kjoho (1716-1736)","Kanpo (1741-1744)","Enkjo (1744-1748)","Horjaku (1751-1764)","Meiva (1764-1772)","Kjova (1801-1804)","Tenpo (1830-1844)","Koka (1844-1848)","Bunkju (1861-1864)","Genđi (1864-1865)","Keiko (1865-1868)","Meiđi","Taišo","Šova","Haisei","Faravadin","Ordibehešt","Kordad","Šahrivar","Dej","Pre RK","RK","Tl","Bhi","Van","Vol","Nkh","Nhl","Kho","Ngc","Nyo","Mph","Ngo","Bhimbidvwane","iNdlovana","iNdlovu-lenkhulu","iNkhwekhweti","iNhlaba","Kholwane","iNgci","iNyoni","iMphala","Lweti","iNgongoni","Mso","Tsa","Mgc","Lisontfo","uMsombuluko","Lesibili","Lesitsatfu","Lesine","Lesihlanu","uMgcibelo","Sal","Naba Sambat","Sani","Salus","Rabuq","Camus","Jumqata","Qunxa Sambat","Phe","Kol","Ube","Mme","Mot","Upu","Pha","Leo","Pun","Tsh","Phesekgong","Hlakola","Hlakubele","Mmese","Motsheanong","Phupjane","Phupu","Phata","Leotshe","Mphalane","Pundungwane","Tshitwe","Mma","Moq","Sontaha","Mmantaha","Laboraru","Labohlane","Moqebelo","kiahk","augusti","sö","må","lö","sön","mån","tis","ons","tors","fre","lör","söndag","tisdag","lördag","före Kristus","efter Kristus","före västerländsk tideräkning","västerländsk tideräkning","fm","em","tishrí","heshván","kislév","tevét","shevát","adár I","adár","nisán","ijjár","siván","tammúz","elúl","adár II","Saka-eran","jumada-l-ula","jumada-l-akhira","sha’ban","shawwal","dhu-l-ga’da","dhu-l-hijja","Taika (645–650)","Hakuchi (650–671)","Hakuhō (672–686)","Shuchō (686–701)","Taihō (701–704)","Keiun (704–708)","Wadō (708–715)","Reiki (715–717)","Yōrō (717–724)","Jinki (724–729)","Tempyō (729–749)","Tempyō-kampō (749–749)","Tempyō-shōhō (749–757)","Tempyō-hōji (757–765)","Temphō-jingo (765–767)","Jingo-keiun (767–770)","Hōki (770–780)","Ten-ō (781–782)","Enryaku (782–806)","Daidō (806–810)","Kōnin (810–824)","Tenchō (824–834)","Jōwa (834–848)","Kajō (848–851)","Ninju (851–854)","Saiko (854–857)","Tennan (857–859)","Jōgan (859–877)","Genkei (877–885)","Ninna (885–889)","Kampyō (889–898)","Shōtai (898–901)","Engi (901–923)","Enchō (923–931)","Shōhei (931–938)","Tengyō (938–947)","Tenryaku (947–957)","Tentoku (957–961)","Ōwa (961–964)","Kōhō (964–968)","Anna (968–970)","Tenroku (970–973)","Ten-en (973–976)","Jōgen (976–978)","Tengen (978–983)","Eikan (983–985)","Kanna (985–987)","Ei-en (987–989)","Eiso (989–990)","Shōryaku (990–995)","Chōtoku (995–999)","Chōhō (999–1004)","Kankō (1004–1012)","Chōwa (1012–1017)","Kannin (1017–1021)","Jian (1021–1024)","Manju (1024–1028)","Chōgen (1028–1037)","Chōryaku (1037–1040)","Chōkyū (1040–1044)","Kantoku (1044–1046)","Eishō (1046–1053)","Tengi (1053–1058)","Kōhei (1058–1065)","Jiryaku (1065–1069)","Enkyū (1069–1074)","Shōho (1074–1077)","Shōryaku (1077–1081)","Eiho (1081–1084)","Ōtoku (1084–1087)","Kanji (1087–1094)","Kaho (1094–1096)","Eichō (1096–1097)","Shōtoku (1097–1099)","Kōwa (1099–1104)","Chōji (1104–1106)","Kashō (1106–1108)","Tennin (1108–1110)","Ten-ei (1110–1113)","Eikyū (1113–1118)","Gen-ei (1118–1120)","Hoan (1120–1124)","Tenji (1124–1126)","Daiji (1126–1131)","Tenshō (1131–1132)","Chōshō (1132–1135)","Hoen (1135–1141)","Eiji (1141–1142)","Kōji (1142–1144)","Tenyō (1144–1145)","Kyūan (1145–1151)","Ninpei (1151–1154)","Kyūju (1154–1156)","Hogen (1156–1159)","Heiji (1159–1160)","Eiryaku (1160–1161)","Ōho (1161–1163)","Chōkan (1163–1165)","Eiman (1165–1166)","Nin-an (1166–1169)","Kaō (1169–1171)","Shōan (1171–1175)","Angen (1175–1177)","Jishō (1177–1181)","Yōwa (1181–1182)","Juei (1182–1184)","Genryuku (1184–1185)","Bunji (1185–1190)","Kenkyū (1190–1199)","Shōji (1199–1201)","Kennin (1201–1204)","Genkyū (1204–1206)","Ken-ei (1206–1207)","Shōgen (1207–1211)","Kenryaku (1211–1213)","Kenpō (1213–1219)","Shōkyū (1219–1222)","Jōō (1222–1224)","Gennin (1224–1225)","Karoku (1225–1227)","Antei (1227–1229)","Kanki (1229–1232)","Jōei (1232–1233)","Tempuku (1233–1234)","Bunryaku (1234–1235)","Katei (1235–1238)","Ryakunin (1238–1239)","En-ō (1239–1240)","Ninji (1240–1243)","Kangen (1243–1247)","Hōji (1247–1249)","Kenchō (1249–1256)","Kōgen (1256–1257)","Shōka (1257–1259)","Shōgen (1259–1260)","Bun-ō (1260–1261)","Kōchō (1261–1264)","Bun-ei (1264–1275)","Kenji (1275–1278)","Kōan (1278–1288)","Shōō (1288–1293)","Einin (1293–1299)","Shōan (1299–1302)","Kengen (1302–1303)","Kagen (1303–1306)","Tokuji (1306–1308)","Enkei (1308–1311)","Ōchō (1311–1312)","Shōwa (1312–1317)","Bunpō (1317–1319)","Genō (1319–1321)","Genkyō (1321–1324)","Shōchū (1324–1326)","Kareki (1326–1329)","Gentoku (1329–1331)","Genkō (1331–1334)","Kemmu (1334–1336)","Engen (1336–1340)","Kōkoku (1340–1346)","Shōhei (1346–1370)","Kentoku (1370–1372)","Bunchū (1372–1375)","Tenju (1375–1379)","Kōryaku (1379–1381)","Kōwa (1381–1384)","Genchū (1384–1392)","Meitoku (1384–1387)","Kakei (1387–1389)","Kōō (1389–1390)","Meitoku (1390–1394)","Ōei (1394–1428)","Shōchō (1428–1429)","Eikyō (1429–1441)","Kakitsu (1441–1444)","Bun-an (1444–1449)","Hōtoku (1449–1452)","Kyōtoku (1452–1455)","Kōshō (1455–1457)","Chōroku (1457–1460)","Kanshō (1460–1466)","Bunshō (1466–1467)","Ōnin (1467–1469)","Bunmei (1469–1487)","Chōkyō (1487–1489)","Entoku (1489–1492)","Meiō (1492–1501)","Bunki (1501–1504)","Eishō (1504–1521)","Taiei (1521–1528)","Kyōroku (1528–1532)","Tenmon (1532–1555)","Kōji (1555–1558)","Eiroku (1558–1570)","Genki (1570–1573)","Tenshō (1573–1592)","Bunroku (1592–1596)","Keichō (1596–1615)","Genwa (1615–1624)","Kan-ei (1624–1644)","Shōho (1644–1648)","Keian (1648–1652)","Shōō (1652–1655)","Meiryaku (1655–1658)","Manji (1658–1661)","Kanbun (1661–1673)","Enpō (1673–1681)","Tenwa (1681–1684)","Jōkyō (1684–1688)","Genroku (1688–1704)","Hōei (1704–1711)","Shōtoku (1711–1716)","Kyōhō (1716–1736)","Genbun (1736–1741)","Kanpō (1741–1744)","Enkyō (1744–1748)","Kan-en (1748–1751)","Hōryaku (1751–1764)","Meiwa (1764–1772)","An-ei (1772–1781)","Tenmei (1781–1789)","Kansei (1789–1801)","Kyōwa (1801–1804)","Bunka (1804–1818)","Bunsei (1818–1830)","Tenpō (1830–1844)","Kōka (1844–1848)","Kaei (1848–1854)","Ansei (1854–1860)","Man-en (1860–1861)","Bunkyū (1861–1864)","Genji (1864–1865)","Keiō (1865–1868)","farvardin","ordibehesht","khordād","tir","mordād","shahrivar","mehr","ābān","āzar","dey","bahman","esfand","före R.K.","R.K.","Bds$","Tk","BM$","BN$","BR$","BS$","BZ$","CAN$","RD$","EG£","GY$","Ikr","JM$","NKr","SY£","TW$","VN₫","före vår tideräkning","vår tideräkning","mkw","mpi","mtu","msb","mun","mts","mku","mkm","mkb","mwezi ya kwanja","mwezi ya pili","mwezi ya tatu","mwezi ya ine","mwezi ya tanu","mwezi ya sita","mwezi ya saba","mwezi ya munane","mwezi ya tisa","mwezi ya kumi","mwezi ya kumi na moya","mwezi ya kumi ya mbili","yen","pil","tat","ine","tan","sit","siku ya yenga","siku ya kwanza","siku ya pili","siku ya tatu","siku ya ine","siku ya tanu","siku ya sita","mbele ya Y","kisha ya Y","mbele ya Yezu Kristo","kisha ya Yezu Kristo","ya asubuyi","ya muchana","மா1","மா2","மா3","மா4","மா5","மா6","மா7","மா8","மா9","மா10","மா11","மா12","மாதம்1","மாதம்2","மாதம்3","மாதம்4","மாதம்5","மாதம்6","மாதம்7","மாதம்8","மாதம்9","மாதம்10","மாதம்11","மாதம்12","ஜ","பி","மா","ஏ","மே","ஜூ","ஆ","செ","அ","ந","டி","ஜன.","பிப்.","மார்.","ஏப்.","ஜூன்","ஜூலை","ஆக.","செப்.","அக்.","நவ.","டிச.","ஜனவரி","பிப்ரவரி","மார்ச்","ஏப்ரல்","ஆகஸ்ட்","செப்டம்பர்","அக்டோபர்","நவம்பர்","டிசம்பர்","ஞா","தி","பு","வி","வெ","ச","ஞாயிறு","திங்கள்","செவ்வாய்","புதன்","வியாழன்","வெள்ளி","சனி","கி.மு.","கி.பி.","கிறிஸ்துவுக்கு முன்","அனோ டோமினி","முற்பகல்","பிற்பகல்","Rs.","S$","{weekday}, {day}, {month} {year} {hour}:{minute}:{second}","{weekday}, {day}, {month} {year} {hour}:{minute}:{second} {ampm}","{day}, {month} {year}","జ","ఫి","మా","ఏ","మే","జూ","జు","ఆ","సె","అ","న","డి","జన","ఫిబ్ర","మార్చి","ఏప్రి","జూన్","జులై","ఆగ","సెప్టెం","అక్టో","నవం","డిసెం","జనవరి","ఫిబ్రవరి","ఎప్రిల్","ఆగస్టు","సెప్టెంబర్","అక్టోబర్","నవంబర్","డిసెంబర్","ఆది","సోమ","మం","బుధ","గురు","శుక్ర","శని","మంగళ","ఆదివారం","సోమవారం","మంగళవారం","బుధవారం","గురువారం","శుక్రవారం","శనివారం","ఈసాపూర్వ.","సన్.","క్రీపూ","క్రీశ","చైత్రం","వైశాఖం","జ్యేష్ఠం","ఆషాఢం","శ్రావణం","భాద్రపదం","ఆశ్వయుజం","కార్తీకం","మార్గశిరం","పుష్యం","మాఘం","ఫల్గుణం","శక","Muk","Dun","Mod","Ped","Sok","Tib","Poo","Orara","Omuk","Okwamg'","Odung'el","Omaruk","Omodok'king'ol","Ojola","Opedel","Osokosokoma","Otibar","Olabor","Opoo","Bar","Aar","Uni","Ung","Nakaejuma","Nakaebarasa","Nakaare","Nakauni","Nakaung'on","Nakakany","Nakasabiti","Taparachu","Ebongi","Янв","Фев","Мар","Апр","Май","Июн","Июл","Авг","Сен","Окт","Ноя","Дек","Январ","Феврал","Март","Апрел","Август","Сентябр","Октябр","Ноябр","Декабр","Яшб","Дшб","Сшб","Чшб","Пшб","Ҷмъ","Шнб","Якшанбе","Душанбе","Сешанбе","Чоршанбе","Панҷшанбе","Ҷумъа","Шанбе","ПеМ","ПаМ","Пеш аз милод","пе. чо.","па. чо.","พ.ศ.","พุทธศักราช","เทาท์","บาบา","ฮาเทอร์","เคียฟ","โทบา","อัมเชอร์","บารัมฮัท","บาราเมาดา","บาชันส์","พาโอนา","อีเปป","เมสรา","นาซี","เมสเคอเรม","เตเกมท","เฮดาร์","ทาฮ์ซัส","เทอร์","เยคาทิท","เมกาบิต","เมียเซีย","เจนบอต","เซเน","ฮัมเล","เนแฮซ","พากูเมน","ม.ค.","ก.พ.","มี.ค.","เม.ย.","พ.ค.","มิ.ย.","ก.ค.","ส.ค.","ก.ย.","ต.ค.","พ.ย.","ธ.ค.","มกราคม","กุมภาพันธ์","มีนาคม","เมษายน","พฤษภาคม","มิถุนายน","กรกฎาคม","สิงหาคม","กันยายน","ตุลาคม","พฤศจิกายน","ธันวาคม","อา.","จ.","อ.","พ.","พฤ.","ศ.","ส.","วันอาทิตย์","วันจันทร์","วันอังคาร","วันพุธ","วันพฤหัสบดี","วันศุกร์","วันเสาร์","ก่อน ค.ศ.","ค.ศ.","ก.ส.ศ.","ส.ศ.","ปีก่อน ค.ศ.","ปีก่อนคริสต์ศักราช","คริสต์ศักราช","ก่อนสามัญศักราช","สามัญศักราช","ก่อนเที่ยง","หลังเที่ยง","ทิชรี","เฮวาน","กีสเลฟ","เตเวต","เชวัต","อาดาร์ I","อาดาร์","นิสซาน","อิยาร์","สีวัน","ตามูซ","อัฟ","เอลอุล","อาดาร์ II","ย.ศ.","จิตรา","วิสาขา","เชษฐา","อัษฎา","ศรวณา","พัตรา","อัศวิชา","การติกา","มฤคศิรา","ปุษยา","มาฆะ","ผลคุณี","ม.ศ.","มุฮัร.","เศาะ.","รอบี 1","รอบี 2","ญุมา 1","ญุมา 2","เราะ.","ชะอ์.","เราะมะ.","เชาว.","ซุลกิอฺ.","ซุลหิจ.","มุฮะร์รอม","ซอฟาร์","รอบี I","รอบี II","จุมาดา I","จุมาดา II","รอจับ","ชะอะบาน","รอมะดอน","เชาวัล","ดฮุุอัลกิดะห์","ดฮุอัลฮิจจะห์","ฮ.ศ.","ฮิจเราะห์ศักราช","ทะอิกะ (645-650)","ฮะกุชิ (650-671)","ฮากุโฮ (672-686)","ชุโช (686-701)","ทะอิโฮ (701-704)","เคอุง (704-708)","วะโด (708-715)","เรกิ (715-717)","โยโร (717-724)","จิงกิ (724-729)","เท็มเพียว (729-749)","เท็มเพียว-คัมโป (749-749)","เท็มเพียว-โชโฮ (749-757)","เท็มเพียว-โฮจิ (757-765)","เท็มเพียว-จิงโงะ (765-767)","จิงโงะ-เคอุง (767-770)","โฮกิ (770-780)","เท็นโอ (781-782)","เอ็นเรียะกุ (782-806)","ดะอิโด (806-810)","โคนิง (810-824)","เท็นโช (824-834)","โชวะ (834-848)","คะโจ (848-851)","นินจุ (851-854)","ซะอิโกะ (854-857)","เท็นนัง (857-859)","โจงัง (859-877)","เก็งเก (877-885)","นินนะ (885-889)","คัมเพียว (889-898)","โชตะอิ (898-901)","เอ็งงิ (901-923)","เอ็นโช (923-931)","โชเฮ (931-938)","เท็งเงียว (938-947)","เท็นเรียะกุ (947-957)","เท็นโตะกุ (957-961)","โอวะ (961-964)","โคโฮ (964-968)","อันนะ (968-970)","เท็นโระกุ (970-973)","เท็นเอ็ง (973-976)","โจเง็ง (976-978)","เท็งเง็ง (978-983)","เอกัง (983-985)","คันนะ (985-987)","เอเอ็ง (987-989)","เอโซ (989-990)","โชเรียะกุ (990-995)","โชโตะกุ (995-999)","โชโฮ (999-1004)","คันโก (1004-1012)","โชวะ (1012-1017)","คันนิง (1017-1021)","จิอัง (1021-1024)","มันจุ (1024-1028)","โชเง็ง (1028-1037)","โชเรียะกุ (1037-1040)","โชคีว (1040-1044)","คันโตะกุ (1044-1046)","เอโช (1046-1053)","เท็งงิ (1053-1058)","โคเฮ (1058-1065)","จิเรียะกุ (1065-1069)","เอ็งคีว (1069-1074)","โชโฮ (1074-1077)","โชเรียะกุ (1077-1081)","เอโฮะ (1081-1084)","โอโตะกุ (1084-1087)","คันจิ (1087-1094)","คะโฮะ (1094-1096)","เอโช (1096-1097)","โชโตะกุ (1097-1099)","โควะ (1099-1104)","โชจิ (1104-1106)","คะโช (1106-1108)","เท็นนิง (1108-1110)","เท็นเอ (1110-1113)","เอกีว (1113-1118)","เก็นเอ (1118-1120)","โฮะอัง (1120-1124)","เท็นจิ (1124-1126)","ดะอิจิ (1126-1131)","เท็นโช (1131-1132)","โชโช (1132-1135)","โฮะเอ็ง (1135-1141)","เอจิ (1141-1142)","โคจิ (1142-1144)","เท็นโย (1144-1145)","คีวอัง (1145-1151)","นิมเป (1151-1154)","คีวจุ (1154-1156)","โฮะเง็ง (1156-1159)","เฮจิ (1159-1160)","เอเรียะกุ (1160-1161)","โอโฮ (1161-1163)","โชกัง (1163-1165)","เอมัง (1165-1166)","นินอัง (1166-1169)","คะโอ (1169-1171)","โชอัง (1171-1175)","อังเง็ง (1175-1177)","จิโช (1177-1181)","โยวะ (1181-1182)","จุเอ (1182-1184)","เก็นเรียะกุ (1184-1185)","บุนจิ (1185-1190)","เค็งกีว (1190-1199)","โชจิ (1199-1201)","เค็นนิง (1201-1204)","เก็งกีว (1204-1206)","เค็นเอ (1206-1207)","โชเก็ง (1207-1211)","เค็นเรียะกุ (1211-1213)","เค็มโป (1213-1219)","โชกีว (1219-1222)","โจโอ (1222-1224)","เก็นนิง (1224-1225)","คะโระกุ (1225-1227)","อันเต (1227-1229)","คังกิ (1229-1232)","โจเอ (1232-1233)","เท็มปุกุ (1233-1234)","บุนเรียะกุ (1234-1235)","คะเต (1235-1238)","เรียะกุนิง (1238-1239)","เอ็นโอ (1239-1240)","นินจิ (1240-1243)","คังเง็ง (1243-1247)","โฮจิ (1247-1249)","เค็นโช (1249-1256)","โคเง็ง (1256-1257)","โชกะ (1257-1259)","โชเง็ง (1259-1260)","บุนโอ (1260-1261)","โคโช (1261-1264)","บุนเอ (1264-1275)","เค็นจิ (1275-1278)","โคอัง (1278-1288)","โชโอ (1288-1293)","เอนิง (1293-1299)","โชอัง (1299-1302)","เค็งเง็ง (1302-1303)","คะเง็ง (1303-1306)","โทะกุจิ (1306-1308)","เอ็งเก (1308-1311)","โอโช (1311-1312)","โชวะ (1312-1317)","บุมโป (1317-1319)","เก็นโอ (1319-1321)","เก็งเกียว (1321-1324)","โชชู (1324-1326)","คะเระกิ (1326-1329)","เก็นโตะกุ (1329-1331)","เก็งโก (1331-1334)","เค็มมุ (1334-1336)","เอ็งเง็ง (1336-1340)","โคโกะกุ (1340-1346)","โชเฮ (1346-1370)","เค็นโตะกุ (1370-1372)","บุนชู (1372-1375)","เท็นจุ (1375-1379)","โคเรียะกุ (1379-1381)","โควะ (1381-1384)","เก็นชู (1384-1392)","เมโตะกุ (1384-1387)","คะเค (1387-1389)","โคโอ (1389-1390)","เมโตะกุ (1390-1394)","โอเอ (1394-1428)","โชโช (1428-1429)","เอเกียว (1429-1441)","คะกิสึ (1441-1444)","บุนอัง (1444-1449)","โฮโตะกุ (1449-1452)","เคียวโตะกุ (1452-1455)","โคโช (1455-1457)","โชโระกุ (1457-1460)","คันโช (1460-1466)","บุนโช (1466-1467)","โอนิง (1467-1469)","บุมเม (1469-1487)","โชเกียว (1487-1489)","เอ็นโตะกุ (1489-1492)","เมโอ (1492-1501)","บุงกิ (1501-1504)","เอโช (1504-1521)","ทะอิเอ (1521-1528)","เคียวโระกุ (1528-1532)","เท็มมน (1532-1555)","โคจิ (1555-1558)","เอโระกุ (1558-1570)","เก็งกิ (1570-1573)","เท็นโช (1573-1592)","บุนโระกุ (1592-1596)","เคโช (1596-1615)","เก็งวะ (1615-1624)","คันเอ (1624-1644)","โชโฮ (1644-1648)","เคอัง (1648-1652)","โชโอ (1652-1655)","เมเรียะกุ (1655-1658)","มันจิ (1658-1661)","คัมบุง (1661-1673)","เอ็มโป (1673-1681)","เท็นวะ (1681-1684)","โจเกียว (1684-1688)","เก็นโระกุ (1688-1704)","โฮเอ (1704-1711)","โชโตะกุ (1711-1716)","เคียวโฮ (1716-1736)","เก็มบุง (1736-1741)","คัมโป (1741-1744)","เอ็งเกียว (1744-1748)","คันเอ็ง (1748-1751)","โฮเรียะกุ (1751-1764)","เมวะ (1764-1772)","อันเอ (1772-1781)","เท็มเม (1781-1789)","คันเซ (1789-1801)","เคียววะ (1801-1804)","บุงกะ (1804-1818)","บุนเซ (1818-1830)","เท็มโป (1830-1844)","โคกะ (1844-1848)","คะเอ (1848-1854)","อันเซ (1854-1860)","มันเอ็ง (1860-1861)","บุงกีว (1861-1864)","เก็นจิ (1864-1865)","เคโอ (1865-1868)","เมจิ","ทะอิโช","โชวะ","เฮเซ","ฟาร์วาร์ดิน","ออร์ดิเบเฮชต์","คอร์แดด","เตอร์","มอร์แดด","ชาหริวาร์","เมฮร์","อะบาน","อะซาร์","เดย์","บาฮ์มาน","เอสฟานด์","ปีเปอร์เซีย","ปีก่อนไต้หวัน","ไต้หวัน","ጥሪ","ለካቲ","ሰነ","ሓምለ","ነሓሰ","ሕዳር","ታሕሳ","ለካቲት","ጥቅምቲ","ታሕሳስ","ሰንበት","ሰኑይ","ሰሉስ","ሓሙስ","ዓርቢ","ቀዳም","ንጉሆ ሰዓተ","ድሕር ሰዓት","ኤፕረ","ኤፕረል","ሠሉስ","ኃሙስ","ሰ/ዓ","ሰኖ","ታላሸ","ኣረር","ከሚሽ","ጅምዓ","ሰ/ን","ሰንበት ዓባይ","ታላሸኖ","ኣረርባዓ","ጅምዓት","ሰንበት ንኢሽ","ቀደም ሰርምዕል","ሓቆ ስርምዕል","Fer","Tlh","Mop","Mor","See","Phu","Dip","Ngw","Sed","Ferikgong","Tlhakole","Mopitlo","Moranang","Motsheganang","Seetebosigo","Phukwi","Phatwe","Lwetse","Diphalane","Ngwanatsele","Sedimonthole","Tla","Tshipi","Mosopulogo","Labotlhano","Matlhatso","Sān","Fēp","Maʻa","ʻEpe","Mē","Siu","ʻAok","ʻOka","Nōv","Tīs","Sānuali","Fēpueli","Maʻasi","ʻEpeleli","Sune","Siulai","ʻAokosi","Sepitema","ʻOkatopa","Nōvema","Tīsema","Sāp","Mōn","Tus","Pul","Tuʻa","Fal","Tok","Tūs","Sāpate","Mōnite","Tūsite","Pulelulu","Tuʻapulelulu","Falaite","Tokonaki","KāMā","TāSā","TS","ki muʻa","taʻu ʻo Sīsū","TF","{day} {month} {year} {weekday}  {hour}:{minute}:{second}","{day} {month} {year} {weekday} {ampm} {hour}:{minute}:{second}","{day} {month} {year} {weekday}","Tût","Bâbe","Keyhek","Tûbe","Imşir","Bermuhat","Bermude","Peyştes","Bune","Ebip","Mısrî","Nesî","Tikimt","Hidar","Yakatit","Magabit","Ginbot","Nehasa","Pagumiene","Ş","Oca","Şub","Nis","Haz","Tem","Ağu","Eyl","Eki","Ocak","Şubat","Mart","Mayıs","Haziran","Temmuz","Ağustos","Eylül","Ekim","Kasım","Aralık","Pa","Pt","Ça","Pe","Cu","Ct","Paz","Pzt","Çar","Per","Cum","Pazar","Pazartesi","Salı","Çarşamba","Perşembe","Cuma","Cumartesi","MÖ","MS","İÖ","İS","Milattan Önce","Milattan Sonra","ÖÖ","ÖS","Tişri","Heşvan","Şevat","Veadar","İyar","Muharrem","Safer","Rebiülevvel","Rebiülahir","Cemaziyelevvel","Cemaziyelahir","Recep","Şaban","Ramazan","Şevval","Zilkade","Zilhicce","Hicri","Ferverdin","Ordibeheşt","Hordad","Şehriver","Azer","Behmen","Esfend","%{number}","-%{number}","₺","Kul","Dzi","Mha","Ndz","Huk","N'w","Sunguti","Nyenyenyani","Nyenyankulu","Dzivamisoko","Mudyaxihi","Khotavuxika","Mawuwani","Mhawuri","Ndzhati","Hukuri","N'wendzamhala","Bir","Har","Musumbhunuku","Ravumbirhi","Ravunharhu","Ravumune","Ravuntlhanu","Mugqivela","Ibr","Cut","Kṭu","Nwa","Duj","Yebrayer","Ibrir","Yulyuz","Cutanbir","Kṭuber","Nwanbir","Dujanbir","Asa","Ayn","Asn","Akr","Akw","Asm","Asḍ","Asamas","Aynas","Asinas","Akras","Akwas","Asimwas","Asiḍyas","ZƐ","ḌƐ","Zdat Ɛisa (TAƔ)","Ḍeffir Ɛisa (TAƔ)","Zdat azal","Ḍeffir aza","тот.","баб.","хат.","кіх.","тоб.","амш.","барам.","барм.","баш.","баун.","аб.","мис.","нас.","Бабе","Кіхак","Тобе","Абіб","Насі","мес.","тек.","хед.","тах.","тер.","єкат.","мег.","міяз.","ген.","хам.","нех.","паг.","мескерема","текемта","хедара","тахсаса","тера","єкатіта","мегабіта","міязія","генбота","сене","хамле","нехасе","пагумена","Л","Б","К","Т","Ч","В","Ж","Г","січ.","лют.","бер.","квіт.","трав.","черв.","лип.","серп.","вер.","жовт.","лист.","груд.","січня","лютого","березня","квітня","травня","червня","липня","серпня","вересня","жовтня","листопада","грудня","Нд","Пн","Вт","Ср","Чт","Пт","неділя","понеділок","вівторок","середа","четвер","пʼятниця","до н.е.","до нашої ери","нашої ери","дп","пп","Тішри","Марчешван","Числьов","Тебет","Нісан","Іар","Аб","чайт.","вайс.","джай.","асад.","шрав.","бхад.","асв.","кар.","агр.","паус.","маг.","фаль.","джайстха","шравана","бхадра","асвіна","картіка","аграхаяна","пауса","фальгуна","мух.","саф.","раб. I","раб. II","джум. I","джум. II","радж.","шааб.","рам.","дав.","зу-ль-к.","зу-ль-х.","Рабі I","Рабі II","Джумада I","Джумада II","Даввал","Зу-ль-каада","Зу-ль-хіджа","Тайка (645–650)","Хакуті (650–671)","Хакухо (672–686)","Сютьо (686–701)","Тайхо (701–704)","Кейун (704–708)","Вадо (708–715)","Рейкі (715–717)","Йоро (717–724)","Дзінгі (724–729)","Темпьо (729–749)","Темпьо-кампо (749–749)","Темпьо-сьохо (749–757)","Темпьо-ходзі (757–765)","Темпьо-дзінго (765–767)","Дзінго кейун (767–770)","Хокі (770–780)","Тен’о (781–782)","Енряку (782–806)","Дайдо (806–810)","Конін (810–824)","Тентьо (824–834)","Сьова (834–848)","Кадзьо (848–851)","Ніндзю (851–854)","Сайко (854–857)","Теннан (857–859)","Дзьоган (859–877)","Генкей (877–885)","Нінна (885–889)","Кампьо (889–898)","Сьотай (898–901)","Енгі (901–923)","Ентьо (923–931)","Сьохей (931–938)","Тенгьо (938–947)","Тенряку (947–957)","Тентоку (957–961)","Ова (961–964)","Кохо (964–968)","Анна (968–970)","Тенроку (970–973)","Тен’ен (973–976)","Дзьоген (976–978)","Тенген (978–983)","Ейкан (983–985)","Канна (985–987)","Ейен (987–989)","Ейсо (989–990)","Сьоряку (990–995)","Тьотоку (995–999)","Тьохо (999–1004)","Канко (1004–1012)","Тьова (1012–1017)","Каннін (1017–1021)","Дзіан (1021–1024)","Мандзю (1024–1028)","Тьоген (1028–1037)","Тьоряку (1037–1040)","Тьокю (1040–1044)","Кантоку (1044–1046)","Ейсьо (1046–1053)","Тенгі (1053–1058)","Кохей (1058–1065)","Дзіряку (1065–1069)","Енкю (1069–1074)","Сьохо (1074–1077)","Сьоряку (1077–1081)","Ейхо (1081–1084)","Отоку (1084–1087)","Кандзі (1087–1094)","Кахо (1094–1096)","Ейсьо (1096–1097)","Сьотоку (1097–1099)","Кова (1099–1104)","Тьодзі (1104–1106)","Кадзьо (1106–1108)","Теннін (1108–1110)","Тен’ей (1110–1113)","Ейкю (1113–1118)","Ген’ей (1118–1120)","Хоан (1120–1124)","Тендзі (1124–1126)","Дайдзі (1126–1131)","Тенсьо (1131–1132)","Тьосьо (1132–1135)","Хоен (1135–1141)","Ейдзі (1141–1142)","Кодзі (1142–1144)","Теньо (1144–1145)","Кюан (1145–1151)","Німпей (1151–1154)","Кюдзю (1154–1156)","Хоген (1156–1159)","Хейдзі (1159–1160)","Ейряку (1160–1161)","Охо (1161–1163)","Тьокан (1163–1165)","Ейман (1165–1166)","Нін’ан (1166–1169)","Као (1169–1171)","Сьоан (1171–1175)","Анген (1175–1177)","Дзісьо (1177–1181)","Йова (1181–1182)","Дзюей (1182–1184)","Генряку (1184–1185)","Бундзі (1185–1190)","Кенкю (1190–1199)","Сьодзі (1199–1201)","Кеннін (1201–1204)","Генкю (1204–1206)","Кен’ей (1206–1207)","Сьоген (1207–1211)","Кенряку (1211–1213)","Кенпо (1213–1219)","Дзьокю (1219–1222)","Дзьоо (1222–1224)","Геннін (1224–1225)","Кароку (1225–1227)","Антей (1227–1229)","Канкі (1229–1232)","Дзюей (1232–1233)","Темпуку (1233–1234)","Бунряку (1234–1235)","Катей (1235–1238)","Рякунін (1238–1239)","Ен’о (1239–1240)","Ніндзі (1240–1243)","Канген (1243–1247)","Хейдзі (1247–1249)","Кентьо (1249–1256)","Коген (1256–1257)","Сьока (1257–1259)","Сьоген (1259–1260)","Бун’о (1260–1261)","Котьо (1261–1264)","Бун’ей (1264–1275)","Кендзі (1275–1278)","Коан (1278–1288)","Сьоо (1288–1293)","Ейнін (1293–1299)","Сьоан (1299–1302)","Кенген (1302–1303)","Каген (1303–1306)","Токудзі (1306–1308)","Енкей (1308–1311)","Отьо (1311–1312)","Сьова (1312–1317)","Бумпо (1317–1319)","Ген’о (1319–1321)","Генкьо (1321–1324)","Сьотю (1324–1326)","Карекі (1326–1329)","Гентоку (1329–1331)","Генко (1331–1334)","Кемму (1334–1336)","Ейген (1336–1340)","Кококу (1340–1346)","Сьохей (1346–1370)","Кентоку (1370–1372)","Бунтю (1372–1375)","Тендзю (1375–1379)","Коряку (1379–1381)","Кова (1381–1384)","Гентю (1384–1392)","Мейтоку (1384–1387)","Какей (1387–1389)","Коо (1389–1390)","Мейтоку (1390–1394)","Оей (1394–1428)","Сьотьо (1428–1429)","Ейкьо (1429–1441)","Какіцу (1441–1444)","Буннан (1444–1449)","Хотоку (1449–1452)","Кьотоку (1452–1455)","Косьо (1455–1457)","Тьороку (1457–1460)","Кансьо (1460–1466)","Бунсьо (1466–1467)","Онін (1467–1469)","Буммей (1469–1487)","Тьокьо (1487–1489)","Ентоку (1489–1492)","Мейо (1492–1501)","Бункі (1501–1504)","Ейсьо (1504–1521)","Тайей (1521–1528)","Кьороку (1528–1532)","Теммон (1532–1555)","Кодзі (1555–1558)","Ейроку (1558–1570)","Генкі (1570–1573)","Тенсьо (1573–1592)","Бунроку (1592–1596)","Кейтьо (1596–1615)","Генна (1615–1624)","Кан’ей (1624–1644)","Сьохо (1644–1648)","Кейан (1648–1652)","Сьоо (1652–1655)","Мейряку (1655–1658)","Мандзі (1658–1661)","Камбун (1661–1673)","Емпо (1673–1681)","Тенва (1681–1684)","Дзьокьо (1684–1688)","Генроку (1688–1704)","Хоей (1704–1711)","Сьотоку (1711–1716)","Кьохо (1716–1736)","Гембун (1736–1741)","Канпо (1741–1744)","Енкьо (1744–1748)","Кан’ен (1748–1751)","Хоряку (1751–1764)","Мейва (1764–1772)","Ан’ей (1772–1781)","Теммей (1781–1789)","Кансей (1789–1801)","Кьова (1801–1804)","Бунка (1804–1818)","Бунсей (1818–1830)","Тенпо (1830–1844)","Кока (1844–1848)","Каей (1848–1854)","Ансей (1854–1860)","Ман’ен (1860–1861)","Бункю (1861–1864)","Гендзі (1864–1865)","Кейо (1865–1868)","Мейдзі","Тайсьо","Хейсей","фар.","орд.","хор.","тір.","мор.","шах.","мех.","абан.","азер.","дея","бах.","есф.","Фарвардін","Ордібехешт","Тір","Шахрівер","Не число","крб.","مئی","سوموار","بدھ","ق م","عیسوی سن","قبل مسیح","قبل دوپہر","بعد دوپہر","ر بیع الاول","ر بیع الثانی","ذوالقعدۃ","ذوالحجۃ","یہ عدد نہیں","{currency}{number}‎","-{currency}{number}‎","Y","Yanv","Iyun","Iyul","Avg","Noya","Dek","Yanvar","Fevral","Aprel","Avgust","Sentyabr","Oktyabr","Noyabr","Dekabr","Yaksh","Dush","Sesh","Chor","Shan","yakshanba","dushanba","seshanba","chorshanba","payshanba","shanba","M.A.","Robiʼ ul-avval","Robiʼ ul-oxir","Jumad ul-avval","Jumad ul-oxir","Shaʼbon","Ramazon","Shavvol","Zul-qaʼda","Zul-hijja","soʻm","فبر","مار","اپر","اگس","سپت","اکت","نوم","ی.","د.","س.","چ.","پ.","ج.","ش.","Якш","Душ","Сеш","Чор","Пай","Шан","якшанба","душанба","сешанба","чоршанба","пайшанба","жума","шанба","М.А.","Э","Муҳаррам","Рабиул-аввал","Рабиул-охир","Жумодиул-уло","Жумодиул-ухро","Ражаб","Шаъбон","Рамазон","Шаввол","Зил-қаъда","Зил-ҳижжа","сўм","ꖨꕪꖃ ꔞꕮ","ꕒꕡꖝꖕ","ꕾꖺ","ꖢꖕ","ꖑꕱ","ꗛꔕ","ꕢꕌ","ꕭꖃ","ꔞꘋꕔꕿ ꕸꖃꗏ","ꖨꕪꕱ ꗏꕮ","ꕞꕌꔵ","ꗳꗡꘉ","ꕚꕞꕚ","ꕉꕞꕒ","ꕉꔤꕆꕢ","ꕉꔤꕀꕮ","ꔻꔬꔳ","luukao kemã","ɓandaɓu","vɔɔ","fulu","goo","kɔnde","saah","galo","kenpkato ɓololɔ","luukao lɔma","lahadi","tɛɛnɛɛ","talata","alaba","aimisa","aijima","siɓiti","Luh","Ṱhf","Lam","Shu","Lwi","Lwa","Ṱha","Khu","Ḽar","Phando","Luhuhi","Ṱhafamuhwe","Lambamai","Shundunthule","Fulwi","Fulwana","Ṱhangule","Khubvumedzi","Tshimedzi","Ḽara","Nyendavhusiku","Swo","Vhi","Ṋa","Ṱan","Swondaha","Musumbuluwo","Ḽavhuvhili","Ḽavhuraru","Ḽavhuṋa","Ḽavhuṱanu","Mugivhela","{hour}:{minute}:{second}  {weekday}, {day} {month} {year}","{hour}:{minute}:{second} {ampm} {weekday}, {day} {month} {year}","thg 1","thg 2","thg 3","thg 4","thg 5","thg 6","thg 7","thg 8","thg 9","thg 10","thg 11","thg 12","tháng 1","tháng 2","tháng 3","tháng 4","tháng 5","tháng 6","tháng 7","tháng 8","tháng 9","tháng 10","tháng 11","tháng 12","CN","T2","T3","T4","T5","T6","T7","Th 2","Th 3","Th 4","Th 5","Th 6","Th 7","Chủ Nhật","Thứ Hai","Thứ Ba","Thứ Tư","Thứ Năm","Thứ Sáu","Thứ Bảy","tr. CN","sau CN","CH","Trước R.O.C","mäz","prl","gst","ton","janul","febul","mäzil","prilul","mayul","yunul","yulul","gustul","setul","tobul","novul","dekul","mu.","tu.","ve.","dö.","zä.","sudel","mudel","tudel","vedel","dödel","fridel","zädel","b. t. kr.","p. t. kr.","posz.","büz.","short","Ö","W","Hor","Brá","Hei","Öig","Her","Wím","Win","Chr","Jenner","Hornig","Märze","Abrille","Meije","Bráčet","Heiwet","Öigšte","Herbštmánet","Wímánet","Wintermánet","Chrištmánet","Män","Ziš","Mit","Fró","Sunntag","Mäntag","Zištag","Mittwuč","Fróntag","Fritag","Samštag","n. Chr","руб","ወጋ","ሳይኖ","ማቆሳኛ","አሩዋ","ሃሙሳ","አርባ","ቄራ","አዳ ዎዴ","ግሮተታ ላይታ","ማለዶ","ቃማ","Aga","Janyuwari","Februwari","Matshi","Epreli","Okthoba","Caw","Bin","Cawe","Lwesibini","Lwesithathu","Lwesine","Lwesihlanu","umnyaka wokuzalwa kukaYesu","Sabi","Bala","Kubi","Kusa","Kuna","Kuta","Muka","Sabiiti","Owokubili","Owokusatu","Olokuna","Olokutaanu","Olomukaaga","AZ","AF","Kulisto nga azilawo","Kulisto nga affile","Munkyo","Eigulo","o.1","o.2","o.3","o.4","o.5","o.6","o.7","o.8","o.9","o.10","o.11","o.12","pikítíkítie, oólí ú kutúan","siɛyɛ́, oóli ú kándíɛ","ɔnsúmbɔl, oóli ú kátátúɛ","mesiŋ, oóli ú kénie","ensil, oóli ú kátánuɛ","ɔsɔn","efute","pisuyú","imɛŋ i puɔs","imɛŋ i putúk,oóli ú kátíɛ","makandikɛ","pilɔndɔ́","sd","md","mw","et","kl","fl","ss","sɔ́ndiɛ","móndie","muányáŋmóndie","metúkpíápɛ","kúpélimetúkpiapɛ","feléte","séselé","+J.C.","katikupíen Yésuse","ékélémkúnupíén n","kiɛmɛ́ɛm","kisɛ́ndɛ","Shɛ́rɛ́","Èrèlè","Ɛrɛ̀nà","Ìgbé","Ɛ̀bibi","Òkúdu","Agɛmɔ","Ògún","Owewe","Ɔ̀wàrà","Bélú","Ɔ̀pɛ̀","Oshù Shɛ́rɛ́","Oshù Èrèlè","Oshù Ɛrɛ̀nà","Oshù Ìgbé","Oshù Ɛ̀bibi","Oshù Òkúdu","Oshù Agɛmɔ","Oshù Ògún","Oshù Owewe","Oshù Ɔ̀wàrà","Oshù Bélú","Oshù Ɔ̀pɛ̀","Àìkú","Ajé","Ìsɛ́gun","Ɔjɔ́rú","Ɔjɔ́bɔ","Ɛtì","Àbámɛ́ta","Ɔjɔ́ Àìkú","Ɔjɔ́ Ajé","Ɔjɔ́ Ìsɛ́gun","Ɔjɔ́ Ɛtì","Ɔjɔ́ Àbámɛ́ta","Saju Kristi","Lehin Kristi","Àárɔ̀","Ɔ̀sán","Ṣẹ́rẹ́","Ẹrẹ̀nà","Ẹ̀bibi","Agẹmọ","Ọ̀wàrà","Ọ̀pẹ̀","Oṣù Ṣẹ́rẹ́","Oṣù Èrèlè","Oṣù Ẹrẹ̀nà","Oṣù Ìgbé","Oṣù Ẹ̀bibi","Oṣù Òkúdu","Oṣù Agẹmọ","Oṣù Ògún","Oṣù Owewe","Oṣù Ọ̀wàrà","Oṣù Bélú","Oṣù Ọ̀pẹ̀","Ìsẹ́gun","Ọjọ́rú","Ọjọ́bọ","Ẹtì","Àbámẹ́ta","Ọjọ́ Àìkú","Ọjọ́ Ajé","Ọjọ́ Ìsẹ́gun","Ọjọ́ Ẹtì","Ọjọ́ Àbámẹ́ta","Àárọ̀","Ọ̀sán","ⴰⵙⵉⵎⵡⴰⵙ","{year}年{month}月{day}日{weekday} {hour}:{minute}:{second}","{year}年{month}月{day}日{weekday} {ampm}{hour}:{minute}:{second}","{year}年{month}月{day}日{weekday}","佛历","13月","一月","十三月","周日","周一","周二","周三","周四","周五","周六","星期日","星期一","星期二","星期三","星期四","星期五","星期六","公元前","公元","上午","下午","闰7月","闰七月","希伯来历","印度历","回历","大化 (645–650)","白雉 (650–671)","白凤 (672–686)","朱鸟 (686–701)","大宝 (701–704)","庆云 (704–708)","和铜 (708–715)","灵龟 (715–717)","养老 (717–724)","神龟 (724–729)","天平 (729–749)","天平感宝 (749–749)","天平胜宝 (749–757)","天平宝字 (757–765)","天平神护 (765–767)","神护景云 (767–770)","宝龟 (770–780)","天应 (781–782)","延历 (782–806)","大同 (806–810)","弘仁 (810–824)","天长 (824–834)","承和 (834–848)","嘉祥 (848–851)","仁寿 (851–854)","齐衡 (854–857)","天安 (857–859)","贞观 (859–877)","元庆 (877–885)","仁和 (885–889)","宽平 (889–898)","昌泰 (898–901)","延喜 (901–923)","延长 (923–931)","承平 (931–938)","天庆 (938–947)","天历 (947–957)","天德 (957–961)","应和 (961–964)","康保 (964–968)","安和 (968–970)","天禄 (970–973)","天延 (973–976)","贞元 (976–978)","天元 (978–983)","永观 (983–985)","宽和 (985–987)","永延 (987–989)","永祚 (989–990)","正历 (990–995)","长德 (995–999)","长保 (999–1004)","宽弘 (1004–1012)","长和 (1012–1017)","宽仁 (1017–1021)","治安 (1021–1024)","万寿 (1024–1028)","长元 (1028–1037)","长历 (1037–1040)","长久 (1040–1044)","宽德 (1044–1046)","永承 (1046–1053)","天喜 (1053–1058)","康平 (1058–1065)","治历 (1065–1069)","延久 (1069–1074)","承保 (1074–1077)","正历 (1077–1081)","永保 (1081–1084)","应德 (1084–1087)","宽治 (1087–1094)","嘉保 (1094–1096)","永长 (1096–1097)","承德 (1097–1099)","康和 (1099–1104)","长治 (1104–1106)","嘉承 (1106–1108)","天仁 (1108–1110)","天永 (1110–1113)","永久 (1113–1118)","元永 (1118–1120)","保安 (1120–1124)","天治 (1124–1126)","大治 (1126–1131)","天承 (1131–1132)","长承 (1132–1135)","保延 (1135–1141)","永治 (1141–1142)","康治 (1142–1144)","天养 (1144–1145)","久安 (1145–1151)","仁平 (1151–1154)","久寿 (1154–1156)","保元 (1156–1159)","平治 (1159–1160)","永历 (1160–1161)","应保 (1161–1163)","长宽 (1163–1165)","永万 (1165–1166)","仁安 (1166–1169)","嘉应 (1169–1171)","承安 (1171–1175)","安元 (1175–1177)","治承 (1177–1181)","养和 (1181–1182)","寿永 (1182–1184)","元历 (1184–1185)","文治 (1185–1190)","建久 (1190–1199)","正治 (1199–1201)","建仁 (1201–1204)","元久 (1204–1206)","建永 (1206–1207)","承元 (1207–1211)","建历 (1211–1213)","建保 (1213–1219)","承久 (1219–1222)","贞应 (1222–1224)","元仁 (1224–1225)","嘉禄 (1225–1227)","安贞 (1227–1229)","宽喜 (1229–1232)","贞永 (1232–1233)","天福 (1233–1234)","文历 (1234–1235)","嘉祯 (1235–1238)","历仁 (1238–1239)","延应 (1239–1240)","仁治 (1240–1243)","宽元 (1243–1247)","宝治 (1247–1249)","建长 (1249–1256)","康元 (1256–1257)","正嘉 (1257–1259)","正元 (1259–1260)","文应 (1260–1261)","弘长 (1261–1264)","文永 (1264–1275)","建治 (1275–1278)","弘安 (1278–1288)","正应 (1288–1293)","永仁 (1293–1299)","正安 (1299–1302)","干元 (1302–1303)","嘉元 (1303–1306)","德治 (1306–1308)","延庆 (1308–1311)","应长 (1311–1312)","正和 (1312–1317)","文保 (1317–1319)","元应 (1319–1321)","元亨 (1321–1324)","正中 (1324–1326)","嘉历 (1326–1329)","元德 (1329–1331)","元弘 (1331–1334)","建武 (1334–1336)","延元 (1336–1340)","兴国 (1340–1346)","正平 (1346–1370)","建德 (1370–1372)","文中 (1372–1375)","天授 (1375–1379)","康历 (1379–1381)","弘和 (1381–1384)","元中 (1384–1392)","至德 (1384–1387)","嘉庆 (1387–1389)","康应 (1389–1390)","明德 (1390–1394)","应永 (1394–1428)","正长 (1428–1429)","永享 (1429–1441)","嘉吉 (1441–1444)","文安 (1444–1449)","宝德 (1449–1452)","享德 (1452–1455)","康正 (1455–1457)","长禄 (1457–1460)","宽正 (1460–1466)","文正 (1466–1467)","应仁 (1467–1469)","文明 (1469–1487)","长享 (1487–1489)","延德 (1489–1492)","明应 (1492–1501)","文龟 (1501–1504)","永正 (1504–1521)","大永 (1521–1528)","享禄 (1528–1532)","天文 (1532–1555)","弘治 (1555–1558)","永禄 (1558–1570)","元龟 (1570–1573)","天正 (1573–1592)","文禄 (1592–1596)","庆长 (1596–1615)","元和 (1615–1624)","宽永 (1624–1644)","正保 (1644–1648)","庆安 (1648–1652)","承应 (1652–1655)","明历 (1655–1658)","万治 (1658–1661)","宽文 (1661–1673)","延宝 (1673–1681)","天和 (1681–1684)","贞享 (1684–1688)","元禄 (1688–1704)","宝永 (1704–1711)","正德 (1711–1716)","享保 (1716–1736)","元文 (1736–1741)","宽保 (1741–1744)","延享 (1744–1748)","宽延 (1748–1751)","宝历 (1751–1764)","明和 (1764–1772)","安永 (1772–1781)","天明 (1781–1789)","宽政 (1789–1801)","享和 (1801–1804)","文化 (1804–1818)","文政 (1818–1830)","天保 (1830–1844)","弘化 (1844–1848)","嘉永 (1848–1854)","安政 (1854–1860)","万延 (1860–1861)","文久 (1861–1864)","元治 (1864–1865)","庆应 (1865–1868)","波斯历","{year} 年 {month} 月 {day} 日 ({weekday}){hour}:{minute}:{second}","{year} 年 {month} 月 {day} 日 ({weekday}){ampm}{hour}:{minute}:{second}","{year} 年 {month} 月 {day} 日 ({weekday}","{year} 年 {month} 月 {day}","{year} 年 {month}","一","佛曆","週日","週一","週二","週三","週四","週五","週六","西元前","西元","提斯利月","瑪西班月","基斯流月","提別月","細罷特月","亞達月 I","亞達月","尼散月","以珥月","西彎月","搭模斯月","埃波月","以祿月","亞達月 II","創世紀元","制檀邏月","吠舍佉月","逝瑟吒月","頞沙荼月","室羅伐拏月","婆羅鉢陀月","頞涇縛庚闍月","迦剌底迦月","末伽始羅月","報沙月","磨祛月","頗勒窶拏月","印度曆","穆哈蘭姆月","色法爾月","賴比月 I","賴比月 II","主馬達月 I","主馬達月 II","賴哲卜月","舍爾邦月","賴買丹月","閃瓦魯月","都爾喀爾德月","都爾黑哲月","伊斯蘭曆","白鳳","大寶","靈龜","神龜","天平感寶","天平勝寶","天平寶字","寶龜","天應","延曆","仁壽","齊衡","貞觀","寬平","天曆","天德","應和","天祿","永觀","寬和","正曆","長德","寬弘","寬仁","萬壽","長曆","寬德","治曆","承曆","應德","寬治","承德","久壽","永曆","應保","長寬","永萬","嘉應","壽永","元曆","建曆","貞應","嘉祿","寬喜","文曆","曆仁","延應","寬元","寶治","文應","正應","德治","應長","元應","嘉曆","元德","興國","建德","康曆","至德","康應","明德","應永","寶德","享德","長祿","寬正","應仁","延德","明應","文龜","享祿","永祿","元龜","文祿","寬永","承應","明曆","萬治","寬文","延寶","元祿","寶永","正德","寬保","寬延","寶曆","寬政","萬延","慶應","伊朗曆","民國前","民國","非數值","Mas","Januwari","Mashi","Apreli","Septhemba","Isonto","Umsombuluko","Ulwesibili","Ulwesithathu","Ulwesine","Ulwesihlanu","Umgqibelo","Msombuluko","Lwesibili","Ekuseni","Ntambama","I-NaN"],b=[];b[0]=[[a[0],a[1],a[2],a[3],a[4],a[5],a[6],a[7],a[8],a[9],a[10],a[11],a[12]],{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[16],"pattern12":a[17]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[18]},{"year":a[13],"month":a[14],"day":a[13],"pattern":a[19]},{"year":a[13],"month":a[15],"day":a[15],"pattern":a[20]},{"year":a[13],"month":a[15],"pattern":a[21]},{"year":a[13],"month":a[14],"pattern":a[22]},{"month":a[14],"day":a[13],"pattern":a[23]},{"month":a[15],"day":a[15],"pattern":a[24]},{"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[25],"pattern12":a[26]},{"hour":a[13],"minute":a[15],"pattern":a[27],"pattern12":a[28]},[a[29]],[a[30],a[31],a[32],a[33],a[34],a[35],a[36],a[37],a[38],a[39],a[40],a[41]],[a[42],a[43],a[44],a[45],a[46],a[47],a[48],a[49],a[50],a[51],a[52],a[53],a[54]],[a[55],a[56]],[a[57],a[58],a[59],a[60],a[61],a[62],a[63],a[64],a[65],a[66],a[67],a[68],a[69]],[a[55]],[a[70],a[71],a[72],a[73],a[74],a[75],a[76],a[77],a[78],a[79],a[80],a[81]],[a[82],a[83],a[84],a[85],a[86],a[87],a[88],a[89],a[90],a[91],a[92],a[93]],[a[94],a[95],a[96],a[97],a[98],a[99],a[100]],[a[101],a[102],a[103],a[104],a[105],a[106],a[107]],[a[108],a[109]],{"am":a[110],"pm":a[111]},[a[112],a[113],a[114],a[115],a[116],a[117],a[118],a[119],a[120],a[121],a[122],a[123],a[124],a[125]],[a[126]],[a[127],a[128],a[129],a[130],a[131],a[132],a[133],a[134],a[135],a[136],a[137],a[138]],[a[139]],[a[140],a[141],a[142],a[143],a[144],a[145],a[146],a[147],a[148],a[149],a[150],a[151]],[a[152],a[153],a[154],a[155],a[156],a[157],a[158],a[159],a[160],a[161],a[162],a[163]],[a[164]],[a[165],a[166],a[167],a[168],a[169],a[170],a[171],a[172],a[173],a[174],a[175],a[176],a[177],a[178],a[179],a[180],a[181],a[182],a[183],a[184],a[185],a[186],a[187],a[188],a[189],a[190],a[191],a[192],a[193],a[194],a[195],a[196],a[197],a[198],a[199],a[200],a[201],a[202],a[203],a[204],a[205],a[206],a[207],a[208],a[209],a[210],a[211],a[212],a[213],a[214],a[215],a[216],a[217],a[218],a[219],a[220],a[221],a[222],a[223],a[224],a[225],a[226],a[227],a[228],a[229],a[230],a[231],a[232],a[233],a[234],a[235],a[236],a[237],a[238],a[239],a[240],a[241],a[242],a[243],a[244],a[245],a[246],a[247],a[248],a[249],a[250],a[251],a[252],a[253],a[254],a[255],a[256],a[257],a[258],a[259],a[260],a[261],a[262],a[263],a[264],a[265],a[266],a[267],a[268],a[269],a[270],a[271],a[272],a[273],a[274],a[275],a[276],a[277],a[278],a[279],a[280],a[281],a[282],a[283],a[284],a[285],a[286],a[287],a[288],a[289],a[290],a[291],a[292],a[293],a[294],a[295],a[296],a[297],a[298],a[299],a[300],a[301],a[302],a[303],a[304],a[305],a[306],a[307],a[308],a[309],a[310],a[311],a[312],a[313],a[314],a[315],a[316],a[317],a[318],a[319],a[320],a[321],a[322],a[323],a[324],a[325],a[326],a[327],a[328],a[329],a[330],a[331],a[332],a[333],a[334],a[335],a[336],a[337],a[338],a[339],a[340],a[341],a[342],a[343],a[344],a[345],a[346],a[347],a[348],a[349],a[350],a[351],a[352],a[353],a[354],a[355],a[356],a[357],a[358],a[359],a[360],a[361],a[362],a[363],a[364],a[365],a[366],a[367],a[368],a[369],a[370],a[371],a[372],a[373],a[374],a[375],a[376],a[377],a[378],a[379],a[380],a[381],a[382],a[383],a[384],a[385],a[386],a[387],a[388],a[389],a[390],a[391],a[392],a[393],a[394],a[395],a[396],a[397],a[398],a[399],a[400]],[a[165],a[166],a[167],a[168],a[169],a[170],a[171],a[172],a[173],a[174],a[175],a[176],a[177],a[178],a[179],a[180],a[181],a[182],a[183],a[184],a[185],a[186],a[187],a[188],a[189],a[190],a[191],a[192],a[193],a[194],a[195],a[196],a[197],a[198],a[199],a[200],a[201],a[202],a[203],a[204],a[205],a[206],a[207],a[208],a[209],a[210],a[211],a[212],a[213],a[214],a[215],a[216],a[217],a[218],a[219],a[220],a[221],a[222],a[223],a[224],a[225],a[226],a[227],a[228],a[229],a[230],a[231],a[232],a[233],a[234],a[235],a[236],a[237],a[238],a[239],a[240],a[241],a[242],a[243],a[244],a[245],a[246],a[247],a[248],a[249],a[250],a[251],a[252],a[253],a[254],a[255],a[256],a[257],a[258],a[259],a[260],a[261],a[262],a[263],a[264],a[265],a[266],a[267],a[268],a[269],a[270],a[271],a[272],a[273],a[274],a[275],a[276],a[277],a[278],a[279],a[280],a[281],a[282],a[283],a[284],a[285],a[286],a[287],a[288],a[289],a[290],a[291],a[292],a[293],a[294],a[295],a[296],a[297],a[298],a[299],a[300],a[301],a[302],a[303],a[304],a[305],a[306],a[307],a[308],a[309],a[310],a[311],a[312],a[313],a[314],a[315],a[316],a[317],a[318],a[319],a[320],a[321],a[322],a[323],a[324],a[325],a[401],a[327],a[328],a[329],a[402],a[331],a[332],a[333],a[334],a[335],a[336],a[337],a[338],a[339],a[340],a[341],a[342],a[343],a[344],a[345],a[346],a[347],a[348],a[349],a[350],a[351],a[352],a[353],a[354],a[355],a[356],a[357],a[358],a[359],a[360],a[361],a[362],a[363],a[364],a[365],a[366],a[367],a[368],a[369],a[370],a[371],a[372],a[373],a[374],a[375],a[376],a[377],a[378],a[379],a[380],a[381],a[382],a[383],a[384],a[385],a[386],a[387],a[388],a[389],a[390],a[391],a[392],a[393],a[394],a[395],a[396],a[403],a[404],a[405],a[406]],[a[407],a[408],a[409],a[410],a[411],a[412],a[413],a[414],a[415],a[416],a[417],a[418]],[a[419]],[a[420],a[421]],[a[422]],{"positivePattern":a[423],"negativePattern":a[424]},{"positivePattern":a[425],"negativePattern":a[426]},{"positivePattern":a[427],"negativePattern":a[428]},{"decimal":a[429],"group":a[430],"nan":a[431],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DJF":a[438],"ETB":a[439],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[82],a[83],a[84],a[85],a[86],a[87],a[88],a[457],a[90],a[91],a[92],a[93]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ERN":a[458],"ETB":a[439],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ETB":a[439],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[459],"pattern12":a[460]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[461]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[462]},{"month":a[13],"day":a[13],"year":a[13],"pattern":a[463]},{"month":a[13],"year":a[13],"pattern":a[464]},{"month":a[14],"year":a[13],"pattern":a[465]},{"month":a[13],"day":a[13],"pattern":a[466]},[a[467],a[468],a[397],a[469],a[397],a[467],a[467],a[469],a[399],a[470],a[471],a[472]],[a[473],a[474],a[475],a[476],a[477],a[478],a[479],a[480],a[481],a[482],a[483],a[484]],[a[485],a[486],a[487],a[488],a[477],a[489],a[490],a[491],a[492],a[493],a[494],a[495]],[a[496],a[497],a[498],a[499],a[500],a[501],a[502]],[a[503],a[504],a[505],a[506],a[507],a[508],a[509]],[a[510],a[511],a[512],a[513],a[514],a[515],a[516]],[a[517],a[518],a[519],a[520]],[a[517],a[518],a[521],a[522]],[a[523],a[524]],{"am":a[525],"pm":a[526]},{"positivePattern":a[527],"negativePattern":a[528]},{"decimal":a[430],"group":a[529],"nan":a[431],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NAD":a[530],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZAR":a[531]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZAR":a[531]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[532],"pattern12":a[533]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[534]},{"day":a[13],"month":a[13],"year":a[13],"pattern":a[535]},{"day":a[13],"month":a[14],"pattern":a[536]},{"day":a[13],"month":a[13],"pattern":a[537]},[a[538],a[539],a[540],a[541],a[542],a[543],a[544],a[545],a[546],a[547],a[548],a[549]],[a[550],a[551],a[552],a[553],a[554],a[555],a[556],a[557],a[558],a[559],a[560],a[561]],[a[562],a[563],a[564],a[565],a[566],a[567],a[568]],[a[569],a[570],a[571],a[572],a[573],a[574],a[575]],[a[576],a[577]],[a[578],a[579]],{"am":a[580],"pm":a[581]},{"positivePattern":a[582],"negativePattern":a[583]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"month":a[14],"day":a[13],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[584],"pattern12":a[585]},{"weekday":a[14],"month":a[14],"day":a[13],"year":a[13],"pattern":a[586]},{"year":a[13],"month":a[13],"day":a[13],"pattern":a[587]},[a[588],a[589],a[590],a[591],a[592],a[593],a[594],a[595],a[596],a[597],a[598],a[599]],[a[600],a[601],a[602],a[603],a[604],a[605],a[606],a[607],a[608],a[609],a[610],a[611]],[a[612],a[613],a[614],a[615],a[616],a[617],a[618]],[a[619],a[620],a[621],a[622],a[623],a[624],a[625]],[a[626],a[627]],[a[628],a[629]],{"am":a[630],"pm":a[631]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GHS":a[632],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"month":a[14],"day":a[13],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[633],"pattern12":a[634]},{"weekday":a[14],"month":a[14],"day":a[13],"year":a[13],"pattern":a[635]},[a[636],a[637]],[a[638],a[639],a[640],a[641],a[642],a[643],a[644],a[645],a[646],a[647],a[648],a[649],a[650]],[a[651],a[652],a[653],a[654],a[655],a[656],a[657],a[658],a[659],a[660],a[661],a[662],a[663]],[a[664],a[665],a[653],a[666],a[655],a[667],a[668],a[669],a[670],a[660],a[661],a[662],a[671]],[a[672],a[673],a[674],a[675],a[676],a[677],a[677],a[678],a[679],a[678],a[680],a[681]],[a[682],a[683],a[684],a[685],a[686],a[687],a[688],a[689],a[690],a[691],a[692],a[693]],[a[694],a[695],a[684],a[696],a[686],a[687],a[688],a[697],a[698],a[699],a[700],a[701]],[a[702],a[703],a[674],a[704],a[705],a[706],a[707]],[a[708],a[709],a[710],a[711],a[712],a[713],a[714]],[a[708],a[709],a[715],a[711],a[712],a[713],a[714]],[a[716],a[717]],{"am":a[718],"pm":a[719]},[a[720],a[721],a[722],a[723],a[724],a[725],a[726],a[727],a[728],a[729],a[730],a[731]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ETB":a[733],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[734],"pattern12":a[735]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[736]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[737]},{"day":a[13],"month":a[13],"year":a[13],"pattern":a[738]},{"month":a[13],"year":a[13],"pattern":a[739]},{"day":a[13],"month":a[13],"pattern":a[740]},[a[741]],[a[742],a[743],a[744],a[745],a[746],a[747],a[748],a[749],a[750],a[751],a[752],a[753],a[754]],[a[755],a[756],a[757],a[758],a[759],a[760],a[761],a[762],a[763],a[764],a[765],a[766],a[767]],[a[768],a[769],a[770],a[771],a[772],a[773],a[774],a[775],a[776],a[777],a[778],a[779]],[a[780],a[781],a[782],a[783],a[784],a[785],a[786],a[787],a[788],a[789],a[790],a[791]],[a[792],a[793],a[794],a[795],a[796],a[797],a[798]],[a[799],a[770]],[a[800],a[801]],{"am":a[802],"pm":a[770]},[a[803],a[804],a[805],a[806],a[807],a[808],a[809],a[810],a[811],a[812],a[813],a[814],a[815],a[816]],[a[802]],[a[817],a[818],a[819],a[820],a[821],a[822],a[823],a[824],a[825],a[826],a[827],a[828]],[a[829],a[830],a[831],a[832],a[833],a[834],a[835],a[836],a[837],a[838],a[839],a[840]],[a[841]],[a[842],a[843],a[844],a[845],a[846],a[847],a[848],a[849],a[850],a[851],a[852],a[853],a[854],a[855],a[856],a[857],a[858],a[859],a[860],a[861],a[862],a[863],a[864],a[865],a[866],a[867],a[868],a[869],a[870],a[871],a[872],a[873],a[874],a[875],a[876],a[877],a[878],a[879],a[880],a[881],a[882],a[883],a[884],a[885],a[886],a[887],a[888],a[889],a[890],a[891],a[892],a[893],a[894],a[895],a[896],a[897],a[898],a[899],a[900],a[901],a[902],a[903],a[904],a[905],a[906],a[907],a[908],a[909],a[910],a[911],a[912],a[913],a[914],a[915],a[916],a[917],a[918],a[919],a[920],a[921],a[922],a[923],a[924],a[925],a[926],a[927],a[928],a[929],a[930],a[931],a[932],a[933],a[934],a[935],a[936],a[937],a[938],a[939],a[940],a[941],a[942],a[943],a[944],a[945],a[946],a[947],a[948],a[949],a[950],a[951],a[952],a[953],a[954],a[955],a[956],a[957],a[958],a[959],a[960],a[961],a[962],a[963],a[964],a[965],a[966],a[967],a[968],a[969],a[970],a[971],a[972],a[973],a[974],a[975],a[976],a[977],a[978],a[979],a[980],a[981],a[982],a[983],a[984],a[985],a[986],a[987],a[988],a[989],a[990],a[991],a[992],a[993],a[994],a[995],a[996],a[997],a[998],a[999],a[1000],a[1001],a[1002],a[1003],a[1004],a[1005],a[1006],a[1007],a[1008],a[1009],a[1010],a[1011],a[1012],a[1013],a[1014],a[1015],a[1016],a[1017],a[1018],a[1019],a[1020],a[1021],a[1022],a[1023],a[1024],a[1025],a[1026],a[1027],a[1028],a[1029],a[1030],a[1031],a[1032],a[1033],a[1034],a[1035],a[1036],a[1037],a[1038],a[1039],a[1040],a[1041],a[1042],a[1043],a[1044],a[1045],a[1046],a[1047],a[1048],a[1049],a[1050],a[1051],a[1052],a[1053],a[1054],a[1055],a[1056],a[1057],a[1058],a[1059],a[1060],a[1061],a[1062],a[1063],a[1064],a[1065],a[1066],a[1067],a[1068],a[1069],a[1070],a[1071],a[1072],a[1073],a[1074],a[1075],a[1076],a[1077]],[a[1078],a[1079],a[1080],a[1081],a[1082],a[1083],a[1084],a[1085],a[1086],a[1087],a[1088],a[1089]],[a[1090]],[a[420],a[1091]],[a[1092]],{"decimal":a[1093],"group":a[1094],"nan":a[1095],"percent":a[1096],"infinity":a[433]},{"AED":a[1097],"AUD":a[732],"BHD":a[1098],"BRL":a[435],"CAD":a[436],"CNY":a[1099],"DZD":a[1100],"EGP":a[1101],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[1102],"IQD":a[1103],"JOD":a[1104],"JPY":a[445],"KMF":a[1105],"KRW":a[446],"KWD":a[1106],"LBP":a[1107],"LYD":a[1108],"MAD":a[1109],"MRO":a[1110],"MXN":a[447],"NZD":a[448],"OMR":a[1111],"QAR":a[1112],"SAR":a[1113],"SDD":a[1114],"SDG":a[1115],"SSP":a[1116],"SYP":a[1117],"THB":a[449],"TND":a[1118],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[1119],"YER":a[1120]},{"AED":a[1097],"AUD":a[732],"BHD":a[1098],"BRL":a[435],"CAD":a[436],"CNY":a[1099],"DJF":a[438],"DZD":a[1100],"EGP":a[1101],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[1102],"IQD":a[1103],"JOD":a[1104],"JPY":a[445],"KMF":a[1105],"KRW":a[446],"KWD":a[1106],"LBP":a[1107],"LYD":a[1108],"MAD":a[1109],"MRO":a[1110],"MXN":a[447],"NZD":a[448],"OMR":a[1111],"QAR":a[1112],"SAR":a[1113],"SDD":a[1114],"SDG":a[1115],"SSP":a[1116],"SYP":a[1117],"THB":a[449],"TND":a[1118],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[1119],"YER":a[1120]},{"year":a[13],"month":a[13],"pattern":a[1121]},[a[1122],a[769],a[770],a[771],a[770],a[1122],a[1122],a[771],a[776],a[771],a[773],a[779]],[a[1123],a[1124],a[782],a[1125],a[1126],a[1127],a[1128],a[1129],a[788],a[789],a[790],a[791]],{"decimal":a[430],"group":a[429],"nan":a[431],"percent":a[432],"infinity":a[433]},{"AED":a[1097],"AUD":a[732],"BHD":a[1098],"BRL":a[435],"CAD":a[436],"CNY":a[1099],"DZD":a[1100],"EGP":a[1101],"ERN":a[458],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[1102],"IQD":a[1103],"JOD":a[1104],"JPY":a[445],"KMF":a[1105],"KRW":a[446],"KWD":a[1106],"LBP":a[1107],"LYD":a[1108],"MAD":a[1109],"MRO":a[1110],"MXN":a[447],"NZD":a[448],"OMR":a[1111],"QAR":a[1112],"SAR":a[1113],"SDD":a[1114],"SDG":a[1115],"SSP":a[1116],"SYP":a[1117],"THB":a[449],"TND":a[1118],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[1119],"YER":a[1120]},[a[777],a[1130],a[1131],a[773],a[771],a[1132],a[1133],a[1131],a[771],a[1133],a[1133],a[777]],[a[1134],a[807],a[809],a[810],a[811],a[1135],a[813],a[814],a[815],a[1136],a[1137],a[1138]],[a[1134],a[807],a[809],a[810],a[811],a[1135],a[813],a[814],a[815],a[1139],a[1137],a[1138]],{"AED":a[1097],"AUD":a[732],"BHD":a[1098],"BRL":a[435],"CAD":a[436],"CNY":a[1099],"DZD":a[1100],"EGP":a[1101],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[1102],"IQD":a[1103],"JOD":a[1104],"JPY":a[445],"KMF":a[1105],"KRW":a[446],"KWD":a[1106],"LBP":a[1107],"LYD":a[1108],"MAD":a[1109],"MRO":a[1110],"MXN":a[447],"NZD":a[448],"OMR":a[1111],"QAR":a[1112],"SAR":a[1113],"SDD":a[1114],"SSP":a[1116],"SYP":a[1117],"THB":a[449],"TND":a[1118],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[1119],"YER":a[1120]},[a[768],a[769],a[770],a[771],a[770],a[773],a[774],a[775],a[1130],a[777],a[778],a[779]],[a[780],a[781],a[782],a[783],a[1126],a[785],a[1140],a[1141],a[1142],a[789],a[1143],a[1144]],[a[768],a[769],a[770],a[1145],a[772],a[773],a[774],a[775],a[1130],a[777],a[778],a[779]],[a[780],a[781],a[782],a[1146],a[784],a[785],a[786],a[1147],a[1148],a[789],a[790],a[1149]],{"AED":a[1097],"AUD":a[732],"BHD":a[1098],"BRL":a[435],"CAD":a[436],"CNY":a[1099],"DZD":a[1100],"EGP":a[1101],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[1102],"IQD":a[1103],"JOD":a[1104],"JPY":a[445],"KMF":a[1105],"KRW":a[446],"KWD":a[1106],"LBP":a[1107],"LYD":a[1108],"MAD":a[1109],"MRO":a[1110],"MXN":a[447],"NZD":a[448],"OMR":a[1111],"QAR":a[1112],"SAR":a[1113],"SDD":a[1114],"SDG":a[1115],"SOS":a[399],"SSP":a[1116],"SYP":a[1117],"THB":a[449],"TND":a[1118],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[1119],"YER":a[1120]},{"AED":a[1097],"AUD":a[732],"BHD":a[1098],"BRL":a[435],"CAD":a[436],"CNY":a[1099],"DZD":a[1100],"EGP":a[1101],"EUR":a[440],"GBP":a[1150],"HKD":a[442],"ILS":a[443],"INR":a[1102],"IQD":a[1103],"JOD":a[1104],"JPY":a[445],"KMF":a[1105],"KRW":a[446],"KWD":a[1106],"LBP":a[1107],"LYD":a[1108],"MAD":a[1109],"MRO":a[1110],"MXN":a[447],"NZD":a[448],"OMR":a[1111],"QAR":a[1112],"SAR":a[1113],"SDD":a[1114],"SDG":a[1115],"SSP":a[441],"SYP":a[1117],"THB":a[449],"TND":a[1118],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[1119],"YER":a[1120]},[a[1151],a[1152],a[1153],a[1154],a[1155],a[1156],a[1157],a[1158],a[1159],a[1160],a[1161],a[1162]],[a[1163],a[1164],a[1153],a[1154],a[1155],a[1156],a[1157],a[1165],a[1166],a[1167],a[1168],a[1169]],[a[1170],a[1171],a[1172],a[1173],a[1174],a[1175],a[1176]],[a[1177],a[1178],a[1179],a[1180],a[1181],a[1182],a[1183]],[a[1184],a[1185]],{"am":a[1186],"pm":a[1187]},[a[1188]],[a[473],a[474],a[1189],a[476],a[477],a[478],a[479],a[1190],a[481],a[482],a[483],a[1191]],[a[1192],a[1193],a[1194],a[1195],a[477],a[1196],a[1197],a[1198],a[1199],a[1200],a[1201],a[1202]],[a[1203],a[1204],a[1205],a[1206],a[1207],a[1208],a[1209]],[a[1210],a[1211],a[1212],a[1213],a[1214],a[1215],a[1216]],[a[1217],a[1218]],[a[1219],a[1220]],{"am":a[1221],"pm":a[1222]},{"positivePattern":a[1223],"negativePattern":a[1224]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"TZS":a[1225],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1226],a[468],a[397],a[469],a[397],a[1226],a[1226],a[469],a[399],a[470],a[1227],a[469]],[a[1228],a[1229],a[1230],a[1231],a[1232],a[1233],a[1234],a[1235],a[1236],a[1237],a[1238],a[1239]],[a[1240],a[1241],a[1242],a[1243],a[1244],a[1245],a[1246],a[1247],a[1248],a[1249],a[1250],a[1251]],[a[1252],a[1253],a[1254],a[1255],a[1256],a[1257],a[1258]],[a[1259],a[1260],a[1230],a[1261],a[1262],a[1263],a[1264]],[a[1265],a[1266],a[1267],a[1268],a[1269],a[1270],a[1271]],[a[1272],a[1273]],{"am":a[126],"pm":a[1274]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[13],"month":a[14],"year":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1276],"pattern12":a[1277]},{"day":a[13],"month":a[14],"year":a[13],"weekday":a[14],"pattern":a[1278]},{"day":a[15],"month":a[15],"year":a[13],"pattern":a[1279]},{"month":a[15],"year":a[13],"pattern":a[1280]},{"day":a[15],"month":a[15],"pattern":a[1281]},[a[638],a[639],a[640],a[641],a[642],a[643],a[644],a[645],a[646],a[647],a[648],a[649]],[a[1282],a[1283],a[1230],a[1284],a[1232],a[1285],a[1286],a[1287],a[1288],a[1289],a[1290],a[1291]],[a[1292],a[1293],a[1294],a[1295],a[1232],a[1296],a[1297],a[1298],a[1299],a[1300],a[1301],a[1302]],[a[1303],a[1304],a[1305],a[1306],a[1307],a[1308],a[1309]],[a[1310],a[1311],a[1312],a[1313],a[1314],a[1315],a[1316]],[a[1317],a[1318]],[a[1319],a[1320]],{"AUD":a[434],"AZN":a[1321],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1322],"pattern12":a[1323]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[1324]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[1325]},{"month":a[14],"year":a[13],"pattern":a[1326]},[a[1327],a[1328],a[1329],a[1330],a[1331],a[1332],a[1333],a[1334],a[1335],a[1336],a[1337],a[1338]],[a[1339],a[1340],a[1341],a[1342],a[1343],a[1344],a[1345]],{"AUD":a[434],"AZN":a[1346],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1347],a[1348],a[1349],a[1350],a[1351],a[1352],a[1353],a[1354],a[1355],a[1356],a[1232],a[1357]],[a[1358],a[1359],a[1360],a[1361],a[1362],a[1363],a[1364],a[1365],a[1366],a[1367],a[1368],a[1369]],[a[1370],a[1371],a[1372],a[1373],a[1374],a[1375],a[1376]],[a[1377],a[1378],a[1379],a[1380],a[1381],a[1382],a[1383]],[a[1384],a[1385]],[a[1386],a[1387]],{"am":a[1388],"pm":a[1389]},{"positivePattern":a[1390],"negativePattern":a[1391]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1392],"pattern12":a[1393]},{"day":a[13],"month":a[13],"year":a[13],"pattern":a[1279]},{"month":a[13],"year":a[13],"pattern":a[1280]},{"day":a[13],"month":a[13],"pattern":a[1281]},{"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1394],"pattern12":a[1395]},{"hour":a[13],"minute":a[15],"pattern":a[1396],"pattern12":a[1397]},[a[1398],a[1399],a[1398],a[1400],a[1401],a[1402],a[1399],a[1403],a[1404],a[1400],a[1399],a[1398]],[a[1405],a[1406],a[1407],a[1408],a[1409],a[1410],a[1411],a[1412],a[1413],a[1414],a[1415],a[1416]],[a[1417],a[1418],a[1419],a[1420],a[1409],a[1421],a[1422],a[1423],a[1424],a[1425],a[1426],a[1427]],[a[1428],a[1429],a[1430],a[1431],a[1432],a[1433],a[1434]],[a[1435],a[1436],a[1437],a[1438],a[1439],a[1440],a[1441]],[a[1442],a[1443]],{"am":a[1444],"pm":a[1445]},{"AUD":a[434],"BRL":a[435],"BYR":a[1446],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[1447],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[473],a[474],a[1189],a[1448],a[477],a[478],a[479],a[1449],a[481],a[482],a[483],a[1450]],[a[1192],a[1193],a[1194],a[1451],a[477],a[1196],a[1197],a[1452],a[1199],a[1200],a[1201],a[1453]],[a[1454],a[1455],a[1456],a[1457],a[1458],a[1459],a[1460]],[a[1461],a[1462]],[a[1463],a[1464]],{"am":a[1465],"pm":a[1466]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZMW":a[1467]},[a[1468],a[1469],a[1470],a[1471],a[1472],a[1473],a[100],a[1474],a[1475],a[1476],a[1477],a[1478]],[a[1479],a[1480],a[1481],a[1482],a[1483],a[1484],a[1485],a[1486],a[1487],a[1488],a[1489],a[1490]],[a[1491],a[1469],a[1492],a[1493],a[1494],a[1495],a[1496]],[a[1497],a[1498],a[1499],a[1500],a[1501],a[1502],a[1503]],[a[1504],a[1505]],{"am":a[1506],"pm":a[1507]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1508],"pattern12":a[1509]},{"day":a[13],"month":a[15],"year":a[13],"pattern":a[1279]},{"day":a[13],"month":a[15],"pattern":a[1281]},[a[1510],a[1511],a[1401],a[1512],a[1401],a[1513],a[1513],a[1512],a[1398],a[1514],a[1515],a[1516]],[a[1517],a[1518],a[1329],a[1519],a[1331],a[1520],a[1521],a[1522],a[1523],a[1524],a[1525],a[1526]],[a[1527],a[1528],a[1329],a[1529],a[1331],a[1520],a[1521],a[1334],a[1530],a[1531],a[1532],a[1533]],[a[1428],a[1429],a[1534],a[1431],a[1535],a[1433],a[1434]],[a[1536],a[1537],a[1538],a[1539],a[1540],a[1541],a[1542]],[a[1543]],[a[1544],a[1545],a[1546],a[1547]],[a[1544],a[1545]],{"am":a[1548],"pm":a[1549]},[a[1550],a[1551],a[1552],a[1553],a[1554],a[1555],a[1556],a[1557],a[1558],a[1559],a[1560],a[1561],a[1562],a[1563]],[a[1564],a[1565],a[1566],a[1567],a[1568],a[1569],a[1570],a[1571],a[1572],a[1573],a[1574],a[1575]],[a[1576],a[1577],a[1578],a[1579],a[1580],a[1581],a[1582],a[1583],a[1584],a[1585],a[1586],a[1587]],{"BGN":a[1588],"EUR":a[440],"RUB":a[1589],"USD":a[1590],"XAF":a[453],"XOF":a[455],"XPF":a[456]},[a[1591],a[1229],a[1230],a[1592],a[1593],a[1594],a[1595],a[1596],a[1597],a[1598],a[1599],a[1600]],[a[1601],a[1602],a[1603],a[1604],a[1593],a[1605],a[1606],a[1596],a[1607],a[1608],a[1609],a[1610]],[a[1611],a[1612],a[1613],a[1614],a[1615],a[1616],a[1617]],[a[1618],a[1619],a[1620],a[1621],a[1622],a[1623],a[1624]],[a[1625],a[1626]],[a[1627],a[1628]],{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1629],"pattern12":a[1630]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[1631]},[a[1632],a[1633],a[1634],a[1635],a[1155],a[1156],a[1636],a[1637],a[1638],a[1639],a[1640],a[1641]],[a[1642],a[1643],a[1644],a[1645],a[1155],a[1156],a[1157],a[1646],a[1647],a[1648],a[1649],a[1650]],[a[1651],a[1652],a[1653],a[1654],a[1655],a[1656],a[1657]],[a[1658],a[1171],a[1172],a[1173],a[1659],a[1660],a[1176]],[a[1661],a[1662],a[1663],a[1664],a[1665],a[1666],a[1667]],[a[1668],a[1669]],{"am":a[1670],"pm":a[1671]},[a[1672],a[1673],a[1674],a[1675],a[1676],a[1677],a[1678],a[1679],a[1680],a[1681],a[1682],a[1683]],[a[1684]],[a[1685],a[1686],a[1687],a[1688],a[1689],a[1690],a[1691],a[1692],a[1693],a[1694],a[1695],a[1696]],[a[1697]],{"positivePattern":a[582],"negativePattern":a[1698]},{"nan":a[1699]},{"AUD":a[434],"BDT":a[1700],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"year":a[13],"month":a[14],"day":a[13],"pattern":a[1701]},[a[1702],a[1703],a[1704],a[1705],a[1706],a[1707],a[1708],a[1709],a[1710],a[1711],a[1712],a[1713]],[a[1714],a[1715],a[1716],a[1717],a[1718],a[1719],a[1720],a[1721],a[1722],a[1723],a[1724],a[1725]],[a[1726],a[1727],a[1728],a[1729],a[1730],a[1731],a[1732]],[a[1733],a[1734],a[1735],a[1736],a[1737],a[1738],a[1739]],[a[1740],a[1741]],{"am":a[1742],"pm":a[1743]},{"nan":a[1744]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[1275],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1745],a[1746],a[1747],a[1748],a[1749],a[1750],a[1751],a[1752],a[1753],a[1754],a[1755],a[1756]],[a[1757],a[1758],a[1759],a[1760],a[1749],a[1761],a[1762],a[1752],a[1763],a[1754],a[1755],a[1764]],[a[1765],a[1766],a[1767],a[1768],a[1769],a[1770],a[1771]],[a[1772],a[1773],a[1759],a[1774],a[1775],a[1776],a[1777]],{"day":a[13],"month":a[14],"pattern":a[1778]},[a[1779],a[1780],a[1781],a[1782],a[1783],a[1784],a[1785],a[1786],a[1787],a[1788],a[1789],a[1790]],[a[1791],a[1792],a[1793],a[1794],a[1795],a[1796],a[1797]],[a[1798],a[1799],a[1800],a[1801],a[1802],a[1803],a[1804]],[a[1805],a[1806]],{"am":a[1807],"pm":a[1808]},[a[1809],a[1810],a[1811],a[1812],a[1813],a[1814],a[1815],a[1816],a[1817],a[1818],a[1819],a[1820]],{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[1821],"pattern12":a[1822]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"pattern":a[1823]},{"day":a[15],"month":a[14],"year":a[13],"pattern":a[1824]},{"day":a[15],"month":a[14],"pattern":a[1825]},{"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[25],"pattern12":a[26]},{"hour":a[15],"minute":a[15],"pattern":a[27],"pattern12":a[28]},[a[1826],a[1229],a[1230],a[1284],a[1827],a[1828],a[1829],a[1830],a[1831],a[1289],a[1832],a[1833]],[a[1834],a[1835],a[1294],a[1836],a[1827],a[1837],a[1838],a[1839],a[1840],a[1841],a[1842],a[1843]],[a[1844],a[1845],a[1846],a[1847],a[1848],a[1849],a[1850]],[a[1851],a[1852],a[1853],a[1854],a[1855],a[1856],a[1857]],[a[1858],a[1859]],[a[1860],a[1861]],{"am":a[1862],"pm":a[1863]},[a[1864],a[1865],a[1866],a[1867],a[1868],a[1869],a[1870],a[1871],a[1872],a[1873],a[1874],a[1875]],[a[1876],a[1877],a[1878],a[1879],a[1880],a[1881],a[1882],a[1883],a[1884],a[1885],a[1886],a[1887]],{"AUD":a[434],"BAM":a[1217],"BRL":a[435],"CAD":a[436],"CNY":a[437],"CZK":a[1888],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PLN":a[1889],"RSD":a[1890],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1891]],[a[1892],a[1893],a[1894],a[1895],a[1896],a[1897],a[1898],a[1899],a[1900],a[1901],a[1902],a[1903],a[1904]],[a[1905],a[1906],a[1907],a[1908],a[1909],a[1910],a[1911],a[1912],a[1913],a[1914],a[1915],a[1916],a[1917]],[a[1918],a[1919],a[1920],a[1921],a[1922],a[1923],a[1924],a[1925],a[1926],a[1927],a[1928],a[1929]],[a[1930],a[1931],a[1329],a[1529],a[1922],a[1932],a[1933],a[1334],a[1934],a[1935],a[1936],a[1937]],[a[1938],a[1939],a[1940],a[1941],a[1942],a[1943],a[1944]],[a[1945],a[1946],a[1947],a[1948],a[1949],a[1950],a[1441]],[a[1951],a[1952]],[a[1953],a[1954]],[a[1955],a[1956]],{"am":a[1957],"pm":a[1958]},[a[1959],a[1960],a[1961],a[1962],a[1963],a[1964],a[1965],a[1966],a[1967],a[1968],a[1969],a[1970],a[1971],a[1972]],[a[1973],a[1974],a[1975],a[1976],a[1977],a[1978],a[1979],a[1980],a[1981],a[1982],a[1983],a[1984]],[a[1985]],[a[1986],a[1987],a[1988],a[1989],a[1990],a[1991],a[1992],a[1993],a[1994],a[1995],a[1996],a[1997]],[a[1998],a[1999],a[2000],a[2001],a[2002],a[2003],a[2004],a[2005],a[2006],a[1585],a[2007],a[2008]],[a[2009]],[a[2010],a[2011],a[2012],a[2013],a[2014],a[2015],a[2016],a[2017],a[2018],a[2019],a[2020],a[2021],a[2022],a[2023],a[2024],a[2025],a[2026],a[2027],a[2028],a[2029],a[2030],a[2031],a[2032],a[2033],a[2034],a[2035],a[2036],a[2037],a[2038],a[2039],a[2040],a[2041],a[2042],a[2043],a[2044],a[2045],a[2046],a[2047],a[2048],a[2049],a[2050],a[2051],a[2052],a[2053],a[2054],a[2055],a[2056],a[2057],a[2058],a[2059],a[2060],a[2061],a[2062],a[2063],a[2064],a[2065],a[2066],a[2067],a[2068],a[2069],a[2070],a[2071],a[2072],a[2073],a[2074],a[2075],a[2076],a[2077],a[2078],a[2079],a[2080],a[2081],a[2082],a[2083],a[2084],a[2085],a[2086],a[2087],a[2088],a[2089],a[2090],a[2091],a[2092],a[2093],a[2094],a[2095],a[2096],a[2097],a[2098],a[2099],a[2100],a[2101],a[2102],a[2103],a[2104],a[2105],a[2106],a[2107],a[2108],a[2109],a[2110],a[2111],a[2112],a[2113],a[2114],a[2115],a[2116],a[2117],a[2118],a[2119],a[2120],a[2121],a[2122],a[2123],a[2124],a[2125],a[2126],a[2127],a[2128],a[2129],a[2130],a[2131],a[2132],a[2133],a[2134],a[2135],a[2136],a[2137],a[2138],a[2139],a[2140],a[2141],a[2142],a[2143],a[2144],a[2145],a[2146],a[2147],a[2148],a[2149],a[2150],a[2151],a[2152],a[2153],a[2154],a[2155],a[2156],a[2157],a[2158],a[2159],a[2160],a[2161],a[2162],a[2163],a[2164],a[2165],a[2166],a[2167],a[2168],a[2169],a[2170],a[2171],a[2172],a[2173],a[2174],a[2175],a[2176],a[2177],a[2178],a[2179],a[2180],a[2181],a[2182],a[2183],a[2184],a[2185],a[2186],a[2187],a[2188],a[2189],a[2190],a[2191],a[2192],a[2193],a[2194],a[2195],a[2196],a[2197],a[2198],a[2199],a[2200],a[2201],a[2202],a[2203],a[2204],a[2205],a[2206],a[2207],a[2208],a[2209],a[2210],a[2211],a[2212],a[2213],a[2214],a[2215],a[2216],a[2217],a[2218],a[2219],a[2220],a[2221],a[2222],a[2223],a[2224],a[2225],a[2226],a[2227],a[2228],a[2229],a[2230],a[2231],a[2232],a[2233],a[2234],a[2235],a[2236],a[2237],a[2238],a[2239],a[2240],a[2241],a[2242],a[2243],a[2244],a[2245]],[a[2246],a[2247],a[2248],a[2249],a[2250],a[2251],a[2252],a[2253],a[2254],a[2255],a[2256],a[2257]],[a[2258]],[a[2259],a[2260]],{"AUD":a[434],"BAM":a[2261],"BRL":a[435],"CAD":a[436],"CNY":a[437],"CZK":a[2262],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PLN":a[2263],"RSD":a[2264],"THB":a[449],"TRY":a[2265],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2266],a[2267],a[2268],a[2269],a[2270],a[2271],a[2272],a[2273],a[2274],a[2275],a[2276],a[2277]],[a[2278],a[2279],a[2268],a[2280],a[2281],a[2282],a[2283],a[2284],a[2285],a[2275],a[2286],a[2287]],[a[2288],a[2289],a[2290],a[2291],a[2292],a[2293],a[2294]],[a[2295],a[2289],a[2290],a[2296],a[2292],a[2293],a[2297]],[a[2298],a[2299]],{"am":a[2300],"pm":a[2301]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ERN":a[458],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2302]],[a[2303],a[2304],a[2305],a[2306],a[2307],a[2308],a[2309],a[2310],a[2311],a[2312],a[2313],a[2314]],[a[2315],a[2316],a[2317],a[2318],a[2319],a[2320],a[2321],a[2322],a[2323],a[2324],a[2325],a[2326]],[a[2327],a[2328],a[2317],a[2329],a[2319],a[2320],a[2330],a[2331],a[2332],a[2333],a[2334],a[2335]],[a[2336],a[2337],a[2338],a[2339],a[2340],a[2341],a[2342]],[a[2343],a[2344],a[2345],a[2346],a[2347],a[2348],a[2349]],[a[2350],a[2351]],[a[2350],a[2351],a[2352],a[2353]],[a[2354],a[2355],a[2352],a[2356]],{"am":a[2357],"pm":a[2358]},{"AUD":a[732],"CAD":a[436],"CNY":a[1275],"ESP":a[2359],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XPF":a[456]},{"AUD":a[732],"CAD":a[436],"CNY":a[1275],"ESP":a[2359],"EUR":a[440],"FRF":a[468],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XPF":a[456]},[a[2360],a[2361],a[2362],a[2363],a[2364],a[2365],a[2366],a[2367],a[2368],a[2369],a[2370],a[2371]],[a[2372],a[2373],a[2374],a[2375],a[2376],a[2377],a[2378],a[2379],a[2380],a[2381],a[2382],a[2383]],[a[2384],a[2385],a[2386],a[2387],a[2388],a[2389],a[2390]],[a[2391],a[2392],a[2393],a[2394],a[2395],a[2396],a[2397]],[a[2398],a[2399]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"UGX":a[2400],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"month":a[14],"day":a[13],"year":a[13],"pattern":a[2401]},[a[2402],a[2403],a[2404],a[2405],a[2406],a[2407],a[2408],a[2409],a[2410],a[2411],a[2412],a[2413]],[a[2414],a[2415],a[2416],a[2417],a[2418],a[2419],a[2420],a[2421],a[2422],a[2423],a[2424],a[2425]],[a[2426],a[2427],a[2428],a[2429],a[2430],a[2431],a[2432]],[a[2433],a[2434],a[2435],a[2436],a[2437],a[2438],a[2439]],[a[2440],a[2441]],[a[2442],a[2443]],{"am":a[2444],"pm":a[2445]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[2446],"pattern12":a[2447]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[2448]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[2449]},{"day":a[13],"month":a[13],"year":a[13],"pattern":a[2449]},{"day":a[13],"month":a[14],"pattern":a[1825]},{"day":a[13],"month":a[13],"pattern":a[1825]},[a[2450],a[2451],a[2452],a[2453],a[2454],a[2455],a[2456],a[2457],a[2458],a[2459],a[2460],a[2461]],[a[2462],a[2463],a[2464],a[2465],a[2466],a[2467],a[2468],a[2469],a[2470],a[2471],a[2472],a[2473]],[a[2474],a[2475],a[2476],a[2477],a[2478],a[2479],a[2480]],[a[2481],a[2482],a[2483],a[2484],a[2485],a[2486],a[2487]],[a[2488],a[2489]],[a[2490],a[2491]],[a[2492],a[421]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"CSK":a[2493],"CZK":a[1888],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"TWD":a[450],"USD":a[451],"XAF":a[453],"XBB":a[2494],"XCD":a[454],"XEU":a[2495],"XOF":a[455],"XPF":a[456]},[a[2496],a[2497],a[397],a[2498],a[397],a[397],a[2499],a[469],a[397],a[400],a[398],a[2500]],[a[2501],a[2502],a[2503],a[2504],a[2505],a[2506],a[2507],a[2508],a[2509],a[2510],a[2511],a[2512]],[a[2513],a[2514],a[2503],a[2504],a[2505],a[2515],a[2516],a[2508],a[2509],a[2517],a[2518],a[2519]],[a[2520],a[2521],a[504],a[2522],a[2523],a[2524],a[509]],[a[1772],a[2525],a[2526],a[2527],a[2528],a[1753],a[2529]],[a[2530],a[2531],a[2532],a[2533],a[2534],a[2535],a[2536]],[a[1308],a[470]],[a[2537],a[2312]],[a[2538],a[2539]],{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[2540],"pattern12":a[2541]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[2542]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[1824]},[a[2543],a[2316],a[2544],a[2545],a[1827],a[2546],a[2321],a[2547],a[2548],a[2549],a[2325],a[2550]],[a[1834],a[1835],a[2551],a[1836],a[1827],a[1837],a[1838],a[2552],a[2553],a[2554],a[2555],a[2556]],[a[2557],a[1254],a[2558],a[2559],a[2560],a[2561],a[2562]],[a[2563],a[1321],a[2564],a[2565],a[2566],a[2567],a[2568]],[a[2569],a[2570],a[2571],a[2572],a[2573],a[2574],a[2575]],[a[2576],a[2577],a[2578],a[2579]],[a[2580],a[2581],a[2582],a[2583]],[a[2580],a[2581],a[2584],a[2585]],[a[420],a[2586]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DKK":a[2587],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2588],a[2589],a[2590],a[2591],a[2592],a[2593],a[2594],a[2595],a[2596],a[2597],a[2598],a[2599]],[a[2600],a[2601],a[2602],a[2603],a[2604],a[2605],a[2606],a[2607],a[2608],a[2609],a[2610],a[2611]],[a[2612],a[2613],a[2589],a[2590],a[2591],a[2592],a[2614]],[a[2615],a[2616],a[2617],a[2618],a[2619],a[2620],a[2621]],[a[2622],a[577]],[a[2623],a[2624]],{"am":a[2625],"pm":a[2626]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[2628],"pattern12":a[2629]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[1823]},[a[2630],a[2631],a[2632],a[2633],a[2505],a[1196],a[2634],a[2635],a[2636],a[2637],a[2638],a[2639]],[a[2640],a[2641],a[2632],a[488],a[2505],a[1196],a[2634],a[2642],a[492],a[493],a[494],a[2643]],[a[496],a[2644],a[498],a[2645],a[500],a[2646],a[502]],[a[2647],a[2648],a[2649],a[2650],a[2651],a[2652],a[2653]],[a[2654],a[2655],a[2656],a[2657]],[a[2654],a[2655],a[2658],a[2659]],[a[2654],a[2655],a[2660],a[2661]],{"am":a[2662],"pm":a[2663]},{"ATS":a[2664],"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2665],a[2631],a[2632],a[2633],a[2505],a[1196],a[2634],a[2635],a[2636],a[2637],a[2638],a[2639]],[a[2666],a[2641],a[2632],a[488],a[2505],a[1196],a[2634],a[2642],a[492],a[493],a[494],a[2643]],{"positivePattern":a[527],"negativePattern":a[2667]},{"decimal":a[429],"group":a[2668],"nan":a[431],"percent":a[432],"infinity":a[433]},{"ATS":a[2664],"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"LUF":a[468],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2669],a[2670],a[475],a[2671],a[2522],a[2672],a[2673],a[2674],a[2675],a[482],a[2676],a[2677]],[a[2678],a[2679],a[2680],a[2681],a[2522],a[2682],a[2683],a[2674],a[2684],a[2685],a[2686],a[2687]],[a[1207],a[2688],a[2689],a[2690],a[2691],a[2692],a[2693]],[a[2694],a[2695],a[2696],a[2697],a[1214],a[2698],a[2699]],[a[2700],a[2701]],[a[2702],a[2703]],{"am":a[2704],"pm":a[2705]},{"decimal":a[429],"group":a[529],"nan":a[431],"percent":a[432],"infinity":a[433]},[a[2706],a[2707],a[2708],a[2709],a[2710],a[2711],a[2712],a[2713],a[2714],a[1232],a[2715],a[2716]],[a[2717],a[2718],a[2719],a[2720],a[2721],a[2722],a[2723],a[2724],a[2725],a[2726],a[2727],a[2728]],[a[2729],a[2730],a[2731],a[2732],a[2733],a[2734],a[2735]],[a[2736],a[2737],a[2738],a[2739],a[2740],a[2741],a[2742]],[a[2743],a[2744]],[a[2745],a[2746]],{"am":a[2747],"pm":a[2748]},[a[509],a[2749],a[504],a[2750],a[2522],a[2520],a[2751],a[2674],a[2752],a[2753],a[2754],a[2755]],[a[2756],a[2757],a[2758],a[2759],a[2760],a[2761],a[2762],a[2674],a[2763],a[2764],a[2765],a[2766]],[a[2767],a[2768],a[96],a[2690],a[2769],a[2770],a[2771]],[a[2772],a[2773],a[2774],a[2775],a[2776],a[2777],a[2778]],[a[2779],a[2780]],[a[2781],a[2782]],{"weekday":a[14],"year":a[13],"month":a[14],"day":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[2783],"pattern12":a[2784]},{"weekday":a[14],"year":a[13],"month":a[14],"day":a[13],"pattern":a[2785]},{"year":a[13],"month":a[13],"day":a[13],"pattern":a[20]},{"year":a[13],"month":a[13],"pattern":a[21]},{"year":a[13],"month":a[14],"pattern":a[2786]},{"month":a[14],"day":a[13],"pattern":a[2787]},{"month":a[13],"day":a[13],"pattern":a[24]},[a[2788],a[2789],a[2790],a[641],a[2791],a[2792],a[2793],a[2794],a[646],a[2795],a[2796],a[2797]],[a[2788],a[2789],a[2790],a[2798],a[2791],a[2792],a[2793],a[2794],a[2799],a[2795],a[2796],a[649]],[a[2800],a[2801],a[2802],a[2803],a[2804],a[2805],a[2806],a[2807],a[2808],a[2809],a[2810],a[2811]],[a[2812],a[2813],a[2814],a[2815],a[1731],a[2816],a[2817]],[a[1734],a[1735],a[2818],a[1737],a[2819],a[1739],a[1733]],{"am":a[2820],"pm":a[2821]},[a[2822]],{"decimal":a[429],"group":a[430],"nan":a[2823],"percent":a[432],"infinity":a[2824]},{"AUD":a[732],"BRL":a[435],"BTN":a[2825],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"INR":a[444],"JPY":a[445],"KRW":a[2826],"MXN":a[447],"NZD":a[448],"THB":a[2827],"TWD":a[450],"USD":a[451],"VND":a[452],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2828],a[2829],a[2830],a[2591],a[2831],a[2832],a[2833],a[2834],a[2835],a[2597],a[2598],a[2836]],[a[2837],a[2838],a[2839],a[2840],a[2841],a[2842],a[2843],a[2844],a[2845],a[2846],a[2847],a[2848]],[a[2849],a[2850],a[2851],a[2852],a[2853],a[2854],a[2855]],[a[2856],a[2857],a[2858],a[2859],a[2860],a[2861],a[2862]],[a[2863],a[2864]],[a[2865],a[2866]],{"am":a[2867],"pm":a[2868]},{"month":a[14],"day":a[13],"year":a[13],"pattern":a[2869]},{"pattern":a[2870],"pattern12":a[2871]},[a[2872],a[2873],a[2874],a[2875],a[2876],a[2877],a[2878],a[2879],a[2880],a[2881],a[2882],a[2883]],[a[2884],a[2885],a[2886],a[2887],a[2888],a[2889],a[2890],a[2891],a[2892],a[2893],a[2894],a[2895]],[a[2872],a[2873],a[2874],a[2875],a[2876],a[2877],a[2878],a[2879],a[2880],a[2881],a[2882],a[2883],a[2896]],[a[2884],a[2885],a[2886],a[2887],a[2888],a[2889],a[2890],a[2891],a[2892],a[2893],a[2894],a[2895],a[2897]],[a[2898],a[2898],a[2899],a[2900],a[2898],a[2901],a[2902],a[2898],a[2900],a[2903],a[2900],a[2898]],[a[2884],a[2885],a[2886],a[2904],a[2905],a[2889],a[2890],a[2891],a[2892],a[2893],a[2894],a[2895]],[a[2906],a[2907],a[2908],a[2909],a[2910],a[2911],a[2912]],[a[2913],a[2914],a[2915],a[2916],a[2917],a[2918],a[2919]],[a[2920],a[2921]],[a[2922],a[2923]],{"am":a[2924],"pm":a[2925]},[a[2884],a[2885],a[2886],a[2887],a[2888],a[2889],a[2890],a[2891],a[2892],a[2893],a[2894],a[2895],a[2897],a[2890]],[a[2884],a[2885],a[2886],a[2887],a[2888],a[2889],a[2890],a[2926],a[2892],a[2893],a[2894],a[2895],a[2897],a[2890]],[a[2924]],[a[2884],a[2885],a[2886],a[2875],a[2888],a[2889],a[2890],a[2891],a[2892],a[2893],a[2894],a[2895]],[a[2927],a[421]],{"decimal":a[429],"group":a[430],"nan":a[2928],"percent":a[432],"infinity":a[433]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GHS":a[632],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[2929],"pattern12":a[2930]},[a[2931],a[2932],a[2933],a[2934],a[2933],a[2931],a[2931],a[2934],a[2935],a[2936],a[2937],a[2938]],[a[2939],a[2940],a[2941],a[2942],a[2943],a[2944],a[2945],a[2946],a[2947],a[2948],a[2949],a[2950]],[a[2951],a[2952],a[2953],a[2954],a[2955],a[2956],a[2957],a[2958],a[2959],a[2960],a[2961],a[2962]],[a[2963],a[2964],a[2965],a[2966],a[2967],a[2968],a[2969]],[a[2970],a[2971],a[2972],a[2973],a[2974],a[2975],a[2976]],[a[2977],a[2978],a[2979],a[2980],a[2981],a[2982],a[2983]],[a[2984],a[2985],a[2986],a[2987]],{"am":a[2988],"pm":a[2989]},[a[2988]],[a[2990],a[421]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GRD":a[2991],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"month":a[14],"day":a[13],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[2992],"pattern12":a[2993]},[a[2994],a[2995],a[2996],a[2997],a[2998],a[2999],a[3000],a[3001],a[3002],a[3003],a[3004],a[3005]],[a[3006],a[3007],a[3008],a[3009],a[3010],a[3011],a[3012],a[3013],a[3014],a[3015],a[3016],a[3017]],[a[473],a[474],a[475],a[476],a[3018],a[478],a[479],a[480],a[481],a[3019],a[483],a[1191]],[a[3020],a[3021],a[3022],a[488],a[3018],a[3023],a[3024],a[2642],a[492],a[3025],a[494],a[3026]],[a[2520],a[3027],a[3028],a[3029],a[3030],a[3031],a[509]],[a[3032],a[3033],a[3034],a[3035],a[3036],a[3037],a[3038]],[a[3039],a[3040],a[3041],a[3042],a[3043],a[3044],a[3045]],[a[3046],a[469]],[a[1461],a[1462],a[1184],a[1185]],[a[3047],a[3048],a[3049],a[3050]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3051],"pattern12":a[3052]},{"AUD":a[530],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BBD":a[530],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BMD":a[530],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"BSD":a[530],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3051],"pattern12":a[3052]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"pattern":a[534]},{"day":a[15],"month":a[14],"year":a[13],"pattern":a[462]},{"day":a[15],"month":a[15],"year":a[13],"pattern":a[535]},{"month":a[15],"year":a[13],"pattern":a[464]},{"day":a[15],"month":a[14],"pattern":a[536]},{"day":a[15],"month":a[15],"pattern":a[537]},{"AUD":a[434],"BRL":a[435],"BWP":a[1227],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"BZD":a[530],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[530],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[530],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[530],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[3053],a[3054],a[3055],a[3056],a[3055],a[3053],a[3053],a[3057],a[3058],a[3059],a[3060],a[3061]],[a[3062],a[3063],a[3064],a[3065],a[3066],a[3067],a[3068],a[3069],a[3070],a[3071],a[3072],a[3073]],[a[3074],a[3075],a[3076],a[3077],a[3066],a[3067],a[3078],a[3079],a[3080],a[3081],a[3082],a[3083]],[a[3084],a[3085],a[3086],a[3087],a[3088],a[3089],a[3090]],[a[3091],a[3092],a[3093],a[3094],a[3095],a[3096],a[3097]],[a[3098],a[3099]],[a[3100],a[3101],a[1184],a[1185]],[a[3102],a[3103],a[3049],a[3050]],{"am":a[3104],"pm":a[3105]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ERN":a[458],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"FJD":a[530],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"FKP":a[441],"GBP":a[1150],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"am":a[3106],"pm":a[3107]},[a[3108]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GHS":a[632],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[1150],"GIP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GMD":a[472],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GYD":a[530],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3109],"pattern12":a[3110]},[a[165],a[166],a[167],a[168],a[169],a[170],a[171],a[172],a[173],a[174],a[175],a[176],a[177],a[178],a[179],a[180],a[181],a[182],a[183],a[184],a[185],a[186],a[187],a[188],a[189],a[190],a[191],a[192],a[193],a[194],a[195],a[196],a[197],a[198],a[199],a[200],a[201],a[202],a[203],a[204],a[205],a[206],a[207],a[208],a[209],a[210],a[211],a[212],a[213],a[214],a[215],a[216],a[217],a[218],a[219],a[220],a[221],a[222],a[223],a[224],a[225],a[226],a[227],a[228],a[229],a[230],a[231],a[232],a[233],a[234],a[235],a[236],a[237],a[238],a[239],a[240],a[241],a[242],a[243],a[244],a[245],a[246],a[247],a[248],a[249],a[250],a[251],a[252],a[253],a[254],a[255],a[256],a[257],a[258],a[259],a[260],a[261],a[262],a[263],a[264],a[265],a[266],a[267],a[268],a[269],a[270],a[271],a[272],a[273],a[274],a[275],a[276],a[277],a[278],a[279],a[280],a[281],a[282],a[283],a[284],a[285],a[286],a[287],a[288],a[289],a[290],a[291],a[292],a[293],a[294],a[295],a[296],a[297],a[298],a[299],a[300],a[301],a[302],a[303],a[304],a[305],a[306],a[307],a[308],a[309],a[310],a[311],a[312],a[313],a[314],a[315],a[316],a[317],a[318],a[319],a[320],a[321],a[322],a[323],a[324],a[325],a[3111],a[327],a[328],a[329],a[3112],a[331],a[332],a[333],a[334],a[335],a[336],a[337],a[338],a[339],a[340],a[341],a[342],a[343],a[344],a[345],a[346],a[347],a[348],a[349],a[350],a[351],a[352],a[353],a[354],a[355],a[356],a[357],a[358],a[359],a[360],a[361],a[362],a[363],a[364],a[365],a[366],a[367],a[368],a[369],a[370],a[371],a[372],a[373],a[374],a[375],a[376],a[377],a[378],a[379],a[380],a[381],a[382],a[383],a[384],a[385],a[386],a[387],a[388],a[389],a[390],a[391],a[392],a[393],a[394],a[395],a[396],a[403],a[404],a[405],a[406]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[530],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"am":a[3113],"pm":a[3114]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3115],"pattern12":a[3116]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[3117]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JMD":a[530],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"KYD":a[530],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"LRD":a[530],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZAR":a[531]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MGA":a[3118],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MOP":a[3119],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3120],"pattern12":a[3121]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"pattern":a[461]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[1150],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MUR":a[3122],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MWK":a[2863],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NAD":a[530],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NGN":a[3123],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[13],"month":a[15],"year":a[13],"pattern":a[535]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[530],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PGK":a[1467],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PHP":a[3124],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PKR":a[3122],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RWF":a[3125],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SBD":a[530],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SCR":a[3126],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SGD":a[530],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[1150],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SHP":a[441],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SLL":a[3127],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[1150],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SSP":a[441],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"ANG":a[3128],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SZL":a[2498],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TOP":a[3129],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TTD":a[530],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"TZS":a[1225],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"UGX":a[2400],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"VUV":a[3130],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"year":a[13],"month":a[15],"day":a[15],"pattern":a[587]},{"month":a[15],"day":a[15],"pattern":a[466]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZMW":a[1467]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3109],"pattern12":a[3110]},{"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"pattern":a[1631]},{"day":a[15],"month":a[14],"year":a[13],"pattern":a[1325]},[a[1826],a[1229],a[1230],a[1284],a[1827],a[1828],a[1829],a[3132],a[1831],a[1289],a[1832],a[1833]],[a[3133],a[3134],a[3135],a[3136],a[3137],a[3138],a[3139],a[3140],a[3141],a[3142],a[3143],a[3144]],[a[2706],a[3145],a[1254],a[3146],a[3147],a[3148],a[1258]],[a[3149],a[3150],a[3151],a[3152],a[3153],a[3154],a[3155]],[a[3156],a[3157]],{"am":a[3158],"pm":a[3159]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3160],"pattern12":a[3161]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[3162]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[3163]},{"month":a[14],"year":a[13],"pattern":a[3164]},{"day":a[13],"month":a[14],"pattern":a[3165]},[a[2498],a[468],a[397],a[469],a[397],a[467],a[467],a[469],a[399],a[470],a[471],a[472]],[a[3166],a[2316],a[2544],a[2318],a[3167],a[2546],a[2321],a[3168],a[3169],a[2324],a[2325],a[3170]],[a[3171],a[3172],a[3173],a[2329],a[3174],a[3138],a[3139],a[3175],a[3176],a[2333],a[3177],a[3178]],[a[3179],a[3180],a[3181],a[3182],a[3183],a[3184],a[3185]],[a[3186],a[3187],a[2544],a[3188],a[3189],a[3190],a[3191]],[a[3192],a[3193],a[1267],a[1268],a[3194],a[3195],a[3196]],[a[3197],a[3198],a[3199],a[3200]],[a[3201],a[3202]],[a[3203],a[421]],{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"month":a[13],"year":a[13],"pattern":a[3205]},{"ARS":a[530],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"BOB":a[3206],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"day":a[15],"month":a[15],"year":a[13],"pattern":a[3207]},{"month":a[15],"year":a[13],"pattern":a[3205]},{"day":a[15],"month":a[15],"pattern":a[1778]},{"positivePattern":a[425],"negativePattern":a[2667]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"CLP":a[530],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"COP":a[530],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"CRC":a[3208],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"CUP":a[530],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"DOP":a[530],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"GTQ":a[3209],"MXN":a[447],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"HNL":a[3210],"MXN":a[447],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[3160],"pattern12":a[3161]},[a[2498],a[468],a[504],a[469],a[3211],a[3212],a[3213],a[3214],a[399],a[470],a[471],a[472]],[a[3166],a[3215],a[3216],a[2318],a[3217],a[2546],a[2321],a[2322],a[2323],a[2324],a[2325],a[3170]],[a[3218],a[3219],a[3220],a[3221],a[3222],a[3223],a[3224]],[a[3186],a[3187],a[2544],a[3225],a[3189],a[3226],a[3227]],[a[1272],a[1273],a[3199],a[3200]],{"ANG":a[3228],"AOA":a[3229],"ARS":a[3204],"AUD":a[732],"AWG":a[3230],"CNY":a[437],"ESP":a[2359],"MXN":a[530],"ZMW":a[3231]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"NIO":a[3232],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"month":a[15],"day":a[15],"year":a[13],"pattern":a[463]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"PAB":a[3233],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"PEN":a[3234],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"PHP":a[3124],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"positivePattern":a[527],"negativePattern":a[3235]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"PYG":a[3236],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"JPY":a[1275],"MXN":a[447],"USD":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[451],"UYU":a[530],"XAF":a[453],"XCD":a[454]},{"ARS":a[3204],"AUD":a[732],"CAD":a[436],"ESP":a[2359],"EUR":a[440],"MXN":a[447],"USD":a[530],"VEF":a[3237],"XAF":a[453],"XCD":a[454]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3238],"pattern12":a[3239]},{"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3240],"pattern12":a[3241]},[a[467],a[3242],a[397],a[469],a[397],a[467],a[467],a[469],a[399],a[470],a[471],a[472]],[a[3243],a[3244],a[3245],a[1284],a[3246],a[3247],a[3248],a[3249],a[3250],a[1289],a[1832],a[3251]],[a[3252],a[3253],a[3245],a[3254],a[3246],a[3247],a[3248],a[2552],a[2553],a[3255],a[2555],a[3256]],[a[1227],a[2498],a[398],a[1467],a[471],a[531],a[3210]],[a[3257],a[3258],a[3259],a[3260],a[3261],a[3262],a[3263]],[a[3264],a[3265],a[3266],a[1827]],[a[3264],a[3265]],[a[3267],a[3268]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EEK":a[2587],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[3269]],[a[3270],a[470],a[397],a[469],a[397],a[2498],a[3270],a[469],a[2496],a[3270],a[469],a[469]],[a[3271],a[3272],a[2544],a[3273],a[3274],a[3275],a[3276],a[3277],a[3278],a[3279],a[3280],a[3281]],[a[3282],a[3283],a[3284],a[3285],a[3286],a[3287],a[3288],a[3289],a[3290],a[3291],a[3292],a[3293]],[a[3294],a[3295],a[3296],a[3297],a[3298],a[3299],a[3300]],[a[3301],a[3302],a[3303],a[3304],a[3305],a[3306],a[3307]],[a[3308],a[3309]],[a[3310],a[421]],{"positivePattern":a[3311],"negativePattern":a[3312]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ESP":a[2359],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[3313],a[3314],a[3315],a[3316],a[3317],a[3318],a[3319],a[3320],a[3321],a[3322],a[3323],a[3324]],[a[3325],a[3326],a[3327],a[3328],a[3329],a[3330],a[3331],a[3332],a[3333],a[3334],a[3335],a[3336]],[a[3337],a[3338],a[3339],a[3340],a[3341],a[3342],a[3343]],[a[3344],a[3345],a[3346],a[3347],a[3348],a[3349],a[3350]],[a[3351],a[3352]],[a[3353],a[3354]],{"am":a[3355],"pm":a[3356]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3357],"pattern12":a[3358]},[a[3359],a[769],a[770],a[1131],a[770],a[3359],a[3359],a[3360],a[776],a[3360],a[773],a[779]],[a[992],a[3361],a[782],a[3362],a[3363],a[3364],a[3365],a[3366],a[3367],a[3368],a[3369],a[3370]],[a[3371],a[3372],a[3373],a[3374],a[3375],a[3376],a[3377],a[3378],a[3379],a[3380],a[3381],a[3382]],[a[3383],a[3384],a[3385],a[3386],a[3387],a[1122],a[1130]],[a[3388],a[3389],a[3390],a[3391],a[3392],a[3393],a[3394]],[a[3395],a[770]],[a[3396],a[3397]],[a[3398],a[3399]],{"am":a[3400],"pm":a[3401]},[a[1133],a[1132],a[3402],a[3403],a[1130],a[1131],a[772],a[773],a[3360],a[776],a[1133],a[1131],a[3360],a[772]],[a[3404],a[3405],a[3406],a[3407],a[807],a[809],a[3408],a[3409],a[3410],a[3411],a[813],a[814],a[3412],a[3413]],[a[770],a[802],a[3414],a[3414],a[1122],a[1122],a[3414],a[1130],a[3414],a[1130],a[3415],a[3415]],[a[829],a[830],a[3416],a[3417],a[3418],a[3419],a[835],a[836],a[837],a[838],a[3420],a[3421]],[a[3422]],[a[3423]],[a[769],a[3360],a[3424],a[1133],a[770],a[1130],a[770],a[1131],a[1131],a[779],a[778],a[3360]],[a[3425],a[3426],a[1080],a[3427],a[1082],a[3428],a[1084],a[3429],a[1086],a[3430],a[1088],a[3431]],[a[3432],a[3433],a[3434],a[3435],a[3436],a[3437],a[3438],a[3439],a[3440],a[3441],a[3442],a[3443]],[a[3444]],[a[3445]],[a[3446]],{"positivePattern":a[3447],"negativePattern":a[3448]},{"decimal":a[1093],"group":a[1094],"nan":a[3449],"percent":a[1096],"infinity":a[433]},{"AFN":a[3450],"AUD":a[434],"BRL":a[435],"CAD":a[3451],"CNY":a[3452],"EUR":a[440],"GBP":a[441],"HKD":a[3453],"ILS":a[443],"INR":a[444],"IRR":a[3454],"JPY":a[1275],"KRW":a[446],"MXN":a[3455],"NZD":a[3456],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[3457],"XOF":a[455],"XPF":a[456]},[a[3458],a[3361],a[782],a[3362],a[3459],a[3364],a[3460],a[3366],a[3367],a[3368],a[3369],a[3461]],[a[3462]],[a[3463],a[3464],a[3465],a[542],a[3466],a[3467],a[3468],a[3469],a[3470],a[3471],a[3472],a[3473]],[a[3474],a[3475],a[3476],a[3477],a[3478],a[3479],a[3480],a[3481],a[3482],a[3483],a[3484],a[3485]],[a[3486],a[3487],a[3488],a[1353],a[3489],a[3490],a[3491]],[a[3492],a[3493],a[3494],a[3495],a[3496],a[3497],a[3498]],[a[3499],a[3500]],[a[3501],a[3502]],{"am":a[3503],"pm":a[3504]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GNF":a[3505],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MRO":a[3506],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[398],a[400],a[397],a[400],a[398],a[1467],a[400],a[2498],a[399],a[3210],a[397],a[467]],[a[3507],a[3508],a[3509],a[3510],a[3511],a[3512],a[3513],a[3514],a[3515],a[3516],a[3517],a[3518]],[a[3519],a[1254],a[2558],a[3520],a[2560],a[3521],a[3522]],[a[3523],a[3524],a[3525],a[3526],a[3527],a[3528],a[3529]],[a[3530],a[3531],a[3532],a[3533]],[a[3534],a[3535],a[3536],a[3537]],[a[3538],a[3539],a[3540],a[3541]],{"am":a[3542],"pm":a[3543]},[a[3544],a[3545],a[3546],a[3547],a[3548],a[3549],a[3550],a[3551],a[3552],a[3553],a[3554],a[3555],a[3556],a[3557]],[a[3558],a[3559],a[3560],a[3561],a[3562],a[3563],a[3564],a[3565],a[3566],a[3567],a[3568],a[3569]],{"decimal":a[430],"group":a[529],"nan":a[3570],"percent":a[432],"infinity":a[433]},{"EUR":a[440],"GBP":a[441],"JPY":a[1275],"USD":a[530],"XAF":a[453],"XOF":a[455]},[a[2498],a[1227],a[397],a[469],a[397],a[400],a[400],a[469],a[399],a[470],a[471],a[472]],[a[3571],a[3572],a[475],a[3573],a[3018],a[3574],a[3575],a[1190],a[3576],a[482],a[3577],a[1450]],[a[3578],a[3579],a[3580],a[3581],a[3582],a[3583],a[3584],a[3585],a[3586],a[3587],a[3588],a[3589]],[a[3590],a[3591],a[504],a[3592],a[3593],a[3594],a[509]],[a[3595],a[1773],a[475],a[3596],a[3597],a[3598],a[100]],[a[3599],a[3600],a[3601],a[3602],a[3603],a[3604],a[3605]],{"weekday":a[14],"year":a[13],"month":a[14],"day":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3606],"pattern12":a[3607]},{"weekday":a[14],"year":a[13],"month":a[14],"day":a[13],"pattern":a[3608]},[a[1826],a[1229],a[1230],a[1284],a[3246],a[1828],a[1829],a[3249],a[1831],a[1289],a[1832],a[1600]],[a[1834],a[1835],a[3609],a[3610],a[3246],a[1837],a[1838],a[2552],a[2553],a[2554],a[2555],a[3611]],[a[3519],a[3612],a[3613],a[1255],a[3614],a[2561],a[3615]],[a[3616],a[3617],a[3618],a[3619],a[3620],a[3621],a[3622]],[a[3623],a[3624],a[3625],a[3626],a[3627],a[3628],a[3629]],[a[2576],a[2577],a[3630],a[3631]],[a[2580],a[2581],a[3632],a[3633]],[a[3634],a[3635],a[3636],a[3637]],{"am":a[3638],"pm":a[3639]},{"decimal":a[430],"group":a[429],"nan":a[3640],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DKK":a[2587],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[0],a[1],a[2],a[3],a[3641],a[4],a[5],a[6],a[7],a[8],a[9],a[10],a[11],a[12]],[a[3642]],[a[3643]],[a[3644]],[a[3645],a[3646],a[3647],a[3648],a[3649],a[3650],a[3651],a[3652],a[3653],a[3654],a[3655],a[3656]],[a[3657],a[3658],a[3659],a[3660],a[3661],a[3662],a[3663],a[3664],a[3665],a[3666],a[3667],a[3668]],[a[398],a[3046],a[400],a[1467],a[398],a[469],a[3046],a[3046],a[3046],a[3046],a[469],a[397],a[471]],[a[3669],a[3670],a[3671],a[3672],a[3673],a[3674],a[3675],a[3676],a[3677],a[3678],a[3679],a[3680],a[3681]],[a[3669],a[3682],a[3683],a[3684],a[3685],a[3686],a[3687],a[3688],a[3689],a[3690],a[3691],a[3692],a[3693]],[a[3694],a[3695]],[a[3696],a[3697]],[a[3698],a[3699],a[3700],a[3701],a[3702],a[3703],a[3704],a[3705],a[3706],a[3707],a[3708],a[3709],a[3710]],[a[3711],a[3712],a[3713],a[3714],a[3702],a[3715],a[3716],a[3717],a[3718],a[3719],a[3720],a[3721],a[3722]],[a[3723],a[3724]],[a[3725],a[3726]],[a[3727],a[3728],a[3609],a[3729],a[3246],a[3730],a[3731],a[3732],a[3169],a[2324],a[2325],a[3733]],[a[3734],a[3735],a[3609],a[3736],a[3246],a[3730],a[3737],a[3732],a[3738],a[3739],a[2334],a[3740]],[a[2706],a[3145],a[1254],a[3146],a[3741],a[3148],a[1258]],[a[3742],a[3187],a[2544],a[1768],a[3743],a[3744],a[3745]],[a[3746],a[3747],a[3748],a[3749],a[3750],a[3751],a[3752]],[a[3753],a[3754]],[a[3753],a[3754],a[3755],a[3756]],[a[3757],a[3758]],[a[398],a[400],a[1467],a[398],a[399],a[469],a[469],a[471],a[2496],a[399],a[398],a[469],a[2498],a[469]],[a[3759],a[3760],a[3761],a[3762],a[3763],a[3764],a[3765],a[3766],a[3767],a[3768],a[3769],a[3770],a[3771],a[3772]],[a[3773],a[3774],a[114],a[3775],a[3776],a[117],a[118],a[3777],a[120],a[121],a[3778],a[2750],a[3779],a[125]],[a[3780]],[a[1308],a[3242],a[467],a[3781],a[399],a[3046],a[3781],a[1467],a[397],a[1227],a[397],a[1227]],[a[3782],a[3783],a[3784],a[3785],a[3786],a[3787],a[3788],a[3789],a[3790],a[3791],a[3792],a[3793]],[a[3794],a[3795],a[3796],a[3797],a[3798],a[3799],a[3800],a[3801],a[3802],a[3803],a[3792],a[3804]],[a[3805],a[1865],a[3806],a[3807],a[3808],a[3809],a[3810],a[3811],a[1872],a[3812],a[3813],a[3814]],[a[3815],a[3559],a[3816],a[3817],a[3818],a[3819],a[3820],a[3821],a[3566],a[3822],a[3823],a[3824]],[a[3825],a[3826]],{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BIF":a[3862],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[530],"CLP":a[3833],"CNY":a[437],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CDF":a[3863],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"DJF":a[438],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"DZD":a[3864],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"GNF":a[3505],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"HTG":a[2499],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KMF":a[3865],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"LUF":a[468],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MGA":a[3118],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MRO":a[3506],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MUR":a[3122],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"RWF":a[3125],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SCR":a[3126],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"SYP":a[3866],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TND":a[3867],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},{"ARS":a[3827],"AUD":a[3828],"BEF":a[2304],"BMD":a[3829],"BND":a[3830],"BRL":a[435],"BSD":a[3831],"BZD":a[3832],"CAD":a[3451],"CLP":a[3833],"CNY":a[3452],"COP":a[3834],"CVE":a[3835],"CYP":a[3836],"EGP":a[3837],"EUR":a[440],"FJD":a[3838],"FKP":a[3839],"FRF":a[468],"GBP":a[3840],"GIP":a[3841],"HKD":a[3453],"IEP":a[3842],"ILP":a[3843],"ILS":a[443],"INR":a[444],"ITL":a[3844],"JPY":a[3845],"KRW":a[446],"LBP":a[3846],"LRD":a[3847],"MTP":a[3848],"MXN":a[3455],"NAD":a[3849],"NZD":a[3456],"RHD":a[3850],"SBD":a[3851],"SDG":a[3852],"SGD":a[3853],"SHP":a[3854],"SRD":a[3855],"SSP":a[3856],"THB":a[449],"TTD":a[3857],"TWD":a[3858],"USD":a[3859],"UYU":a[3860],"VND":a[452],"VUV":a[3130],"WST":a[3131],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[3861]},[a[3868],a[468],a[397],a[469],a[397],a[467],a[3210],a[469],a[399],a[470],a[471],a[472]],[a[3869],a[3870],a[475],a[3871],a[2505],a[3872],a[3873],a[3874],a[3576],a[3875],a[483],a[3876]],[a[3877],a[3878],a[3879],a[3880],a[2505],a[3881],a[3873],a[3882],a[3883],a[3884],a[2765],a[3885]],[a[1259],a[1766],a[1230],a[1261],a[3886],a[3887],a[1264]],[a[3888],a[3889],a[3890],a[3891],a[3892],a[3893],a[3894]],[a[3895],a[3896]],{"am":a[3897],"pm":a[3898]},[a[3899],a[3900],a[3901],a[3902],a[3903],a[3904],a[3905],a[3906],a[3907],a[3908],a[3909],a[3910]],[a[3911],a[3912],a[3901],a[3913],a[3914],a[3915],a[3905],a[3916],a[3917],a[3918],a[3919],a[3920]],[a[3921],a[3922],a[3923],a[3924],a[3925],a[3926],a[3927]],[a[3928],a[3929],a[3930],a[3931],a[3932],a[3933],a[3934]],[a[3935],a[1462]],[a[3936],a[3048]],{"ANG":a[3937],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[3938],a[3939],a[3940],a[3941],a[3942],a[3943],a[3944],a[3945],a[3946],a[3947],a[3909],a[3948]],[a[3949],a[3950],a[3951],a[3952],a[3953],a[3954],a[3955],a[3956],a[3957],a[3958],a[3959],a[3960]],[a[3961],a[3962],a[3963],a[3964],a[3965],a[3966],a[3967]],[a[3968],a[3969],a[3970],a[3971],a[3972],a[3973],a[3974]],[a[3975],a[3048]],{"am":a[2901],"pm":a[3976]},{"day":a[13],"month":a[13],"pattern":a[1778]},[a[1226],a[468],a[397],a[469],a[397],a[1226],a[1226],a[469],a[399],a[470],a[471],a[472]],[a[3977],a[1229],a[1230],a[1231],a[3246],a[3978],a[3979],a[1235],a[1236],a[3980],a[1832],a[1833]],[a[3981],a[3982],a[3173],a[2329],a[3983],a[3984],a[3985],a[3175],a[3986],a[3987],a[3143],a[3144]],[a[1259],a[3988],a[3989],a[3990],a[3991],a[3148],a[3227]],[a[1259],a[1766],a[1230],a[3992],a[3993],a[3994],a[3227]],[a[3192],a[3988],a[1267],a[3995],a[3996],a[3997],a[3196]],[a[1272],a[1273],a[1184],a[1185]],[a[3201],a[3998]],{"AUD":a[434],"BRL":a[435],"CAD":a[3451],"CNY":a[437],"ESP":a[2359],"EUR":a[440],"GBP":a[441],"HKD":a[3453],"ILS":a[443],"INR":a[444],"JPY":a[3845],"KRW":a[446],"MXN":a[3455],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[473],a[474],a[3999],a[476],a[2505],a[478],a[479],a[480],a[481],a[482],a[483],a[4000]],[a[2666],a[2641],a[2632],a[488],a[2505],a[1196],a[2634],a[4001],a[4002],a[4003],a[4004],a[4005]],[a[4006],a[4007],a[4008],a[2645],a[4009],a[2646],a[502]],[a[4010],a[4011],a[4012],a[4013],a[4014],a[4015],a[4016]],[a[2654],a[2655],a[4017],a[4018]],{"am":a[2662],"pm":a[4019]},{"decimal":a[429],"group":a[4020],"nan":a[431],"percent":a[432],"infinity":a[433]},{"ATS":a[2664],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[4021],a[4022],a[4023],a[4024],a[4025],a[4026],a[4027],a[4028],a[4029],a[4028],a[4030],a[4031]],[a[4032],a[4033],a[4034],a[4035],a[4025],a[4036],a[4037],a[4038],a[4039],a[4040],a[4041],a[4042]],[a[4043],a[4044],a[4034],a[4035],a[4025],a[4036],a[4037],a[4038],a[4045],a[4046],a[4047],a[4048]],[a[4049],a[4050],a[4051],a[4052],a[4053],a[4054],a[4055]],[a[4056],a[4057],a[4058],a[4059],a[4060],a[4061],a[4062]],[a[4063],a[4064],a[4065],a[4066],a[4067],a[4068],a[4069]],[a[4070],a[4071]],[a[4072],a[4073],a[4074],a[4075]],[a[4076],a[4073]],[a[4077],a[474],a[1189],a[476],a[477],a[478],a[4078],a[4079],a[481],a[482],a[3577],a[1450]],[a[4080],a[4081],a[1194],a[4082],a[477],a[1196],a[4083],a[1198],a[1199],a[4084],a[4085],a[1453]],[a[4086],a[4087],a[4088],a[4089],a[4090],a[4091],a[4092]],[a[4093],a[4094],a[4095],a[4096],a[4097],a[4098],a[4099]],[a[4100],a[4101]],[a[4102],a[4103]],{"am":a[4104],"pm":a[4105]},[a[4106],a[4107],a[4108],a[4109],a[4110],a[4111],a[4112],a[4113],a[4114],a[4115],a[4116],a[4117]],[a[4118],a[4119],a[4108],a[4120],a[4110],a[4121],a[4122],a[4113],a[4123],a[4124],a[4125],a[4126]],[a[4127],a[4128],a[4129],a[4130],a[4131],a[4132],a[4133]],[a[4134],a[4135],a[4136],a[4137],a[4138],a[4139],a[4140]],[a[473],a[4141],a[475],a[4142],a[3018],a[4143],a[4144],a[4145],a[3038],a[482],a[4146],a[1450]],[a[4147],a[4148],a[4149],a[4150],a[4151],a[4152],a[4153],a[4154],a[4155],a[1200],a[4156],a[4157]],[a[4158],a[3590],a[4159],a[4160],a[4161],a[4162],a[4163]],[a[4164],a[4165],a[2774],a[4166],a[4167],a[4168],a[4169]],[a[4170],a[4171]],[a[4172],a[4173]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"GHS":a[632],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NGN":a[3123],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NGN":a[3123],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[13],"month":a[13],"year":a[15],"pattern":a[535]},{"month":a[13],"year":a[15],"pattern":a[464]},[a[4174],a[4175],a[4176],a[4177],a[477],a[4178],a[4179],a[4180],a[4181],a[4182],a[4183],a[4184]],[a[4185],a[4186],a[4187],a[4188],a[477],a[4189],a[4190],a[4191],a[4192],a[4193],a[4194],a[4195]],[a[4196],a[4197],a[4198],a[4199],a[4200],a[4201],a[4202]],[a[4203],a[4204],a[4205],a[4206],a[4207],a[4208],a[4209]],{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[4210],"pattern12":a[4211]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[4212]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[4213]},{"day":a[13],"month":a[14],"pattern":a[4214]},[a[4215],a[4216],a[4217],a[4218],a[4219],a[4220],a[4221],a[4222],a[4223],a[4224],a[4225],a[4226]],[a[4227],a[4228],a[4217],a[4229],a[4219],a[4230],a[4231],a[4232],a[4233],a[4234],a[4235],a[4236]],[a[4237],a[4238],a[4239],a[4240],a[4241],a[4242],a[4243]],[a[4244],a[4245],a[4246],a[4247],a[4248],a[4249],a[4250]],[a[4251],a[4252],a[4253],a[4254],a[4255],a[4256],a[4257]],[a[4258],a[4259],a[1184],a[1185]],[a[4260],a[4261]],{"am":a[4262],"pm":a[4263]},[a[4264],a[4265],a[4266],a[4267],a[4268],a[4269],a[4270],a[4271],a[4272],a[4273],a[4274],a[4275],a[4276],a[4277]],[a[4278],a[4279],a[4280],a[4281],a[4282],a[4283],a[4284],a[4285],a[4286],a[4287],a[4288],a[4275],a[4289],a[4290]],[a[4291]],[a[4292],a[4293],a[4294],a[4295],a[4296],a[4297],a[4298],a[4299],a[4300],a[4301],a[4302],a[4303]],[a[4292],a[4293],a[4304],a[4305],a[4306],a[4307],a[4298],a[4299],a[4300],a[4301],a[4302],a[4303]],[a[4308]],[a[4309],a[166],a[167],a[168],a[169],a[170],a[171],a[172],a[173],a[174],a[175],a[176],a[177],a[178],a[179],a[180],a[181],a[182],a[183],a[184],a[185],a[186],a[187],a[188],a[4310],a[190],a[191],a[192],a[193],a[194],a[195],a[196],a[197],a[198],a[199],a[200],a[201],a[202],a[203],a[204],a[205],a[206],a[207],a[208],a[209],a[210],a[211],a[212],a[213],a[214],a[215],a[216],a[217],a[218],a[219],a[220],a[221],a[222],a[223],a[224],a[225],a[226],a[227],a[228],a[229],a[230],a[231],a[232],a[233],a[234],a[235],a[236],a[237],a[4311],a[239],a[240],a[241],a[242],a[243],a[244],a[245],a[246],a[247],a[248],a[249],a[250],a[251],a[252],a[253],a[254],a[255],a[256],a[257],a[258],a[259],a[260],a[261],a[262],a[263],a[264],a[265],a[266],a[267],a[268],a[269],a[270],a[271],a[272],a[273],a[274],a[275],a[276],a[277],a[278],a[279],a[280],a[281],a[282],a[283],a[284],a[285],a[286],a[287],a[288],a[289],a[290],a[291],a[292],a[293],a[294],a[295],a[296],a[297],a[298],a[299],a[300],a[301],a[302],a[303],a[304],a[305],a[306],a[307],a[308],a[309],a[310],a[311],a[312],a[313],a[314],a[315],a[316],a[317],a[318],a[319],a[320],a[321],a[322],a[323],a[324],a[325],a[401],a[327],a[328],a[329],a[402],a[331],a[332],a[333],a[334],a[335],a[336],a[337],a[338],a[339],a[340],a[341],a[342],a[343],a[344],a[345],a[346],a[347],a[348],a[349],a[350],a[351],a[352],a[353],a[354],a[355],a[356],a[357],a[358],a[359],a[360],a[361],a[362],a[363],a[364],a[365],a[366],a[367],a[368],a[369],a[370],a[371],a[372],a[373],a[374],a[375],a[376],a[377],a[378],a[379],a[380],a[381],a[382],a[383],a[384],a[385],a[386],a[387],a[388],a[389],a[390],a[391],a[392],a[393],a[394],a[395],a[396],a[403],a[404],a[405],a[406]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILP":a[4312],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[3120],"pattern12":a[3121]},[a[4313],a[4314],a[4315],a[4316],a[4317],a[4318],a[4319],a[4320],a[4321],a[4322],a[4323],a[4324],a[4325]],[a[4326],a[4327],a[4328],a[4329],a[4330],a[4331],a[4332],a[4329],a[4333],a[4329],a[4334],a[4335]],[a[4336],a[4337],a[4338],a[4339],a[4340],a[4341],a[4342],a[4343],a[4344],a[4345],a[4346],a[4347]],[a[4348],a[4349],a[4338],a[4350],a[4340],a[4341],a[4351],a[4352],a[4353],a[4354],a[4355],a[4356]],[a[4357],a[4358],a[4359],a[4360],a[4361],a[4362],a[4363]],[a[4364],a[4365],a[1793],a[4366],a[4367],a[4368],a[4369]],[a[4370],a[4371],a[4372],a[4373],a[4374],a[4375],a[4376]],[a[4377],a[4378]],{"am":a[4379],"pm":a[4380]},[a[4381],a[4382],a[4383],a[4384],a[4385],a[1813],a[4386],a[4387],a[4388],a[4389],a[4390],a[4391]],[a[4392]],[a[4393],a[4394],a[4395],a[4396],a[4397],a[4398],a[4399],a[4400],a[4401],a[4402],a[4403],a[4404]],[a[4405],a[4406],a[4407],a[4408],a[4409],a[4410],a[4411],a[4412],a[4413],a[4414],a[4415],a[4416],a[4417],a[4418],a[4419],a[4420],a[4421],a[4422],a[4423],a[4424],a[4425],a[4426],a[4427],a[4428],a[4429],a[4430],a[4431],a[4432],a[4433],a[4434],a[4435],a[4436],a[4437],a[4438],a[4439],a[4440],a[4441],a[4442],a[4443],a[4444],a[4445],a[4446],a[4447],a[4448],a[4449],a[4450],a[4451],a[4452],a[4453],a[4454],a[4455],a[4456],a[4457],a[4458],a[4459],a[4460],a[4461],a[4462],a[4463],a[4464],a[4465],a[4466],a[4467],a[4468],a[4469],a[4470],a[4471],a[4472],a[4473],a[4474],a[4475],a[4476],a[4477],a[4478],a[4479],a[4480],a[4481],a[4482],a[4483],a[4484],a[4485],a[4486],a[4487],a[4488],a[4489],a[4490],a[4491],a[4492],a[4493],a[4494],a[4495],a[4496],a[4497],a[4498],a[4499],a[4500],a[4501],a[4502],a[4503],a[4504],a[4505],a[4506],a[4507],a[4508],a[4509],a[4510],a[4511],a[4512],a[4513],a[4514],a[4515],a[4516],a[4517],a[4518],a[4519],a[4520],a[4521],a[4522],a[4523],a[4524],a[4525],a[4526],a[4527],a[4528],a[4529],a[4530],a[4531],a[4532],a[4533],a[4534],a[4535],a[4536],a[4537],a[4538],a[4539],a[4540],a[4541],a[4542],a[4543],a[4544],a[4545],a[4546],a[4547],a[4548],a[4549],a[4550],a[4551],a[4552],a[4553],a[4554],a[4555],a[4556],a[4557],a[4558],a[4559],a[4560],a[4561],a[4562],a[4563],a[4564],a[4565],a[4566],a[4567],a[4568],a[4569],a[4570],a[4571],a[4572],a[4573],a[4574],a[4575],a[4576],a[4577],a[4578],a[4579],a[4580],a[4581],a[4582],a[4583],a[4584],a[4585],a[4586],a[4587],a[4588],a[4589],a[4590],a[4591],a[4592],a[4593],a[4594],a[4595],a[4596],a[4597],a[4598],a[4599],a[4600],a[4601],a[4602],a[4603],a[4604],a[4605],a[4606],a[4607],a[4608],a[4609],a[4610],a[4611],a[4612],a[4613],a[4614],a[4615],a[4616],a[4617],a[4618],a[4619],a[4620],a[4621],a[4622],a[4623],a[4624],a[4625],a[4626],a[4627],a[4628],a[4629],a[4630],a[4631],a[4632],a[4633],a[4634],a[4635],a[4636],a[4637],a[4638],a[4639],a[4640]],[a[4641],a[4642],a[4643],a[4644],a[4645],a[4646],a[4647],a[4648],a[4649],a[4650],a[4651],a[4652]],{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[1821],"pattern12":a[1822]},{"month":a[13],"year":a[13],"pattern":a[4653]},[a[4654],a[4655],a[4656],a[4657],a[4658],a[4659],a[4660],a[4661],a[4662],a[4663],a[4664],a[4665]],[a[4666],a[4667],a[4668],a[4669],a[4670],a[4671],a[2457],a[4672],a[4673],a[2460],a[4674],a[2461]],[a[4675],a[4676],a[4677],a[4678],a[4679],a[4680],a[4681],a[4682],a[4683],a[4684],a[4685],a[4686]],[a[4687],a[1462]],[a[4688],a[4689],a[4690],a[4691]],[a[4692],a[4693]],[a[4694],a[421]],{"BAM":a[1217],"EUR":a[440],"HRK":a[4695],"JPY":a[1275],"XAF":a[453],"XOF":a[455]},{"EUR":a[440],"HRK":a[4695],"JPY":a[1275],"XAF":a[453],"XOF":a[455]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[4696],"pattern12":a[4697]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[4698]},{"year":a[13],"month":a[14],"day":a[13],"pattern":a[4699]},{"year":a[13],"month":a[15],"day":a[15],"pattern":a[4700]},{"year":a[13],"month":a[13],"pattern":a[4701]},{"year":a[13],"month":a[14],"pattern":a[4701]},{"month":a[13],"day":a[13],"pattern":a[4702]},{"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[25],"pattern12":a[4703]},{"hour":a[13],"minute":a[15],"pattern":a[27],"pattern12":a[4704]},[a[577]],[a[467],a[468],a[397],a[4705],a[397],a[467],a[467],a[469],a[4706],a[470],a[471],a[472]],[a[2543],a[3215],a[4707],a[4708],a[4709],a[4710],a[4711],a[2547],a[4712],a[2549],a[2325],a[2550]],[a[4713],a[4714],a[4715],a[4716],a[4717],a[4718],a[4719],a[4720],a[4721],a[4722],a[2555],a[2556]],[a[3242],a[400],a[1467],a[4723],a[4724],a[1227],a[4725]],[a[4726],a[4727],a[4728],a[4729],a[4730],a[4731],a[4732]],[a[4733],a[4734]],[a[4735],a[4736],a[4737],a[4738]],[a[4739],a[4740]],{"am":a[4741],"pm":a[4742]},[a[4743],a[4744],a[4745],a[4746],a[4747],a[4748],a[4749],a[4750],a[4751],a[4752],a[122],a[4753],a[124],a[4754]],[a[4755]],[a[4756],a[141],a[4757],a[4758],a[4759],a[4760],a[4761],a[4762],a[148],a[4763],a[4764],a[4765]],[a[4766],a[153],a[4767],a[4768],a[4769],a[4770],a[4771],a[4772],a[4773],a[4774],a[4775],a[4776]],[a[4777]],[a[4778],a[421]],{"HUF":a[4779],"JPY":a[1275],"USD":a[530],"XAF":a[453],"XOF":a[455],"XPF":a[456]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[4780],"pattern12":a[4781]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[4782]},{"year":a[13],"month":a[14],"pattern":a[4783]},[a[4784],a[4785],a[4786],a[4787],a[4786],a[4784],a[4784],a[4788],a[4789],a[4784],a[4790],a[4791]],[a[4792],a[4793],a[4794],a[4795],a[4796],a[4797],a[4798],a[4799],a[4800],a[4801],a[4802],a[4803]],[a[4804],a[4805],a[4806],a[4807],a[4808],a[4809],a[4810],a[4811],a[4812],a[4813],a[4814],a[4815]],[a[4816],a[4817],a[4818],a[4819],a[4820],a[4821],a[4822]],[a[4823],a[4824],a[4825],a[4826],a[4827],a[4828],a[4829]],[a[4830],a[4831]],{"am":a[4832],"pm":a[4833]},{"AMD":a[4834],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1826],a[1229],a[1230],a[1284],a[3246],a[1828],a[1829],a[3249],a[1831],a[4835],a[1832],a[1833]],[a[4836],a[4837],a[4838],a[1836],a[3983],a[3138],a[3139],a[4839],a[3738],a[3739],a[2334],a[4840]],[a[1259],a[1766],a[1230],a[4841],a[4842],a[3994],a[1264]],[a[4843],a[4844],a[4845],a[4846],a[4847],a[4848],a[4849]],[a[4850],a[4851]],[a[4852],a[4853]],[a[473],a[474],a[475],a[476],a[477],a[478],a[479],a[4079],a[481],a[482],a[483],a[484]],[a[1192],a[1193],a[4854],a[488],a[477],a[1196],a[2634],a[4855],a[492],a[493],a[494],a[495]],[a[4856],a[4857],a[4858],a[4859],a[98],a[2612],a[100]],[a[4860],a[4861],a[4862],a[4863],a[4864],a[4865],a[4866]],[a[4867],a[397]],[a[4867],a[397],a[4868],a[4869]],[a[140],a[141],a[142],a[143],a[144],a[145],a[146],a[147],a[148],a[4870],a[150],a[151]],[a[152],a[153],a[154],a[155],a[156],a[157],a[158],a[4871],a[4872],a[4873],a[162],a[163]],[a[165],a[166],a[167],a[168],a[169],a[170],a[171],a[172],a[173],a[174],a[175],a[176],a[177],a[178],a[179],a[180],a[181],a[182],a[183],a[184],a[185],a[186],a[187],a[188],a[189],a[190],a[191],a[192],a[193],a[194],a[195],a[196],a[197],a[198],a[199],a[200],a[201],a[202],a[203],a[204],a[205],a[206],a[207],a[208],a[209],a[210],a[211],a[212],a[213],a[214],a[215],a[216],a[217],a[218],a[219],a[220],a[221],a[222],a[223],a[224],a[225],a[226],a[227],a[228],a[229],a[230],a[231],a[232],a[233],a[234],a[235],a[236],a[237],a[238],a[239],a[240],a[241],a[242],a[243],a[244],a[245],a[246],a[247],a[248],a[249],a[250],a[251],a[252],a[253],a[254],a[255],a[256],a[257],a[258],a[259],a[260],a[261],a[262],a[263],a[264],a[265],a[266],a[267],a[268],a[269],a[270],a[271],a[272],a[273],a[274],a[275],a[276],a[277],a[278],a[279],a[280],a[281],a[282],a[283],a[284],a[285],a[286],a[287],a[288],a[289],a[290],a[291],a[292],a[293],a[294],a[295],a[296],a[297],a[298],a[299],a[300],a[301],a[302],a[303],a[304],a[305],a[306],a[307],a[308],a[309],a[310],a[311],a[312],a[313],a[314],a[315],a[316],a[317],a[318],a[319],a[320],a[321],a[322],a[323],a[324],a[325],a[326],a[327],a[328],a[329],a[330],a[331],a[332],a[333],a[334],a[335],a[336],a[337],a[338],a[339],a[340],a[341],a[342],a[343],a[344],a[345],a[346],a[347],a[348],a[349],a[350],a[351],a[352],a[353],a[354],a[355],a[356],a[357],a[358],a[359],a[360],a[361],a[362],a[363],a[364],a[365],a[366],a[367],a[368],a[369],a[370],a[371],a[372],a[373],a[374],a[375],a[376],a[377],a[378],a[379],a[380],a[381],a[382],a[383],a[384],a[385],a[386],a[387],a[388],a[389],a[390],a[391],a[392],a[393],a[394],a[395],a[396],a[403],a[404],a[405],a[406]],[a[4874],a[421]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"IDR":a[4875],"ILS":a[443],"INR":a[3122],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[4876],a[474],a[2854],a[1448],a[2760],a[4877],a[479],a[4878],a[481],a[4879],a[483],a[1450]],[a[4880],a[4881],a[4882],a[4883],a[2760],a[4884],a[4885],a[4886],a[1199],a[4887],a[1201],a[1453]],[a[4888],a[4889],a[4890],a[4891],a[4892],a[4893],a[3038]],[a[4894],a[4895],a[4896],a[4897],a[4898],a[4899],a[4900]],[a[4901],a[4902]],[a[4903],a[4904]],{"am":a[4905],"pm":a[4906]},[a[4907],a[4908],a[4909],a[4910],a[4911],a[4912],a[4913],a[4914],a[4915],a[4916],a[4917],a[4918]],[a[4919],a[4920],a[4921],a[4922],a[4923],a[4924],a[4925]],[a[4926],a[4927],a[4928],a[4929],a[4930],a[4931],a[4932]],[a[4933],a[4934]],{"am":a[4935],"pm":a[4936]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[4937],"pattern12":a[4938]},[a[467],a[468],a[397],a[469],a[397],a[467],a[467],a[4705],a[399],a[470],a[471],a[472]],[a[2543],a[2316],a[2544],a[2545],a[4939],a[4710],a[4711],a[4940],a[2548],a[2549],a[4941],a[2326]],[a[4942],a[4943],a[3609],a[3610],a[4939],a[4944],a[4945],a[4946],a[2553],a[4722],a[4947],a[3611]],[a[4948],a[4949],a[4950],a[3221],a[4951],a[4952],a[4953]],[a[4954],a[4955],a[4956],a[4957],a[4958],a[4959],a[4960]],[a[3623],a[4961],a[4962],a[4963],a[4964],a[4965],a[4966]],[a[4967],a[4968],a[3630],a[3631]],{"am":a[4969],"pm":a[4970]},{"AUD":a[434],"BRL":a[435],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"ISK":a[2587],"JPY":a[445],"KRW":a[446],"NZD":a[448],"THB":a[449],"TWD":a[450],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[532],"pattern12":a[533]},[a[4971]],[a[2499],a[468],a[397],a[469],a[397],a[2499],a[3210],a[469],a[399],a[470],a[471],a[472]],[a[4972],a[1229],a[1230],a[1284],a[4973],a[4974],a[4975],a[1235],a[1236],a[4976],a[1832],a[4977]],[a[4978],a[4979],a[3173],a[4980],a[4981],a[4982],a[4983],a[3175],a[4984],a[4985],a[2334],a[4986]],[a[1259],a[1766],a[1230],a[4841],a[4987],a[3994],a[1264]],[a[4988],a[4989],a[4990],a[4991],a[4992],a[4993],a[3155]],[a[2350],a[2351],a[1184],a[1185]],[a[4994],a[2586]],{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[4995],"pattern12":a[4996]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[4997]},{"year":a[13],"month":a[14],"day":a[13],"pattern":a[4998]},{"year":a[13],"month":a[14],"pattern":a[4999]},{"month":a[14],"day":a[13],"pattern":a[5000]},{"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[25],"pattern12":a[5001]},{"hour":a[13],"minute":a[15],"pattern":a[27],"pattern12":a[5002]},[a[5003]],[a[5004],a[5005],a[5006],a[5007],a[5008],a[5009],a[5010],a[5011],a[5012],a[5013],a[5014],a[5015]],[a[5016],a[5017],a[5018],a[5019],a[5020],a[5021],a[5022],a[5023],a[5024],a[5025],a[5026],a[5027]],[a[5028],a[5029],a[5030],a[5031],a[5032],a[5033],a[5034],a[5035],a[5036],a[5037],a[5038],a[5039],a[5040]],[a[5041],a[5042],a[5043],a[5044],a[5045],a[5046],a[5047],a[5048],a[5049],a[5050],a[5051],a[5052],a[5053]],[a[5054],a[5055],a[5056],a[5057],a[5058],a[5059],a[5060],a[5061],a[5062],a[5063],a[5064],a[5065]],[a[5066],a[5067],a[5068],a[5069],a[5070],a[5071],a[5072]],[a[5073],a[5074],a[5075],a[5076],a[5077],a[5078],a[5079]],[a[5080],a[5081],a[5082]],{"am":a[5083],"pm":a[5084]},[a[638],a[639],a[640],a[641],a[642],a[643],a[644],a[645],a[646],a[647],a[648],a[649],a[650],a[644]],[a[5085],a[5086],a[5087],a[5088],a[5089],a[5090],a[5091],a[5092],a[5093],a[5094],a[5095],a[5096],a[5097],a[5098]],[a[5099],a[5100],a[5101],a[5102],a[5103],a[5104],a[5105],a[5106],a[5107],a[5108],a[5109],a[5110]],[a[5111]],[a[5112],a[5113],a[5114],a[5115],a[5116],a[5117],a[5118],a[5119],a[5120],a[5121],a[5122],a[5123]],[a[5124],a[5125],a[5126],a[5127],a[5128],a[5129],a[5130],a[5131],a[5132],a[5133],a[5134],a[5135],a[5136],a[5137],a[5138],a[5139],a[5140],a[5141],a[5142],a[5143],a[5144],a[5145],a[5146],a[5147],a[5148],a[5149],a[5150],a[5151],a[5152],a[5153],a[5154],a[5155],a[5156],a[5157],a[5158],a[5159],a[5160],a[5161],a[5162],a[5163],a[5164],a[5165],a[5166],a[5167],a[5168],a[5169],a[5170],a[5171],a[5172],a[5173],a[5174],a[5175],a[5176],a[5177],a[5178],a[5179],a[5180],a[5181],a[5182],a[5183],a[5184],a[5185],a[5186],a[5187],a[5188],a[5189],a[5190],a[5191],a[5192],a[5193],a[5194],a[5195],a[5196],a[5197],a[5198],a[5199],a[5200],a[5201],a[5202],a[5203],a[5204],a[5205],a[5206],a[5207],a[5208],a[5209],a[5210],a[5211],a[5212],a[5213],a[5214],a[5215],a[5216],a[5217],a[5218],a[5219],a[5220],a[5221],a[5222],a[5223],a[5224],a[5225],a[5226],a[5227],a[5228],a[5229],a[5230],a[5231],a[5232],a[5233],a[5234],a[5235],a[5236],a[5237],a[5238],a[5239],a[5240],a[5241],a[5242],a[5243],a[5244],a[5245],a[5246],a[5247],a[5248],a[5249],a[5250],a[5251],a[5252],a[5253],a[5254],a[5255],a[5256],a[5257],a[5258],a[5259],a[5260],a[5261],a[5262],a[5263],a[5264],a[5265],a[5266],a[5267],a[5268],a[5269],a[5270],a[5271],a[5272],a[5273],a[5274],a[5275],a[5276],a[5277],a[5278],a[5279],a[5280],a[5281],a[5282],a[5283],a[5284],a[5285],a[5286],a[5287],a[5288],a[5289],a[5290],a[5291],a[5292],a[5293],a[5294],a[5295],a[5296],a[5297],a[5298],a[5299],a[5300],a[5301],a[5302],a[5303],a[5304],a[5305],a[5306],a[5307],a[5308],a[5309],a[5310],a[5311],a[5312],a[5313],a[5314],a[5315],a[5316],a[5317],a[5318],a[5319],a[5320],a[5321],a[5322],a[5323],a[5324],a[5325],a[5326],a[5327],a[5328],a[5329],a[5330],a[5331],a[5332],a[5333],a[5334],a[5335],a[5336],a[5337],a[5338],a[5339],a[5340],a[5341],a[5342],a[5343],a[5344],a[5345],a[5346],a[5347],a[5348],a[5349],a[5350],a[5351],a[5352],a[5353],a[5354],a[5355],a[397],a[398],a[399],a[400]],[a[5124],a[5125],a[5126],a[5127],a[5128],a[5129],a[5130],a[5131],a[5132],a[5133],a[5134],a[5135],a[5136],a[5137],a[5138],a[5139],a[5140],a[5141],a[5142],a[5143],a[5144],a[5145],a[5146],a[5147],a[5148],a[5149],a[5150],a[5151],a[5152],a[5153],a[5154],a[5155],a[5156],a[5157],a[5158],a[5159],a[5160],a[5161],a[5162],a[5163],a[5164],a[5165],a[5166],a[5167],a[5168],a[5169],a[5170],a[5171],a[5172],a[5173],a[5174],a[5175],a[5176],a[5177],a[5178],a[5179],a[5180],a[5181],a[5182],a[5183],a[5184],a[5185],a[5186],a[5187],a[5188],a[5189],a[5190],a[5191],a[5192],a[5193],a[5194],a[5195],a[5196],a[5197],a[5198],a[5199],a[5200],a[5201],a[5202],a[5203],a[5204],a[5205],a[5206],a[5207],a[5208],a[5209],a[5210],a[5211],a[5212],a[5213],a[5214],a[5215],a[5216],a[5217],a[5218],a[5219],a[5220],a[5221],a[5222],a[5223],a[5224],a[5225],a[5226],a[5227],a[5228],a[5229],a[5230],a[5231],a[5232],a[5233],a[5234],a[5235],a[5236],a[5237],a[5238],a[5239],a[5240],a[5241],a[5242],a[5243],a[5244],a[5245],a[5246],a[5247],a[5248],a[5249],a[5250],a[5251],a[5252],a[5253],a[5254],a[5255],a[5256],a[5257],a[5258],a[5259],a[5260],a[5261],a[5262],a[5263],a[5264],a[5265],a[5266],a[5267],a[5268],a[5269],a[5270],a[5271],a[5272],a[5273],a[5274],a[5275],a[5276],a[5277],a[5278],a[5279],a[5280],a[5281],a[5282],a[5283],a[5284],a[5285],a[5286],a[5287],a[5288],a[5289],a[5290],a[5291],a[5292],a[5293],a[5294],a[5295],a[5296],a[5297],a[5298],a[5299],a[5300],a[5301],a[5302],a[5303],a[5304],a[5305],a[5306],a[5307],a[5308],a[5309],a[5310],a[5311],a[5312],a[5313],a[5314],a[5315],a[5316],a[5317],a[5318],a[5319],a[5320],a[5321],a[5322],a[5323],a[5324],a[5325],a[5326],a[5327],a[5328],a[5329],a[5330],a[5331],a[5332],a[5333],a[5334],a[5335],a[5336],a[5337],a[5338],a[5339],a[5340],a[5341],a[5342],a[5343],a[5344],a[5345],a[5346],a[5347],a[5348],a[5349],a[5350],a[5351],a[5352],a[5353],a[5354],a[5355],a[5356],a[5357],a[5358],a[5359]],[a[5360],a[5361],a[5362],a[5363],a[5364],a[5365],a[5366],a[5367],a[5368],a[5369],a[5370],a[5371]],[a[5372],a[5373]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[5374],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[5375],"KRW":a[5376],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"month":a[13],"day":a[13],"year":a[13],"pattern":a[5377]},[a[5378],a[5379],a[5380],a[5381],a[5382],a[5383],a[5384],a[5385],a[5386],a[5387],a[5388],a[5389]],[a[5390],a[5391],a[5392],a[5393],a[5394],a[5395],a[5396]],[a[5397],a[5398]],{"am":a[5399],"pm":a[5400]},[a[473],a[474],a[1189],a[476],a[477],a[478],a[479],a[1190],a[481],a[482],a[483],a[484]],[a[1192],a[1193],a[1194],a[5401],a[477],a[5402],a[5403],a[5404],a[1199],a[1200],a[1201],a[1202]],[a[1203],a[1204],a[1205],a[1206],a[1207],a[5405],a[1209]],[a[5406],a[5407],a[1212],a[5408],a[1214],a[1215],a[1216]],[a[5409],a[5410]],{"am":a[5411],"pm":a[5412]},[a[5413],a[5414],a[5415],a[5416],a[5415],a[5413],a[5413],a[5416],a[5417],a[5418],a[5419],a[5420]],[a[5421],a[5422],a[5423],a[5424],a[5425],a[5426],a[5427],a[5428],a[5429],a[5430],a[5431],a[5432]],[a[5433],a[5434],a[5435],a[5436],a[5437],a[5438],a[5439],a[5440],a[5441],a[5442],a[5443],a[5444]],[a[5445],a[5446],a[5447],a[5448],a[5449],a[5450],a[5451]],[a[5452],a[5453],a[5454],a[5455],a[5456],a[5457],a[5458]],[a[5459],a[5460],a[5461],a[5462],a[5463],a[5464],a[5465]],[a[5466],a[5467]],[a[5468],a[5469]],{"am":a[5470],"pm":a[5471]},[a[5472],a[5473],a[5474],a[5475],a[5476],a[5477],a[5478],a[5479],a[5480],a[5481],a[5482],a[5483],a[5484],a[5485]],[a[5486],a[5487],a[5488],a[5489],a[5490],a[5491],a[5492],a[5493],a[5494],a[5495],a[5496],a[5497]],[a[5498],a[5499],a[5500],a[5501],a[5502],a[5503],a[5504],a[5505],a[5506],a[5507],a[5508],a[5509]],[a[5510],a[5511],a[5512],a[5513],a[5514],a[5515],a[5516],a[5517],a[5518],a[5519],a[5520],a[5521]],{"decimal":a[430],"group":a[529],"nan":a[5522],"percent":a[432],"infinity":a[433]},{"BRL":a[435],"CAD":a[436],"EUR":a[440],"GBP":a[441],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[5523],a[5524],a[5525],a[5526],a[3018],a[4143],a[4144],a[5527],a[5528],a[5529],a[5530],a[5531]],[a[5532],a[5533],a[5534],a[5535],a[5536],a[5537],a[5538],a[5539],a[5540],a[5541],a[5542],a[5543]],[a[5544],a[5545],a[5546],a[5547],a[5548],a[5549],a[5550]],[a[5551],a[5552],a[5553],a[5554],a[5555],a[5556],a[5557]],[a[5558],a[5559]],[a[5560],a[5561]],{"am":a[5562],"pm":a[5563]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DZD":a[3864],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2828],a[5564],a[5565],a[2591],a[5566],a[5567],a[5568],a[5569],a[5570],a[5571],a[5572],a[5573]],[a[5574],a[5575],a[5576],a[5577],a[5578],a[5579],a[5580],a[5581],a[5582],a[5583],a[5584],a[5585]],[a[5586],a[5587],a[5588],a[5589],a[5590],a[5591],a[5592]],[a[5593],a[5594],a[5595],a[5596],a[5597],a[5598],a[5599]],[a[5600],a[5601]],[a[5602],a[5603]],{"am":a[5604],"pm":a[5605]},[a[5606],a[5607],a[5608],a[5609],a[5610],a[5611],a[5612],a[5613],a[5614],a[5615],a[5616],a[5617]],[a[5618],a[5619],a[5620],a[5621],a[5622],a[5623],a[5624]],[a[5625],a[5626],a[5627],a[5628],a[5629],a[5630],a[5631]],[a[5632],a[5633]],[a[5634],a[5635]],{"am":a[5636],"pm":a[5637]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[5638],"pattern12":a[5639]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[5640]},{"month":a[14],"year":a[13],"pattern":a[5641]},[a[473],a[3870],a[475],a[3573],a[2505],a[478],a[479],a[1190],a[3576],a[3875],a[5642],a[5643]],[a[5644],a[5645],a[5646],a[3581],a[5647],a[5648],a[5649],a[5650],a[5651],a[5652],a[5653],a[5654]],[a[5655],a[5656],a[5657],a[5658],a[5659],a[5660],a[1258]],[a[544],a[5661],a[3702],a[5662],a[5663],a[5664],a[1264]],[a[5665],a[5666],a[5667],a[5668],a[5669],a[5670],a[5671]],[a[626],a[5672]],[a[5673],a[5674]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1207],a[2688],a[2689],a[2690],a[2691],a[5675],a[5676]],[a[2694],a[5677],a[5678],a[2697],a[5679],a[5680],a[5681]],[a[2702],a[5682]],{"am":a[5683],"pm":a[5684]},[a[5685],a[5686],a[5687],a[5688],a[5689],a[5690],a[5691],a[5692],a[5693],a[5694],a[5695],a[5696]],[a[5697],a[5698],a[5699],a[5700],a[5701],a[5702],a[5703],a[5704],a[5705],a[5706],a[5707],a[5708]],[a[5709],a[5710],a[5711],a[5712],a[5713],a[5714],a[2855]],[a[2856],a[5715],a[2858],a[5716],a[2860],a[2861],a[5717]],{"am":a[5718],"pm":a[5719]},[a[5720],a[5721],a[5722],a[5723],a[5724],a[5725],a[5726],a[5727],a[5728],a[5729],a[5730],a[5731]],[a[5732],a[5733],a[5734],a[5735],a[5736],a[5737],a[5738],a[5739],a[5740],a[5741],a[5742],a[5743]],[a[5744],a[5745],a[5746],a[5747],a[5748],a[5749],a[5750]],[a[5751],a[5752],a[5753],a[5754],a[5755],a[5756],a[5757]],[a[5758],a[5759]],{"am":a[5760],"pm":a[5761]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"KZT":a[5762],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[15],"month":a[15],"year":a[13],"pattern":a[5763]},{"month":a[15],"year":a[13],"pattern":a[465]},[a[5764],a[5765],a[5766],a[5767],a[5768],a[5769],a[5770],a[5771],a[5772],a[5773],a[648],a[5774]],[a[5775],a[3747],a[3748],a[5776],a[5777],a[5778],a[5779]],[a[1826],a[1229],a[1230],a[1284],a[1827],a[1828],a[1829],a[3249],a[1831],a[1289],a[1832],a[1833]],[a[5780],a[5781],a[5782],a[5783],a[5784],a[1837],a[1838],a[5785],a[5786],a[5787],a[5788],a[5789]],[a[1264],a[5790],a[1230],a[5791],a[5792],a[5793],a[5794]],[a[5795],a[5796],a[5797],a[5798],a[5799],a[5800],a[5801]],[a[5802],a[5803],a[2582],a[2583]],[a[5804],a[5805],a[2582],a[2583]],[a[5806],a[5807],a[2584],a[2585]],{"am":a[5808],"pm":a[5809]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DKK":a[2587],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NOK":a[5810],"NZD":a[448],"SEK":a[5811],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[1491],a[5812],a[5813],a[5814],a[5815],a[5816],a[5817],a[5818],a[5819],a[5820],a[1471],a[5821]],[a[5822],a[5823],a[5824],a[5825],a[5826],a[5827],a[5828],a[5829],a[5830],a[5831],a[5832],a[5833]],[a[1475],a[1471],a[5821],a[5834],a[5835],a[5836],a[5837]],[a[5838],a[5839],a[5840],a[5841],a[5842],a[5843],a[5844]],[a[126],a[5845]],[a[5846],a[5847]],{"am":a[5848],"pm":a[5849]},{"day":a[13],"month":a[13],"year":a[13],"pattern":a[3207]},[a[5850]],[a[5851],a[5852],a[5853],a[5854],a[5855],a[5856],a[5857],a[5858],a[5859],a[5860],a[5861],a[5862]],[a[5863],a[5864],a[5865],a[5866],a[5867],a[5868],a[5869]],[a[5870],a[5871],a[5872]],[a[5873],a[5874]],{"am":a[5875],"pm":a[5876]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KHR":a[5877],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[13],"month":a[14],"year":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[5878],"pattern12":a[5879]},{"day":a[13],"month":a[14],"year":a[13],"weekday":a[14],"pattern":a[5880]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[5881]},[a[5882],a[5883],a[5884],a[5885],a[5886],a[5887],a[5888],a[5889],a[5890],a[5891],a[5892],a[5893]],[a[5894],a[5895],a[5884],a[5896],a[5886],a[5887],a[5897],a[5898],a[5899],a[5900],a[5901],a[5902]],[a[5903],a[5904],a[5905],a[5906],a[5886],a[5907],a[5908],a[5909],a[5910],a[5911],a[5912],a[5913]],[a[5914],a[5915],a[5916],a[5917],a[5918],a[5919],a[5920]],[a[5921],a[5922],a[5923],a[5924],a[5925],a[5926],a[5927]],[a[5928],a[5929],a[5930],a[5931],a[5932],a[5933],a[5934]],[a[5935],a[5936],a[5937]],[a[5938],a[5939]],{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[5940],"pattern12":a[5941]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[5942]},{"year":a[13],"month":a[14],"day":a[13],"pattern":a[5943]},{"year":a[13],"month":a[13],"day":a[13],"pattern":a[4700]},{"year":a[13],"month":a[14],"pattern":a[5944]},[a[5945]],[a[5946],a[5947],a[5948],a[5949],a[5950],a[5951],a[5952],a[5953],a[5954],a[5955],a[5956],a[5957],a[5958]],[a[5959],a[5960],a[5961],a[5962],a[5963],a[5964],a[5965],a[5966],a[5967],a[5968],a[5969],a[5970]],[a[5971],a[5972],a[5973],a[5974],a[5975],a[5976],a[5977],a[5978],a[5979],a[5980],a[5981],a[5982],a[5983]],[a[5984],a[5985],a[5986],a[5987],a[5988],a[5989],a[5990]],[a[5991],a[5992],a[5993],a[5994],a[5995],a[5996],a[5997]],[a[5998],a[5999]],[a[6000],a[6001]],{"am":a[6002],"pm":a[6003]},[a[6004],a[6005],a[6006],a[6007],a[6008],a[6009],a[6010],a[6011],a[6012],a[6013],a[6014],a[6015],a[6016],a[6017]],[a[6018],a[6019],a[6020],a[6021],a[6022],a[6023],a[6024],a[6025],a[6026],a[6027],a[6028],a[6029],a[6030],a[6031],a[6032],a[6033],a[6034],a[6035],a[6036],a[6037],a[6038],a[6039],a[6040],a[6041],a[6042],a[6043],a[6044],a[6045],a[6046],a[6047],a[6048],a[6049],a[6050],a[6051],a[6052],a[6053],a[6054],a[6055],a[6056],a[6057],a[6058],a[6059],a[6060],a[6061],a[6062],a[6063],a[6064],a[6065],a[6066],a[6067],a[6068],a[6069],a[6070],a[6071],a[6072],a[6073],a[6074],a[6075],a[6076],a[6077],a[6078],a[6079],a[6080],a[6081],a[6082],a[6083],a[6084],a[6085],a[6086],a[6087],a[6088],a[6089],a[6090],a[6091],a[6092],a[6093],a[6094],a[6095],a[6096],a[6097],a[6098],a[6099],a[6100],a[6101],a[6102],a[6103],a[6104],a[6105],a[6106],a[6107],a[6108],a[6109],a[6110],a[6111],a[6112],a[6113],a[6114],a[6115],a[6116],a[6117],a[6118],a[6119],a[6120],a[6121],a[6122],a[6123],a[6124],a[6125],a[6126],a[6127],a[6128],a[6129],a[6130],a[6131],a[6132],a[6133],a[6134],a[6135],a[6136],a[6137],a[6138],a[6139],a[6140],a[6141],a[6142],a[6143],a[6144],a[6145],a[6146],a[6147],a[6148],a[6149],a[6150],a[6151],a[6152],a[6153],a[6154],a[6155],a[6156],a[6157],a[6158],a[6159],a[6160],a[6161],a[6162],a[6163],a[6164],a[6165],a[6166],a[6167],a[6168],a[6169],a[6170],a[6171],a[6172],a[6173],a[6174],a[6175],a[6176],a[6177],a[6178],a[6179],a[6180],a[6181],a[6182],a[6183],a[6184],a[6185],a[6186],a[6187],a[6188],a[6189],a[6190],a[6191],a[6192],a[6193],a[6194],a[6195],a[6196],a[6197],a[6198],a[6199],a[6200],a[6201],a[6202],a[6203],a[6204],a[6205],a[6206],a[6207],a[6208],a[6209],a[6210],a[6211],a[6212],a[6213],a[6214],a[6215],a[6216],a[6217],a[6218],a[6219],a[6220],a[6221],a[6222],a[6223],a[6224],a[6225],a[6226],a[6227],a[6228],a[6229],a[6230],a[6231],a[6232],a[6233],a[6234],a[6235],a[6236],a[6237],a[6238],a[6239],a[6240],a[6241],a[6242],a[6243],a[6244],a[6245],a[6246],a[6247],a[6248],a[6249],a[6250],a[6251],a[6252],a[6253]],[a[6254],a[6255]],{"positivePattern":a[425],"negativePattern":a[6256]},[a[6257],a[1780],a[4338],a[6258],a[1783],a[4341],a[6259],a[6260],a[6261],a[6262],a[6263],a[6264]],[a[4364],a[4365],a[6265],a[4366],a[4367],a[4368],a[4369]],[a[6266],a[4371],a[6267],a[4373],a[4374],a[4375],a[4376]],[a[6268],a[6269]],{"am":a[6270],"pm":a[6271]},[a[6272],a[6273],a[6274],a[3374],a[6275],a[6276],a[6277],a[3378],a[6278],a[6279],a[3381],a[3382]],[a[6280],a[6281],a[6282],a[6283],a[6284],a[6285],a[6286]],[a[6287],a[6288],a[6282],a[6283],a[6284],a[6285],a[6286]],[a[6289],a[6290]],[a[6291],a[6292]],[a[829],a[830],a[6293],a[6294],a[3418],a[3419],a[835],a[836],a[837],a[838],a[6295],a[6296]],{"decimal":a[429],"group":a[430],"nan":a[431],"percent":a[432],"infinity":a[6297]},[a[6298],a[6299],a[1194],a[6300],a[477],a[1196],a[1197],a[1198],a[1199],a[1200],a[1201],a[1202]],[a[1203],a[1204],a[6301],a[1206],a[1207],a[5405],a[1209]],[a[6302],a[6303],a[6304],a[6305],a[1214],a[1215],a[6306]],[a[6307],a[6308]],{"am":a[6309],"pm":a[6310]},[a[6311],a[6312],a[6313],a[6314],a[6315],a[6316],a[6317],a[6318],a[6319],a[6320],a[6321],a[6322]],[a[6323],a[6324],a[6325],a[6326],a[6327],a[6328],a[6329],a[6330],a[6331],a[6332],a[6333],a[6334]],[a[3337],a[6335],a[6336],a[6337],a[6338],a[6339],a[6340]],[a[6341],a[6342],a[6343],a[6344],a[6345],a[6346],a[6347]],[a[6348],a[6349]],[a[6350],a[6351]],{"am":a[6352],"pm":a[6353]},[a[473],a[6354],a[6355],a[476],a[6356],a[478],a[479],a[6357],a[6358],a[482],a[483],a[4000]],[a[6359],a[6360],a[6361],a[6362],a[6356],a[6363],a[6364],a[6365],a[4002],a[4003],a[4004],a[4005]],[a[2520],a[3027],a[505],a[2522],a[1755],a[3031],a[509]],[a[4006],a[2644],a[498],a[6366],a[4009],a[2646],a[502]],[a[6367],a[6368],a[6369],a[6370],a[6371],a[6372],a[6373]],[a[6374],a[6375],a[2656],a[2657]],[a[6376],a[6377],a[6378],a[2661]],{"am":a[6379],"pm":a[6380]},{"decimal":a[430],"group":a[529],"nan":a[3640],"percent":a[432],"infinity":a[433]},[a[1745],a[6381],a[2527],a[1748],a[2522],a[6382],a[6383],a[4092],a[6384],a[6385],a[1755],a[6386]],[a[6387],a[6388],a[6389],a[6390],a[6391],a[6392],a[6393],a[6394],a[6395],a[6396],a[6397],a[6398]],[a[1772],a[1773],a[6399],a[6400],a[6401],a[6402],a[2529]],[a[6403],a[6404],a[6405],a[6406],a[6407],a[6408],a[6409]],{"year":a[13],"day":a[13],"month":a[14],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[6410],"pattern12":a[6411]},{"year":a[13],"day":a[13],"month":a[14],"weekday":a[14],"pattern":a[6412]},{"year":a[13],"day":a[13],"month":a[14],"pattern":a[6413]},{"year":a[13],"month":a[14],"pattern":a[6414]},[a[6415],a[6416],a[6417],a[6418],a[6417],a[6419],a[6419],a[6418],a[6420],a[6421],a[6422],a[6423]],[a[6424],a[6425],a[6426],a[1519],a[1331],a[6427],a[6428],a[1522],a[6429],a[1524],a[6430],a[1526]],[a[6431],a[6432],a[1329],a[6433],a[1331],a[6434],a[6435],a[1334],a[6436],a[6437],a[6438],a[6439]],[a[6440],a[6441],a[6442],a[6443],a[6444],a[6445],a[6446]],[a[6447],a[6448],a[6449],a[6450],a[6451],a[6452],a[6453]],[a[6454],a[5759]],[a[6454],a[5759],a[6455]],[a[6456],a[5759]],{"am":a[6457],"pm":a[6458]},{"decimal":a[430],"group":a[529],"nan":a[6459],"percent":a[432],"infinity":a[433]},{"CNY":a[437],"EUR":a[440],"JPY":a[445],"KGS":a[6460],"THB":a[449],"VND":a[452],"XAF":a[453],"XOF":a[455],"XPF":a[456]},[a[6461],a[6462],a[6463],a[6464],a[6465],a[6466],a[6467],a[6468],a[6469],a[6470],a[6471],a[6472]],[a[6473],a[6474],a[6475],a[6476],a[6477],a[6478],a[6479],a[6480],a[6481],a[6482],a[6483],a[6484]],[a[6485],a[6486],a[6487],a[6488],a[1207],a[1208],a[6489]],[a[6490],a[6491],a[6492],a[6493],a[6494],a[6495],a[6496]],[a[6497],a[6498]],[a[6499],a[6500]],{"am":a[6501],"pm":a[6502]},[a[473],a[474],a[475],a[6503],a[2854],a[4877],a[479],a[4145],a[6504],a[6505],a[483],a[484]],[a[6506],a[6507],a[6508],a[6509],a[6510],a[6363],a[6511],a[6512],a[6513],a[6514],a[1201],a[1202]],[a[100],a[6515],a[6516],a[6517],a[6518],a[6519],a[6520]],[a[6521],a[6522],a[6523],a[6524],a[6525],a[6526],a[6527]],[a[6528],a[6529]],[a[6530],a[6531],a[6532],a[6533],a[6534],a[6535],a[6536],a[6537],a[6538],a[6539],a[6540],a[6541]],[a[6542],a[6543],a[6544],a[6545],a[6546],a[6547],a[6548]],[a[1282],a[6549],a[6550],a[6551],a[3246],a[6552],a[6553],a[6554],a[6555],a[6556],a[6557],a[6558]],[a[6559],a[6560],a[6561],a[6562],a[6563],a[6564],a[6565],a[6566],a[6567],a[6568],a[6569],a[6570]],[a[6571],a[6572],a[6573],a[6574],a[6575],a[6576],a[6577]],[a[6578],a[6579],a[6580],a[6581],a[6582],a[6583],a[6584]],[a[6585],a[6586]],[a[6587],a[6588]],{"am":a[6589],"pm":a[6590]},{"AOA":a[3229],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CDF":a[3863],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CDF":a[3863],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[6591]],[a[6592],a[6593],a[6594],a[6595],a[6596],a[6597],a[6598],a[6599],a[6600],a[6601],a[6602],a[6603],a[6604]],[a[6592],a[6593],a[6594],a[6595],a[6596],a[6605],a[6598],a[6599],a[6600],a[6601],a[6602],a[6603],a[6606]],[a[6607],a[6608],a[6609],a[6610],a[6611],a[6612],a[6613],a[6614],a[6615],a[6616],a[6617],a[6618],a[6619]],[a[6620],a[6621],a[6622],a[6623],a[6624],a[6625],a[6626],a[6627],a[6628],a[6629],a[6630],a[6631]],[a[6632],a[6633],a[6634],a[6635],a[6636],a[6637],a[6638],a[6639],a[6640],a[6641],a[6642],a[6643]],[a[6644],a[6645],a[6646],a[6647],a[6648],a[6649],a[6650]],[a[6651],a[6652]],{"am":a[6653],"pm":a[6654]},[a[6655],a[6656],a[6657],a[6658],a[6659],a[6660],a[6661],a[6662],a[6663],a[6664],a[6665],a[6666],a[6667],a[6668]],[a[6655],a[6656],a[6657],a[6658],a[6659],a[6660],a[6661],a[6662],a[6669],a[6664],a[6665],a[6666],a[6667],a[6670]],[a[6671],a[6672],a[6673],a[6674],a[6675],a[6676],a[6677],a[6678],a[6679],a[6680],a[6681],a[6682]],[a[6671],a[6672],a[6673],a[6674],a[6675],a[6676],a[6677],a[6678],a[6679],a[6683],a[6681],a[6682]],[a[6623]],[a[6684],a[6685],a[6686],a[6687],a[6688],a[6689],a[6690],a[6691],a[6692],a[6693],a[6694],a[6695]],[a[6696],a[6697],a[6698],a[6699],a[6700],a[6701],a[6702],a[6703],a[6704],a[6705],a[6706],a[6707]],[a[6708],a[6709],a[6710],a[6711],a[6712],a[6713],a[6714],a[6715],a[6716],a[6717],a[6718],a[6719],a[6720],a[6721],a[6722],a[6723],a[6724],a[6725],a[6726],a[6727],a[6728],a[6729],a[6730],a[6731],a[6732],a[6733],a[6734],a[6735],a[6736],a[6737],a[6738],a[6739],a[6740],a[6741],a[6742],a[6743],a[6744],a[6745],a[6746],a[6747],a[6748],a[6749],a[6750],a[6751],a[6752],a[6753],a[6754],a[6755],a[6756],a[6757],a[6758],a[6759],a[6760],a[6761],a[6762],a[6763],a[6764],a[6765],a[6766],a[6767],a[6768],a[6769],a[6770],a[6771],a[6772],a[6773],a[6774],a[6775],a[6776],a[6777],a[6778],a[6779],a[6780],a[6781],a[6782],a[6783],a[6784],a[6785],a[6786],a[6787],a[6788],a[6789],a[6790],a[6791],a[6792],a[6793],a[6794],a[6795],a[6796],a[6797],a[6798],a[6799],a[6800],a[6801],a[6802],a[6803],a[6804],a[6805],a[6806],a[6807],a[6808],a[6809],a[6810],a[6811],a[6812],a[6813],a[6814],a[6815],a[6816],a[6817],a[6818],a[6819],a[6820],a[6821],a[6822],a[6823],a[6824],a[6825],a[6826],a[6827],a[6828],a[6829],a[6830],a[6831],a[6832],a[6833],a[6834],a[6835],a[6836],a[6837],a[6838],a[6839],a[6840],a[6841],a[6842],a[6843],a[6844],a[6845],a[6846],a[6847],a[6848],a[6849],a[6850],a[6851],a[6852],a[6853],a[6854],a[6855],a[6856],a[6857],a[6858],a[6859],a[6860],a[6861],a[6862],a[6863],a[6864],a[6865],a[6866],a[6867],a[6868],a[6869],a[6870],a[6871],a[6872],a[6873],a[6874],a[6875],a[6876],a[6877],a[6878],a[6879],a[6880],a[6881],a[6882],a[6883],a[6884],a[6885],a[6886],a[6887],a[6888],a[6889],a[6890],a[6891],a[6892],a[6893],a[6894],a[6895],a[6896],a[6897],a[6898],a[6899],a[6900],a[6901],a[6902],a[6903],a[6904],a[6905],a[6906],a[6907],a[6908],a[6909],a[6910],a[6911],a[6912],a[6913],a[6914],a[6915],a[6916],a[6917],a[6918],a[6919],a[6920],a[6921],a[6922],a[6923],a[6924],a[6925],a[6926],a[6927],a[6928],a[6929],a[6930],a[6931],a[6932],a[6933],a[6934],a[6935],a[6936],a[6937],a[6938],a[6939],a[6940],a[6941],a[6942],a[6943]],[a[6944],a[6945],a[6946],a[6947],a[6948],a[6949],a[6950],a[6951],a[6952],a[6953],a[6954],a[6955]],[a[6956],a[6945],a[6946],a[6947],a[6957],a[6949],a[6950],a[6951],a[6958],a[6953],a[6959],a[6955]],[a[6960]],[a[6961],a[421]],{"decimal":a[430],"group":a[429],"nan":a[6962],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"LAK":a[6963],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[16],"pattern12":a[17]},{"month":a[15],"day":a[13],"pattern":a[24]},[a[399],a[3242],a[1467],a[3046],a[2499],a[3046],a[3210],a[531],a[531],a[399],a[3210],a[2499]],[a[6964],a[6965],a[6966],a[6967],a[6968],a[6969],a[6970],a[6971],a[6972],a[6973],a[6974],a[6975]],[a[6976],a[6977],a[6978],a[6979],a[6980],a[6981],a[6982],a[6983],a[6984],a[6985],a[6986],a[6987]],[a[6988],a[6989],a[6990],a[6991],a[6992],a[6993],a[6994]],[a[6995],a[6996],a[6997],a[6998],a[6999],a[7000],a[7001]],[a[7002],a[7003],a[7004],a[7005],a[7006],a[7007],a[7008]],[a[4688],a[7009],a[7010],a[7011]],[a[4688],a[7009],a[7012],a[7013]],[a[7014],a[7015],a[7016],a[7017]],{"am":a[7018],"pm":a[7019]},{"EUR":a[440],"LTL":a[7020]},[a[7021],a[3873],a[7022],a[7023],a[7024],a[7025],a[7026],a[7027],a[7028],a[1773],a[2592],a[7029]],[a[7030],a[7031],a[7032],a[7033],a[7034],a[7035],a[7036],a[7037],a[7038],a[7039],a[7040],a[7041]],[a[7024],a[7042],a[7043],a[7044],a[7045],a[7046],a[7047]],[a[7048],a[7049],a[7050],a[7051],a[7052],a[7053],a[7054]],[a[7055],a[7056]],[a[7057],a[7058]],{"am":a[7059],"pm":a[7060]},[a[7061],a[7062],a[7063],a[7064],a[7065],a[7066],a[7067],a[7068],a[7069],a[7070],a[7071],a[7072]],[a[7073],a[7074],a[7075],a[7076],a[7077],a[7078],a[7079],a[7080],a[7081],a[7082],a[7083],a[7084]],[a[7085],a[7086],a[7087],a[7088],a[7089],a[7090],a[7091]],[a[7092],a[7093],a[7094],a[7095],a[7096],a[7097],a[7098]],[a[7099],a[7100]],{"am":a[7101],"pm":a[7102]},[a[473],a[474],a[475],a[476],a[477],a[478],a[479],a[1190],a[481],a[482],a[483],a[484]],[a[7103],a[7104],a[7105],a[7106],a[4161],a[7107],a[7108]],[a[7109],a[1211],a[1212],a[1213],a[7110],a[7111],a[1216]],[a[7112],a[7113]],{"positivePattern":a[425],"negativePattern":a[7114]},[a[3727],a[3215],a[2551],a[2545],a[7115],a[7116],a[7117],a[2547],a[3169],a[2549],a[2325],a[2550]],[a[7118],a[7119],a[2551],a[7120],a[7115],a[7121],a[7122],a[7123],a[7124],a[7125],a[7126],a[7127]],[a[7128],a[6989],a[7129],a[6991],a[7130],a[7131],a[2752]],[a[7132],a[7133],a[7134],a[7135],a[7136],a[7137],a[7138]],[a[7139],a[7140],a[7141],a[7142]],[a[7139],a[7140]],[a[7143],a[7144]],{"am":a[7145],"pm":a[7146]},[a[7147],a[7148],a[7149],a[7150],a[7151],a[7152],a[7153],a[7154],a[7155],a[7156],a[7157],a[7158],a[7159],a[7160]],[a[7161],a[7162],a[7163],a[7164],a[7165],a[7166],a[7167],a[7168],a[7169],a[7170],a[7171],a[7172]],{"decimal":a[430],"group":a[529],"nan":a[7173],"percent":a[432],"infinity":a[433]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"LVL":a[7174],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7175],a[7176],a[7177],a[7178],a[7179],a[7180],a[7181],a[7182],a[7183],a[7184],a[7185],a[7186]],[a[7187],a[7188],a[7189],a[7190],a[7191],a[7192],a[7193],a[7194],a[7195],a[7196],a[7197],a[7198]],[a[7199],a[6491],a[7200],a[7201],a[7202],a[7203],a[7204]],[a[5600],a[7205]],[a[7206],a[7207]],{"am":a[7208],"pm":a[7209]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"TZS":a[1225],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7210],a[7211],a[7212],a[7213],a[7214],a[7215],a[7216],a[7217],a[7218],a[2389],a[7219],a[7220]],[a[7221],a[7222],a[1194],a[7223],a[7224],a[7225],a[7226],a[7227],a[1199],a[7228],a[1201],a[7229]],[a[7230],a[7231],a[7232],a[7233],a[7234],a[5689],a[7235]],[a[2856],a[7236],a[7237],a[7238],a[7239],a[7240],a[1216]],[a[2863],a[7241]],[a[7242],a[7243]],{"am":a[7244],"pm":a[7245]},[a[1591],a[1283],a[1230],a[7246],a[3146],a[7247],a[7248],a[3980],a[1831],a[1289],a[1832],a[1600]],[a[7249],a[7250],a[3609],a[3736],a[3146],a[7247],a[7251],a[3980],a[7252],a[7253],a[7254],a[7255]],[a[7256],a[7257],a[1230],a[4841],a[7258],a[7259],a[6340]],[a[7260],a[7261],a[3748],a[7262],a[7263],a[7264],a[7265]],[a[7266],a[7267]],[a[7268],a[7269]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MUR":a[3122],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[473],a[474],a[475],a[476],a[7270],a[7271],a[7272],a[7273],a[481],a[482],a[483],a[484]],[a[7274],a[7275],a[7276],a[7277],a[7270],a[7278],a[7279],a[7280],a[7281],a[7282],a[7283],a[7284]],[a[7285],a[7286],a[96],a[7287],a[7288],a[7289],a[7290]],[a[7291],a[7292],a[2774],a[7293],a[7294],a[7295],a[7296]],[a[7297],a[7298]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MGA":a[3118],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7299],a[7300],a[7301],a[7302],a[5567],a[7303],a[100],a[1474],a[1475],a[1476],a[7304],a[7305]],[a[7306],a[7307],a[7308],a[7309],a[7310],a[7311],a[7312],a[7313],a[7314],a[7315],a[7316],a[7317]],[a[100],a[1204],a[1205],a[1206],a[2769],a[5405],a[1209]],[a[7318],a[1211],a[1212],a[1213],a[7319],a[1215],a[1216]],[a[7320],a[7321]],[a[7322],a[7323]],{"am":a[7324],"pm":a[7325]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"MZN":a[7326],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7327],a[7328],a[7329],a[7330],a[7331],a[7332],a[7333],a[7334],a[7335],a[647],a[648],a[649]],[a[7336],a[7337],a[7338],a[7339],a[7340],a[7341],a[7342],a[7343],a[7344],a[7345],a[7346],a[7347]],[a[7348],a[7337],a[7338],a[7339],a[7340],a[7341],a[7342],a[7343],a[7344],a[7345],a[7346],a[7347]],[a[638],a[639],a[640],a[641],a[642],a[643],a[644]],[a[7349],a[7350],a[7351],a[7352],a[7353],a[7354],a[7355]],{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7356],"pattern12":a[7357]},[a[7358],a[1511],a[1401],a[1512],a[1401],a[7358],a[7358],a[1512],a[1398],a[1514],a[1515],a[1516]],[a[7359],a[6425],a[6426],a[1519],a[1922],a[7360],a[7361],a[1522],a[1523],a[1524],a[1525],a[1526]],[a[7362],a[1528],a[1329],a[1529],a[1922],a[1932],a[1933],a[1334],a[1530],a[1531],a[1532],a[1533]],[a[7363],a[7364],a[640],a[7365],a[7366],a[7367],a[644]],[a[7363],a[7364],a[7368],a[7365],a[7366],a[7367],a[7369]],[a[7370],a[1537],a[1538],a[7371],a[7372],a[7373],a[7374]],[a[1546],a[1952]],{"am":a[7375],"pm":a[7376]},{"BRL":a[435],"CAD":a[436],"EUR":a[440],"MKD":a[7377],"MXN":a[447],"USD":a[451],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7378],a[7379],a[7380],a[7381],a[7382],a[7383],a[7384],a[7385],a[7386],a[7387],a[7388],a[7389]],[a[7390],a[7391],a[7392],a[7393],a[7382],a[7383],a[7384],a[7394],a[7395],a[7396],a[7397],a[7398]],[a[7399],a[7400],a[7401],a[7402],a[7403],a[7404],a[7405]],[a[7406],a[7407],a[7408],a[7409],a[7410],a[7411],a[7412]],[a[7413],a[7414],a[7415],a[7416],a[7417],a[7418],a[7419]],[a[7420],a[7421],a[7422],a[7423]],[a[7424],a[7425]],[a[7426],a[7427],a[7428],a[7429],a[7430],a[7431],a[7432],a[7433],a[7434],a[7435],a[7436],a[7437]],[a[7438]],[a[7439],a[7440],a[7441],a[7442],a[7443],a[7444],a[7445],a[7446],a[7447],a[7448],a[7449],a[7450]],[a[7451]],{"decimal":a[429],"group":a[430],"nan":a[7452],"percent":a[432],"infinity":a[433]},[a[7453],a[7454],a[7455],a[7456],a[7457],a[7458],a[7459],a[7460],a[7461],a[7462],a[7463],a[7464]],[a[7465],a[7466],a[7467],a[7468],a[7469],a[7470],a[7471],a[7472],a[7473],a[7474],a[7475],a[7476]],[a[7477],a[7478],a[7479],a[7480],a[7481],a[7482],a[7483]],[a[7484],a[7485],a[7486],a[7487],a[7488],a[7489],a[7490]],[a[7491],a[7492]],[a[7493],a[7494],a[7495],a[7496]],[a[7497],a[7498]],{"am":a[7499],"pm":a[7500]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"JPY":a[445],"KRW":a[446],"MNT":a[7501],"MXN":a[447],"NZD":a[448],"THB":a[449],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7502],"pattern12":a[7503]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[7504]},[a[7505],a[7506],a[4328],a[7507],a[1783],a[4331],a[4332],a[7508],a[7509],a[7508],a[7510],a[7511]],[a[7512],a[7513],a[4338],a[7514],a[1783],a[4341],a[6259],a[7515],a[7516],a[7517],a[7518],a[7519]],[a[6257],a[1780],a[4338],a[6258],a[1783],a[4341],a[6259],a[7520],a[7521],a[7522],a[6263],a[6264]],[a[4370],a[4371],a[7523],a[4373],a[4374],a[4375],a[4376]],[a[7524],a[1806],a[7525],a[7526]],[a[7527],a[7528]],{"am":a[7529],"pm":a[7530]},[a[7531]],[a[3212],a[7532],a[7533],a[7534],a[2522],a[4162],a[3213],a[7535],a[7536],a[2753],a[7537],a[7538]],[a[473],a[474],a[1189],a[476],a[477],a[478],a[479],a[7539],a[481],a[482],a[483],a[1450]],[a[1192],a[1193],a[1189],a[488],a[477],a[478],a[1197],a[7540],a[492],a[493],a[494],a[7541]],[a[467],a[468],a[397],a[469],a[397],a[467],a[467],a[470],a[399],a[470],a[471],a[472]],[a[7542],a[7543],a[2752],a[7544],a[7545],a[4162],a[509]],[a[7546],a[7547],a[4858],a[4859],a[7548],a[2612],a[100]],[a[7549],a[7550],a[4862],a[4863],a[7551],a[7552],a[4866]],[a[7553],a[7554]],{"am":a[7555],"pm":a[7556]},{"AUD":a[434],"BND":a[530],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"MYR":a[7557],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"MYR":a[7557],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"MYR":a[7557],"NZD":a[448],"SGD":a[530],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[473],a[7558],a[475],a[476],a[7559],a[7560],a[7561],a[7562],a[3576],a[7563],a[483],a[7564]],[a[7565],a[7566],a[7567],a[488],a[7568],a[7569],a[7570],a[7571],a[7572],a[7573],a[7574],a[7575]],[a[7576],a[7577],a[7578],a[7579],a[7580],a[7581],a[2771]],[a[7582],a[7583],a[7584],a[7585],a[7586],a[7587],a[7588]],[a[7589],a[7590],a[7591],a[7592]],[a[7593],a[7594]],{"am":a[7595],"pm":a[7596]},[a[7597],a[7598],a[7599],a[7600],a[7601],a[7602],a[7603],a[7604],a[7605],a[7606],a[7607],a[7608]],[a[7609],a[7610],a[7611],a[7612],a[7613],a[7614],a[7615],a[7616],a[7617],a[7618],a[7619],a[7620]],[a[7621],a[7622],a[7623],a[7624],a[7625],a[7626],a[7627]],[a[7628],a[7629],a[7630],a[7631],a[7632],a[7633],a[7634]],[a[2622],a[7635]],[a[7636],a[7637]],{"am":a[7638],"pm":a[7639]},[a[7640],a[7641],a[7642],a[7643],a[7644],a[7645],a[7646],a[7647],a[7648],a[7649],a[7650],a[7651]],[a[7652],a[7653],a[7654],a[7655],a[7656],a[7657],a[7658]],[a[7659],a[7660],a[7661],a[7662]],[a[7663],a[7664]],{"am":a[7665],"pm":a[7666]},[a[7667]],{"decimal":a[429],"group":a[430],"nan":a[7668],"percent":a[432],"infinity":a[433]},{},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MMK":a[1467],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7669],a[7670],a[7671],a[7672],a[7673],a[7674],a[7675],a[7676],a[7677],a[7678],a[7679],a[7680]],[a[7681],a[504],a[2755],a[7682],a[507],a[3031],a[3038]],[a[7683],a[7684],a[7685],a[7686],a[7687],a[7688],a[7689]],[a[7690],a[7691]],{"am":a[7692],"pm":a[7693]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NAD":a[530],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7694],"pattern12":a[7695]},[a[2543],a[2316],a[2544],a[2545],a[3246],a[2546],a[2321],a[2547],a[2548],a[2549],a[2325],a[2326]],[a[1834],a[1835],a[3609],a[1836],a[3246],a[1837],a[1838],a[2552],a[2553],a[2554],a[2555],a[3611]],[a[7696],a[3220],a[7697],a[7698],a[7699],a[7700],a[7701]],[a[2580],a[2581],a[7702],a[2579]],[a[2580],a[2581],a[7702],a[7703]],[a[2580],a[2581],a[7704],a[7705]],{"EUR":a[440],"GBP":a[441],"NOK":a[2587],"XOF":a[455]},[a[7706],a[7707],a[7708],a[7709],a[7710],a[7711],a[7712],a[7713],a[7714],a[2594],a[7715],a[7716]],[a[7717],a[7718],a[7719],a[7720],a[7721],a[7722],a[7723],a[7724],a[7725],a[7726],a[7727],a[7728]],[a[7681],a[7729],a[2771],a[1473],a[7730],a[7731],a[7732]],[a[7733],a[7734],a[7735],a[7736],a[7737],a[7738],a[7739]],[a[7740],a[7741]],{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7742],"pattern12":a[7743]},[a[7744],a[7745],a[7746],a[7747],a[7748],a[7749],a[7750],a[7751],a[7752],a[7753],a[7754],a[7755]],[a[4348],a[7756],a[4338],a[7757],a[1783],a[1784],a[4351],a[7758],a[7759],a[7760],a[7761],a[7762]],[a[4348],a[7763],a[4338],a[7764],a[4340],a[1784],a[4351],a[4352],a[7759],a[7760],a[7761],a[7765]],[a[7766],a[4365],a[7767],a[4366],a[7768],a[4368],a[4369]],[a[7769],a[4371],a[7770],a[4373],a[7771],a[4375],a[4376]],[a[7772],a[7773]],[a[4382],a[7774],a[7775],a[7776],a[7777],a[7778],a[7779],a[7780],a[7781],a[4390],a[1819],a[7782]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NPR":a[7783],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[7784],a[7785],a[7786],a[7787],a[7788],a[7789],a[7790]],{"am":a[7791],"pm":a[7792]},[a[7793],a[7794],a[7795],a[7796],a[7797],a[47],a[48],a[7798],a[50],a[7799],a[7800],a[7801],a[7802]],[a[7803],a[7804],a[59],a[60],a[7805],a[7806],a[7807],a[7808],a[65],a[7809],a[67],a[7810],a[7811]],[a[2543],a[2316],a[7812],a[2545],a[7813],a[2546],a[2321],a[2547],a[2548],a[2549],a[2325],a[2550]],[a[5780],a[5781],a[7814],a[1836],a[7813],a[1837],a[1838],a[7815],a[2553],a[2554],a[2555],a[2556]],[a[7816],a[1254],a[2706],a[7817],a[1252],a[7818],a[7819]],[a[7820],a[7821],a[7822],a[7823],a[7824],a[7825],a[7826]],[a[517],a[518],a[519],a[7827]],[a[7828],a[7829],a[521],a[522]],[a[7830],a[524],a[7831],a[7832]],[a[7833],a[7834],a[114],a[115],a[7835],a[7836],a[118],a[119],a[7837],a[121],a[7838],a[123],a[7839],a[7840]],[a[127],a[7841],a[7842],a[7843],a[7844],a[7845],a[7846],a[7847],a[135],a[7848],a[7849],a[7850]],[a[7851],a[141],a[142],a[143],a[7852],a[7853],a[146],a[7854],a[148],a[7855],a[7856],a[7857]],[a[7858],a[153],a[7859],a[7860],a[7861],a[7862],a[158],a[7863],a[160],a[7864],a[7865],a[7866]],[a[7867]],{"positivePattern":a[527],"negativePattern":a[7868]},{"AUD":a[732],"AWG":a[3230],"BRL":a[435],"CAD":a[3232],"CNY":a[437],"EUR":a[440],"FJD":a[7869],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SBD":a[7870],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455]},{"AUD":a[732],"BRL":a[435],"CAD":a[3232],"CNY":a[437],"EUR":a[440],"FJD":a[7869],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SBD":a[7870],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455]},{"AUD":a[732],"BRL":a[435],"CAD":a[3232],"CNY":a[437],"EUR":a[440],"FJD":a[7869],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SBD":a[7870],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455]},{"ANG":a[3128],"AUD":a[732],"BRL":a[435],"CAD":a[3232],"CNY":a[437],"EUR":a[440],"FJD":a[7869],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SBD":a[7870],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455]},{"AUD":a[732],"BRL":a[435],"CAD":a[3232],"CNY":a[437],"EUR":a[440],"FJD":a[7869],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SBD":a[7870],"SRD":a[530],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455]},[a[7871],a[7872],a[7873],a[7874],a[7875],a[7876],a[7877],a[7878],a[7879],a[7880],a[7881],a[7882]],[a[7883],a[7884],a[7885],a[7886],a[7887],a[7888],a[7889],a[7890],a[7891],a[7892],a[7893],a[7894]],[a[3337],a[3338],a[3339],a[3340],a[3341],a[7895],a[7896]],[a[3344],a[7897],a[7898],a[7899],a[7900],a[7901],a[7902]],[a[7903],a[7904]],[a[7905],a[7906]],{"am":a[7907],"pm":a[7908]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7909],"pattern12":a[7910]},{"month":a[13],"year":a[13],"pattern":a[465]},[a[2543],a[2316],a[3609],a[2545],a[3246],a[1837],a[1838],a[2547],a[2548],a[2549],a[2325],a[2326]],[a[7696],a[7911],a[7912],a[7698],a[7699],a[7700],a[4953]],[a[2569],a[7913],a[7914],a[2572],a[2573],a[2574],a[7915]],[a[2580],a[2581],a[2578],a[2579]],[a[2580],a[2581],a[7916],a[7917]],{"am":a[7918],"pm":a[7919]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NOK":a[2587],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7920],"pattern12":a[7921]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[7922]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[7923]},[a[7924],a[7925],a[7926],a[7927],a[7928],a[7929],a[7930],a[7931],a[7932],a[7933],a[7934],a[7935]],[a[7936],a[7937],a[7938],a[7939],a[7940],a[7941],a[7942]],[a[7943],a[7944]],[a[7945],a[7946]],{"am":a[7947],"pm":a[7948]},[a[473],a[474],a[7949],a[476],a[7270],a[478],a[479],a[7950],a[481],a[482],a[7951],a[1450]],[a[7952],a[7953],a[7954],a[7955],a[7956],a[1196],a[7957],a[7958],a[1199],a[1200],a[7959],a[1453]],[a[7681],a[7729],a[7960],a[5567],a[7961],a[7962],a[7963]],[a[7964],a[7965],a[7966],a[7967],a[7968],a[7969],a[7970]],[a[473],a[474],a[7949],a[7971],a[477],a[478],a[479],a[1190],a[3576],a[482],a[7972],a[1450]],[a[7973],a[7974],a[7975],a[7976],a[477],a[3023],a[7977],a[7978],a[7979],a[7980],a[7981],a[7982]],[a[7681],a[7983],a[7984],a[7301],a[7961],a[7962],a[7985]],[a[7986],a[7987],a[7988],a[7989],a[7990],a[7991],a[7992]],{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[7993],"pattern12":a[7994]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[7995]},[a[7996],a[7997],a[7998],a[7999],a[8000],a[8001],a[8002],a[8003],a[8004],a[8005],a[8006],a[8007]],[a[8008],a[7997],a[8009],a[7999],a[8010],a[8011],a[8012],a[8013],a[8014],a[8015],a[8006],a[8016]],[a[8017],a[8018],a[8019],a[8020],a[8021],a[8022],a[8023]],[a[8024],a[8025],a[8026],a[8027],a[8028],a[8029],a[8030]],[a[5632],a[8031]],[a[8032],a[8033]],{"am":a[8034],"pm":a[8035]},[a[8036],a[8037],a[8038],a[8039],a[8040],a[8041],a[8042],a[8043],a[8044],a[8045],a[2529],a[8046]],[a[8047],a[8048],a[8049],a[8050],a[8051],a[8052],a[8053],a[8054],a[8055],a[8056],a[8057],a[8058]],[a[8059],a[8060],a[8061],a[8062],a[98],a[2613],a[5545]],[a[8063],a[8064],a[8065],a[8066],a[8067],a[8068],a[8069]],[a[8070],a[8071]],{"am":a[8072],"pm":a[8073]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ETB":a[439],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8074],a[8075],a[8076],a[8077],a[8078],a[8079],a[8080],a[8081],a[8082],a[8083],a[8084],a[8085]],[a[8086],a[8087],a[8088],a[8089],a[8090],a[8091],a[8092]],[a[8093],a[8094],a[8095],a[8096],a[8097],a[8098],a[8099]],[a[6424],a[6425],a[6426],a[1519],a[1409],a[8100],a[8101],a[1522],a[6429],a[1524],a[6430],a[1526]],[a[8102],a[8103],a[8104],a[8105],a[8106],a[8100],a[8101],a[8107],a[8108],a[8109],a[8110],a[8111]],[a[8112],a[8113],a[8114],a[8115],a[8116],a[8117],a[8118]],[a[8119],a[8120],a[8121],a[8122],a[8123],a[8124],a[8125]],[a[8126],a[8127]],{"am":a[8128],"pm":a[8129]},{"decimal":a[430],"group":a[529],"nan":a[8130],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8131],a[8132],a[8133],a[8134],a[8135],a[8136],a[8137],a[8134],a[8138],a[8134],a[8139],a[8140]],[a[8141],a[8142],a[8143],a[8144],a[8145],a[8146],a[8147],a[8148],a[8149],a[8150],a[8151],a[8152]],[a[8153],a[8154],a[8155],a[8156],a[8157],a[8158],a[8159]],[a[8160],a[8161],a[8162],a[8163],a[8164],a[8165],a[8166]],[a[8167],a[8168],a[8169],a[8170]],[a[3371],a[8171],a[3373],a[3374],a[8172],a[3376],a[8173],a[3378],a[6278],a[3380],a[3381],a[3382]],[a[8174],a[8175],a[8176],a[8177],a[8178],a[8179],a[8180]],[a[8181],a[8182]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PKR":a[3414],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[2902],a[8183],a[2901],a[2903],a[2901],a[8184],a[8183],a[2902],a[8185],a[8186],a[8183],a[8187]],[a[8188],a[8189],a[1230],a[8190],a[1827],a[8191],a[4671],a[8192],a[8193],a[8194],a[2460],a[8195]],[a[8196],a[8197],a[8198],a[8199],a[8200],a[8201],a[8202],a[8203],a[8204],a[8205],a[4684],a[8206]],[a[8207],a[8208],a[8209],a[8210],a[8211],a[8212],a[8213]],[a[8214],a[8215],a[8216],a[8217],a[8218],a[8219],a[2487]],[a[8220],a[4691]],[a[8221],a[8222],a[8223],a[8224],a[8225],a[117],a[118],a[119],a[7837],a[8226],a[122],a[8227],a[124],a[125]],[a[140],a[141],a[142],a[143],a[8228],a[8229],a[8230],a[8231],a[148],a[8232],a[8233],a[8234]],[a[152],a[153],a[154],a[155],a[8235],a[8236],a[8237],a[8238],a[160],a[8239],a[8240],a[8241]],[a[8242],a[8243],a[8244],a[410],a[8245],a[8246],a[413],a[8247],a[8248],a[8249],a[417],a[418]],[a[8250],a[8251]],{"BRL":a[435],"EUR":a[440],"PLN":a[8252],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8253],a[8254],a[3373],a[3374],a[3375],a[3376],a[3377],a[8255],a[3379],a[3380],a[3381],a[3382]],{"am":a[8256],"pm":a[8257]},[a[8258],a[8259],a[8260],a[8261],a[8262],a[8263],a[8264],a[8265],a[8266],a[8267],a[8268],a[8269]],{"decimal":a[1093],"group":a[1094],"percent":a[1096]},{"AFN":a[3450],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8270],a[8271],a[8272],a[8273],a[8274],a[8275],a[8276],a[8277],a[8278],a[8279],a[8280],a[8281]],[a[1826],a[1283],a[1230],a[1231],a[3246],a[1828],a[1829],a[1235],a[1236],a[3980],a[1832],a[8282]],[a[8283],a[8284],a[8285],a[2329],a[3983],a[8286],a[8287],a[3175],a[3986],a[3987],a[3143],a[8288]],[a[1259],a[8289],a[3702],a[8290],a[8291],a[8292],a[3227]],[a[3192],a[8293],a[8294],a[8295],a[8296],a[8297],a[3196]],[a[8298],a[8299]],[a[8300],a[421]],{"AOA":a[3229],"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PTE":a[8301],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PTE":a[8301],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MOP":a[3119],"MXN":a[447],"NZD":a[448],"PTE":a[8301],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"MZN":a[7326],"NZD":a[448],"PTE":a[8301],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[8302],"pattern12":a[8303]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"pattern":a[8304]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[535]},{"month":a[14],"year":a[13],"pattern":a[464]},{"day":a[13],"month":a[14],"pattern":a[537]},[a[7327],a[8305],a[7329],a[8306],a[8307],a[8308],a[8309],a[8310],a[8311],a[39],a[40],a[41]],[a[473],a[3870],a[475],a[3573],a[2505],a[478],a[479],a[1190],a[3576],a[8312],a[483],a[4000]],[a[8313],a[8314],a[8315],a[3581],a[8316],a[8317],a[8318],a[3585],a[8319],a[8320],a[8321],a[8322]],[a[1252],a[8323],a[5657],a[8324],a[8325],a[8326],a[8327]],[a[1272],a[1273],a[8328],a[8329]],{"am":a[8330],"pm":a[8331]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PTE":a[8301],"STD":a[8332],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8333],a[8334],a[3609],a[3729],a[8335],a[8336],a[8337],a[8338],a[8339],a[2324],a[2325],a[2550]],[a[8340],a[8341],a[3609],a[8342],a[8335],a[8343],a[8344],a[8338],a[8345],a[8346],a[2555],a[2556]],[a[5655],a[8347],a[1254],a[3146],a[8348],a[3148],a[2480]],[a[8349],a[8350],a[3748],a[8351],a[8352],a[8353],a[8354]],[a[8355],a[8356]],[a[8357],a[8358]],{"am":a[3106],"pm":a[8359]},[a[8360],a[8361],a[8362],a[8363],a[8364],a[8365],a[8366],a[8367],a[8368],a[8369],a[8370],a[8371]],[a[8372],a[8373],a[8374],a[8375],a[8376],a[8377],a[8378],a[8379],a[8380],a[8381],a[8382],a[8383]],[a[8384],a[8385],a[8386],a[8387],a[8388],a[8389],a[8390]],[a[8391],a[8392],a[8393],a[8394],a[8395],a[8396],a[8397]],[a[8398],a[8399]],[a[8400],a[8401]],{"am":a[8402],"pm":a[8403]},{"AUD":a[434],"BIF":a[3862],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8404]],[a[8405]],[a[42],a[43],a[8406],a[45],a[46],a[47],a[48],a[49],a[50],a[51],a[52],a[8407],a[8408]],[a[8409],a[8410],a[3713],a[8411],a[3702],a[8412],a[8413],a[8414],a[8415],a[8416],a[8417],a[8418],a[8419]],[a[2496],a[468],a[397],a[469],a[397],a[2496],a[2496],a[469],a[399],a[470],a[471],a[472]],[a[8420],a[2316],a[2544],a[2545],a[3246],a[8421],a[8422],a[2547],a[3169],a[2324],a[2325],a[2550]],[a[8423],a[8424],a[8425],a[8426],a[3246],a[8427],a[8428],a[2552],a[8429],a[8430],a[8431],a[8432]],[a[1755],a[3591],a[504],a[3592],a[8433],a[8434],a[8435]],[a[8436],a[1773],a[475],a[8437],a[8438],a[8439],a[8440]],[a[8441],a[8442],a[8443],a[8444],a[3886],a[8445],a[8446]],[a[8447],a[8448],a[8449],a[8450]],[a[8451],a[8452]],[a[8453],a[8454],a[114],a[115],a[8455],a[117],a[118],a[119],a[120],a[121],a[8456],a[123],a[124],a[125]],{"EUR":a[440],"MDL":a[3210],"XAF":a[453],"XOF":a[455],"XPF":a[456]},{"EUR":a[440],"XAF":a[453],"XOF":a[455],"XPF":a[456]},[a[8457],a[8458],a[8459],a[8460],a[8461],a[8462],a[8463],a[8464],a[8465],a[8466],a[8467],a[8468]],[a[8469],a[8470],a[8471],a[8472],a[1207],a[5405],a[1208]],[a[8473],a[8474],a[8475],a[8476],a[1214],a[1215],a[8477]],[a[8478],a[8479]],{"am":a[8480],"pm":a[8481]},[a[8482],a[8483],a[8484],a[8485],a[8486],a[1897],a[1898],a[8487],a[8488],a[8489],a[8490],a[8491],a[1904]],[a[1905],a[1906],a[1907],a[1908],a[1909],a[8492],a[8493],a[8494],a[1913],a[8495],a[8496],a[8497],a[8498]],[a[6424],a[1518],a[8499],a[1519],a[1409],a[8500],a[8501],a[1522],a[8502],a[1524],a[8503],a[1526]],[a[8504],a[8505],a[8499],a[8506],a[1409],a[8500],a[8501],a[8507],a[8508],a[8509],a[8510],a[8511]],[a[8512],a[1429],a[1534],a[1431],a[1535],a[1433],a[1434]],[a[8513],a[8514],a[1538],a[7371],a[8515],a[8516],a[8517]],[a[8518],a[1443]],[a[8519],a[8520],a[1443]],[a[8521],a[1960],a[1961],a[1962],a[1963],a[1964],a[1965],a[1966],a[8522],a[1968],a[8523],a[1970],a[8524],a[1972]],[a[8525],a[8526],a[8527],a[8528],a[1977],a[8529],a[8530],a[1980],a[8531],a[1982],a[8532],a[8533]],[a[8534],a[1999],a[8535],a[8536],a[8537],a[8538],a[8539],a[8540],a[2006],a[8541],a[8542],a[8543]],[a[8544],a[8545],a[8546],a[8547],a[8548],a[8549],a[8550],a[8551],a[8552],a[8553],a[8554],a[8555],a[8556],a[8557],a[8558],a[8559],a[8560],a[8561],a[8562],a[8563],a[8564],a[8565],a[8566],a[8567],a[8568],a[8569],a[8570],a[8571],a[8572],a[8573],a[8574],a[8575],a[8576],a[8577],a[8578],a[8579],a[8580],a[8581],a[8582],a[8583],a[8584],a[8585],a[8586],a[8587],a[8588],a[8589],a[8590],a[8591],a[8592],a[8593],a[8594],a[8595],a[8596],a[8597],a[8598],a[8599],a[8600],a[8601],a[8602],a[8603],a[8604],a[8605],a[8606],a[8607],a[8608],a[8609],a[8610],a[8611],a[8612],a[8613],a[8614],a[8615],a[8616],a[8617],a[8618],a[8619],a[8620],a[8621],a[8622],a[8623],a[8624],a[8625],a[8626],a[8627],a[8628],a[8629],a[8630],a[8631],a[8632],a[8633],a[8634],a[8635],a[8636],a[8637],a[8638],a[8639],a[8640],a[8641],a[8642],a[8643],a[8644],a[8645],a[8646],a[8647],a[8648],a[8649],a[8650],a[8651],a[8652],a[8653],a[8654],a[8655],a[8656],a[8657],a[8658],a[8659],a[8660],a[8661],a[8662],a[8663],a[8664],a[8665],a[8666],a[8667],a[8668],a[8669],a[8670],a[8671],a[8672],a[8673],a[8674],a[8675],a[8676],a[8677],a[8678],a[8679],a[8680],a[8681],a[8682],a[8683],a[8684],a[8685],a[8686],a[8687],a[8688],a[8689],a[8690],a[8691],a[8692],a[8693],a[8694],a[8695],a[8696],a[8697],a[8698],a[8699],a[8700],a[8701],a[8702],a[8703],a[8704],a[8705],a[8706],a[8707],a[8708],a[8709],a[8710],a[8711],a[8712],a[8713],a[8714],a[8715],a[8716],a[8717],a[8718],a[8719],a[8720],a[8721],a[8722],a[8723],a[8724],a[8725],a[8726],a[8727],a[8728],a[8729],a[8730],a[8731],a[8732],a[8733],a[8734],a[8735],a[8736],a[8737],a[8738],a[8739],a[8740],a[8741],a[8742],a[8743],a[8744],a[8745],a[8746],a[8747],a[8748],a[8749],a[8750],a[8751],a[8752],a[8753],a[8754],a[8755],a[8756],a[8757],a[8758],a[8759],a[8760],a[8761],a[8762],a[8763],a[8764],a[8765],a[8766],a[8767],a[8768],a[8769],a[8770],a[8771],a[8772],a[8773],a[8774],a[8775],a[8776],a[8777],a[8778],a[8779]],[a[8780],a[2247],a[8781],a[2249],a[2250],a[8782],a[2252],a[2253],a[8783],a[8784],a[2256],a[8785]],{"decimal":a[430],"group":a[529],"nan":a[8786],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"BYR":a[1446],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"THB":a[449],"TMT":a[8787],"TWD":a[450],"UAH":a[8788],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[8789]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KGS":a[6460],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"RUR":a[1446],"THB":a[449],"TMT":a[8787],"TWD":a[450],"UAH":a[8788],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[8789]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"KZT":a[5762],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"RUR":a[1446],"THB":a[449],"TMT":a[8787],"TWD":a[450],"UAH":a[8788],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[8789]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MDL":a[3210],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"RUR":a[1446],"THB":a[449],"TMT":a[8787],"TWD":a[450],"UAH":a[8788],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[8789]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"RUR":a[1446],"THB":a[449],"TMT":a[8787],"TWD":a[450],"UAH":a[8788],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"XXX":a[8789]},[a[8790],a[8791],a[8792],a[8793],a[8794],a[8795],a[8796],a[8388],a[8797],a[8798],a[8799],a[8800]],[a[8801],a[8802],a[8803],a[8804],a[8805],a[8806],a[8380],a[8807],a[8808],a[8809],a[8810],a[8811]],[a[8812],a[8385],a[8386],a[8387],a[8388],a[8389],a[8390]],[a[8813],a[8814],a[8815],a[8816],a[8817],a[8818],a[8819]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RWF":a[3125],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8820],a[8821],a[8822],a[8823],a[8824],a[8825],a[8826],a[8827],a[8828],a[8829],a[8830],a[8831]],[a[8832],a[8833],a[8834],a[8835],a[8836],a[8837],a[8838],a[8839],a[8840],a[8841],a[8842],a[8843]],[a[8844],a[8845],a[8846],a[8847],a[8848],a[8849],a[8850]],[a[8851],a[8852],a[8853],a[8854],a[8855],a[8856],a[8857]],[a[8858],a[8859]],{"am":a[8860],"pm":a[8861]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[1589],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8862],a[8863],a[8864],a[8865],a[8866],a[8867],a[8868],a[8869],a[8870],a[8871],a[8872],a[8873]],[a[8874],a[8875],a[8876],a[8877],a[8878],a[8879],a[8880],a[8881],a[8882],a[8883],a[8884],a[8885]],[a[8886],a[8887],a[8865],a[2851],a[8867],a[8868],a[612]],[a[8888],a[8889],a[8890],a[8891],a[8892],a[8893],a[8894]],[a[8895],a[8896]],{"am":a[8897],"pm":a[8898]},{"month":a[14],"day":a[13],"year":a[13],"pattern":a[8899]},[a[8900],a[8901],a[8902],a[8903],a[8904],a[8905],a[8906],a[8907],a[8908],a[7985],a[8909],a[8910]],[a[8911],a[8912],a[8913],a[8914],a[8915],a[8916],a[8917],a[8918],a[8919],a[8920],a[8921],a[8922]],[a[1491],a[1204],a[1205],a[1206],a[1207],a[5405],a[1209]],[a[8923],a[1211],a[1212],a[1213],a[8924],a[1215],a[1216]],[a[626],a[7635]],[a[8925],a[8926]],{"am":a[8927],"pm":a[8928]},[a[8929],a[8930],a[8931],a[8932],a[8933],a[8934],a[8935],a[8936],a[8937],a[8938],a[8939],a[8940]],[a[8941],a[8942],a[8943],a[8944],a[8945],a[8946],a[8947],a[8948],a[8949],a[8950],a[8951],a[8952]],[a[8953],a[8954],a[8955],a[8956],a[8957],a[8958],a[8959]],[a[8960],a[8961],a[8962],a[8963],a[8964],a[8965],a[8966]],[a[8967],a[8968]],[a[8969],a[8970]],{"am":a[8971],"pm":a[8972]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DKK":a[8973],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NOK":a[2587],"NZD":a[448],"SEK":a[5811],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[8974],a[8975],a[8976],a[8977],a[8978],a[8979],a[8980],a[8981],a[8982],a[8983],a[8984],a[8985]],[a[8986],a[8987],a[8988],a[8989],a[8990],a[8991],a[8992]],[a[473],a[3870],a[475],a[3573],a[2505],a[478],a[479],a[480],a[3576],a[3875],a[483],a[1191]],[a[8313],a[8993],a[8994],a[3581],a[8316],a[8317],a[8318],a[8995],a[8319],a[8996],a[8321],a[8997]],[a[2767],a[8998],a[8999],a[2850],a[9000],a[9001],a[100]],[a[9002],a[9003],a[9004],a[9005],a[9006],a[9007],a[9008]],[a[9009],a[1462]],[a[8298],a[3048]],[a[2694],a[2695],a[2696],a[2697],a[5679],a[2698],a[2699]],[a[9010],a[8044],a[9011],a[2614],a[9012],a[9013],a[9014],a[9015],a[7729],a[9016],a[9017],a[9018]],[a[9019],a[9020],a[9021],a[9022],a[9023],a[9024],a[9025],a[9026],a[9027],a[9028],a[9029],a[9030]],[a[9031],a[9032],a[9033],a[9034],a[9035],a[9036],a[9037]],[a[9038],a[9039],a[9040],a[9041],a[9042],a[9043],a[9044]],[a[9045],a[9046]],[a[9047],a[9048]],{"am":a[9049],"pm":a[9050]},[a[9051],a[9052],a[9053],a[9054],a[9055],a[9056],a[9057],a[9058],a[9059],a[9060],a[9061],a[9062]],[a[9063],a[9064],a[9065],a[9066],a[9067],a[9068],a[9069],a[9070],a[9071],a[9072],a[9073],a[9074]],[a[9075],a[9076],a[9077],a[9078],a[9079],a[9080],a[9081]],[a[9082],a[9083],a[9084],a[9085],a[9086],a[9087],a[9088]],[a[9089],a[9090]],[a[9091],a[9092]],{"am":a[9093],"pm":a[9094]},[a[9095],a[9096],a[9097],a[9098],a[1232],a[6552],a[6553],a[9099],a[9100],a[9101],a[9102],a[9103]],[a[9104],a[9105],a[9106],a[9107],a[9108],a[9109],a[9110],a[9111],a[9112],a[9113],a[9114],a[9115]],[a[9116],a[9117],a[9118],a[9119],a[9120],a[9121],a[9122]],[a[9123],a[9124],a[9125],a[9126],a[9127],a[9128],a[9129]],[a[9130],a[9131]],[a[9132],a[9133]],{"am":a[9134],"pm":a[9135]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[9136],"pattern12":a[9137]},{"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[1394],"pattern12":a[9138]},{"hour":a[13],"minute":a[15],"pattern":a[1396],"pattern12":a[9139]},[a[9140],a[9141],a[9142],a[9143],a[9144],a[9145],a[9145],a[9143],a[9146],a[9147],a[9148],a[9149]],[a[9150],a[9151],a[9152],a[9153],a[9154],a[9155],a[9156],a[9157],a[9158],a[9159],a[9160],a[9161]],[a[9162],a[9163],a[9152],a[9153],a[9154],a[9155],a[9156],a[9164],a[9165],a[9166],a[9167],a[9168]],[a[9169],a[9170],a[9171],a[9172],a[9173],a[9174],a[9175]],[a[9176],a[9177],a[9178],a[9179],a[9180],a[9174],a[9175]],[a[9176],a[9177],a[9181],a[9179],a[9182],a[9183],a[9184]],[a[9185],a[9186],a[9187],a[9188]],[a[9189],a[9190]],{"am":a[9191],"pm":a[9192]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"LKR":a[9193],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[1279]},[a[1826],a[1229],a[1230],a[1284],a[9194],a[9195],a[9196],a[3249],a[1831],a[1289],a[1832],a[1833]],[a[9197],a[9198],a[8198],a[9199],a[9200],a[9201],a[9202],a[9203],a[9204],a[9205],a[9206],a[9207]],[a[7961],a[9208],a[2674],a[9209],a[6994],a[9210],a[503]],[a[2474],a[2475],a[9211],a[2477],a[7001],a[9212],a[2480]],[a[9213],a[9214],a[9215],a[9216],a[9217],a[9218],a[2487]],[a[9219],a[2489]],{"EUR":a[440],"ILS":a[443],"MXN":a[9220],"NZD":a[448],"PLN":a[1889],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[9221],a[3976],a[2901],a[2900],a[2901],a[9221],a[9221],a[2900],a[2902],a[9222],a[9223],a[2898]],[a[2543],a[2316],a[2544],a[2545],a[1827],a[2546],a[2321],a[9224],a[2548],a[2549],a[2325],a[2550]],[a[1834],a[1835],a[9225],a[1836],a[1827],a[9226],a[9227],a[1839],a[2553],a[2554],a[2555],a[2556]],[a[9228],a[8208],a[2566],a[9229],a[9230],a[9231],a[8213]],[a[9232],a[9233],a[9234],a[9235],a[9236],a[9237],a[2487]],[a[9238],a[7009],a[9239]],[a[9240],a[9241]],{"am":a[9242],"pm":a[9243]},{"AUD":a[434],"BRL":a[435],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[9244],a[9245],a[8006],a[9246],a[9247],a[9248],a[9249],a[5569],a[9250],a[99],a[9251],a[9252]],[a[9253],a[9254],a[9255],a[9256],a[9257],a[9258],a[9259],a[9260],a[9261],a[9262],a[9263],a[9264]],[a[9265],a[9266],a[9267],a[9268],a[9269],a[9270],a[2833]],[a[9271],a[9272],a[9004],a[9005],a[9273],a[9007],a[9274]],[a[9275],a[9276]],[a[9277],a[9278],a[2529],a[9279],a[9001],a[9280],a[9281],a[9282],a[9283],a[8872],a[9284],a[9285]],[a[9286],a[9287],a[9288],a[9289],a[9290],a[9291],a[9292],a[9293],a[9294],a[9295],a[9296],a[9297]],[a[9298],a[7547],a[96],a[97],a[7548],a[2613],a[100]],[a[9299],a[9300],a[9301],a[9302],a[9303],a[9304],a[107]],[a[9305],a[9306]],[a[9307],a[9308]],{"am":a[9309],"pm":a[9310]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"DJF":a[438],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SOS":a[399],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"ETB":a[439],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SOS":a[399],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SOS":a[399],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SOS":a[399],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[467],a[399],a[397],a[1227],a[397],a[3209],a[1467],a[2499],a[399],a[398],a[471],a[472]],[a[473],a[9311],a[475],a[9312],a[9313],a[9314],a[8001],a[9315],a[9316],a[9317],a[9318],a[9319]],[a[9320],a[9321],a[3609],a[9322],a[1827],a[9323],a[9324],a[9325],a[9326],a[9327],a[9328],a[9329]],[a[9330],a[9331],a[475],a[9332],a[9333],a[9334],a[9316]],[a[9335],a[9336],a[9337],a[9338],a[9339],a[9340],a[9341]],[a[9342],a[9343],a[9344],a[9345]],[a[9342],a[9343],a[9346],a[9345]],[a[9347],a[9348],a[9349],a[9350]],{"am":a[9351],"pm":a[9352]},{"ALL":a[9353],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"ALL":a[9353],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MKD":a[9354],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[9355],"pattern12":a[9356]},{"hour":a[15],"minute":a[15],"second":a[15],"pattern":a[1394],"pattern12":a[1395]},{"hour":a[15],"minute":a[15],"pattern":a[1396],"pattern12":a[1397]},[a[1938],a[1939],a[1940],a[9357],a[1942],a[1943],a[1944]],[a[1941],a[1938],a[1939],a[1940],a[1942],a[1943],a[1944]],[a[1948],a[1945],a[1946],a[1947],a[1949],a[1950],a[1441]],[a[1930],a[1931],a[1329],a[1529],a[1922],a[1923],a[1924],a[1334],a[1934],a[1935],a[1936],a[1937]],[a[1945],a[1946],a[1947],a[7371],a[1949],a[1950],a[1441]],[a[9358],a[43],a[44],a[45],a[46],a[9359],a[48],a[9360],a[9361],a[51],a[52],a[53],a[7802]],[a[57],a[58],a[59],a[60],a[61],a[9362],a[63],a[64],a[65],a[66],a[67],a[9363],a[69]],[a[1834],a[1835],a[1294],a[1836],a[1827],a[1828],a[1829],a[1839],a[1840],a[1841],a[1842],a[1843]],[a[1844],a[1845],a[1846],a[9364],a[1848],a[1849],a[1850]],[a[9232],a[9365],a[1853],a[9235],a[1855],a[1856],a[1857]],[a[1858],a[9366]],[a[9367],a[9368],a[114],a[115],a[9369],a[117],a[118],a[119],a[7837],a[121],a[122],a[123],a[124],a[125]],[a[9370],a[9371],a[9372],a[9373],a[131],a[9374],a[133],a[134],a[9375],a[9376],a[9377],a[9378]],[a[140],a[141],a[9379],a[9380],a[9381],a[9382],a[9383],a[9384],a[148],a[9385],a[9386],a[9387]],[a[9388],a[153],a[154],a[155],a[156],a[157],a[9389],a[9390],a[160],a[9391],a[9392],a[9393]],[a[165],a[9394],a[9395],a[9396],a[9397],a[170],a[9398],a[172],a[9399],a[174],a[9400],a[9401],a[9402],a[9403],a[9404],a[9405],a[9406],a[9407],a[9408],a[9409],a[9410],a[9411],a[9412],a[9413],a[189],a[190],a[9414],a[9415],a[193],a[9416],a[9417],a[9418],a[9419],a[9420],a[9421],a[9422],a[9423],a[202],a[9424],a[9425],a[9426],a[206],a[207],a[9427],a[209],a[210],a[9428],a[212],a[213],a[9429],a[9430],a[9431],a[9432],a[9433],a[9434],a[9435],a[221],a[9436],a[9437],a[9438],a[225],a[9439],a[9440],a[9441],a[9442],a[9443],a[9444],a[9445],a[9446],a[9447],a[9448],a[236],a[9449],a[9450],a[9451],a[9452],a[9453],a[9454],a[243],a[9455],a[9456],a[246],a[9457],a[9458],a[9459],a[9460],a[251],a[9461],a[9462],a[9463],a[9464],a[256],a[9465],a[258],a[9466],a[9467],a[9468],a[9469],a[263],a[264],a[9470],a[9471],a[267],a[9472],a[9473],a[9474],a[9475],a[9476],a[9477],a[9478],a[9479],a[9480],a[277],a[9481],a[9482],a[9483],a[9484],a[9485],a[9486],a[284],a[285],a[286],a[9487],a[288],a[9488],a[290],a[9489],a[9490],a[293],a[294],a[9491],a[9492],a[9493],a[9494],a[9495],a[9496],a[9497],a[302],a[9498],a[9499],a[9500],a[306],a[9501],a[308],a[309],a[9502],a[311],a[9503],a[9504],a[9505],a[9506],a[9507],a[9508],a[318],a[319],a[9509],a[9510],a[322],a[9511],a[9512],a[325],a[9513],a[327],a[9514],a[9515],a[9516],a[331],a[332],a[9517],a[334],a[9518],a[9519],a[9520],a[338],a[339],a[9521],a[9522],a[9523],a[9524],a[9525],a[9526],a[9527],a[347],a[9528],a[349],a[9529],a[351],a[9530],a[353],a[9531],a[355],a[9532],a[357],a[358],a[9533],a[360],a[9534],a[9535],a[363],a[9536],a[365],a[9537],a[9538],a[368],a[369],a[9539],a[9540],a[9541],a[373],a[9542],a[9543],a[9544],a[377],a[9545],a[9546],a[380],a[9547],a[9548],a[383],a[384],a[385],a[9549],a[387],a[388],a[9550],a[9551],a[391],a[392],a[393],a[9552],a[9553],a[9554],a[9555],a[9556],a[9557],a[9558]],[a[9559],a[9560],a[9561],a[410],a[411],a[9562],a[413],a[414],a[415],a[9563],a[417],a[418]],[a[9564],a[9565]],{"AUD":a[434],"BAM":a[1217],"BRL":a[435],"CAD":a[436],"CNY":a[437],"CZK":a[1888],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PLN":a[1889],"RSD":a[1890],"THB":a[449],"TRY":a[9566],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[9567],a[9568],a[9569],a[7709],a[9570],a[9571],a[9572],a[9573],a[9574],a[9575],a[7715],a[9576]],[a[9577],a[9578],a[9579],a[7720],a[9580],a[9581],a[9582],a[9583],a[9584],a[9585],a[9586],a[9587]],[a[7681],a[9588],a[7960],a[9589],a[7961],a[7962],a[9590]],[a[9591],a[9592],a[9593],a[9594],a[9595],a[9596],a[9597]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"SZL":a[2498],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZAR":a[531]},[a[9017],a[5545],a[9598],a[4859],a[8040],a[2612],a[70]],[a[9599],a[9600],a[9601],a[9602],a[9603],a[9604],a[9605]],[a[9606],a[9607],a[9608],a[9609],a[9610],a[473],a[9611],a[9612],a[9613],a[9575],a[9614],a[9615]],[a[9616],a[9617],a[9618],a[9619],a[9620],a[9621],a[9622],a[9623],a[9624],a[9625],a[9626],a[9627]],[a[7681],a[9628],a[7984],a[7301],a[7961],a[7962],a[9629]],[a[9630],a[9631],a[7988],a[9632],a[7990],a[9633],a[9634]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"LSL":a[397],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZAR":a[531]},[a[3669],a[3682],a[3683],a[9635],a[3685],a[3686],a[3687],a[3688],a[3689],a[3690],a[3691],a[3692],a[3693]],[a[5780],a[5781],a[3609],a[1836],a[1827],a[1837],a[1838],a[9636],a[2553],a[2554],a[2555],a[2556]],[a[9637],a[9638],a[2558],a[2559],a[2560],a[2561],a[9639]],[a[9640],a[9641],a[9642],a[9643],a[9644],a[9645],a[9646]],[a[9647],a[7913],a[9648],a[2572],a[2573],a[2574],a[9649]],[a[9650],a[9651],a[9652],a[9653]],{"am":a[9654],"pm":a[9655]},[a[9656],a[9657],a[9658],a[9659],a[9660],a[9661],a[9662],a[9663],a[9664],a[9665],a[9666],a[3770],a[9667]],[a[9656],a[9657],a[9658],a[9659],a[9660],a[9661],a[9662],a[9663],a[9664],a[9665],a[9666],a[3770],a[9667],a[9668]],[a[9669]],[a[3558],a[3559],a[3560],a[3561],a[9670],a[9671],a[3820],a[9672],a[3566],a[9673],a[9674],a[9675]],[a[9676],a[9677],a[9678],a[9679],a[9680],a[9681],a[9682],a[9683],a[9684],a[9685],a[9686],a[9687],a[9688],a[9689],a[9690],a[9691],a[9692],a[9693],a[9694],a[9695],a[9696],a[9697],a[9698],a[9699],a[9700],a[9701],a[9702],a[9703],a[9704],a[9705],a[9706],a[9707],a[9708],a[9709],a[9710],a[9711],a[9712],a[9713],a[9714],a[9715],a[9716],a[9717],a[9718],a[9719],a[9720],a[9721],a[9722],a[9723],a[9724],a[9725],a[9726],a[9727],a[9728],a[9729],a[9730],a[9731],a[9732],a[9733],a[9734],a[9735],a[9736],a[9737],a[9738],a[9739],a[9740],a[9741],a[9742],a[9743],a[9744],a[9745],a[9746],a[9747],a[9748],a[9749],a[9750],a[9751],a[9752],a[9753],a[9754],a[9755],a[9756],a[9757],a[9758],a[9759],a[9760],a[9761],a[9762],a[9763],a[9764],a[9765],a[9766],a[9767],a[9768],a[9769],a[9770],a[9771],a[9772],a[9773],a[9774],a[9775],a[9776],a[9777],a[9778],a[9779],a[9780],a[9781],a[9782],a[9783],a[9784],a[9785],a[9786],a[9787],a[9788],a[9789],a[9790],a[9791],a[9792],a[9793],a[9794],a[9795],a[9796],a[9797],a[9798],a[9799],a[9800],a[9801],a[9802],a[9803],a[9804],a[9805],a[9806],a[9807],a[9808],a[9809],a[9810],a[9811],a[9812],a[9813],a[9814],a[9815],a[9816],a[9817],a[9818],a[9819],a[9820],a[9821],a[9822],a[9823],a[9824],a[9825],a[9826],a[9827],a[9828],a[9829],a[9830],a[9831],a[9832],a[9833],a[9834],a[9835],a[9836],a[9837],a[9838],a[9839],a[9840],a[9841],a[9842],a[9843],a[9844],a[9845],a[9846],a[9847],a[9848],a[9849],a[9850],a[9851],a[9852],a[9853],a[9854],a[9855],a[9856],a[9857],a[9858],a[9859],a[9860],a[9861],a[9862],a[9863],a[9864],a[9865],a[9866],a[9867],a[9868],a[9869],a[9870],a[9871],a[9872],a[9873],a[9874],a[9875],a[9876],a[9877],a[9878],a[9879],a[9880],a[9881],a[9882],a[9883],a[9884],a[9885],a[9886],a[9887],a[9888],a[9889],a[9890],a[9891],a[9892],a[9893],a[9894],a[9895],a[9896],a[9897],a[9898],a[9899],a[9900],a[9901],a[9902],a[9903],a[9904],a[9905],a[9906],a[9907],a[403],a[404],a[405],a[406]],[a[9908],a[9909],a[9910],a[9911],a[9912],a[9913],a[9914],a[9915],a[9916],a[9917],a[9918],a[9919]],[a[9920],a[9921]],{"AUD":a[732],"BBD":a[9922],"BDT":a[9923],"BMD":a[9924],"BND":a[9925],"BRL":a[9926],"BSD":a[9927],"BZD":a[9928],"CAD":a[9929],"CNY":a[437],"DKK":a[8973],"DOP":a[9930],"EGP":a[9931],"EUR":a[440],"GBP":a[1150],"GYD":a[9932],"HKD":a[442],"ILS":a[443],"INR":a[444],"ISK":a[9933],"JMD":a[9934],"JPY":a[445],"KRW":a[446],"LVL":a[7174],"MXN":a[447],"NOK":a[9935],"NZD":a[448],"SEK":a[2587],"SYP":a[9936],"THB":a[2827],"TWD":a[9937],"USD":a[451],"VND":a[9938],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[9650],a[9651],a[9939],a[9940]],[a[7103],a[7104],a[7105],a[7106],a[1207],a[7107],a[7108]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"TZS":a[1225],"UGX":a[2400],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[9941],a[9942],a[9943],a[6575],a[6576],a[6574],a[9944],a[9945],a[9946],a[9947],a[9948],a[9949]],[a[9950],a[9951],a[9952],a[9953],a[9954],a[9955],a[9956],a[9957],a[9958],a[9959],a[9960],a[9961]],[a[9962],a[2731],a[9963],a[9964],a[9965],a[9966],a[9967]],[a[9968],a[9969],a[9970],a[9971],a[9972],a[9973],a[9974]],[a[9975],a[9976]],[a[9977],a[9978]],{"am":a[9979],"pm":a[9980]},[a[9981],a[9982],a[9983],a[9984],a[9985],a[9986],a[9987],a[9988],a[9989],a[9990],a[9991],a[9992]],[a[9993],a[9994],a[9995],a[9996],a[9997],a[9998],a[9999],a[10000],a[10001],a[10002],a[10003],a[10004]],[a[10005],a[10006],a[10007],a[10008],a[10009],a[10010],a[10010],a[10011],a[10012],a[10013],a[10014],a[10015]],[a[10016],a[10017],a[10018],a[10019],a[10009],a[10020],a[10021],a[10022],a[10023],a[10024],a[10025],a[10026]],[a[10027],a[10028],a[10029],a[10030],a[10009],a[10020],a[10021],a[10031],a[10032],a[10033],a[10034],a[10035]],[a[10036],a[10037],a[10012],a[10038],a[10039],a[10040],a[10041]],[a[10042],a[10043],a[10044],a[10045],a[10046],a[10047],a[10048]],[a[10049],a[10050]],[a[10051],a[10052]],{"am":a[10053],"pm":a[10054]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"LKR":a[10055],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"MYR":a[7557],"NZD":a[448],"SGD":a[10056],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"weekday":a[14],"day":a[13],"month":a[14],"year":a[13],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[10057],"pattern12":a[10058]},{"day":a[13],"month":a[14],"year":a[13],"pattern":a[10059]},[a[10060],a[10061],a[10062],a[10063],a[10064],a[10065],a[10066],a[10067],a[10068],a[10069],a[10070],a[10071]],[a[10072],a[10073],a[10074],a[10075],a[10064],a[10076],a[10077],a[10078],a[10079],a[10080],a[10081],a[10082]],[a[10083],a[10084],a[10074],a[10085],a[10064],a[10076],a[10077],a[10086],a[10087],a[10088],a[10089],a[10090]],[a[10091],a[10092],a[10093],a[10094],a[10095],a[10096],a[10097]],[a[10091],a[10092],a[10098],a[10094],a[10095],a[10096],a[10097]],[a[10099],a[10100],a[10101],a[10102],a[10103],a[10104],a[10105]],[a[10106],a[10107]],[a[10108],a[10109],a[1184],a[1185]],[a[10110],a[10111],a[10112],a[10113],a[10114],a[10115],a[10116],a[10117],a[10118],a[10119],a[10120],a[10121]],[a[10122]],[a[7301],a[10123],a[7299],a[10124],a[475],a[10125],a[7272],a[10126],a[10127],a[10128],a[9278],a[10129]],[a[10130],a[10131],a[10132],a[10133],a[10134],a[10135],a[10136],a[10137],a[10138],a[10139],a[10140],a[10141]],[a[2612],a[10142],a[10143],a[10144],a[10145],a[2591],a[100]],[a[10146],a[10147],a[10148],a[10149],a[10150],a[10151],a[10152]],{"am":a[10153],"pm":a[10154]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KES":a[2627],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"UGX":a[2400],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[10155],a[10156],a[10157],a[10158],a[10159],a[10160],a[10161],a[10162],a[10163],a[10164],a[10165],a[10166]],[a[10167],a[10168],a[10169],a[10170],a[10159],a[10160],a[10161],a[10171],a[10172],a[10173],a[10174],a[10175]],[a[10176],a[10177],a[10178],a[10179],a[10180],a[10181],a[10182]],[a[10183],a[10184],a[10185],a[10186],a[10187],a[10188],a[10189]],[a[10190],a[10191]],[a[10192],a[10191]],{"am":a[10193],"pm":a[10194]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TJS":a[6460],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[10195]],[a[10196]],[a[10197],a[10198],a[10199],a[10200],a[10201],a[10202],a[10203],a[10204],a[10205],a[10206],a[10207],a[10208],a[10209]],[a[10210],a[10211],a[10212],a[10213],a[10214],a[10215],a[10216],a[10217],a[10218],a[10219],a[10220],a[10221],a[10222]],[a[10223],a[10224],a[10225],a[10226],a[10227],a[10228],a[10229],a[10230],a[10231],a[10232],a[10233],a[10234]],[a[10235],a[10236],a[10237],a[10238],a[10239],a[10240],a[10241],a[10242],a[10243],a[10244],a[10245],a[10246]],[a[10247],a[10248],a[10249],a[10250],a[10251],a[10252],a[10253]],[a[10254],a[10255],a[10256],a[10257],a[10258],a[10259],a[10260]],[a[10261],a[10262],a[10263],a[10264]],[a[10265],a[10262],a[10263],a[10264]],[a[10266],a[10267],a[10268],a[10269]],{"am":a[10270],"pm":a[10271]},[a[10272],a[10273],a[10274],a[10275],a[10276],a[10277],a[10278],a[10279],a[10280],a[10281],a[10282],a[10283],a[10284],a[10285]],[a[10286]],[a[10287],a[10288],a[10289],a[10290],a[10291],a[10292],a[10293],a[10294],a[10295],a[10296],a[10297],a[10298]],[a[10299]],[a[10300],a[10301],a[10302],a[10303],a[10304],a[10305],a[10306],a[10307],a[10308],a[10309],a[10310],a[10311]],[a[10312],a[10313],a[10314],a[10315],a[10316],a[10317],a[10318],a[10319],a[10320],a[10321],a[10322],a[10323]],[a[10324]],[a[10325]],[a[10326],a[10327],a[10328],a[10329],a[10330],a[10331],a[10332],a[10333],a[10334],a[10335],a[10336],a[10337],a[10338],a[10339],a[10340],a[10341],a[10342],a[10343],a[10344],a[10345],a[10346],a[10347],a[10348],a[10349],a[10350],a[10351],a[10352],a[10353],a[10354],a[10355],a[10356],a[10357],a[10358],a[10359],a[10360],a[10361],a[10362],a[10363],a[10364],a[10365],a[10366],a[10367],a[10368],a[10369],a[10370],a[10371],a[10372],a[10373],a[10374],a[10375],a[10376],a[10377],a[10378],a[10379],a[10380],a[10381],a[10382],a[10383],a[10384],a[10385],a[10386],a[10387],a[10388],a[10389],a[10390],a[10391],a[10392],a[10393],a[10394],a[10395],a[10396],a[10397],a[10398],a[10399],a[10400],a[10401],a[10402],a[10403],a[10404],a[10405],a[10406],a[10407],a[10408],a[10409],a[10410],a[10411],a[10412],a[10413],a[10414],a[10415],a[10416],a[10417],a[10418],a[10419],a[10420],a[10421],a[10422],a[10423],a[10424],a[10425],a[10426],a[10427],a[10428],a[10429],a[10430],a[10431],a[10432],a[10433],a[10434],a[10435],a[10436],a[10437],a[10438],a[10439],a[10440],a[10441],a[10442],a[10443],a[10444],a[10445],a[10446],a[10447],a[10448],a[10449],a[10450],a[10451],a[10452],a[10453],a[10454],a[10455],a[10456],a[10457],a[10458],a[10459],a[10460],a[10461],a[10462],a[10463],a[10464],a[10465],a[10466],a[10467],a[10468],a[10469],a[10470],a[10471],a[10472],a[10473],a[10474],a[10475],a[10476],a[10477],a[10478],a[10479],a[10480],a[10481],a[10482],a[10483],a[10484],a[10485],a[10486],a[10487],a[10488],a[10489],a[10490],a[10491],a[10492],a[10493],a[10494],a[10495],a[10496],a[10497],a[10498],a[10499],a[10500],a[10501],a[10502],a[10503],a[10504],a[10505],a[10506],a[10507],a[10508],a[10509],a[10510],a[10511],a[10512],a[10513],a[10514],a[10515],a[10516],a[10517],a[10518],a[10519],a[10520],a[10521],a[10522],a[10523],a[10524],a[10525],a[10526],a[10527],a[10528],a[10529],a[10530],a[10531],a[10532],a[10533],a[10534],a[10535],a[10536],a[10537],a[10538],a[10539],a[10540],a[10541],a[10542],a[10543],a[10544],a[10545],a[10546],a[10547],a[10548],a[10549],a[10550],a[10551],a[10552],a[10553],a[10554],a[10555],a[10556],a[10557],a[10558],a[10559],a[10560],a[10561]],[a[10562],a[10563],a[10564],a[10565],a[10566],a[10567],a[10568],a[10569],a[10570],a[10571],a[10572],a[10573]],[a[10574]],[a[10575],a[10576]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[10577],a[10578],a[657],a[658],a[659],a[10579],a[10580],a[10581],a[651],a[652],a[10582],a[10583]],[a[10577],a[10584],a[668],a[669],a[670],a[10579],a[10580],a[10581],a[664],a[10585],a[10582],a[10586]],[a[10587],a[10588],a[10589],a[711],a[10590],a[10591],a[10592]],{"am":a[10593],"pm":a[10594]},[a[682],a[683],a[684],a[10595],a[686],a[687],a[688],a[689],a[690],a[691],a[692],a[693]],[a[694],a[695],a[684],a[10596],a[686],a[687],a[688],a[697],a[698],a[699],a[700],a[701]],[a[10587],a[10588],a[10597],a[711],a[10598],a[10591],a[10592]],[a[10599],a[10600],a[10601],a[10602],a[10603],a[10604],a[10605]],[a[10606],a[10600],a[10607],a[10608],a[10603],a[10609],a[10610]],{"am":a[10611],"pm":a[10612]},[a[10613],a[10614],a[10615],a[10616],a[9610],a[10617],a[10618],a[9612],a[7715],a[10619],a[10620],a[10621]],[a[10622],a[10623],a[10624],a[10625],a[10626],a[10627],a[10628],a[10629],a[10630],a[10631],a[10632],a[10633]],[a[9615],a[7983],a[7984],a[7301],a[7961],a[10634],a[7949]],[a[10635],a[10636],a[7988],a[7989],a[7990],a[10637],a[10638]],{"AUD":a[434],"BRL":a[435],"BWP":a[1227],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456],"ZAR":a[531]},[a[399],a[468],a[397],a[2498],a[397],a[399],a[399],a[469],a[399],a[470],a[471],a[398]],[a[10639],a[10640],a[10641],a[10642],a[10643],a[3032],a[10644],a[10645],a[481],a[10646],a[10647],a[10648]],[a[10649],a[10650],a[10651],a[10652],a[10643],a[10653],a[10654],a[10655],a[10656],a[10657],a[10658],a[10659]],[a[10660],a[10661],a[10662],a[10663],a[10664],a[10665],a[10666]],[a[10660],a[10661],a[10667],a[10663],a[10664],a[10665],a[10666]],[a[10668],a[10669],a[10670],a[10671],a[10672],a[10673],a[10674]],[a[10675],a[10676],a[1217],a[10677]],[a[1217],a[10677]],[a[10678],a[10679],a[1217],a[10677]],{"decimal":a[429],"group":a[430],"nan":a[10680],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TOP":a[3129],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"day":a[13],"month":a[14],"year":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[10681],"pattern12":a[10682]},{"day":a[13],"month":a[14],"year":a[13],"weekday":a[14],"pattern":a[10683]},[a[10684],a[10685],a[7795],a[10686],a[10687],a[10688],a[10689],a[10690],a[10691],a[10692],a[10693],a[10694],a[10695]],[a[57],a[10696],a[10697],a[60],a[410],a[10698],a[10699],a[7808],a[10700],a[66],a[67],a[10701],a[10702]],[a[470],a[10703],a[397],a[471],a[397],a[400],a[398],a[469],a[2498],a[2498],a[1467],a[469]],[a[10704],a[10705],a[475],a[10706],a[3018],a[10707],a[10708],a[10709],a[10710],a[10711],a[2592],a[2769]],[a[10712],a[10713],a[10714],a[119],a[10715],a[10716],a[10717],a[10718],a[10719],a[10720],a[10721],a[10722]],[a[10723],a[10724],a[509],a[10725],a[10726],a[10727],a[10728]],[a[10729],a[10730],a[9598],a[10731],a[10732],a[10733],a[4089]],[a[10734],a[10735],a[10736],a[10737],a[10738],a[10739],a[10740]],[a[10741],a[10742],a[10743],a[10744]],[a[10745],a[10746]],{"am":a[10747],"pm":a[10748]},[a[10749],a[10750],a[114],a[115],a[10751],a[10752],a[118],a[119],a[10753],a[121],a[122],a[123],a[124],a[125]],[a[10754],a[10755],a[10756],a[10757],a[10758],a[10759],a[10760],a[10761],a[10762],a[10763],a[10764],a[10765]],[a[10766]],[a[10767],a[10768],a[10769],a[410],a[411],a[10770],a[413],a[414],a[10771],a[416],a[10772],a[10773]],{"positivePattern":a[10774],"negativePattern":a[10775]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TRY":a[10776],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[3032],a[5544],a[10777],a[10778],a[8046],a[9572],a[2526],a[10779],a[10780],a[9571],a[10781],a[10782]],[a[10783],a[10784],a[10785],a[10786],a[10787],a[10788],a[10789],a[10790],a[10791],a[7722],a[10792],a[10793]],[a[7681],a[8909],a[10794],a[10795],a[7961],a[10614],a[2833]],[a[7733],a[10796],a[10797],a[10798],a[10799],a[10800],a[10801]],[a[5523],a[5526],a[475],a[10802],a[3018],a[4143],a[4144],a[5527],a[10803],a[10804],a[10805],a[10806]],[a[5532],a[10807],a[2758],a[10808],a[5536],a[5537],a[10809],a[5539],a[10810],a[10811],a[10812],a[10813]],[a[10814],a[10815],a[10816],a[10817],a[10818],a[10819],a[10820]],[a[10821],a[10822],a[10823],a[10824],a[10825],a[10826],a[10827]],[a[10828],a[10829]],[a[10830],a[10831]],{"am":a[10832],"pm":a[10833]},[a[10834],a[10835],a[10836],a[10837],a[10838],a[10839],a[10840],a[10841],a[10842],a[10843],a[10844],a[10845],a[10846]],[a[8482],a[10847],a[8484],a[10848],a[10849],a[1897],a[1898],a[8487],a[8488],a[8489],a[10850],a[8491],a[10851]],[a[10852],a[10853],a[10854],a[10855],a[10856],a[10857],a[10858],a[10859],a[10860],a[6429],a[10861],a[10862],a[10863]],[a[10864],a[10865],a[10866],a[10867],a[10868],a[10869],a[10870],a[10871],a[10872],a[10873],a[10874],a[10875],a[10876]],[a[6420],a[10877],a[10878],a[10879],a[10880],a[10881],a[10877],a[6420],a[10882],a[10883],a[10877],a[10884]],[a[10885],a[10886],a[10887],a[10888],a[10889],a[10890],a[10891],a[10892],a[10893],a[10894],a[10895],a[10896]],[a[10897],a[10898],a[10899],a[10900],a[10901],a[10902],a[10903],a[10904],a[10905],a[10906],a[10907],a[10908]],[a[10909],a[10910],a[10911],a[10912],a[10913],a[10914],a[8850]],[a[10915],a[10916],a[10917],a[10918],a[10919],a[10920],a[1441]],[a[10921],a[1952]],[a[10922],a[10923]],{"am":a[10924],"pm":a[10925]},[a[10926],a[10927],a[10928],a[10929],a[1963],a[1964],a[1965],a[10930],a[10931],a[1968],a[8523],a[10932],a[1971],a[1972]],[a[10933],a[10934],a[10935],a[10936],a[10937],a[10938],a[10939],a[10940],a[10941],a[10942],a[10943],a[10944]],[a[1564],a[1565],a[10945],a[1567],a[10946],a[10947],a[10948],a[10949],a[10950],a[10951],a[1574],a[10952]],[a[10953],a[10954],a[10955],a[10956],a[10957],a[10958],a[10959],a[10960],a[10961],a[10962],a[10963],a[10964]],[a[8534],a[1999],a[10965],a[10966],a[10967],a[10968],a[8539],a[8540],a[2006],a[10969],a[10970],a[10971]],[a[10972],a[10973],a[10974],a[10975],a[10976],a[10977],a[10978],a[10979],a[10980],a[10981],a[10982],a[10983],a[10984],a[10985],a[10986],a[10987],a[10988],a[10989],a[10990],a[10991],a[10992],a[10993],a[10994],a[10995],a[10996],a[10997],a[10998],a[10999],a[11000],a[11001],a[11002],a[11003],a[11004],a[11005],a[11006],a[11007],a[11008],a[11009],a[11010],a[11011],a[11012],a[11013],a[11014],a[11015],a[11016],a[11017],a[11018],a[11019],a[11020],a[11021],a[11022],a[11023],a[11024],a[11025],a[11026],a[11027],a[11028],a[11029],a[11030],a[11031],a[11032],a[11033],a[11034],a[11035],a[11036],a[11037],a[11038],a[11039],a[11040],a[11041],a[11042],a[11043],a[11044],a[11045],a[11046],a[11047],a[11048],a[11049],a[11050],a[11051],a[11052],a[11053],a[11054],a[11055],a[11056],a[11057],a[11058],a[11059],a[11060],a[11061],a[11062],a[11063],a[11064],a[11065],a[11066],a[11067],a[11068],a[11069],a[11070],a[11071],a[11072],a[11073],a[11074],a[11075],a[11076],a[11077],a[11078],a[11079],a[11080],a[11081],a[11082],a[11083],a[11084],a[11085],a[11086],a[11087],a[11088],a[11089],a[11090],a[11091],a[11092],a[11093],a[11094],a[11095],a[11096],a[11097],a[11098],a[11099],a[11100],a[11101],a[11102],a[11103],a[11104],a[11105],a[11106],a[11107],a[11108],a[11109],a[11110],a[11111],a[11112],a[11113],a[11114],a[11115],a[11116],a[11117],a[11118],a[11119],a[11120],a[11121],a[11122],a[11123],a[11124],a[11125],a[11126],a[11127],a[11128],a[11129],a[11130],a[11131],a[11132],a[11133],a[11134],a[11135],a[11136],a[11137],a[11138],a[11139],a[11140],a[11141],a[11142],a[11143],a[11144],a[11145],a[11146],a[11147],a[11148],a[11149],a[11150],a[11151],a[11152],a[11153],a[11154],a[11155],a[11156],a[11157],a[11158],a[11159],a[11160],a[11161],a[11162],a[11163],a[11164],a[11165],a[11166],a[11167],a[11168],a[11169],a[11170],a[11171],a[11172],a[11173],a[11174],a[11175],a[11176],a[11177],a[11178],a[11179],a[11180],a[11181],a[11182],a[11183],a[11184],a[11185],a[11186],a[11187],a[11188],a[11189],a[11190],a[11191],a[11192],a[11193],a[11194],a[11195],a[11196],a[11197],a[11198],a[11199],a[11200],a[11201],a[11202],a[11203],a[11204],a[11205],a[8778],a[11206]],[a[11207],a[11208],a[11209],a[11210],a[11211],a[11212],a[11213],a[11214],a[11215],a[11216],a[11217],a[11218]],[a[11219],a[11220],a[8781],a[11221],a[2250],a[11222],a[2252],a[2253],a[8783],a[8784],a[2256],a[2257]],{"decimal":a[430],"group":a[529],"nan":a[11223],"percent":a[432],"infinity":a[433]},{"AUD":a[732],"AZN":a[1346],"EUR":a[440],"GBP":a[441],"JPY":a[1275],"RSD":a[2264],"RUB":a[1589],"UAH":a[8788],"UAK":a[11224],"XAF":a[453],"XOF":a[455],"XPF":a[456]},[a[3371],a[8171],a[3373],a[3374],a[11225],a[3376],a[8173],a[3378],a[6278],a[3380],a[3381],a[3382]],[a[8174],a[11226],a[8176],a[11227],a[8178],a[8179],a[8180]],[a[8174],a[8175],a[8176],a[11227],a[8178],a[8179],a[8180]],[a[11228],a[11229],a[1184],a[1185]],[a[11230],a[11229]],{"am":a[11231],"pm":a[11232]},[a[829],a[830],a[11233],a[11234],a[3418],a[3419],a[835],a[836],a[837],a[838],a[11235],a[11236]],[a[11228],a[11229]],{"decimal":a[429],"group":a[430],"nan":a[11237],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"PKR":a[3122],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"positivePattern":a[11238],"negativePattern":a[11239]},[a[11240],a[468],a[397],a[469],a[397],a[2496],a[2496],a[469],a[399],a[470],a[471],a[472]],[a[11241],a[3870],a[475],a[476],a[3018],a[11242],a[11243],a[11244],a[4857],a[482],a[11245],a[11246]],[a[11247],a[11248],a[10714],a[11249],a[3018],a[11242],a[11243],a[11250],a[11251],a[11252],a[11253],a[11254]],[a[11255],a[11256],a[11257],a[11258],a[8002],a[2612],a[11259]],[a[11260],a[11261],a[11262],a[11263],a[11264],a[1623],a[11265]],[a[11266],a[2498]],[a[152],a[153],a[11267],a[11268],a[11269],a[11270],a[158],a[11271],a[11272],a[11273],a[11274],a[11275]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"UZS":a[11276],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[992],a[11277],a[11278],a[11279],a[3363],a[3376],a[3365],a[11280],a[11281],a[11282],a[11283],a[3370]],[a[11284],a[11285],a[11286],a[11287],a[11288],a[11289],a[11290]],{"AFN":a[3450],"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"UZS":a[11276],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[11291],a[11292],a[11293],a[11294],a[11295],a[6452],a[11296]],[a[11297],a[11298],a[11299],a[11300],a[11301],a[11302],a[11303]],[a[11304],a[11305]],[a[11306],a[1999],a[11307],a[11308],a[11309],a[11310],a[11311],a[11312],a[11313],a[11314],a[11315],a[11316]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"UZS":a[11317],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[11318],a[11319],a[11320],a[11321],a[11322],a[643],a[644],a[11323],a[11324],a[11325],a[11326],a[11327]],[a[11328],a[11329],a[11330],a[11331],a[11332],a[11333],a[11334]],{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[446],"LRD":a[530],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[11335],a[11336],a[11337],a[11338],a[11339],a[643],a[644],a[11340],a[11341],a[11342],a[11343],a[11344]],[a[11345],a[11346],a[11347],a[11348],a[11349],a[11350],a[11351]],[a[9612],a[11352],a[11353],a[11354],a[11355],a[11356],a[11357],a[11358],a[11359],a[9615],a[11360],a[9010]],[a[11361],a[11362],a[11363],a[11364],a[11365],a[11366],a[11367],a[11368],a[11369],a[11370],a[11371],a[11372]],[a[11373],a[8909],a[11374],a[7301],a[11375],a[11376],a[2833]],[a[11377],a[11378],a[11379],a[11380],a[11381],a[11382],a[11383]],{"hour":a[13],"minute":a[15],"second":a[15],"weekday":a[14],"day":a[15],"month":a[14],"year":a[13],"pattern":a[11384],"pattern12":a[11385]},{"day":a[15],"month":a[13],"pattern":a[1778]},[a[11386],a[11387],a[11388],a[11389],a[11390],a[11391],a[11392],a[11393],a[11394],a[11395],a[11396],a[11397]],[a[11398],a[11399],a[11400],a[11401],a[11402],a[11403],a[11404],a[11405],a[11406],a[11407],a[11408],a[11409]],[a[11410],a[11411],a[11412],a[11413],a[11414],a[11415],a[11416]],[a[11410],a[11417],a[11418],a[11419],a[11420],a[11421],a[11422]],[a[11423],a[11424],a[11425],a[11426],a[11427],a[11428],a[11429]],[a[11430],a[11431]],{"am":a[3185],"pm":a[11432]},[a[11433],a[421]],[a[1826],a[1229],a[11434],a[11435],a[1232],a[6552],a[6553],a[11436],a[1236],a[11437],a[1832],a[1291]],[a[11438],a[11439],a[11440],a[11441],a[11442],a[11443],a[11444],a[11445],a[11446],a[11447],a[11448],a[11449]],[a[4948],a[11450],a[11451],a[11452],a[11453],a[7700],a[11454]],[a[11455],a[11456],a[11457],a[11458],a[11459],a[11460],a[11461]],[a[11462],a[11463]],{"am":a[11464],"pm":a[11465]},{"day":a[13],"month":a[11466],"pattern":a[1825]},[a[467],a[400],a[397],a[469],a[397],a[3046],a[400],a[11467],a[400],a[11468],a[11468],a[1308]],[a[4876],a[11469],a[3999],a[3573],a[477],a[11470],a[11471],a[11472],a[11473],a[11474],a[11475],a[11476]],[a[11477],a[11478],a[11479],a[11480],a[11481],a[11482],a[11483],a[11484],a[11485],a[11486],a[11487],a[11488]],[a[3032],a[11489],a[11490],a[11491],a[11492],a[3037],a[5548]],[a[11493],a[11494],a[11495],a[11496],a[11497],a[11498],a[11499]],[a[2654],a[11500]],{"decimal":a[430],"group":a[4020],"nan":a[431],"percent":a[432],"infinity":a[433]},{"AUD":a[434],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[446],"MXN":a[447],"NZD":a[448],"RUB":a[11501],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[11502],a[11503],a[11504],a[11505],a[11506],a[11507],a[11508]],[a[11509],a[11510]],{"am":a[11511],"pm":a[11512]},[a[473],a[474],a[7949],a[1448],a[7270],a[478],a[479],a[11513],a[481],a[482],a[483],a[1450]],[a[11514],a[11515],a[11516],a[11517],a[7956],a[1196],a[7957],a[7227],a[1199],a[11518],a[1201],a[1453]],[a[11519],a[7729],a[11520],a[5567],a[7730],a[7962],a[7732]],[a[11521],a[7734],a[11522],a[11523],a[11524],a[11525],a[7739]],[a[1461],a[11526]],[a[11527],a[11528],a[11529],a[11530],a[11531],a[11532],a[11533]],[a[11534],a[6522],a[11535],a[11536],a[11537],a[11538],a[11539]],[a[11540],a[11541]],[a[11542],a[11543]],{"am":a[11544],"pm":a[11545]},[a[11546],a[11547],a[11548],a[11549],a[11550],a[11551],a[11552],a[11553],a[11554],a[11555],a[11556],a[11557]],[a[11558],a[11559],a[11560],a[11561],a[11562],a[11563],a[11564],a[11565],a[11566],a[11567],a[11568],a[11569]],[a[11570],a[11571],a[11572],a[11573],a[11574],a[11575],a[11576]],[a[11577],a[11578],a[11579],a[11580],a[11581],a[11582],a[11583]],[a[6349],a[11584]],[a[11585],a[11586]],{"am":a[11587],"pm":a[11588]},[a[11589],a[11590],a[11591],a[11592],a[11593],a[11594],a[11595],a[11596],a[11597],a[11598],a[11599],a[11600]],[a[11601],a[11602],a[11603],a[11604],a[11605],a[11606],a[11607],a[11608],a[11609],a[11610],a[11611],a[11612]],[a[11613],a[11614],a[11615],a[11616],a[11617],a[11618],a[11619]],[a[11620],a[11621],a[11622],a[11616],a[11617],a[11623],a[11624]],[a[576],a[9050]],[a[11625],a[11626]],{"am":a[11627],"pm":a[11628]},[a[11629],a[11590],a[11630],a[11592],a[11631],a[11594],a[11632],a[11596],a[11597],a[11633],a[11599],a[11634]],[a[11635],a[11636],a[11637],a[11638],a[11639],a[11640],a[11641],a[11642],a[11643],a[11644],a[11645],a[11646]],[a[11613],a[11614],a[11647],a[11648],a[11649],a[11650],a[11651]],[a[11652],a[11653],a[11654],a[11648],a[11649],a[11655],a[11656]],{"am":a[11657],"pm":a[11658]},[a[9082],a[9083],a[9084],a[9085],a[9086],a[11659],a[9088]],{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[11660],"pattern12":a[11661]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[11662]},[a[11663]],[a[5054],a[5055],a[5056],a[5057],a[5058],a[5059],a[5060],a[5061],a[5062],a[5063],a[5064],a[5065],a[11664]],[a[11665],a[5017],a[5018],a[5019],a[5020],a[5021],a[5022],a[5023],a[5024],a[5025],a[5026],a[5027],a[11666]],[a[11665],a[5017],a[5018],a[5019],a[5020],a[5021],a[5022],a[5023],a[5024],a[5025],a[5026],a[5027]],[a[11667],a[11668],a[11669],a[11670],a[11671],a[11672],a[11673]],[a[11674],a[11675],a[11676],a[11677],a[11678],a[11679],a[11680]],[a[11681],a[11682]],{"am":a[11683],"pm":a[11684]},[a[5054],a[5055],a[5056],a[5057],a[5058],a[5059],a[5060],a[5061],a[5062],a[5063],a[5064],a[5065],a[11664],a[11685]],[a[11665],a[5017],a[5018],a[5019],a[5020],a[5021],a[5022],a[5023],a[5024],a[5025],a[5026],a[5027],a[11666],a[11686]],[a[11687]],[a[11688]],[a[11689]],[a[11690],a[11691],a[11692],a[11693],a[11694],a[11695],a[11696],a[11697],a[11698],a[11699],a[11700],a[11701],a[11702],a[11703],a[11704],a[11705],a[11706],a[11707],a[11708],a[11709],a[11710],a[11711],a[11712],a[11713],a[11714],a[11715],a[11716],a[11717],a[11718],a[11719],a[11720],a[11721],a[11722],a[11723],a[11724],a[11725],a[11726],a[11727],a[11728],a[11729],a[11730],a[11731],a[11732],a[11733],a[11734],a[11735],a[11736],a[11737],a[11738],a[11739],a[11740],a[11741],a[11742],a[11743],a[11744],a[11745],a[11746],a[11747],a[11748],a[11749],a[11750],a[11751],a[11752],a[11753],a[11754],a[11755],a[11756],a[11757],a[11758],a[11759],a[11760],a[11761],a[11762],a[11763],a[11764],a[11765],a[11766],a[11767],a[11768],a[11769],a[11770],a[11771],a[11772],a[11773],a[11774],a[11775],a[11776],a[11777],a[11778],a[11779],a[11780],a[11781],a[11782],a[11783],a[11784],a[11785],a[11786],a[11787],a[11788],a[11789],a[11790],a[11791],a[11792],a[11793],a[11794],a[11795],a[11796],a[11797],a[11798],a[11799],a[11800],a[11801],a[11802],a[11803],a[11804],a[11805],a[11806],a[11807],a[11808],a[11809],a[11810],a[11811],a[11812],a[11813],a[11814],a[11815],a[11816],a[11817],a[11818],a[11819],a[11820],a[11821],a[11822],a[11823],a[11824],a[11825],a[11826],a[11827],a[11828],a[11829],a[11830],a[11831],a[11832],a[11833],a[11834],a[11835],a[11836],a[11837],a[11838],a[11839],a[11840],a[11841],a[11842],a[11843],a[11844],a[11845],a[11846],a[11847],a[11848],a[11849],a[11850],a[11851],a[11852],a[11853],a[11854],a[11855],a[11856],a[11857],a[11858],a[11859],a[11860],a[11861],a[11862],a[11863],a[11864],a[11865],a[11866],a[11867],a[11868],a[11869],a[11870],a[11871],a[11872],a[11873],a[11874],a[11875],a[11876],a[11877],a[11878],a[11879],a[11880],a[11881],a[11882],a[11883],a[11884],a[11885],a[11886],a[11887],a[11888],a[11889],a[11890],a[11891],a[11892],a[11893],a[11894],a[11895],a[11896],a[11897],a[11898],a[11899],a[11900],a[11901],a[11902],a[11903],a[11904],a[11905],a[11906],a[11907],a[11908],a[11909],a[11910],a[11911],a[11912],a[11913],a[11914],a[11915],a[11916],a[11917],a[11918],a[11919],a[11920],a[11921],a[5356],a[5357],a[5358],a[5359]],[a[11922]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[5375],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[5376],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[5375],"EUR":a[440],"GBP":a[441],"HKD":a[530],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[5376],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"year":a[13],"month":a[13],"day":a[13],"pattern":a[4998]},{"year":a[13],"month":a[13],"pattern":a[4999]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[5376],"MOP":a[3119],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[5376],"MXN":a[447],"NZD":a[448],"SGD":a[530],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"hour":a[13],"minute":a[15],"second":a[15],"pattern":a[11923],"pattern12":a[11924]},{"year":a[13],"month":a[14],"day":a[13],"weekday":a[14],"pattern":a[11925]},{"year":a[13],"month":a[14],"day":a[13],"pattern":a[11926]},{"year":a[13],"month":a[14],"pattern":a[11927]},[a[11928],a[5005],a[5006],a[5007],a[5008],a[5009],a[5010],a[5011],a[5012],a[5013],a[5014],a[5015]],{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[5375],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[445],"KRW":a[5376],"MOP":a[3119],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[451],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[11929]],[a[5066],a[11928],a[5005],a[5006],a[5007],a[5008],a[5009]],[a[11930],a[11931],a[11932],a[11933],a[11934],a[11935],a[11936]],[a[11937],a[11938],a[11681],a[11682]],[a[11939],a[11940],a[11941],a[11942],a[11943],a[11944],a[11945],a[11946],a[11947],a[11948],a[11949],a[11950],a[11951],a[11952]],[a[11953]],[a[11954],a[11955],a[11956],a[11957],a[11958],a[11959],a[11960],a[11961],a[11962],a[11963],a[11964],a[11965]],[a[11966]],[a[11967],a[11968],a[11969],a[11970],a[11971],a[11972],a[11973],a[11974],a[11975],a[11976],a[11977],a[11978]],[a[11979]],[a[5124],a[5125],a[11980],a[5127],a[11981],a[5129],a[5130],a[11982],a[5132],a[11983],a[5134],a[11984],a[11985],a[11986],a[5138],a[5139],a[11987],a[11988],a[11989],a[5143],a[5144],a[5145],a[5146],a[5147],a[11990],a[11991],a[5150],a[11992],a[5152],a[5153],a[11993],a[5155],a[5156],a[5157],a[5158],a[5159],a[11994],a[11995],a[11996],a[5163],a[5164],a[11997],a[5166],a[5167],a[5168],a[11998],a[11999],a[5171],a[5172],a[12000],a[12001],a[5175],a[12002],a[5177],a[12003],a[5179],a[12004],a[5181],a[12005],a[5183],a[12006],a[5185],a[5186],a[5187],a[12007],a[5189],a[5190],a[12008],a[5192],a[12009],a[12010],a[5195],a[5196],a[12011],a[5198],a[5199],a[5200],a[5201],a[5202],a[5203],a[5204],a[5205],a[5206],a[5207],a[5208],a[5209],a[5210],a[5211],a[5212],a[5213],a[5214],a[5215],a[12012],a[5217],a[5218],a[12013],a[12014],a[12015],a[12016],a[5223],a[12017],a[5225],a[5226],a[5227],a[5228],a[12018],a[12019],a[5231],a[5232],a[5233],a[5234],a[5235],a[5236],a[5237],a[12020],a[5239],a[5240],a[12021],a[5242],a[12022],a[5244],a[12023],a[5246],a[5247],a[12024],a[5249],a[12025],a[12026],a[5252],a[12027],a[12028],a[5255],a[5256],a[5257],a[5258],a[12029],a[5260],a[5261],a[5262],a[5263],a[12030],a[5265],a[5266],a[5267],a[5268],a[12031],a[5270],a[12032],a[5272],a[5273],a[12033],a[5275],a[5276],a[12034],a[12035],a[5279],a[5280],a[5281],a[12036],a[5283],a[12037],a[5285],a[5286],a[12038],a[5288],a[5289],a[12039],a[5291],a[12040],a[12041],a[12042],a[5295],a[5296],a[5297],a[5298],a[12043],a[12044],a[5301],a[12045],a[12046],a[5304],a[12047],a[5306],a[5307],a[12048],a[12049],a[12050],a[5311],a[5312],a[12051],a[5314],a[5315],a[12052],a[12053],a[5318],a[12054],a[5320],a[5321],a[12055],a[5323],a[5324],a[12056],a[12057],a[12058],a[12059],a[12060],a[5330],a[5331],a[12061],a[12062],a[12063],a[5335],a[5336],a[12064],a[5338],a[12065],a[12066],a[5341],a[5342],a[5343],a[12067],a[5345],a[5346],a[5347],a[5348],a[5349],a[5350],a[5351],a[12068],a[5353],a[5354],a[12069],a[5356],a[5357],a[5358],a[5359]],[a[12070]],[a[12071],a[12072]],{"decimal":a[429],"group":a[430],"nan":a[12073],"percent":a[432],"infinity":a[433]},{"AUD":a[732],"BRL":a[435],"CAD":a[436],"CNY":a[437],"EUR":a[440],"GBP":a[441],"HKD":a[442],"ILS":a[443],"INR":a[444],"JPY":a[1275],"KRW":a[5376],"MXN":a[447],"NZD":a[448],"THB":a[449],"TWD":a[450],"USD":a[530],"VND":a[452],"XAF":a[453],"XCD":a[454],"XOF":a[455],"XPF":a[456]},[a[473],a[474],a[12074],a[476],a[7270],a[478],a[479],a[11513],a[481],a[482],a[483],a[1450]],[a[12075],a[11515],a[12076],a[12077],a[7956],a[1196],a[7957],a[7227],a[12078],a[11518],a[1201],a[1453]],[a[12079],a[12080],a[12081],a[12082],a[12083],a[12084],a[12085]],[a[7681],a[9588],a[7960],a[5567],a[7730],a[7962],a[7732]],[a[7733],a[12086],a[12087],a[11523],a[11524],a[11525],a[7739]],{"am":a[12088],"pm":a[12089]},{"decimal":a[429],"group":a[430],"nan":a[12090],"percent":a[432],"infinity":a[433]}];b[1]=[[b[0][1],b[0][2],b[0][3],b[0][4],b[0][5],b[0][6],b[0][7],b[0][8],b[0][9],b[0][10]],{"short":b[0][11]},{"long":b[0][12]},{"long":b[0][13]},{"short":b[0][14]},{"long":b[0][15]},{"short":b[0][16]},{"short":b[0][17],"long":b[0][18]},{"short":b[0][19],"long":b[0][20]},{"short":b[0][21]},{"long":b[0][23]},{"short":b[0][24]},{"long":b[0][25]},{"short":b[0][26]},{"short":b[0][27],"long":b[0][28]},{"short":b[0][29]},{"narrow":b[0][30],"short":b[0][31]},{"long":b[0][32]},{"short":b[0][33]},{"short":b[0][34]},{"decimal":b[0][36],"currency":b[0][37],"percent":b[0][38]},{"latn":b[0][39]},{"short":b[0][17],"long":b[0][41]},[b[0][44],b[0][45],b[0][46],b[0][47],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][52],"long":b[0][53]},{"narrow":b[0][54],"short":b[0][55],"long":b[0][56]},{"narrow":b[0][57],"short":b[0][58],"long":b[0][59]},{"decimal":b[0][36],"currency":b[0][61],"percent":b[0][38]},{"latn":b[0][62]},[b[0][65],b[0][66],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][70],"long":b[0][71]},{"short":b[0][72],"long":b[0][73]},{"short":b[0][74],"long":b[0][75]},{"decimal":b[0][36],"currency":b[0][77],"percent":b[0][38]},[b[0][79],b[0][80],b[0][3],b[0][81],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][82],"long":b[0][83]},{"short":b[0][84],"long":b[0][85]},{"short":b[0][86],"long":b[0][87]},[b[0][90],b[0][91],b[0][46],b[0][67],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][92]},{"narrow":b[0][93],"short":b[0][94],"long":b[0][95]},{"narrow":b[0][96],"short":b[0][97],"long":b[0][98]},{"narrow":b[0][99],"short":b[0][100],"long":b[0][101]},{"short":b[0][92],"long":b[0][102]},{"short":b[0][27],"long":b[0][104]},[b[0][106],b[0][107],b[0][108],b[0][109],b[0][110],b[0][49],b[0][68],b[0][111],b[0][9],b[0][10]],{"short":b[0][112]},{"narrow":b[0][93],"long":b[0][113]},{"long":b[0][114]},{"narrow":b[0][115],"short":b[0][116],"long":b[0][116]},{"narrow":b[0][117],"short":b[0][117],"long":b[0][117]},{"short":b[0][118],"long":b[0][119]},{"long":b[0][121]},{"short":b[0][122]},{"narrow":b[0][123],"short":b[0][124],"long":b[0][124]},{"short":b[0][125]},{"narrow":b[0][30],"short":b[0][126]},{"long":b[0][127]},{"short":b[0][128]},{"short":b[0][129]},{"arab":b[0][131],"latn":b[0][39]},[b[0][106],b[0][107],b[0][108],b[0][81],b[0][134],b[0][49],b[0][68],b[0][50],b[0][9],b[0][10]],{"narrow":b[0][135],"short":b[0][136],"long":b[0][136]},{"arab":b[0][131],"latn":b[0][137]},{"narrow":b[0][139],"short":b[0][140],"long":b[0][141]},{"narrow":b[0][139],"short":b[0][141],"long":b[0][141]},{"narrow":b[0][143],"short":b[0][144],"long":b[0][144]},{"narrow":b[0][145],"short":b[0][146],"long":b[0][146]},{"short":b[0][149],"long":b[0][150]},{"short":b[0][151],"long":b[0][152]},{"short":b[0][153]},{"decimal":b[0][36],"currency":b[0][61],"percent":b[0][38],"secondaryGroupSize":2},[b[0][79],b[0][80],b[0][46],b[0][67],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][156],"long":b[0][157]},{"short":b[0][158],"long":b[0][159]},{"short":b[0][160],"long":b[0][161]},{"decimal":b[0][36],"currency":b[0][163],"percent":b[0][38]},[b[0][44],b[0][45],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][165],"short":b[0][166],"long":b[0][167]},{"narrow":b[0][168],"short":b[0][169],"long":b[0][170]},{"short":b[0][171]},{"latn":b[0][137]},[b[0][174],b[0][175],b[0][46],b[0][176],b[0][177],b[0][49],b[0][68],b[0][178],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][180],"long":b[0][181]},{"narrow":b[0][182],"short":b[0][182],"long":b[0][183]},{"short":b[0][184],"long":b[0][185]},[b[0][187],b[0][188],b[0][189],b[0][176],b[0][177],b[0][190],b[0][68],b[0][178],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][180],"long":b[0][191]},{"narrow":b[0][182],"short":b[0][182],"long":b[0][192]},{"short":b[0][194],"long":b[0][195]},{"short":b[0][196],"long":b[0][197]},{"short":b[0][198],"long":b[0][199]},{"decimal":b[0][36],"currency":b[0][163],"percent":b[0][201]},[b[0][202],b[0][45],b[0][46],b[0][203],b[0][204],b[0][49],b[0][68],b[0][205],b[0][206],b[0][207]],{"narrow":b[0][208],"short":b[0][209],"long":b[0][210]},{"short":b[0][211],"long":b[0][212]},{"short":b[0][213]},{"short":b[0][216],"long":b[0][217]},{"long":b[0][218]},{"short":b[0][219],"long":b[0][220]},{"short":b[0][223],"long":b[0][224]},{"short":b[0][225],"long":b[0][226]},{"short":b[0][160],"long":b[0][227]},[b[0][229],b[0][45],b[0][46],b[0][230],b[0][204],b[0][49],b[0][68],b[0][231],b[0][9],b[0][10]],{"narrow":b[0][232],"short":b[0][233],"long":b[0][234]},{"narrow":b[0][235],"short":b[0][235],"long":b[0][236]},{"narrow":b[0][237],"short":b[0][238],"long":b[0][239]},{"long":b[0][241]},{"long":b[0][242]},{"short":b[0][27],"long":b[0][243]},{"short":b[0][245],"long":b[0][246]},{"short":b[0][247],"long":b[0][248]},{"short":b[0][249],"long":b[0][250]},[b[0][251],b[0][252],b[0][189],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][253],"short":b[0][254],"long":b[0][254]},{"narrow":b[0][255],"short":b[0][256],"long":b[0][257]},{"short":b[0][258]},{"long":b[0][260]},{"short":b[0][261]},{"short":b[0][27],"long":b[0][262]},{"short":b[0][263]},{"decimal":b[0][36],"currency":b[0][264],"percent":b[0][38],"secondaryGroupSize":2},{"beng":b[0][265],"latn":b[0][39]},[b[0][1],b[0][2],b[0][267],b[0][4],b[0][5],b[0][6],b[0][7],b[0][8],b[0][9],b[0][10]],{"short":b[0][268],"long":b[0][269]},{"short":b[0][270],"long":b[0][271]},{"short":b[0][272]},{"latn":b[0][39],"tibt":b[0][274]},{"short":b[0][276],"long":b[0][277]},{"short":b[0][278],"long":b[0][279]},[b[0][79],b[0][80],b[0][3],b[0][4],b[0][48],b[0][49],b[0][280],b[0][50],b[0][9],b[0][10]],{"long":b[0][281]},{"short":b[0][282],"long":b[0][283]},{"short":b[0][284]},{"long":b[0][286]},[b[0][287],b[0][288],b[0][289],b[0][176],b[0][177],b[0][49],b[0][290],b[0][178],b[0][291],b[0][292]],{"short":b[0][293],"long":b[0][294]},{"short":b[0][295],"long":b[0][296]},{"short":b[0][297],"long":b[0][298]},{"short":b[0][300],"long":b[0][301]},{"short":b[0][303]},{"long":b[0][304]},{"long":b[0][305]},{"short":b[0][306],"long":b[0][307]},{"short":b[0][308],"long":b[0][309]},{"narrow":b[0][310],"short":b[0][311],"long":b[0][312]},{"long":b[0][314]},{"long":b[0][315]},{"short":b[0][316]},{"short":b[0][317],"long":b[0][318]},{"short":b[0][319]},{"narrow":b[0][30],"short":b[0][320]},{"long":b[0][321]},{"short":b[0][322]},{"short":b[0][323]},{"short":b[0][325],"long":b[0][326]},{"short":b[0][327],"long":b[0][328]},{"short":b[0][329]},[b[0][251],b[0][252],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][332]},{"narrow":b[0][179],"short":b[0][179],"long":b[0][179]},{"narrow":b[0][333],"short":b[0][334],"long":b[0][335]},{"narrow":b[0][336],"short":b[0][336],"long":b[0][337]},{"narrow":b[0][338],"short":b[0][339],"long":b[0][340]},{"short":b[0][344],"long":b[0][345]},{"short":b[0][346],"long":b[0][347]},{"short":b[0][219],"long":b[0][348]},[b[0][79],b[0][80],b[0][350],b[0][47],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][351],"long":b[0][352]},{"short":b[0][353],"long":b[0][354]},{"short":b[0][355],"long":b[0][356]},[b[0][359],b[0][360],b[0][361],b[0][362],b[0][48],b[0][49],b[0][363],b[0][364],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][365],"long":b[0][366]},{"narrow":b[0][367],"short":b[0][367],"long":b[0][368]},{"narrow":b[0][369],"short":b[0][370],"long":b[0][370]},{"short":b[0][371]},[b[0][44],b[0][45],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][207]],{"narrow":b[0][373],"short":b[0][374],"long":b[0][375]},{"narrow":b[0][376],"short":b[0][377],"long":b[0][378]},{"narrow":b[0][379],"short":b[0][380],"long":b[0][381]},[b[0][382],b[0][383],b[0][384],b[0][67],b[0][48],b[0][49],b[0][363],b[0][69],b[0][206],b[0][207]],{"narrow":b[0][51],"short":b[0][385],"long":b[0][386]},{"narrow":b[0][387],"short":b[0][388],"long":b[0][389]},{"narrow":b[0][390],"short":b[0][391],"long":b[0][392]},{"short":b[0][393]},{"short":b[0][395],"long":b[0][396]},{"short":b[0][397],"long":b[0][398]},{"short":b[0][399],"long":b[0][400]},[b[0][403],b[0][404],b[0][384],b[0][203],b[0][204],b[0][49],b[0][363],b[0][205],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][405],"long":b[0][406]},{"narrow":b[0][407],"short":b[0][407],"long":b[0][408]},{"narrow":b[0][409],"short":b[0][410],"long":b[0][411]},{"decimal":b[0][36],"currency":b[0][61],"percent":b[0][201]},{"narrow":b[0][51],"short":b[0][414],"long":b[0][415]},{"decimal":b[0][36],"currency":b[0][416],"percent":b[0][201]},{"latn":b[0][417]},{"short":b[0][419],"long":b[0][420]},{"short":b[0][421],"long":b[0][422]},{"short":b[0][423],"long":b[0][424]},{"latn":b[0][426]},{"short":b[0][427],"long":b[0][428]},{"short":b[0][429],"long":b[0][430]},{"short":b[0][431],"long":b[0][432]},{"short":b[0][434],"long":b[0][435]},{"short":b[0][436],"long":b[0][437]},{"short":b[0][438],"long":b[0][439]},[b[0][440],b[0][441],b[0][3],b[0][442],b[0][443],b[0][444],b[0][445],b[0][446],b[0][9],b[0][10]],{"narrow":b[0][447],"short":b[0][448],"long":b[0][449]},{"narrow":b[0][450],"short":b[0][450],"long":b[0][451]},{"decimal":b[0][36],"currency":b[0][37],"percent":b[0][201],"secondaryGroupSize":2},{"latn":b[0][39],"tibt":b[0][454]},{"short":b[0][456],"long":b[0][457]},{"short":b[0][458],"long":b[0][459]},{"short":b[0][460],"long":b[0][461]},[null,b[0][80],b[0][463],b[0][47],b[0][48],b[0][49],b[0][464],b[0][50],null,null],{"short":b[0][465],"long":b[0][466]},{"short":b[0][467],"long":b[0][468]},{"short":b[0][468],"long":b[0][468]},{"narrow":b[0][469],"short":b[0][465],"long":b[0][470]},{"narrow":b[0][471],"short":b[0][471],"long":b[0][472]},{"narrow":b[0][473],"short":b[0][473],"long":b[0][474]},{"short":b[0][476],"long":b[0][477]},{"short":b[0][478]},{"short":b[0][465],"long":b[0][479]},{"short":b[0][480]},{"latn":b[0][481]},[b[0][483],b[0][45],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][484],"short":b[0][485],"long":b[0][486]},{"narrow":b[0][487],"short":b[0][488],"long":b[0][489]},{"short":b[0][490]},{"short":b[0][492]},{"short":b[0][493]},[b[0][495],b[0][80],b[0][350],b[0][47],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][496],"long":b[0][497]},{"short":b[0][498],"long":b[0][499]},{"narrow":b[0][500],"short":b[0][501],"long":b[0][502]},{"narrow":b[0][503],"short":b[0][504],"long":b[0][505]},[b[0][507],b[0][66],b[0][350],b[0][67],b[0][48],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],[b[0][495],b[0][80],b[0][350],b[0][67],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],[b[0][512],b[0][513],b[0][514],b[0][515],b[0][516],b[0][49],b[0][517],b[0][518],b[0][9],b[0][10]],[b[0][495],b[0][80],b[0][350],b[0][4],b[0][5],b[0][49],b[0][7],b[0][8],b[0][9],b[0][10]],{"narrow":b[0][524],"short":b[0][525],"long":b[0][526]},{"narrow":b[0][500],"short":b[0][527],"long":b[0][528]},{"narrow":b[0][529],"short":b[0][530],"long":b[0][531]},[b[0][44],b[0][45],b[0][46],b[0][515],b[0][516],b[0][49],b[0][68],b[0][518],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][25],"long":b[0][25]},{"short":b[0][537]},{"narrow":b[0][179],"short":b[0][27],"long":b[0][28]},[b[0][543],b[0][252],b[0][189],b[0][67],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"narrow":b[0][30],"short":b[0][544]},[b[0][547],b[0][548],b[0][350],b[0][67],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],[b[0][495],b[0][80],b[0][350],b[0][47],b[0][48],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],[b[0][556],b[0][557],b[0][514],b[0][47],b[0][48],b[0][49],b[0][517],b[0][50],b[0][9],b[0][10]],[b[0][495],b[0][80],b[0][350],b[0][563],b[0][48],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],[b[0][556],b[0][557],b[0][514],b[0][583],b[0][48],b[0][49],b[0][517],b[0][584],b[0][9],b[0][10]],[b[0][586],b[0][587],b[0][588],b[0][67],b[0][48],b[0][49],b[0][517],b[0][69],b[0][9],b[0][10]],{"short":b[0][589],"long":b[0][590]},{"short":b[0][591],"long":b[0][592]},{"short":b[0][593]},[b[0][595],b[0][596],b[0][597],b[0][67],b[0][48],b[0][598],b[0][599],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][600],"short":b[0][601],"long":b[0][602]},{"narrow":b[0][603],"short":b[0][604],"long":b[0][605]},{"short":b[0][606],"long":b[0][607]},{"short":b[0][608]},[b[0][595],b[0][596],b[0][597],b[0][67],b[0][610],b[0][598],b[0][599],b[0][69],b[0][9],b[0][10]],[b[0][595],b[0][596],b[0][597],b[0][613],b[0][614],b[0][598],b[0][599],b[0][615],b[0][9],b[0][10]],{"decimal":b[0][36],"currency":b[0][616],"percent":b[0][38]},[b[0][624],b[0][596],b[0][597],b[0][67],b[0][48],b[0][598],b[0][599],b[0][69],b[0][291],b[0][292]],{"narrow":b[0][625],"short":b[0][626],"long":b[0][602]},{"narrow":b[0][627],"short":b[0][628],"long":b[0][605]},{"short":b[0][629],"long":b[0][607]},[b[0][595],b[0][596],b[0][597],b[0][632],b[0][516],b[0][598],b[0][599],b[0][584],b[0][9],b[0][10]],{"decimal":b[0][36],"currency":b[0][636],"percent":b[0][38]},[b[0][79],b[0][80],b[0][350],b[0][47],b[0][48],b[0][598],b[0][7],b[0][50],b[0][9],b[0][10]],[b[0][641],b[0][404],b[0][384],b[0][203],b[0][204],b[0][49],b[0][363],b[0][205],b[0][642],b[0][10]],{"narrow":b[0][643],"short":b[0][644],"long":b[0][645]},{"narrow":b[0][646],"short":b[0][646],"long":b[0][647]},{"narrow":b[0][648],"short":b[0][649],"long":b[0][650]},[b[0][1],b[0][2],b[0][3],b[0][81],b[0][134],b[0][6],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][652]},{"narrow":b[0][653],"short":b[0][654],"long":b[0][655]},{"narrow":b[0][656],"short":b[0][656],"long":b[0][657]},{"short":b[0][658]},{"short":b[0][659]},{"decimal":b[0][36],"currency":b[0][163],"percent":b[0][660]},{"short":b[0][662],"long":b[0][663]},{"short":b[0][664],"long":b[0][665]},{"short":b[0][666],"long":b[0][667]},[b[0][669],b[0][66],b[0][46],b[0][81],b[0][134],b[0][49],b[0][68],b[0][50],b[0][9],b[0][10]],{"narrow":b[0][670],"short":b[0][671],"long":b[0][672]},{"narrow":b[0][673],"short":b[0][674],"long":b[0][674]},{"narrow":b[0][675],"short":b[0][676],"long":b[0][677]},{"narrow":b[0][679],"short":b[0][680],"long":b[0][680]},{"narrow":b[0][681],"short":b[0][682],"long":b[0][682]},{"short":b[0][683],"long":b[0][684]},{"narrow":b[0][685],"short":b[0][686],"long":b[0][687]},{"short":b[0][688],"long":b[0][689]},{"decimal":b[0][36],"currency":b[0][691],"percent":b[0][38]},{"arabext":b[0][692],"latn":b[0][39]},{"narrow":b[0][670],"short":b[0][694],"long":b[0][694]},{"narrow":b[0][685],"short":b[0][686],"long":b[0][686]},{"short":b[0][695],"long":b[0][689]},[b[0][65],b[0][66],b[0][46],b[0][4],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][696],"long":b[0][697]},{"short":b[0][698],"long":b[0][699]},{"short":b[0][700],"long":b[0][701]},[b[0][382],b[0][383],b[0][384],b[0][203],b[0][204],b[0][49],b[0][363],b[0][205],b[0][206],b[0][207]],{"narrow":b[0][705],"short":b[0][706],"long":b[0][706]},{"narrow":b[0][707],"short":b[0][707],"long":b[0][708]},{"narrow":b[0][709],"short":b[0][710],"long":b[0][711]},{"long":b[0][713]},{"short":b[0][27],"long":b[0][714]},{"latn":b[0][715]},{"narrow":b[0][717],"short":b[0][718],"long":b[0][719]},{"narrow":b[0][720],"short":b[0][721],"long":b[0][722]},{"short":b[0][504]},[b[0][723],b[0][724],b[0][3],b[0][4],b[0][5],b[0][6],b[0][7],b[0][446],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][725],"long":b[0][726]},{"narrow":b[0][727],"short":b[0][728],"long":b[0][729]},{"narrow":b[0][730],"short":b[0][731],"long":b[0][732]},{"decimal":b[0][36],"currency":b[0][616],"percent":b[0][201]},{"latn":b[0][734]},{"narrow":b[0][737],"short":b[0][738],"long":b[0][739]},{"narrow":b[0][179],"short":b[0][740],"long":b[0][741]},{"narrow":b[0][742],"short":b[0][743],"long":b[0][744]},{"short":b[0][745],"long":b[0][746]},{"short":b[0][747],"long":b[0][748]},{"short":b[0][749],"long":b[0][750]},{"narrow":b[0][51],"short":b[0][751],"long":b[0][752]},{"narrow":b[0][753],"short":b[0][754],"long":b[0][755]},{"narrow":b[0][756],"short":b[0][757],"long":b[0][758]},{"narrow":b[0][759],"short":b[0][760],"long":b[0][761]},{"short":b[0][24],"long":b[0][762]},{"narrow":b[0][763],"short":b[0][764],"long":b[0][765]},{"narrow":b[0][179],"short":b[0][766],"long":b[0][767]},{"short":b[0][768]},[b[0][65],b[0][66],b[0][46],b[0][4],b[0][5],b[0][49],b[0][68],b[0][446],b[0][9],b[0][10]],[b[0][65],b[0][66],b[0][3],b[0][4],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][787],"short":b[0][788],"long":b[0][789]},{"short":b[0][790],"long":b[0][791]},{"short":b[0][792]},[b[0][65],b[0][66],b[0][46],b[0][515],b[0][516],b[0][49],b[0][68],b[0][518],b[0][9],b[0][10]],{"short":b[0][794],"long":b[0][795]},{"short":b[0][796],"long":b[0][797]},{"short":b[0][798],"long":b[0][799]},[b[0][1],b[0][2],b[0][3],b[0][4],b[0][5],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],{"short":b[0][801],"long":b[0][802]},{"short":b[0][803],"long":b[0][804]},{"short":b[0][798],"long":b[0][805]},[b[0][251],b[0][252],b[0][189],b[0][67],b[0][610],b[0][49],b[0][68],b[0][807],b[0][9],b[0][10]],{"narrow":b[0][808],"short":b[0][809],"long":b[0][810]},{"narrow":b[0][811],"short":b[0][812],"long":b[0][813]},{"short":b[0][814],"long":b[0][815]},[b[0][403],b[0][404],b[0][3],b[0][4],b[0][443],b[0][49],b[0][363],b[0][205],b[0][9],b[0][10]],{"short":b[0][817],"long":b[0][818]},{"short":b[0][819],"long":b[0][820]},{"narrow":b[0][409],"short":b[0][821],"long":b[0][411]},{"latn":b[0][823]},{"narrow":b[0][825],"short":b[0][826],"long":b[0][827]},{"narrow":b[0][828],"short":b[0][829],"long":b[0][830]},{"narrow":b[0][831],"short":b[0][832],"long":b[0][833]},{"decimal":b[0][36],"currency":b[0][37],"percent":b[0][38],"secondaryGroupSize":2},[b[0][79],b[0][80],b[0][3],b[0][4],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][834],"long":b[0][835]},{"short":b[0][836],"long":b[0][837]},{"short":b[0][838],"long":b[0][839]},{"short":b[0][841],"long":b[0][842]},{"short":b[0][843],"long":b[0][844]},{"short":b[0][798]},{"short":b[0][845],"long":b[0][846]},{"short":b[0][847],"long":b[0][848]},{"short":b[0][849],"long":b[0][850]},[b[0][44],b[0][45],b[0][46],b[0][853],b[0][854],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][855],"long":b[0][856]},{"short":b[0][857],"long":b[0][858]},[b[0][859],b[0][860],b[0][861],b[0][203],b[0][204],b[0][49],b[0][862],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][863],"long":b[0][864]},{"narrow":b[0][865],"short":b[0][866],"long":b[0][867]},{"short":b[0][868],"long":b[0][869]},{"narrow":b[0][871],"short":b[0][872],"long":b[0][872]},{"short":b[0][873]},{"short":b[0][874],"long":b[0][875]},{"short":b[0][876]},{"narrow":b[0][30],"short":b[0][877]},[b[0][879],b[0][45],b[0][189],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"long":b[0][880]},{"narrow":b[0][881],"short":b[0][882],"long":b[0][883]},{"narrow":b[0][884],"short":b[0][885],"long":b[0][886]},{"short":b[0][887]},{"narrow":b[0][179],"short":b[0][889],"long":b[0][889]},{"short":b[0][890]},{"short":b[0][27],"long":b[0][891]},{"narrow":b[0][30],"short":b[0][892]},{"long":b[0][893]},[b[0][894],b[0][404],b[0][384],b[0][362],b[0][895],b[0][49],b[0][363],b[0][364],b[0][291],b[0][292]],{"narrow":b[0][896],"short":b[0][897],"long":b[0][898]},{"narrow":b[0][295],"short":b[0][295],"long":b[0][296]},{"narrow":b[0][899],"short":b[0][900],"long":b[0][901]},{"short":b[0][902]},[b[0][905],b[0][906],b[0][907],b[0][908],b[0][909],b[0][910],b[0][7],b[0][911],b[0][912],b[0][913]],{"short":b[0][914]},{"narrow":b[0][179],"short":b[0][179],"long":b[0][12]},{"narrow":b[0][93],"short":b[0][13],"long":b[0][13]},{"narrow":b[0][93],"short":b[0][15],"long":b[0][15]},{"narrow":b[0][915],"short":b[0][916],"long":b[0][917]},{"narrow":b[0][918],"short":b[0][918],"long":b[0][919]},{"narrow":b[0][920],"short":b[0][921],"long":b[0][922]},{"short":b[0][924],"long":b[0][924]},{"short":b[0][925]},{"short":b[0][926],"long":b[0][927]},{"short":b[0][928]},{"narrow":b[0][179],"short":b[0][32],"long":b[0][32]},{"short":b[0][929]},[b[0][931],b[0][932],b[0][189],b[0][176],b[0][177],b[0][933],b[0][68],b[0][178],b[0][9],b[0][10]],{"narrow":b[0][934],"short":b[0][935],"long":b[0][936]},{"narrow":b[0][937],"short":b[0][937],"long":b[0][938]},{"short":b[0][939]},{"short":b[0][942],"long":b[0][943]},{"short":b[0][944],"long":b[0][945]},{"short":b[0][946],"long":b[0][947]},[b[0][202],b[0][45],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][206],b[0][207]],{"narrow":b[0][51],"short":b[0][948],"long":b[0][949]},{"narrow":b[0][950],"short":b[0][950],"long":b[0][951]},{"narrow":b[0][952],"short":b[0][953],"long":b[0][952]},{"short":b[0][23],"long":b[0][23]},{"narrow":b[0][179],"short":b[0][954],"long":b[0][955]},{"narrow":b[0][30],"short":b[0][956]},{"short":b[0][957]},{"short":b[0][959],"long":b[0][960]},{"short":b[0][961],"long":b[0][962]},{"short":b[0][963],"long":b[0][964]},{"short":b[0][179],"long":b[0][966]},{"short":b[0][967],"long":b[0][968]},{"short":b[0][969]},[b[0][971],b[0][404],b[0][384],b[0][203],b[0][895],b[0][49],b[0][363],b[0][205],b[0][9],b[0][10]],{"narrow":b[0][972],"short":b[0][973],"long":b[0][974]},{"narrow":b[0][975],"short":b[0][976],"long":b[0][977]},{"narrow":b[0][978],"short":b[0][731],"long":b[0][732]},[b[0][981],b[0][66],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][291],b[0][292]],{"short":b[0][982]},{"narrow":b[0][983],"short":b[0][984],"long":b[0][985]},{"narrow":b[0][986],"short":b[0][986],"long":b[0][987]},{"narrow":b[0][988],"short":b[0][988],"long":b[0][814]},{"short":b[0][989]},{"decimal":b[0][36],"currency":b[0][416],"percent":b[0][38]},[b[0][990],b[0][991],b[0][992],b[0][81],b[0][134],b[0][993],b[0][994],b[0][50],b[0][995],b[0][996]],{"short":b[0][11],"long":b[0][997]},{"narrow":b[0][998],"short":b[0][999],"long":b[0][999]},{"narrow":b[0][93],"short":b[0][1000],"long":b[0][1000]},{"narrow":b[0][93],"short":b[0][1001],"long":b[0][1001]},{"narrow":b[0][179],"short":b[0][1002],"long":b[0][1002]},{"narrow":b[0][1003],"short":b[0][1003],"long":b[0][1004]},{"narrow":b[0][504],"short":b[0][1005],"long":b[0][1005]},{"narrow":b[0][1007],"short":b[0][1008],"long":b[0][1008]},{"narrow":b[0][179],"short":b[0][1009],"long":b[0][1009]},{"short":b[0][1010]},{"narrow":b[0][179],"short":b[0][1011],"long":b[0][1011]},{"narrow":b[0][1012],"short":b[0][1013]},{"narrow":b[0][179],"short":b[0][1014],"long":b[0][1014]},{"short":b[0][1015]},[b[0][1],b[0][2],b[0][3],b[0][1017],b[0][5],b[0][6],b[0][7],b[0][205],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][1018],"long":b[0][1018]},{"short":b[0][1019],"long":b[0][1019]},{"short":b[0][1020]},{"short":b[0][1022],"long":b[0][1023]},{"short":b[0][1024],"long":b[0][1025]},{"short":b[0][399],"long":b[0][1026]},[b[0][251],b[0][252],b[0][189],b[0][203],b[0][204],b[0][190],b[0][68],b[0][205],b[0][9],b[0][10]],{"narrow":b[0][1028],"short":b[0][1029],"long":b[0][1030]},{"narrow":b[0][1031],"short":b[0][1032],"long":b[0][1033]},{"short":b[0][1034],"long":b[0][1035]},{"short":b[0][1037],"long":b[0][1037]},{"narrow":b[0][179],"short":b[0][1038],"long":b[0][1039]},{"narrow":b[0][179],"short":b[0][1040],"long":b[0][1040]},{"latn":b[0][1041]},{"short":b[0][1043],"long":b[0][1044]},{"short":b[0][1045],"long":b[0][1046]},{"short":b[0][1047],"long":b[0][1048]},{"short":b[0][1051],"long":b[0][1052]},{"short":b[0][1053],"long":b[0][1054]},{"short":b[0][1055],"long":b[0][1056]},{"short":b[0][1022],"long":b[0][1058]},{"short":b[0][1059],"long":b[0][1060]},{"short":b[0][1061],"long":b[0][1062]},[b[0][1064],b[0][1065],b[0][46],b[0][515],b[0][516],b[0][1066],b[0][68],b[0][518],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][1067],"long":b[0][1068]},{"narrow":b[0][1069],"short":b[0][1070],"long":b[0][1071]},{"short":b[0][1072],"long":b[0][1073]},{"short":b[0][1075],"long":b[0][1076]},{"short":b[0][423],"long":b[0][1077]},{"short":b[0][1079],"long":b[0][1080]},{"short":b[0][1081],"long":b[0][1082]},[b[0][44],b[0][45],b[0][46],b[0][613],b[0][614],b[0][49],b[0][68],b[0][615],b[0][9],b[0][10]],{"narrow":b[0][93],"long":b[0][13]},{"narrow":b[0][179],"short":b[0][1084],"long":b[0][1085]},{"short":b[0][1086],"long":b[0][1087]},{"short":b[0][1088]},[b[0][65],b[0][66],b[0][46],b[0][1091],b[0][1092],b[0][49],b[0][68],b[0][518],b[0][9],b[0][10]],{"long":b[0][1093]},{"narrow":b[0][1094],"short":b[0][1094],"long":b[0][1094]},[b[0][79],b[0][80],b[0][350],b[0][4],b[0][5],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][1095],"long":b[0][1096]},{"narrow":b[0][1097],"short":b[0][1097],"long":b[0][1098]},{"narrow":b[0][1099],"short":b[0][1100],"long":b[0][1101]},{"short":b[0][1104],"long":b[0][1105]},{"short":b[0][1106],"long":b[0][1107]},{"short":b[0][1108],"long":b[0][1109]},[b[0][65],b[0][66],b[0][46],b[0][1111],b[0][610],b[0][49],b[0][68],b[0][807],b[0][9],b[0][10]],{"short":b[0][1112]},{"narrow":b[0][179],"short":b[0][1113],"long":b[0][1113]},{"narrow":b[0][1114],"short":b[0][1114],"long":b[0][1114]},{"short":b[0][1115],"long":b[0][1116]},[b[0][1119],b[0][1120],b[0][1121],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][1122],"short":b[0][1123],"long":b[0][1124]},{"narrow":b[0][1125],"short":b[0][1126],"long":b[0][1127]},{"short":b[0][1128],"long":b[0][1129]},[b[0][1130],b[0][1131],b[0][1132],b[0][1133],b[0][909],b[0][1134],b[0][7],b[0][911],b[0][912],b[0][913]],{"short":b[0][1135]},{"short":b[0][1136],"long":b[0][13]},{"narrow":b[0][179],"short":b[0][1137],"long":b[0][1137]},{"short":b[0][1138],"long":b[0][15]},{"narrow":b[0][1137],"short":b[0][1137],"long":b[0][1137]},{"narrow":b[0][1139],"short":b[0][1139],"long":b[0][1140]},{"short":b[0][1141],"long":b[0][1142]},{"short":b[0][1144],"long":b[0][23]},{"narrow":b[0][30],"short":b[0][1145]},{"short":b[0][1146]},{"decimal":b[0][36],"currency":b[0][1147],"percent":b[0][38]},{"long":b[0][1148]},{"short":b[0][1149],"long":b[0][1150]},{"short":b[0][1151]},{"long":b[0][1153]},{"short":b[0][1154],"long":b[0][1155]},{"short":b[0][1156],"long":b[0][1157]},{"long":b[0][1158]},{"latn":b[0][1159]},{"short":b[0][1022],"long":b[0][1160]},{"short":b[0][1161],"long":b[0][1162]},{"short":b[0][399],"long":b[0][1163]},{"short":b[0][1165],"long":b[0][1166]},{"short":b[0][1167],"long":b[0][1168]},{"short":b[0][1169],"long":b[0][1170]},[b[0][359],b[0][360],b[0][361],b[0][4],b[0][5],b[0][49],b[0][363],b[0][8],b[0][9],b[0][10]],{"short":b[0][1172],"long":b[0][1173]},{"narrow":b[0][1174],"short":b[0][1175],"long":b[0][1176]},{"narrow":b[0][1177],"short":b[0][821],"long":b[0][1178]},{"latn":b[0][1180]},{"short":b[0][1181],"long":b[0][1182]},{"short":b[0][1183],"long":b[0][1184]},[b[0][1185],b[0][1186],b[0][1187],b[0][4],b[0][5],b[0][1188],b[0][280],b[0][615],b[0][912],b[0][913]],{"narrow":b[0][1189],"short":b[0][1190],"long":b[0][1191]},{"narrow":b[0][1192],"short":b[0][1192],"long":b[0][1193]},{"narrow":b[0][1194],"short":b[0][1195],"long":b[0][1196]},{"latn":b[0][1198]},{"short":b[0][1200],"long":b[0][1201]},{"short":b[0][1202],"long":b[0][1203]},{"short":b[0][1204],"long":b[0][1205]},{"short":b[0][1207],"long":b[0][1208]},{"short":b[0][1209],"long":b[0][1210]},{"short":b[0][219],"long":b[0][1211]},{"long":b[0][1212]},{"long":b[0][1213]},{"short":b[0][1214],"long":b[0][1215]},{"short":b[0][1216],"long":b[0][1217]},{"short":b[0][1218],"long":b[0][1219]},[b[0][879],b[0][45],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][1223]},{"narrow":b[0][93],"short":b[0][1224],"long":b[0][1225]},{"narrow":b[0][93],"short":b[0][1226],"long":b[0][1226]},{"narrow":b[0][179],"short":b[0][1227],"long":b[0][1228]},{"narrow":b[0][1229],"short":b[0][1229],"long":b[0][1229]},{"short":b[0][1230]},{"short":b[0][1232],"long":b[0][1233]},{"narrow":b[0][179],"short":b[0][1234],"long":b[0][1235]},{"short":b[0][1236]},{"narrow":b[0][179],"short":b[0][1237],"long":b[0][1238]},{"narrow":b[0][30],"short":b[0][1239]},{"narrow":b[0][179],"short":b[0][1240],"long":b[0][1241]},{"short":b[0][1242]},{"short":b[0][1243]},{"latn":b[0][1244]},[b[0][1246],b[0][2],b[0][3],b[0][4],b[0][5],b[0][6],b[0][7],b[0][1247],b[0][291],b[0][292]],{"narrow":b[0][1248],"short":b[0][1249],"long":b[0][1250]},{"narrow":b[0][1251],"short":b[0][1252],"long":b[0][1253]},{"narrow":b[0][1254],"short":b[0][1255],"long":b[0][1256]},{"short":b[0][1259],"long":b[0][1260]},{"short":b[0][1261],"long":b[0][1262]},{"short":b[0][1263],"long":b[0][1264]},{"short":b[0][1266],"long":b[0][1267]},{"short":b[0][1268],"long":b[0][1269]},{"short":b[0][219],"long":b[0][1270]},{"short":b[0][1272],"long":b[0][157]},{"short":b[0][1273],"long":b[0][1274]},{"short":b[0][219],"long":b[0][1275]},{"decimal":b[0][36],"currency":b[0][1276],"percent":b[0][38]},[null,null,null,b[0][203],b[0][177],null,b[0][363],b[0][178],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][1277],"long":b[0][1278]},{"narrow":b[0][1279],"short":b[0][1279],"long":b[0][1280]},{"narrow":b[0][1281],"short":b[0][1282],"long":b[0][1283]},{"long":b[0][1285]},{"short":b[0][27],"long":b[0][1286]},{"latn":b[0][1287]},{"short":b[0][1289],"long":b[0][1290]},{"short":b[0][1024],"long":b[0][1291]},{"short":b[0][1292],"long":b[0][1293]},{"short":b[0][1296],"long":b[0][1297]},{"short":b[0][1298],"long":b[0][1299]},{"short":b[0][1300],"long":b[0][1301]},{"short":b[0][1303],"long":b[0][1304]},{"short":b[0][1305],"long":b[0][1306]},{"short":b[0][1307],"long":b[0][1308]},{"short":b[0][1310],"long":b[0][1311]},{"short":b[0][1312],"long":b[0][1313]},{"short":b[0][219],"long":b[0][1314]},[b[0][79],b[0][80],b[0][350],b[0][67],b[0][48],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],{"short":b[0][1316],"long":b[0][1317]},{"short":b[0][1318],"long":b[0][1319]},{"short":b[0][1320],"long":b[0][1321]},{"narrow":b[0][1324],"short":b[0][1325],"long":b[0][1326]},{"narrow":b[0][1327],"short":b[0][1328],"long":b[0][1328]},[b[0][1329],b[0][45],b[0][46],b[0][203],b[0][204],b[0][49],b[0][68],b[0][205],b[0][9],b[0][10]],{"narrow":b[0][1330],"short":b[0][1331],"long":b[0][1332]},{"narrow":b[0][1333],"short":b[0][1334],"long":b[0][1335]},{"short":b[0][1336]},[b[0][1],b[0][2],b[0][3],b[0][67],b[0][610],b[0][6],b[0][7],b[0][69],b[0][9],b[0][10]],{"short":b[0][1339],"long":b[0][1340]},{"narrow":b[0][1341],"short":b[0][1342],"long":b[0][1343]},{"short":b[0][1344],"long":b[0][1345]},{"long":b[0][1346]},{"short":b[0][1347]},{"short":b[0][27],"long":b[0][1348]},{"short":b[0][1349]},{"decimal":b[0][36],"currency":b[0][77],"percent":b[0][38],"secondaryGroupSize":2},{"latn":b[0][39],"mlym":b[0][1350]},[b[0][723],b[0][724],b[0][3],b[0][442],b[0][443],b[0][6],b[0][7],b[0][446],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][1351],"long":b[0][1352]},{"narrow":b[0][1353],"short":b[0][1353],"long":b[0][1354]},{"narrow":b[0][1355],"short":b[0][1356],"long":b[0][1357]},[b[0][1360],b[0][1361],b[0][189],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][1362],"short":b[0][1363],"long":b[0][1364]},{"narrow":b[0][884],"short":b[0][1149],"long":b[0][1365]},{"short":b[0][1366],"long":b[0][1367]},{"deva":b[0][39],"latn":b[0][39]},[b[0][44],b[0][45],b[0][46],b[0][67],b[0][610],b[0][49],b[0][68],b[0][807],b[0][9],b[0][10]],{"narrow":b[0][1370],"short":b[0][1371],"long":b[0][1372]},{"narrow":b[0][1373],"short":b[0][1371],"long":b[0][1372]},{"narrow":b[0][1374],"short":b[0][1375],"long":b[0][1376]},{"short":b[0][1377]},{"short":b[0][1382],"long":b[0][1383]},{"short":b[0][1384],"long":b[0][1385]},{"short":b[0][1386],"long":b[0][1387]},{"short":b[0][1389],"long":b[0][1390]},{"short":b[0][1391],"long":b[0][1392]},{"short":b[0][1393],"long":b[0][1394]},[b[0][723],b[0][724],b[0][3],b[0][4],b[0][134],b[0][6],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][1396],"long":b[0][1396]},{"narrow":b[0][1397],"short":b[0][1397],"long":b[0][1397]},{"short":b[0][1398],"long":b[0][1399]},{"latn":b[0][1402],"mymr":b[0][1403]},{"short":b[0][498],"long":b[0][1405]},{"short":b[0][1406],"long":b[0][1407]},{"short":b[0][219],"long":b[0][1408]},[b[0][1411],b[0][383],b[0][384],b[0][203],b[0][204],b[0][49],b[0][363],b[0][205],b[0][206],b[0][207]],{"narrow":b[0][51],"short":b[0][1412],"long":b[0][1413]},{"narrow":b[0][1414],"short":b[0][388],"long":b[0][389]},{"narrow":b[0][1415],"short":b[0][1416],"long":b[0][1417]},{"short":b[0][1419],"long":b[0][1420]},{"short":b[0][1421],"long":b[0][1422]},{"short":b[0][219],"long":b[0][1423]},[b[0][1424],b[0][2],b[0][3],b[0][4],b[0][5],b[0][6],b[0][7],b[0][8],b[0][9],b[0][10]],{"narrow":b[0][1425],"short":b[0][1426],"long":b[0][1427]},{"narrow":b[0][1428],"short":b[0][1428],"long":b[0][1429]},{"short":b[0][1430]},{"long":b[0][1431]},{"narrow":b[0][1425],"short":b[0][1426],"long":b[0][1426]},{"narrow":b[0][1428],"short":b[0][1428],"long":b[0][1433]},{"narrow":b[0][179],"long":b[0][12]},{"narrow":b[0][93],"short":b[0][1435],"long":b[0][1435]},{"narrow":b[0][179]},{"narrow":b[0][93],"short":b[0][1436],"long":b[0][1436]},{"narrow":b[0][51],"short":b[0][1437],"long":b[0][1438]},{"narrow":b[0][1439],"short":b[0][1439],"long":b[0][1440]},{"narrow":b[0][1441],"short":b[0][1442],"long":b[0][1443]},{"short":b[0][1444],"long":b[0][1444]},{"narrow":b[0][179],"short":b[0][1445],"long":b[0][1445]},{"narrow":b[0][179],"short":b[0][1446],"long":b[0][1447]},{"short":b[0][1448]},{"decimal":b[0][36],"currency":b[0][1449],"percent":b[0][38]},{"short":b[0][1455],"long":b[0][1456]},{"short":b[0][1457],"long":b[0][1458]},{"short":b[0][1459],"long":b[0][1460]},[b[0][1462],b[0][383],b[0][384],b[0][203],b[0][1463],b[0][49],b[0][363],b[0][205],b[0][9],b[0][10]],{"short":b[0][1464],"long":b[0][1413]},{"short":b[0][1465],"long":b[0][1466]},{"narrow":b[0][1467],"short":b[0][391],"long":b[0][1468]},[b[0][1471],b[0][1472],b[0][1473],b[0][67],b[0][5],b[0][49],b[0][7],b[0][8],b[0][9],b[0][10]],{"short":b[0][1474],"long":b[0][1474]},{"narrow":b[0][1475],"short":b[0][1475],"long":b[0][1475]},{"short":b[0][1476],"long":b[0][1477]},{"short":b[0][1479],"long":b[0][1480]},{"short":b[0][1481],"long":b[0][1482]},{"short":b[0][219]},{"short":b[0][1483],"long":b[0][1484]},{"short":b[0][1485],"long":b[0][1486]},[b[0][1487],b[0][1488],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][1489],"long":b[0][1490]},{"short":b[0][1491],"long":b[0][1492]},{"short":b[0][1493],"long":b[0][1494]},{"short":b[0][1496],"long":b[0][1497]},{"short":b[0][1498],"long":b[0][1499]},{"short":b[0][1500]},{"long":b[0][1503]},{"short":b[0][1504],"long":b[0][1505]},[b[0][879],b[0][45],b[0][3],b[0][4],b[0][177],b[0][49],b[0][68],b[0][178],b[0][9],b[0][10]],{"narrow":b[0][1189],"short":b[0][1506],"long":b[0][1507]},{"short":b[0][1508],"long":b[0][1509]},{"short":b[0][1510]},{"latn":b[0][1512]},[b[0][44],b[0][45],b[0][46],b[0][67],b[0][610],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][1514],"short":b[0][1515],"long":b[0][1515]},{"narrow":b[0][1516],"short":b[0][1516],"long":b[0][1517]},{"short":b[0][1518]},{"narrow":b[0][1514],"short":b[0][1515],"long":b[0][1519]},{"narrow":b[0][1516],"short":b[0][1516],"long":b[0][1520]},{"short":b[0][1518],"long":b[0][1521]},[b[0][879],b[0][45],b[0][46],b[0][230],b[0][177],b[0][49],b[0][68],b[0][231],b[0][9],b[0][10]],{"narrow":b[0][1523],"short":b[0][1524],"long":b[0][1525]},{"narrow":b[0][1526],"short":b[0][1526],"long":b[0][1527]},{"short":b[0][1528],"long":b[0][1528]},{"short":b[0][1529],"long":b[0][1529]},{"narrow":b[0][179],"short":b[0][1530],"long":b[0][1531]},{"narrow":b[0][179],"short":b[0][1532],"long":b[0][1532]},{"short":b[0][1533]},[b[0][1],b[0][2],b[0][3],b[0][4],b[0][134],b[0][6],b[0][7],b[0][50],b[0][9],b[0][10]],{"long":b[0][1535]},{"long":b[0][674]},{"short":b[0][676]},{"long":b[0][1537]},{"arabext":b[0][1538],"latn":b[0][137]},[b[0][595],b[0][596],b[0][597],b[0][515],b[0][516],b[0][598],b[0][599],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][179],"long":b[0][1540]},{"narrow":b[0][51],"short":b[0][1541],"long":b[0][1542]},{"narrow":b[0][1543],"short":b[0][1543],"long":b[0][1544]},{"short":b[0][171],"long":b[0][1545]},{"short":b[0][1546]},[b[0][1551],b[0][1552],b[0][1553],b[0][515],b[0][516],b[0][1554],b[0][1555],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][1556],"long":b[0][1540]},{"narrow":b[0][51],"short":b[0][1557],"long":b[0][1558]},{"narrow":b[0][1559],"short":b[0][1543],"long":b[0][1544]},{"short":b[0][1560],"long":b[0][1545]},{"short":b[0][1563],"long":b[0][1564]},{"short":b[0][1565],"long":b[0][1566]},{"short":b[0][1567],"long":b[0][1568]},{"short":b[0][1570],"long":b[0][1571]},{"short":b[0][1572],"long":b[0][1573]},{"short":b[0][1574],"long":b[0][1575]},{"decimal":b[0][36],"currency":b[0][77],"percent":b[0][201]},[b[0][879],b[0][45],b[0][46],b[0][176],b[0][177],b[0][49],b[0][68],b[0][178],b[0][9],b[0][10]],{"short":b[0][1578],"long":b[0][1579]},{"long":b[0][1580]},{"long":b[0][1581]},{"narrow":b[0][1582],"short":b[0][1583],"long":b[0][1584]},{"narrow":b[0][1585],"short":b[0][1586],"long":b[0][1587]},{"short":b[0][1588],"long":b[0][1589]},{"short":b[0][1590],"long":b[0][1590]},{"short":b[0][1556],"long":b[0][1593]},{"short":b[0][1594],"long":b[0][1595]},{"short":b[0][160],"long":b[0][1596]},{"long":b[0][501]},{"narrow":b[0][93],"short":b[0][1598],"long":b[0][1598]},{"narrow":b[0][93],"short":b[0][1599],"long":b[0][1599]},{"narrow":b[0][1189],"short":b[0][1600],"long":b[0][1601]},{"narrow":b[0][1602],"short":b[0][1602],"long":b[0][1603]},{"narrow":b[0][1604],"short":b[0][1605],"long":b[0][1604]},{"short":b[0][1606],"long":b[0][1606]},{"narrow":b[0][179],"short":b[0][1607],"long":b[0][1607]},{"narrow":b[0][179],"short":b[0][1608],"long":b[0][1608]},{"narrow":b[0][30],"short":b[0][1609]},{"narrow":b[0][179],"short":b[0][1610],"long":b[0][1610]},{"latn":b[0][1611]},{"short":b[0][1617],"long":b[0][1618]},{"short":b[0][1619],"long":b[0][1620]},{"short":b[0][1622],"long":b[0][1623]},{"short":b[0][1624],"long":b[0][1625]},{"short":b[0][1626]},{"short":b[0][1629],"long":b[0][1630]},{"short":b[0][1631],"long":b[0][1632]},{"short":b[0][399],"long":b[0][1633]},[b[0][79],b[0][80],b[0][1635],b[0][47],b[0][48],b[0][49],b[0][7],b[0][50],b[0][9],b[0][10]],{"short":b[0][1636],"long":b[0][1637]},{"short":b[0][1638],"long":b[0][1639]},{"short":b[0][1640],"long":b[0][1641]},{"short":b[0][1643],"long":b[0][1644]},{"short":b[0][1645],"long":b[0][1646]},{"short":b[0][1647],"long":b[0][1648]},{"short":b[0][1651],"long":b[0][1644]},{"short":b[0][1645],"long":b[0][1652]},[b[0][595],b[0][596],b[0][597],b[0][4],b[0][516],b[0][598],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][1653],"long":b[0][1654]},{"short":b[0][1655],"long":b[0][1656]},{"short":b[0][1657],"long":b[0][1658]},{"short":b[0][421],"long":b[0][1659]},{"short":b[0][1660],"long":b[0][1661]},{"short":b[0][1662],"long":b[0][1663]},{"short":b[0][1664],"long":b[0][1665]},{"short":b[0][1667],"long":b[0][1668]},{"short":b[0][1669],"long":b[0][1670]},{"short":b[0][1671],"long":b[0][1672]},{"short":b[0][1674],"long":b[0][1675]},{"short":b[0][1676],"long":b[0][1677]},{"short":b[0][1678],"long":b[0][1679]},[b[0][1681],b[0][2],b[0][3],b[0][442],b[0][443],b[0][6],b[0][7],b[0][446],b[0][1682],b[0][1683]],{"narrow":b[0][1684],"short":b[0][1685],"long":b[0][1686]},{"narrow":b[0][1687],"short":b[0][1688],"long":b[0][1689]},{"short":b[0][1690],"long":b[0][1691]},[b[0][403],b[0][404],b[0][1694],b[0][203],b[0][204],b[0][49],b[0][363],b[0][205],b[0][9],b[0][10]],{"short":b[0][1695],"long":b[0][1696]},{"narrow":b[0][1697],"short":b[0][1698],"long":b[0][1699]},{"short":b[0][1700]},[b[0][403],b[0][404],b[0][384],b[0][362],b[0][48],b[0][49],b[0][363],b[0][364],b[0][9],b[0][10]],{"narrow":b[0][1702],"short":b[0][1703],"long":b[0][1704]},{"narrow":b[0][1705],"short":b[0][1705],"long":b[0][1706]},{"short":b[0][1707],"long":b[0][1708]},{"short":b[0][1711],"long":b[0][1712]},{"short":b[0][1713],"long":b[0][1714]},{"short":b[0][219],"long":b[0][1715]},{"short":b[0][1716],"long":b[0][1717]},{"short":b[0][1718],"long":b[0][1719]},{"short":b[0][1720],"long":b[0][1721]},[b[0][44],b[0][45],b[0][46],b[0][515],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][1727],"short":b[0][1728],"long":b[0][1729]},{"narrow":b[0][1730],"short":b[0][1730],"long":b[0][1731]},{"narrow":b[0][1732],"short":b[0][1733],"long":b[0][1734]},[b[0][1738],b[0][404],b[0][384],b[0][203],b[0][204],b[0][49],b[0][363],b[0][69],b[0][1739],b[0][1740]],{"narrow":b[0][1330],"short":b[0][306],"long":b[0][307]},{"narrow":b[0][1741],"short":b[0][1742],"long":b[0][1743]},{"narrow":b[0][1330],"short":b[0][306],"long":b[0][1744]},{"narrow":b[0][1741],"short":b[0][1741],"long":b[0][1745]},{"long":b[0][1746]},{"long":b[0][1747]},{"narrow":b[0][1702],"short":b[0][293],"long":b[0][1748]},{"narrow":b[0][1749],"short":b[0][1749],"long":b[0][1750]},{"narrow":b[0][1528],"short":b[0][1751],"long":b[0][298]},{"long":b[0][1752]},{"long":b[0][1753]},{"short":b[0][1754],"long":b[0][1755]},{"narrow":b[0][30],"short":b[0][1756]},{"long":b[0][1757]},{"short":b[0][1758]},{"short":b[0][1760],"long":b[0][1761]},{"short":b[0][1762],"long":b[0][1763]},{"short":b[0][1765],"long":b[0][1766]},{"short":b[0][1767],"long":b[0][1768]},{"short":b[0][1769],"long":b[0][1770]},[b[0][65],b[0][66],b[0][46],b[0][4],b[0][5],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"short":b[0][1772],"long":b[0][1772]},{"short":b[0][748],"long":b[0][748]},{"narrow":b[0][51],"short":b[0][1095],"long":b[0][1773]},{"narrow":b[0][1774],"short":b[0][1775],"long":b[0][1776]},{"narrow":b[0][1467],"short":b[0][391],"long":b[0][1777]},{"short":b[0][1779],"long":b[0][1780]},{"short":b[0][765],"long":b[0][765]},{"short":b[0][537],"long":b[0][1781]},{"short":b[0][27],"long":b[0][1782]},{"narrow":b[0][30],"short":b[0][1783]},{"short":b[0][1784],"long":b[0][1784]},{"short":b[0][1785]},{"narrow":b[0][1467],"short":b[0][391],"long":b[0][1787]},[b[0][79],b[0][80],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][807],b[0][9],b[0][10]],{"short":b[0][1022],"long":b[0][157]},{"narrow":b[0][159],"short":b[0][1788],"long":b[0][159]},{"short":b[0][1790],"long":b[0][1791]},{"short":b[0][1792],"long":b[0][1793]},{"short":b[0][1794],"long":b[0][1795]},[b[0][251],b[0][252],b[0][189],b[0][67],b[0][48],b[0][49],b[0][7],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][1797],"long":b[0][1798]},{"narrow":b[0][1799],"short":b[0][1800],"long":b[0][1801]},{"narrow":b[0][1802],"short":b[0][1802],"long":b[0][1803]},{"short":b[0][1804],"long":b[0][1805]},[b[0][1809],b[0][1361],b[0][1810],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][1811],"short":b[0][1812],"long":b[0][1813]},{"narrow":b[0][1814],"short":b[0][1815],"long":b[0][1816]},{"narrow":b[0][1817],"short":b[0][1818],"long":b[0][1817]},{"long":b[0][1819]},{"short":b[0][1820]},{"short":b[0][1821],"long":b[0][1822]},{"short":b[0][1823],"long":b[0][1824]},{"short":b[0][1827],"long":b[0][1828]},{"short":b[0][1829],"long":b[0][1830]},{"short":b[0][1831],"long":b[0][1832]},{"short":b[0][1835],"long":b[0][1836]},{"narrow":b[0][93],"short":b[0][1837],"long":b[0][1837]},{"narrow":b[0][93],"short":b[0][1838],"long":b[0][1838]},{"short":b[0][1839],"long":b[0][1840]},{"narrow":b[0][1841],"short":b[0][1841],"long":b[0][1842]},{"narrow":b[0][1843],"short":b[0][1844],"long":b[0][1845]},{"short":b[0][1847],"long":b[0][1847]},{"short":b[0][1848]},{"narrow":b[0][179],"short":b[0][1849],"long":b[0][1849]},{"short":b[0][1850]},{"narrow":b[0][179],"short":b[0][1851],"long":b[0][1852]},{"short":b[0][1853],"long":b[0][1854]},{"narrow":b[0][30],"short":b[0][1855]},{"narrow":b[0][179],"short":b[0][1856],"long":b[0][1856]},{"short":b[0][1857]},{"short":b[0][1858]},{"short":b[0][1860],"long":b[0][1861]},{"long":b[0][1862]},{"short":b[0][1864],"long":b[0][1865]},{"long":b[0][1866]},{"short":b[0][1867],"long":b[0][1868]},{"short":b[0][1870],"long":b[0][1871]},{"short":b[0][1872],"long":b[0][1873]},[b[0][507],b[0][66],b[0][46],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][1875],"short":b[0][1876],"long":b[0][1877]},{"narrow":b[0][1878],"short":b[0][1879],"long":b[0][1880]},{"narrow":b[0][1881],"short":b[0][1882],"long":b[0][1883]},{"latn":b[0][1884]},[b[0][1886],b[0][1887],b[0][514],b[0][176],b[0][516],b[0][49],b[0][68],b[0][518],b[0][912],b[0][913]],{"long":b[0][1888]},{"long":b[0][1889]},{"narrow":b[0][1890],"short":b[0][1891],"long":b[0][1892]},{"narrow":b[0][1893],"short":b[0][1894],"long":b[0][1895]},{"short":b[0][1896],"long":b[0][1897]},{"long":b[0][1899]},{"short":b[0][1900],"long":b[0][1900]},{"short":b[0][1901]},{"long":b[0][1902]},{"decimal":b[0][36],"currency":b[0][163],"percent":b[0][1903]},{"short":b[0][1905],"long":b[0][1906]},{"short":b[0][1907],"long":b[0][1908]},{"short":b[0][1909],"long":b[0][1910]},{"short":b[0][1911],"long":b[0][1912]},{"short":b[0][1913],"long":b[0][1914]},[b[0][44],b[0][45],b[0][46],b[0][176],b[0][177],b[0][49],b[0][68],b[0][178],b[0][9],b[0][10]],{"short":b[0][1916],"long":b[0][1917]},{"short":b[0][1918],"long":b[0][1919]},{"narrow":b[0][1920],"short":b[0][1921],"long":b[0][1922]},{"narrow":b[0][1923],"short":b[0][1923],"long":b[0][1924]},{"short":b[0][1925],"long":b[0][1926]},{"long":b[0][1928]},{"short":b[0][1929],"long":b[0][1930]},{"short":b[0][1931],"long":b[0][1932]},{"narrow":b[0][30],"short":b[0][1933]},{"short":b[0][1934],"long":b[0][1935]},{"latn":b[0][1936]},[b[0][106],b[0][107],b[0][108],b[0][67],b[0][48],b[0][49],b[0][68],b[0][69],b[0][9],b[0][10]],{"narrow":b[0][51],"short":b[0][1938],"long":b[0][1938]},{"narrow":b[0][1939],"short":b[0][1940],"long":b[0][1940]},{"short":b[0][1941],"long":b[0][1942]},{"short":b[0][27],"long":b[0][1944]},{"short":b[0][1945]},{"arabext":b[0][1946],"latn":b[0][39]},{"narrow":b[0][1939],"short":b[0][1939],"long":b[0][1939]},{"decimal":b[0][36],"currency":b[0][1948],"percent":b[0][38]},{"narrow":b[0][1949],"short":b[0][1950],"long":b[0][1951]},{"narrow":b[0][1952],"short":b[0][1952],"long":b[0][1953]},{"short":b[0][1954]},{"short":b[0][27],"long":b[0][1955]},[b[0][1],b[0][2],b[0][3],b[0][4],b[0][5],b[0][6],b[0][7],b[0][50],b[0][9],b[0][10]],{"narrow":b[0][1949],"short":b[0][1957],"long":b[0][672]},{"narrow":b[0][1952],"short":b[0][1958],"long":b[0][674]},{"narrow":b[0][1189],"short":b[0][1827],"long":b[0][1828]},{"narrow":b[0][1960],"short":b[0][1960],"long":b[0][1961]},{"short":b[0][1962]},{"short":b[0][27],"long":b[0][1963]},{"long":b[0][1965]},{"long":b[0][1966]},{"long":b[0][1968]},{"long":b[0][1969]},{"short":b[0][1970],"long":b[0][1971]},{"short":b[0][1972],"long":b[0][1973]},[b[0][1974],b[0][557],b[0][588],b[0][67],b[0][48],b[0][49],b[0][517],b[0][1975],b[0][9],b[0][10]],{"narrow":b[0][179],"short":b[0][1976],"long":b[0][1977]},{"narrow":b[0][1978],"short":b[0][1979],"long":b[0][1980]},{"narrow":b[0][1981],"short":b[0][1981],"long":b[0][1981]},{"short":b[0][1983]},{"short":b[0][1984],"long":b[0][1985]},{"short":b[0][1986],"long":b[0][1987]},{"short":b[0][1988],"long":b[0][1988]},[b[0][403],b[0][404],b[0][384],b[0][4],b[0][5],b[0][49],b[0][363],b[0][1990],b[0][9],b[0][10]],{"narrow":b[0][1991],"short":b[0][1992],"long":b[0][1993]},{"short":b[0][1994],"long":b[0][1995]},{"short":b[0][1996]},{"latn":b[0][1997]},{"long":b[0][1999]},{"short":b[0][2000]},{"short":b[0][2002],"long":b[0][2003]},{"short":b[0][2004],"long":b[0][2005]},{"short":b[0][219],"long":b[0][2006]},{"short":b[0][2007],"long":b[0][2008]},{"short":b[0][2009],"long":b[0][2010]},{"short":b[0][2012],"long":b[0][2013]},{"short":b[0][2014],"long":b[0][2015]},{"short":b[0][2016],"long":b[0][2017]},{"short":b[0][2019],"long":b[0][2020]},{"short":b[0][2021],"long":b[0][2022]},{"short":b[0][2023],"long":b[0][2024]},{"short":b[0][2026],"long":b[0][2027]},{"short":b[0][2028],"long":b[0][2029]},{"short":b[0][1669],"long":b[0][2031]},[b[0][2032],b[0][2033],b[0][992],b[0][81],b[0][134],b[0][993],b[0][994],b[0][50],b[0][995],b[0][996]],{"short":b[0][2034]},{"narrow":b[0][93],"short":b[0][2035],"long":b[0][2036]},{"narrow":b[0][179],"short":b[0][1002],"long":b[0][2037]},{"narrow":b[0][2038],"short":b[0][2038],"long":b[0][2039]},{"narrow":b[0][2040],"short":b[0][2040],"long":b[0][2040]},{"short":b[0][2042],"long":b[0][2043]},{"short":b[0][2044]},{"short":b[0][2045]},{"short":b[0][2046]},{"narrow":b[0][30],"short":b[0][2047]},{"short":b[0][2048]},[b[0][2032],b[0][2033],b[0][992],b[0][67],b[0][48],b[0][993],b[0][994],b[0][69],b[0][995],b[0][996]],[b[0][2032],b[0][2033],b[0][992],b[0][2051],b[0][2052],b[0][993],b[0][994],b[0][446],b[0][995],b[0][996]],[b[0][2055],b[0][2056],b[0][2057],b[0][67],b[0][48],b[0][2058],b[0][994],b[0][69],b[0][995],b[0][996]],{"narrow":b[0][2059],"short":b[0][2037],"long":b[0][2037]},[b[0][2032],b[0][2033],b[0][992],b[0][81],b[0][134],b[0][993],b[0][994],b[0][446],b[0][995],b[0][996]],{"short":b[0][2061]},{"narrow":b[0][93],"short":b[0][2035],"long":b[0][2035]},{"narrow":b[0][2062],"short":b[0][2063],"long":b[0][2039]},{"narrow":b[0][2064],"short":b[0][2064],"long":b[0][2064]},{"short":b[0][2065],"long":b[0][2065]},{"short":b[0][2066]},{"narrow":b[0][179],"short":b[0][2067],"long":b[0][2067]},{"short":b[0][2068]},{"narrow":b[0][179],"short":b[0][2069],"long":b[0][2069]},{"short":b[0][2070]},{"narrow":b[0][30],"short":b[0][2071]},{"short":b[0][2072]},{"short":b[0][2073]},{"latn":b[0][2074]},{"narrow":b[0][51],"short":b[0][2076],"long":b[0][2077]},{"narrow":b[0][2078],"short":b[0][2079],"long":b[0][2080]},{"latn":b[0][2082]}];b[2]=[{"eras":b[1][1]},{"months":b[1][2]},{"months":b[1][3],"eras":b[1][4]},{"months":b[1][5],"eras":b[1][4]},{"eras":b[1][6]},{"months":b[1][2],"eras":b[1][4]},{"months":b[1][7],"days":b[1][8],"eras":b[1][9],"dayPeriods":b[0][22]},{"months":b[1][10],"eras":b[1][11]},{"months":b[1][12],"eras":b[1][13]},{"months":b[1][14],"eras":b[1][15]},{"eras":b[1][16]},{"months":b[1][17],"eras":b[1][18]},{"eras":b[1][19]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][40]},{"months":b[1][22],"days":b[1][8],"eras":b[1][9],"dayPeriods":b[0][22]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][42]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][43]},{"months":b[1][24],"days":b[1][25],"eras":b[1][26],"dayPeriods":b[0][60]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][28],"currencies":b[0][63]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][28],"currencies":b[0][64]},{"months":b[1][30],"days":b[1][31],"eras":b[1][32],"dayPeriods":b[0][76]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][28],"currencies":b[0][78]},{"months":b[1][35],"days":b[1][36],"eras":b[1][37],"dayPeriods":b[0][88]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][89]},{"months":b[1][3],"eras":b[1][39]},{"months":b[1][40],"eras":b[1][4]},{"months":b[1][41],"days":b[1][42],"eras":b[1][43],"dayPeriods":b[0][103]},{"months":b[1][44],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][105]},{"eras":b[1][46]},{"months":b[1][47],"eras":b[1][4]},{"months":b[1][48],"eras":b[1][4]},{"months":b[1][49],"days":b[1][50],"eras":b[1][51],"dayPeriods":b[0][120]},{"months":b[1][52],"eras":b[1][53]},{"months":b[1][54],"eras":b[1][55]},{"eras":b[1][56]},{"months":b[1][57],"eras":b[1][58]},{"eras":b[1][59]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][60],"currencies":b[0][132]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][60],"currencies":b[0][133]},{"months":b[1][62],"days":b[1][50],"eras":b[1][51],"dayPeriods":b[0][120]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][63],"currencies":b[0][132]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][60],"currencies":b[0][132]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][60],"currencies":b[0][138]},{"months":b[1][64],"days":b[1][50],"eras":b[1][51],"dayPeriods":b[0][120]},{"months":b[1][65],"days":b[1][50],"eras":b[1][51],"dayPeriods":b[0][120]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][63],"currencies":b[0][142]},{"months":b[1][66],"days":b[1][50],"eras":b[1][51],"dayPeriods":b[0][120]},{"months":b[1][67],"days":b[1][50],"eras":b[1][51],"dayPeriods":b[0][120]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][63],"currencies":b[0][132]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][60],"currencies":b[0][147]},{"nu":b[0][130],"patterns":b[1][27],"symbols":b[1][60],"currencies":b[0][148]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][63],"currencies":b[0][132]},{"months":b[1][68],"days":b[1][69],"eras":b[1][70],"dayPeriods":b[0][154]},{"nu":b[0][155],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][78]},{"months":b[1][73],"days":b[1][74],"eras":b[1][75],"dayPeriods":b[0][162]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][21],"currencies":b[0][164]},{"months":b[1][78],"days":b[1][79],"eras":b[1][80],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][173]},{"months":b[1][83],"days":b[1][84],"eras":b[1][85],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][186]},{"months":b[1][87],"days":b[1][88],"eras":b[1][85],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][193]},{"months":b[1][89],"days":b[1][90],"eras":b[1][91],"dayPeriods":b[0][200]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][78]},{"months":b[1][94],"days":b[1][95],"eras":b[1][96],"dayPeriods":b[0][214]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][28],"currencies":b[0][215]},{"months":b[1][97],"days":b[1][98],"eras":b[1][99],"dayPeriods":b[0][221]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][222]},{"months":b[1][100],"days":b[1][101],"eras":b[1][102],"dayPeriods":b[0][228]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][21],"currencies":b[0][164]},{"months":b[1][104],"days":b[1][105],"eras":b[1][106],"dayPeriods":b[0][240]},{"months":b[1][107],"eras":b[1][11]},{"months":b[1][108],"eras":b[1][13]},{"months":b[1][109],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][244]},{"months":b[1][110],"days":b[1][111],"eras":b[1][112],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][78]},{"months":b[1][114],"days":b[1][115],"eras":b[1][116],"dayPeriods":b[0][259]},{"months":b[1][117],"eras":b[1][118]},{"months":b[1][119],"eras":b[1][120]},{"nu":b[0][155],"patterns":b[1][121],"symbols":b[1][122],"currencies":b[0][266]},{"months":b[1][124],"days":b[1][125],"eras":b[1][126],"dayPeriods":b[0][273]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][127],"currencies":b[0][275]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][127],"currencies":b[0][78]},{"months":b[1][128],"days":b[1][129],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][28],"currencies":b[0][78]},{"months":b[1][131],"days":b[1][132],"eras":b[1][133],"dayPeriods":b[0][285]},{"months":b[1][134],"eras":b[1][13]},{"nu":b[0][35],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][78]},{"months":b[1][136],"days":b[1][137],"eras":b[1][138],"dayPeriods":b[0][299]},{"months":b[1][139],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][302]},{"eras":b[1][140]},{"months":b[1][141],"eras":b[1][4]},{"months":b[1][142],"eras":b[1][4]},{"months":b[1][143],"days":b[1][144],"eras":b[1][145],"dayPeriods":b[0][313]},{"months":b[1][146],"eras":b[1][11]},{"months":b[1][147],"eras":b[1][148]},{"months":b[1][149],"eras":b[1][150]},{"eras":b[1][151]},{"months":b[1][152],"eras":b[1][153]},{"eras":b[1][154]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][324]},{"months":b[1][155],"days":b[1][156],"eras":b[1][157],"dayPeriods":b[0][330]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][331]},{"eras":b[1][159]},{"months":b[1][160]},{"months":b[1][161],"days":b[1][162],"eras":b[1][163],"dayPeriods":b[0][341]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][342]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][343]},{"months":b[1][164],"days":b[1][165],"eras":b[1][166],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][349]},{"months":b[1][168],"days":b[1][169],"eras":b[1][170],"dayPeriods":b[0][357]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][358]},{"months":b[1][172],"days":b[1][173],"eras":b[1][174],"dayPeriods":b[0][172]},{"eras":b[1][175]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][372]},{"months":b[1][177],"days":b[1][178],"eras":b[1][179],"dayPeriods":b[0][172]},{"months":b[1][181],"days":b[1][182],"eras":b[1][183],"dayPeriods":b[0][172]},{"eras":b[1][184]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][394]},{"months":b[1][185],"days":b[1][186],"eras":b[1][187],"dayPeriods":b[0][401]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][402]},{"months":b[1][189],"days":b[1][190],"eras":b[1][191],"dayPeriods":b[0][412]},{"nu":b[0][35],"patterns":b[1][192],"symbols":b[1][81],"currencies":b[0][413]},{"months":b[1][193],"days":b[1][190],"eras":b[1][191],"dayPeriods":b[0][412]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][413]},{"nu":b[0][35],"patterns":b[1][194],"symbols":b[1][195],"currencies":b[0][413]},{"nu":b[0][35],"patterns":b[1][192],"symbols":b[1][195],"currencies":b[0][413]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][418]},{"months":b[1][196],"days":b[1][197],"eras":b[1][198],"dayPeriods":b[0][425]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][199],"currencies":b[0][78]},{"months":b[1][200],"days":b[1][201],"eras":b[1][202],"dayPeriods":b[0][433]},{"months":b[1][203],"days":b[1][204],"eras":b[1][205],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][78]},{"months":b[1][207],"days":b[1][208],"eras":b[1][70],"dayPeriods":b[0][452]},{"nu":b[0][453],"patterns":b[1][209],"symbols":b[1][210],"currencies":b[0][455]},{"months":b[1][211],"days":b[1][212],"eras":b[1][213],"dayPeriods":b[0][462]},{"months":b[1][215]},{"months":b[1][216],"eras":b[1][4]},{"months":b[1][217],"eras":b[1][4]},{"months":b[1][218],"days":b[1][219],"eras":b[1][220],"dayPeriods":b[0][475]},{"months":b[1][221],"eras":b[1][222]},{"months":b[1][223],"eras":b[1][13]},{"months":b[1][215],"eras":b[1][15]},{"months":b[1][215],"eras":b[1][18]},{"eras":b[1][224]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][225],"currencies":b[0][482]},{"months":b[1][227],"days":b[1][228],"eras":b[1][229],"dayPeriods":b[0][491]},{"months":b[1][10],"eras":b[1][230]},{"eras":b[1][231]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][494]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][494]},{"months":b[1][233]},{"months":b[1][234],"days":b[1][235],"eras":b[1][236],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][506]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][173]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][508]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][509]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][506]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][510]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][511]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][519]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][520]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][521]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][522]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][523]},{"months":b[1][241],"days":b[1][242],"eras":b[1][243],"dayPeriods":b[0][532]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][533]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][534]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][535]},{"months":b[1][234],"days":b[1][235],"eras":b[1][236],"dayPeriods":b[0][536]},{"months":b[1][245],"eras":b[1][246]},{"months":b[1][247],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][538]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][539]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][540]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][541]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][542]},{"eras":b[1][249]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][545]},{"months":b[1][234],"days":b[1][235],"eras":b[1][236],"dayPeriods":b[0][546]},{"nu":b[0][35],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][506]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][549]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][550]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][551]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][552]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][553]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][554]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][555]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][558]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][559]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][560]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][561]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][562]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][564]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][565]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][566]},{"nu":b[0][35],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][567]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][568]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][569]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][570]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][571]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][572]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][573]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][574]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][575]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][576]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][577]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][578]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][579]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][580]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][581]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][582]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][28],"currencies":b[0][553]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][585]},{"months":b[1][256],"days":b[1][257],"eras":b[1][258],"dayPeriods":b[0][594]},{"months":b[1][260],"days":b[1][261],"eras":b[1][262],"dayPeriods":b[0][341]},{"eras":b[1][263]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][609]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][611]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][612]},{"nu":b[0][35],"patterns":b[1][266],"symbols":b[1][81],"currencies":b[0][617]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][618]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][619]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][620]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][621]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][609]},{"nu":b[0][35],"patterns":b[1][266],"symbols":b[1][81],"currencies":b[0][609]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][609]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][622]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][623]},{"months":b[1][268],"days":b[1][269],"eras":b[1][270],"dayPeriods":b[0][546]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][630]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][631]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][633]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][634]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][635]},{"nu":b[0][35],"patterns":b[1][272],"symbols":b[1][81],"currencies":b[0][637]},{"months":b[1][260],"days":b[1][261],"eras":b[1][262],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][638]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][639]},{"nu":b[0][35],"patterns":b[1][266],"symbols":b[1][81],"currencies":b[0][640]},{"months":b[1][275],"days":b[1][276],"eras":b[1][277],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][651]},{"eras":b[1][279]},{"months":b[1][280],"days":b[1][281],"eras":b[1][282],"dayPeriods":b[0][172]},{"eras":b[1][283]},{"nu":b[0][35],"patterns":b[1][284],"symbols":b[1][81],"currencies":b[0][661]},{"months":b[1][285],"days":b[1][286],"eras":b[1][287],"dayPeriods":b[0][668]},{"months":b[1][289],"days":b[1][290],"eras":b[1][291],"dayPeriods":b[0][678]},{"months":b[1][292],"eras":b[1][11]},{"months":b[1][293],"eras":b[1][294]},{"months":b[1][295],"eras":b[1][296]},{"nu":b[0][690],"patterns":b[1][297],"symbols":b[1][298],"currencies":b[0][693]},{"months":b[1][299],"days":b[1][290],"eras":b[1][291],"dayPeriods":b[0][678]},{"months":b[1][300],"eras":b[1][301]},{"months":b[1][303],"days":b[1][304],"eras":b[1][305],"dayPeriods":b[0][702]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][703]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][704]},{"months":b[1][307],"days":b[1][308],"eras":b[1][309],"dayPeriods":b[0][712]},{"months":b[1][310],"eras":b[1][11]},{"months":b[1][311],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][312],"currencies":b[0][716]},{"months":b[1][313],"days":b[1][314],"eras":b[1][315],"dayPeriods":b[0][172]},{"months":b[1][317],"days":b[1][318],"eras":b[1][319],"dayPeriods":b[0][733]},{"nu":b[0][35],"patterns":b[1][320],"symbols":b[1][321],"currencies":b[0][735]},{"eras":b[1][322]},{"months":b[1][323]},{"months":b[1][324],"eras":b[1][325]},{"months":b[1][326],"eras":b[1][327]},{"months":b[1][328],"days":b[1][329],"eras":b[1][330],"dayPeriods":b[0][172]},{"months":b[1][331],"eras":b[1][332]},{"months":b[1][333],"eras":b[1][246]},{"months":b[1][334],"eras":b[1][15]},{"eras":b[1][335]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][769]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][769]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][770]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][771]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][772]},{"nu":b[0][35],"patterns":b[1][194],"symbols":b[1][195],"currencies":b[0][769]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][773]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][774]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][775]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][776]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][777]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][778]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][779]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][780]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][781]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][782]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][783]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][784]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][785]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][786]},{"months":b[1][338],"days":b[1][339],"eras":b[1][340],"dayPeriods":b[0][793]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][78]},{"months":b[1][342],"days":b[1][343],"eras":b[1][344],"dayPeriods":b[0][546]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][800]},{"months":b[1][346],"days":b[1][347],"eras":b[1][348],"dayPeriods":b[0][806]},{"months":b[1][350],"days":b[1][351],"eras":b[1][352],"dayPeriods":b[0][546]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][816]},{"months":b[1][354],"days":b[1][355],"eras":b[1][356],"dayPeriods":b[0][822]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][357],"currencies":b[0][824]},{"months":b[1][358],"days":b[1][359],"eras":b[1][360],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][361],"symbols":b[1][21],"currencies":b[0][78]},{"months":b[1][363],"days":b[1][364],"eras":b[1][365],"dayPeriods":b[0][840]},{"months":b[1][366],"days":b[1][367],"eras":b[1][368],"dayPeriods":b[0][546]},{"months":b[1][369],"days":b[1][370],"eras":b[1][371],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][851]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][852]},{"months":b[1][373],"days":b[1][374],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][376],"days":b[1][377],"eras":b[1][378],"dayPeriods":b[0][870]},{"months":b[1][379],"eras":b[1][380]},{"months":b[1][381],"eras":b[1][382]},{"eras":b[1][383]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][21],"currencies":b[0][878]},{"months":b[1][385],"eras":b[1][4]},{"months":b[1][386],"days":b[1][387],"eras":b[1][388],"dayPeriods":b[0][888]},{"months":b[1][389],"eras":b[1][390]},{"months":b[1][391],"eras":b[1][15]},{"eras":b[1][392]},{"months":b[1][393],"eras":b[1][18]},{"nu":b[0][35],"patterns":b[1][361],"symbols":b[1][21],"currencies":b[0][358]},{"months":b[1][395],"days":b[1][396],"eras":b[1][397],"dayPeriods":b[0][172]},{"eras":b[1][398]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][903]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][904]},{"eras":b[1][400]},{"months":b[1][401]},{"months":b[1][402],"eras":b[1][4]},{"months":b[1][403],"eras":b[1][4]},{"months":b[1][404],"days":b[1][405],"eras":b[1][406],"dayPeriods":b[0][923]},{"months":b[1][407],"eras":b[1][408]},{"months":b[1][245],"eras":b[1][13]},{"months":b[1][409],"eras":b[1][410]},{"months":b[1][411],"eras":b[1][18]},{"eras":b[1][412]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][930]},{"months":b[1][414],"days":b[1][415],"eras":b[1][416],"dayPeriods":b[0][940]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][941]},{"months":b[1][417],"days":b[1][418],"eras":b[1][419],"dayPeriods":b[0][546]},{"months":b[1][421],"days":b[1][422],"eras":b[1][423],"dayPeriods":b[0][172]},{"months":b[1][424],"eras":b[1][11]},{"months":b[1][425],"eras":b[1][15]},{"eras":b[1][426]},{"eras":b[1][427]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][958]},{"months":b[1][428],"days":b[1][429],"eras":b[1][430],"dayPeriods":b[0][965]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][852]},{"months":b[1][431],"days":b[1][432],"eras":b[1][433],"dayPeriods":b[0][970]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][275]},{"months":b[1][435],"days":b[1][436],"eras":b[1][437],"dayPeriods":b[0][979]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][980]},{"eras":b[1][439]},{"months":b[1][440],"days":b[1][441],"eras":b[1][442],"dayPeriods":b[0][172]},{"eras":b[1][443]},{"nu":b[0][35],"patterns":b[1][444],"symbols":b[1][195],"currencies":b[0][78]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][78]},{"eras":b[1][446]},{"months":b[1][447]},{"months":b[1][448],"eras":b[1][4]},{"months":b[1][449],"eras":b[1][4]},{"months":b[1][450],"days":b[1][451],"eras":b[1][452],"dayPeriods":b[0][1006]},{"months":b[1][453],"eras":b[1][11]},{"months":b[1][454],"eras":b[1][455]},{"months":b[1][456],"eras":b[1][15]},{"eras":b[1][457]},{"months":b[1][458],"eras":b[1][18]},{"eras":b[1][459]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1016]},{"months":b[1][461],"days":b[1][462],"eras":b[1][463],"dayPeriods":b[0][1021]},{"months":b[1][464],"days":b[1][465],"eras":b[1][466],"dayPeriods":b[0][1027]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][164]},{"months":b[1][468],"days":b[1][469],"eras":b[1][470],"dayPeriods":b[0][1036]},{"months":b[1][471],"eras":b[1][11]},{"months":b[1][472],"eras":b[1][15]},{"months":b[1][473],"eras":b[1][18]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][474],"currencies":b[0][1042]},{"months":b[1][475],"days":b[1][476],"eras":b[1][477],"dayPeriods":b[0][1049]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][28],"currencies":b[0][1050]},{"months":b[1][478],"days":b[1][479],"eras":b[1][480],"dayPeriods":b[0][1057]},{"months":b[1][481],"days":b[1][482],"eras":b[1][483],"dayPeriods":b[0][1063]},{"months":b[1][485],"days":b[1][486],"eras":b[1][487],"dayPeriods":b[0][536]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][81],"currencies":b[0][1074]},{"months":b[1][196],"days":b[1][488],"eras":b[1][489],"dayPeriods":b[0][1078]},{"months":b[1][490],"days":b[1][491],"eras":b[1][213],"dayPeriods":b[0][1083]},{"months":b[1][493],"eras":b[1][4]},{"months":b[1][494],"days":b[1][495],"eras":b[1][496],"dayPeriods":b[0][1089]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][1090]},{"months":b[1][498],"days":b[1][499],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][501],"days":b[1][502],"eras":b[1][503],"dayPeriods":b[0][1102]},{"nu":b[0][35],"patterns":b[1][320],"symbols":b[1][321],"currencies":b[0][1103]},{"months":b[1][504],"days":b[1][505],"eras":b[1][506],"dayPeriods":b[0][1110]},{"eras":b[1][508]},{"months":b[1][509],"days":b[1][510],"eras":b[1][511],"dayPeriods":b[0][1117]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1118]},{"months":b[1][513],"days":b[1][514],"eras":b[1][515],"dayPeriods":b[0][172]},{"eras":b[1][517]},{"months":b[1][518],"eras":b[1][4]},{"months":b[1][519]},{"months":b[1][520],"eras":b[1][4]},{"months":b[1][521],"days":b[1][522],"eras":b[1][523],"dayPeriods":b[0][1143]},{"months":b[1][524],"eras":b[1][11]},{"eras":b[1][525]},{"eras":b[1][526]},{"nu":b[0][35],"patterns":b[1][527],"symbols":b[1][21],"currencies":b[0][1074]},{"months":b[1][528],"days":b[1][529],"eras":b[1][530],"dayPeriods":b[0][1152]},{"months":b[1][531],"days":b[1][532],"eras":b[1][533],"dayPeriods":b[0][172]},{"months":b[1][534],"eras":b[1][13]},{"nu":b[0][690],"patterns":b[1][71],"symbols":b[1][535],"currencies":b[0][78]},{"months":b[1][536],"days":b[1][537],"eras":b[1][538],"dayPeriods":b[0][1164]},{"months":b[1][539],"days":b[1][540],"eras":b[1][541],"dayPeriods":b[0][1171]},{"months":b[1][543],"days":b[1][544],"eras":b[1][545],"dayPeriods":b[0][1179]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][546],"currencies":b[0][78]},{"months":b[1][547],"days":b[1][548],"eras":b[1][368],"dayPeriods":b[0][546]},{"months":b[1][550],"days":b[1][551],"eras":b[1][552],"dayPeriods":b[0][1197]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][553],"currencies":b[0][1199]},{"months":b[1][554],"days":b[1][555],"eras":b[1][556],"dayPeriods":b[0][1206]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][164]},{"months":b[1][557],"days":b[1][558],"eras":b[1][559],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][21],"currencies":b[0][349]},{"months":b[1][560],"days":b[1][561],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][358]},{"months":b[1][562],"days":b[1][563],"eras":b[1][564],"dayPeriods":b[0][1220]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][1221]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][1222]},{"eras":b[1][566]},{"months":b[1][567],"eras":b[1][4]},{"months":b[1][568],"eras":b[1][4]},{"months":b[1][569],"days":b[1][570],"eras":b[1][571],"dayPeriods":b[0][1231]},{"months":b[1][572],"eras":b[1][11]},{"months":b[1][573],"eras":b[1][574]},{"months":b[1][575],"eras":b[1][15]},{"eras":b[1][576]},{"months":b[1][577],"eras":b[1][578]},{"eras":b[1][579]},{"nu":b[0][35],"patterns":b[1][266],"symbols":b[1][580],"currencies":b[0][1245]},{"months":b[1][582],"days":b[1][583],"eras":b[1][584],"dayPeriods":b[0][1257]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][1258]},{"months":b[1][585],"days":b[1][586],"eras":b[1][587],"dayPeriods":b[0][1265]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][81],"currencies":b[0][1222]},{"months":b[1][588],"days":b[1][589],"eras":b[1][590],"dayPeriods":b[0][1271]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][21],"currencies":b[0][402]},{"months":b[1][591],"days":b[1][592],"eras":b[1][593],"dayPeriods":b[0][546]},{"nu":b[0][35],"patterns":b[1][594],"symbols":b[1][21],"currencies":b[0][402]},{"months":b[1][596],"days":b[1][597],"eras":b[1][598],"dayPeriods":b[0][1284]},{"months":b[1][599],"eras":b[1][11]},{"months":b[1][600],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][527],"symbols":b[1][601],"currencies":b[0][1288]},{"months":b[1][602],"days":b[1][603],"eras":b[1][604],"dayPeriods":b[0][1294]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1295]},{"months":b[1][605],"days":b[1][606],"eras":b[1][607],"dayPeriods":b[0][1302]},{"months":b[1][608],"days":b[1][609],"eras":b[1][610],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][199],"currencies":b[0][1309]},{"months":b[1][611],"days":b[1][612],"eras":b[1][613],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1315]},{"months":b[1][615],"days":b[1][616],"eras":b[1][617],"dayPeriods":b[0][1322]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][1323]},{"months":b[1][618],"days":b[1][619],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][78]},{"months":b[1][621],"days":b[1][622],"eras":b[1][623],"dayPeriods":b[0][1337]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][1338]},{"months":b[1][625],"days":b[1][626],"eras":b[1][627],"dayPeriods":b[0][172]},{"months":b[1][628],"eras":b[1][629]},{"months":b[1][630],"eras":b[1][631]},{"nu":b[0][35],"patterns":b[1][632],"symbols":b[1][633],"currencies":b[0][358]},{"months":b[1][635],"days":b[1][636],"eras":b[1][637],"dayPeriods":b[0][1358]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][1359]},{"months":b[1][639],"days":b[1][640],"eras":b[1][641],"dayPeriods":b[0][1368]},{"nu":b[0][1369],"patterns":b[1][20],"symbols":b[1][642],"currencies":b[0][358]},{"months":b[1][644]},{"months":b[1][645],"days":b[1][646],"eras":b[1][647],"dayPeriods":b[0][1378]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][1379]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1380]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1381]},{"months":b[1][648],"days":b[1][649],"eras":b[1][650],"dayPeriods":b[0][1388]},{"months":b[1][651],"days":b[1][652],"eras":b[1][653],"dayPeriods":b[0][1395]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][78]},{"months":b[1][655],"days":b[1][656],"eras":b[1][657],"dayPeriods":b[0][1400]},{"nu":b[0][1401],"patterns":b[1][27],"symbols":b[1][658],"currencies":b[0][1404]},{"months":b[1][659],"days":b[1][660],"eras":b[1][661],"dayPeriods":b[0][1409]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1410]},{"months":b[1][663],"days":b[1][664],"eras":b[1][665],"dayPeriods":b[0][546]},{"nu":b[0][35],"patterns":b[1][192],"symbols":b[1][28],"currencies":b[0][1418]},{"months":b[1][666],"days":b[1][667],"eras":b[1][668],"dayPeriods":b[0][172]},{"months":b[1][670],"days":b[1][671],"eras":b[1][672],"dayPeriods":b[0][888]},{"months":b[1][673],"eras":b[1][13]},{"nu":b[0][1369],"patterns":b[1][20],"symbols":b[1][642],"currencies":b[0][1432]},{"months":b[1][674],"days":b[1][675],"eras":b[1][672],"dayPeriods":b[0][1434]},{"months":b[1][676]},{"months":b[1][677],"eras":b[1][4]},{"months":b[1][678]},{"months":b[1][679],"eras":b[1][4]},{"months":b[1][680],"days":b[1][681],"eras":b[1][682],"dayPeriods":b[0][172]},{"months":b[1][683],"eras":b[1][11]},{"months":b[1][684],"eras":b[1][246]},{"months":b[1][685],"eras":b[1][686]},{"nu":b[0][35],"patterns":b[1][687],"symbols":b[1][81],"currencies":b[0][1450]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][1451]},{"nu":b[0][35],"patterns":b[1][687],"symbols":b[1][81],"currencies":b[0][1452]},{"nu":b[0][35],"patterns":b[1][687],"symbols":b[1][81],"currencies":b[0][1453]},{"nu":b[0][35],"patterns":b[1][687],"symbols":b[1][81],"currencies":b[0][1451]},{"nu":b[0][35],"patterns":b[1][687],"symbols":b[1][81],"currencies":b[0][1454]},{"months":b[1][688],"days":b[1][689],"eras":b[1][690],"dayPeriods":b[0][1461]},{"months":b[1][692],"days":b[1][693],"eras":b[1][694],"dayPeriods":b[0][1469]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][1470]},{"months":b[1][696],"days":b[1][697],"eras":b[1][698],"dayPeriods":b[0][1478]},{"months":b[1][699],"days":b[1][700],"eras":b[1][701],"dayPeriods":b[0][172]},{"months":b[1][702],"days":b[1][703],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][705],"days":b[1][706],"eras":b[1][707],"dayPeriods":b[0][1495]},{"months":b[1][708],"days":b[1][709],"eras":b[1][710],"dayPeriods":b[0][1501]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1502]},{"months":b[1][711],"days":b[1][712],"eras":b[1][70],"dayPeriods":b[0][536]},{"months":b[1][714],"days":b[1][715],"eras":b[1][716],"dayPeriods":b[0][1511]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][717],"currencies":b[0][358]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][717],"currencies":b[0][1513]},{"months":b[1][719],"days":b[1][720],"eras":b[1][721],"dayPeriods":b[0][172]},{"months":b[1][722],"days":b[1][723],"eras":b[1][724],"dayPeriods":b[0][172]},{"nu":b[0][690],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][1522]},{"months":b[1][726],"days":b[1][727],"eras":b[1][728],"dayPeriods":b[0][172]},{"months":b[1][729],"eras":b[1][11]},{"months":b[1][730],"eras":b[1][15]},{"months":b[1][731],"eras":b[1][18]},{"eras":b[1][732]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][1534]},{"months":b[1][734],"days":b[1][735],"eras":b[1][736],"dayPeriods":b[0][1536]},{"months":b[1][737],"eras":b[1][18]},{"nu":b[0][690],"patterns":b[1][76],"symbols":b[1][738],"currencies":b[0][1539]},{"months":b[1][740]},{"months":b[1][741],"days":b[1][742],"eras":b[1][743],"dayPeriods":b[0][172]},{"eras":b[1][744]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1547]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1548]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1549]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1550]},{"months":b[1][746]},{"months":b[1][747],"days":b[1][748],"eras":b[1][749],"dayPeriods":b[0][1561]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][1548]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1562]},{"months":b[1][750],"days":b[1][751],"eras":b[1][752],"dayPeriods":b[0][1569]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][357],"currencies":b[0][78]},{"months":b[1][753],"days":b[1][754],"eras":b[1][755],"dayPeriods":b[0][1576]},{"nu":b[0][35],"patterns":b[1][756],"symbols":b[1][81],"currencies":b[0][1577]},{"eras":b[1][758]},{"months":b[1][759],"eras":b[1][4]},{"months":b[1][760],"eras":b[1][4]},{"months":b[1][761],"days":b[1][762],"eras":b[1][763],"dayPeriods":b[0][546]},{"months":b[1][764],"eras":b[1][11]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][1591]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][81],"currencies":b[0][1592]},{"months":b[1][765],"days":b[1][766],"eras":b[1][767],"dayPeriods":b[0][1597]},{"months":b[1][2],"days":b[1][768],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][769],"eras":b[1][4]},{"months":b[1][770],"eras":b[1][4]},{"months":b[1][771],"days":b[1][772],"eras":b[1][773],"dayPeriods":b[0][172]},{"months":b[1][774],"eras":b[1][11]},{"months":b[1][775],"eras":b[1][13]},{"months":b[1][776],"eras":b[1][15]},{"eras":b[1][777]},{"months":b[1][778],"eras":b[1][18]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][779],"currencies":b[0][1612]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][779],"currencies":b[0][1613]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][779],"currencies":b[0][1614]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][779],"currencies":b[0][1615]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][779],"currencies":b[0][1616]},{"months":b[1][780],"days":b[1][781],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][81],"currencies":b[0][1621]},{"months":b[1][782],"days":b[1][783],"eras":b[1][784],"dayPeriods":b[0][1627]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][1628]},{"months":b[1][785],"days":b[1][786],"eras":b[1][787],"dayPeriods":b[0][1634]},{"months":b[1][789],"days":b[1][790],"eras":b[1][791],"dayPeriods":b[0][1642]},{"months":b[1][792],"days":b[1][793],"eras":b[1][794],"dayPeriods":b[0][1649]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][546],"currencies":b[0][1650]},{"months":b[1][795],"days":b[1][796],"eras":b[1][794],"dayPeriods":b[0][1649]},{"months":b[1][798],"days":b[1][799],"eras":b[1][800],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][33],"symbols":b[1][81],"currencies":b[0][1323]},{"months":b[1][196],"days":b[1][801],"eras":b[1][198],"dayPeriods":b[0][1078]},{"months":b[1][802],"days":b[1][803],"eras":b[1][804],"dayPeriods":b[0][1666]},{"nu":b[0][35],"patterns":b[1][266],"symbols":b[1][81],"currencies":b[0][78]},{"months":b[1][805],"days":b[1][806],"eras":b[1][807],"dayPeriods":b[0][1673]},{"months":b[1][808],"days":b[1][809],"eras":b[1][810],"dayPeriods":b[0][1680]},{"months":b[1][812],"days":b[1][813],"eras":b[1][814],"dayPeriods":b[0][1692]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1693]},{"months":b[1][816],"days":b[1][817],"eras":b[1][818],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][28],"currencies":b[0][1701]},{"months":b[1][820],"days":b[1][821],"eras":b[1][822],"dayPeriods":b[0][1709]},{"nu":b[0][35],"patterns":b[1][527],"symbols":b[1][81],"currencies":b[0][1710]},{"months":b[1][823],"days":b[1][824],"eras":b[1][825],"dayPeriods":b[0][172]},{"months":b[1][826],"days":b[1][827],"eras":b[1][828],"dayPeriods":b[0][1722]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1723]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1724]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1725]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1726]},{"months":b[1][830],"days":b[1][831],"eras":b[1][832],"dayPeriods":b[0][1735]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][1736]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][28],"currencies":b[0][1737]},{"months":b[1][834],"days":b[1][835],"eras":b[1][145],"dayPeriods":b[0][313]},{"months":b[1][836],"days":b[1][837],"eras":b[1][145],"dayPeriods":b[0][313]},{"months":b[1][838],"eras":b[1][4]},{"months":b[1][839],"eras":b[1][4]},{"months":b[1][840],"days":b[1][841],"eras":b[1][842],"dayPeriods":b[0][299]},{"months":b[1][843],"eras":b[1][11]},{"months":b[1][844],"eras":b[1][246]},{"months":b[1][845],"eras":b[1][15]},{"eras":b[1][846]},{"months":b[1][847],"eras":b[1][18]},{"eras":b[1][848]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][1759]},{"months":b[1][849],"days":b[1][850],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][28],"currencies":b[0][1764]},{"months":b[1][22],"days":b[1][851],"eras":b[1][9],"dayPeriods":b[0][22]},{"months":b[1][852],"days":b[1][853],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][28],"currencies":b[0][1771]},{"months":b[1][855],"eras":b[1][4]},{"months":b[1][856],"eras":b[1][4]},{"months":b[1][857],"days":b[1][858],"eras":b[1][859],"dayPeriods":b[0][1778]},{"months":b[1][860],"eras":b[1][11]},{"months":b[1][861],"eras":b[1][862]},{"months":b[1][863],"eras":b[1][15]},{"eras":b[1][864]},{"months":b[1][865],"eras":b[1][18]},{"eras":b[1][866]},{"nu":b[0][35],"patterns":b[1][92],"symbols":b[1][546],"currencies":b[0][1786]},{"months":b[1][857],"days":b[1][858],"eras":b[1][867],"dayPeriods":b[0][1778]},{"months":b[1][869],"days":b[1][870],"eras":b[1][187],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1789]},{"months":b[1][871],"days":b[1][872],"eras":b[1][873],"dayPeriods":b[0][1796]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][81],"currencies":b[0][1222]},{"months":b[1][875]},{"months":b[1][876],"days":b[1][877],"eras":b[1][878],"dayPeriods":b[0][1806]},{"nu":b[0][35],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][358]},{"nu":b[0][35],"patterns":b[1][71],"symbols":b[1][21],"currencies":b[0][1807]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][1808]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][1381]},{"months":b[1][880],"days":b[1][881],"eras":b[1][882],"dayPeriods":b[0][172]},{"months":b[1][883],"eras":b[1][884]},{"months":b[1][885],"days":b[1][886],"eras":b[1][787],"dayPeriods":b[0][1825]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1826]},{"months":b[1][887],"days":b[1][888],"eras":b[1][889],"dayPeriods":b[0][1833]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][1834]},{"eras":b[1][890]},{"months":b[1][891],"eras":b[1][4]},{"months":b[1][892],"eras":b[1][4]},{"months":b[1][893],"days":b[1][894],"eras":b[1][895],"dayPeriods":b[0][1846]},{"months":b[1][896],"eras":b[1][897]},{"months":b[1][898],"eras":b[1][899]},{"months":b[1][900],"eras":b[1][901]},{"eras":b[1][902]},{"months":b[1][903],"eras":b[1][904]},{"eras":b[1][905]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1859]},{"months":b[1][906],"days":b[1][907],"eras":b[1][39],"dayPeriods":b[0][1863]},{"months":b[1][908],"days":b[1][909],"eras":b[1][39],"dayPeriods":b[0][1863]},{"months":b[1][908],"days":b[1][910],"eras":b[1][39],"dayPeriods":b[0][1869]},{"months":b[1][911],"days":b[1][912],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][28],"currencies":b[0][1874]},{"months":b[1][914],"days":b[1][915],"eras":b[1][916],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][917],"currencies":b[0][1885]},{"months":b[1][919],"eras":b[1][4]},{"months":b[1][920],"eras":b[1][4]},{"months":b[1][921],"days":b[1][922],"eras":b[1][923],"dayPeriods":b[0][1898]},{"months":b[1][924],"eras":b[1][11]},{"months":b[1][925],"eras":b[1][926]},{"months":b[1][927],"eras":b[1][18]},{"nu":b[0][35],"patterns":b[1][928],"symbols":b[1][81],"currencies":b[0][1904]},{"months":b[1][929],"days":b[1][930],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][196],"days":b[1][801],"eras":b[1][198],"dayPeriods":b[0][425]},{"months":b[1][931],"days":b[1][932],"eras":b[1][933],"dayPeriods":b[0][1915]},{"months":b[1][935],"eras":b[1][4]},{"months":b[1][936],"eras":b[1][4]},{"months":b[1][937],"days":b[1][938],"eras":b[1][939],"dayPeriods":b[0][1927]},{"months":b[1][940],"eras":b[1][11]},{"months":b[1][941],"eras":b[1][13]},{"months":b[1][942],"eras":b[1][15]},{"eras":b[1][943]},{"months":b[1][944],"eras":b[1][18]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][945],"currencies":b[0][1937]},{"months":b[1][947],"days":b[1][948],"eras":b[1][949],"dayPeriods":b[0][1943]},{"months":b[1][950],"eras":b[1][951]},{"nu":b[0][690],"patterns":b[1][71],"symbols":b[1][952],"currencies":b[0][1947]},{"months":b[1][947],"days":b[1][953],"eras":b[1][949],"dayPeriods":b[0][1943]},{"nu":b[0][35],"patterns":b[1][954],"symbols":b[1][952],"currencies":b[0][1947]},{"months":b[1][955],"days":b[1][956],"eras":b[1][957],"dayPeriods":b[0][172]},{"months":b[1][958],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][28],"currencies":b[0][1956]},{"months":b[1][960],"days":b[1][961],"eras":b[1][736],"dayPeriods":b[0][172]},{"nu":b[0][690],"patterns":b[1][76],"symbols":b[1][738],"currencies":b[0][1959]},{"months":b[1][962],"days":b[1][963],"eras":b[1][964],"dayPeriods":b[0][172]},{"months":b[1][965],"eras":b[1][15]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][28],"currencies":b[0][1964]},{"months":b[1][966],"days":b[1][967],"eras":b[1][70],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][1967]},{"months":b[1][968],"days":b[1][969],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][970],"days":b[1][971],"eras":b[1][70],"dayPeriods":b[0][172]},{"months":b[1][973],"days":b[1][974],"eras":b[1][975],"dayPeriods":b[0][1982]},{"eras":b[1][976]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][81],"currencies":b[0][1074]},{"months":b[1][977],"days":b[1][978],"eras":b[1][979],"dayPeriods":b[0][1989]},{"months":b[1][981],"days":b[1][982],"eras":b[1][983],"dayPeriods":b[0][172]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][984],"currencies":b[0][1998]},{"months":b[1][908],"days":b[1][985],"eras":b[1][986],"dayPeriods":b[0][2001]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][357],"currencies":b[0][43]},{"months":b[1][987],"days":b[1][988],"eras":b[1][989],"dayPeriods":b[0][172]},{"months":b[1][557],"days":b[1][990],"eras":b[1][991],"dayPeriods":b[0][2011]},{"nu":b[0][35],"patterns":b[1][76],"symbols":b[1][21],"currencies":b[0][349]},{"months":b[1][992],"days":b[1][993],"eras":b[1][994],"dayPeriods":b[0][2018]},{"months":b[1][995],"days":b[1][996],"eras":b[1][997],"dayPeriods":b[0][2025]},{"months":b[1][998],"days":b[1][999],"eras":b[1][997],"dayPeriods":b[0][2030]},{"months":b[1][805],"days":b[1][1000],"eras":b[1][807],"dayPeriods":b[0][1673]},{"nu":b[0][35],"patterns":b[1][756],"symbols":b[1][28],"currencies":b[0][78]},{"eras":b[1][1002]},{"months":b[1][1003],"eras":b[1][4]},{"months":b[1][1004],"days":b[1][1005],"eras":b[1][1006],"dayPeriods":b[0][2041]},{"months":b[1][1007],"eras":b[1][1008]},{"months":b[1][1004],"eras":b[1][1009]},{"months":b[1][1004],"eras":b[1][1010]},{"eras":b[1][1011]},{"months":b[1][1004],"eras":b[1][1012]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][2049]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][2050]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][2053]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][21],"currencies":b[0][2054]},{"months":b[1][1016]},{"nu":b[0][35],"patterns":b[1][27],"symbols":b[1][21],"currencies":b[0][2060]},{"eras":b[1][1018]},{"months":b[1][1019],"eras":b[1][4]},{"months":b[1][450],"days":b[1][1020],"eras":b[1][1021],"dayPeriods":b[0][2041]},{"months":b[1][1022],"eras":b[1][1023]},{"months":b[1][1024],"eras":b[1][1025]},{"months":b[1][1026],"eras":b[1][1027]},{"eras":b[1][1028]},{"months":b[1][450],"eras":b[1][1029]},{"eras":b[1][1030]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][1031],"currencies":b[0][2075]},{"months":b[1][1032],"days":b[1][1033],"eras":b[1][315],"dayPeriods":b[0][2081]},{"nu":b[0][35],"patterns":b[1][20],"symbols":b[1][1034],"currencies":b[0][64]}];b[3]=[{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][6],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][14],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][17],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][20],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][22],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][24],"ethiopic":b[2][25],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][26],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][27],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][29],"chinese":b[2][1],"coptic":b[2][30],"ethiopic":b[2][31],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][32],"hebrew":b[2][33],"indian":b[2][8],"islamic":b[2][34],"japanese":b[2][35],"persian":b[2][36],"roc":b[2][37]},{"buddhist":b[2][29],"chinese":b[2][1],"coptic":b[2][30],"ethiopic":b[2][31],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][40],"hebrew":b[2][33],"indian":b[2][8],"islamic":b[2][34],"japanese":b[2][35],"persian":b[2][36],"roc":b[2][37]},{"buddhist":b[2][29],"chinese":b[2][1],"coptic":b[2][30],"ethiopic":b[2][31],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][44],"hebrew":b[2][33],"indian":b[2][8],"islamic":b[2][34],"japanese":b[2][35],"persian":b[2][36],"roc":b[2][37]},{"buddhist":b[2][29],"chinese":b[2][1],"coptic":b[2][30],"ethiopic":b[2][31],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][45],"hebrew":b[2][33],"indian":b[2][8],"islamic":b[2][34],"japanese":b[2][35],"persian":b[2][36],"roc":b[2][37]},{"buddhist":b[2][29],"chinese":b[2][1],"coptic":b[2][30],"ethiopic":b[2][31],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][47],"hebrew":b[2][33],"indian":b[2][8],"islamic":b[2][34],"japanese":b[2][35],"persian":b[2][36],"roc":b[2][37]},{"buddhist":b[2][29],"chinese":b[2][1],"coptic":b[2][30],"ethiopic":b[2][31],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][48],"hebrew":b[2][33],"indian":b[2][8],"islamic":b[2][34],"japanese":b[2][35],"persian":b[2][36],"roc":b[2][37]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][53],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][55],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][57],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][59],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][61],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][63],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][65],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][67],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][69],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][71],"hebrew":b[2][72],"indian":b[2][73],"islamic":b[2][74],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][76],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][78],"hebrew":b[2][7],"indian":b[2][79],"islamic":b[2][80],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][82],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][85],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][87],"hebrew":b[2][7],"indian":b[2][88],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][90],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][91],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][93],"chinese":b[2][1],"coptic":b[2][94],"ethiopic":b[2][95],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][96],"hebrew":b[2][97],"indian":b[2][98],"islamic":b[2][99],"japanese":b[2][100],"persian":b[2][101],"roc":b[2][102]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][104],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][106],"chinese":b[2][107],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][108],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][111],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][113],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][115],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][116]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][118],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][119],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][122],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][124],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][126],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][131],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][133],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][134],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][136],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][138],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][139],"coptic":b[2][140],"ethiopic":b[2][141],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][142],"hebrew":b[2][143],"indian":b[2][144],"islamic":b[2][145],"japanese":b[2][10],"persian":b[2][146],"roc":b[2][147]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][149],"hebrew":b[2][150],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][151]},{"buddhist":b[2][0],"chinese":b[2][154],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][155],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][154],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][168],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][107],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][172],"hebrew":b[2][7],"indian":b[2][173],"islamic":b[2][174],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][154],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][155],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][180],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][154],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][182],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][217],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][218],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][219]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][233],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][219]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][240],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][219]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][244],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][246],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][247],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][248]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][250],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][251],"hebrew":b[2][252],"indian":b[2][8],"islamic":b[2][253],"japanese":b[2][10],"persian":b[2][254],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][256],"hebrew":b[2][252],"indian":b[2][8],"islamic":b[2][253],"japanese":b[2][10],"persian":b[2][257],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][258],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][261],"hebrew":b[2][262],"indian":b[2][8],"islamic":b[2][263],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][265],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][266],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][268],"chinese":b[2][269],"coptic":b[2][270],"dangi":b[2][269],"ethiopic":b[2][271],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][272],"hebrew":b[2][273],"indian":b[2][274],"islamic":b[2][275],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][276]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][297],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][299],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][301],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][302],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][304],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][306],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][308],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][309],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][310],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][313],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][314],"hebrew":b[2][315],"indian":b[2][8],"islamic":b[2][316],"japanese":b[2][317],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][319],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][320],"hebrew":b[2][7],"indian":b[2][321],"islamic":b[2][322],"japanese":b[2][323],"persian":b[2][324],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][326],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][327]},{"buddhist":b[2][330],"chinese":b[2][331],"coptic":b[2][332],"ethiopic":b[2][333],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][334],"hebrew":b[2][335],"indian":b[2][336],"islamic":b[2][337],"japanese":b[2][10],"persian":b[2][338],"roc":b[2][339]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][341],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][343],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][107],"coptic":b[2][332],"ethiopic":b[2][333],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][344],"hebrew":b[2][345],"indian":b[2][173],"islamic":b[2][346],"japanese":b[2][347],"persian":b[2][338],"roc":b[2][348]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][350],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][352],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][354],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][356],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][357],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][358]},{"buddhist":b[2][361],"chinese":b[2][362],"coptic":b[2][363],"dangi":b[2][362],"ethiopic":b[2][364],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][365],"hebrew":b[2][366],"indian":b[2][367],"islamic":b[2][368],"japanese":b[2][369],"persian":b[2][370],"roc":b[2][371]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][373],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][374],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][376],"hebrew":b[2][377],"indian":b[2][8],"islamic":b[2][378],"japanese":b[2][10],"persian":b[2][379],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][381],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][383],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][384],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][385],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][387],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][388],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][389],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][390],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][392],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][393],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][395],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][396],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][397],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][399],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][400],"chinese":b[2][1],"coptic":b[2][401],"dangi":b[2][402],"ethiopic":b[2][403],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][404],"hebrew":b[2][405],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][406],"persian":b[2][11],"roc":b[2][407]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][409],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][410],"hebrew":b[2][7],"indian":b[2][411],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][413],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][414],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][415],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][417],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][418],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][420],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][422],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][424],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][426],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][429],"chinese":b[2][107],"coptic":b[2][430],"ethiopic":b[2][431],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][432],"hebrew":b[2][433],"indian":b[2][434],"islamic":b[2][435],"japanese":b[2][436],"persian":b[2][437],"roc":b[2][438]},{"buddhist":b[2][0],"chinese":b[2][107],"coptic":b[2][389],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][440],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][442],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][444],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][446],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][448],"hebrew":b[2][449],"indian":b[2][8],"islamic":b[2][450],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][452],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][454],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][455],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][457],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][459],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][461],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][463],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][465],"hebrew":b[2][7],"indian":b[2][466],"islamic":b[2][467],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][469],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][471],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][473],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][474],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][478],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][479],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][481],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][483],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][485],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][487],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][488],"hebrew":b[2][7],"indian":b[2][489],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][491],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][492],"coptic":b[2][493],"dangi":b[2][494],"ethiopic":b[2][495],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][496],"hebrew":b[2][497],"indian":b[2][498],"islamic":b[2][499],"japanese":b[2][10],"persian":b[2][338],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][506],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][507],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][509],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][510],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][511],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][512],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][513],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][515],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][516],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][519],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][520],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][331],"coptic":b[2][332],"ethiopic":b[2][333],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][522],"hebrew":b[2][523],"indian":b[2][336],"islamic":b[2][524],"japanese":b[2][10],"persian":b[2][525],"roc":b[2][526]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][528],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][529],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][531],"coptic":b[2][332],"ethiopic":b[2][333],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][532],"hebrew":b[2][345],"indian":b[2][336],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][338],"roc":b[2][533]},{"buddhist":b[2][0],"chinese":b[2][538],"coptic":b[2][332],"ethiopic":b[2][333],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][539],"hebrew":b[2][345],"indian":b[2][336],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][338],"roc":b[2][533]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][542],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][544],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][546],"chinese":b[2][331],"coptic":b[2][547],"ethiopic":b[2][548],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][549],"hebrew":b[2][550],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][553],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][554],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][331],"coptic":b[2][555],"ethiopic":b[2][556],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][557],"hebrew":b[2][558],"indian":b[2][559],"islamic":b[2][560],"japanese":b[2][561],"persian":b[2][562],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][568],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][570],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][572],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][573],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][574],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][576],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][577],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][579],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][580],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][582],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][583],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][584],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][586],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][588],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][590],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][591],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][596],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][93],"chinese":b[2][1],"coptic":b[2][94],"ethiopic":b[2][95],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][599],"hebrew":b[2][97],"indian":b[2][98],"islamic":b[2][99],"japanese":b[2][100],"persian":b[2][101],"roc":b[2][102]},{"buddhist":b[2][93],"chinese":b[2][1],"coptic":b[2][94],"ethiopic":b[2][95],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][600],"hebrew":b[2][97],"indian":b[2][98],"islamic":b[2][99],"japanese":b[2][100],"persian":b[2][101],"roc":b[2][102]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][601],"ethiopic":b[2][602],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][603],"hebrew":b[2][604],"indian":b[2][605],"islamic":b[2][606],"japanese":b[2][607],"persian":b[2][608],"roc":b[2][609]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][611],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][613],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][614],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][616],"dangi":b[0][1403],"ethiopic":b[2][617],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][618],"hebrew":b[2][619],"indian":b[2][620],"islamic":b[2][621],"japanese":b[2][622],"persian":b[2][623],"roc":b[2][624]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][616],"dangi":b[0][1403],"ethiopic":b[2][617],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][626],"hebrew":b[2][619],"indian":b[2][620],"islamic":b[2][621],"japanese":b[2][622],"persian":b[2][623],"roc":b[2][624]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][627],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][629],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][631],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][632],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][637],"hebrew":b[2][7],"indian":b[2][638],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][639],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][641],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][643],"chinese":b[2][107],"coptic":b[2][644],"dangi":b[2][107],"ethiopic":b[2][645],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][646],"hebrew":b[2][647],"indian":b[2][648],"islamic":b[2][649],"japanese":b[2][650],"persian":b[2][651],"roc":b[2][652]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][654],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][655],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][656],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][657],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][659],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][661],"ethiopic":b[2][662],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][663],"hebrew":b[2][664],"indian":b[2][8],"islamic":b[2][665],"japanese":b[2][10],"persian":b[2][666],"roc":b[2][120]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][668],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][669],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][670],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][671],"ethiopic":b[2][672],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][673],"hebrew":b[2][674],"indian":b[2][675],"islamic":b[2][676],"japanese":b[2][677],"persian":b[2][678],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][680],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][681],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][683],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][681],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][685],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][686],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][688],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][686],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][690],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][691],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][693],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][695],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][696],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][107],"coptic":b[2][332],"ethiopic":b[2][333],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][697],"hebrew":b[2][345],"indian":b[2][336],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][338],"roc":b[2][698]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][700],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][701],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][703],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][705],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][706],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][708],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][709],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][710],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][711],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]},{"buddhist":b[2][713],"chinese":b[2][362],"coptic":b[2][714],"ethiopic":b[2][714],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][715],"hebrew":b[2][716],"indian":b[2][717],"islamic":b[2][718],"japanese":b[2][719],"persian":b[2][720],"roc":b[2][371]},{"buddhist":b[2][713],"chinese":b[2][725],"coptic":b[2][714],"ethiopic":b[2][714],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][715],"hebrew":b[2][716],"indian":b[2][717],"islamic":b[2][718],"japanese":b[2][719],"persian":b[2][720],"roc":b[2][371]},{"buddhist":b[2][727],"chinese":b[2][362],"coptic":b[2][728],"dangi":b[2][362],"ethiopic":b[2][728],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][729],"hebrew":b[2][730],"indian":b[2][731],"islamic":b[2][732],"japanese":b[2][733],"persian":b[2][734],"roc":b[2][735]},{"buddhist":b[2][0],"chinese":b[2][1],"coptic":b[2][2],"ethiopic":b[2][3],"ethioaa":b[2][4],"generic":b[2][5],"gregory":b[2][737],"hebrew":b[2][7],"indian":b[2][8],"islamic":b[2][9],"japanese":b[2][10],"persian":b[2][11],"roc":b[2][12]}];b[4]=[{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][0]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][1]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][23],"calendars":b[3][2]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][23],"calendars":b[3][2]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][3]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][34],"calendars":b[3][4]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][38],"calendars":b[3][5]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][45],"calendars":b[3][6]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][61],"calendars":b[3][7]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][45],"calendars":b[3][8]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][45],"calendars":b[3][9]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][61],"calendars":b[3][10]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][45],"calendars":b[3][11]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][12]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][72],"calendars":b[3][13]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][77],"calendars":b[3][14]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][82],"calendars":b[3][15]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][86],"calendars":b[3][16]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][17]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][93],"calendars":b[3][18]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][72],"calendars":b[3][19]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][72],"calendars":b[3][20]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][103],"calendars":b[3][21]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][22]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][113],"calendars":b[3][23]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][123],"calendars":b[3][24]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][25]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][130],"calendars":b[3][26]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][135],"calendars":b[3][27]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][135],"calendars":b[3][28]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][29]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][158],"calendars":b[3][30]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][72],"calendars":b[3][31]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][167],"calendars":b[3][32]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][171],"calendars":b[3][33]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][176],"calendars":b[3][34]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][180],"calendars":b[3][35]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][72],"calendars":b[3][36]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][188],"calendars":b[3][37]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][188],"calendars":b[3][38]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][39]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][40]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][41]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][206],"calendars":b[3][42]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][72],"calendars":b[3][43]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][214],"calendars":b[3][44]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][226],"calendars":b[3][45]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][232],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][237],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][238],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][239],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][239],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][240],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][232],"calendars":b[3][47]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][244],"calendars":b[3][48]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][248],"calendars":b[3][49]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][237],"calendars":b[3][50]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][250],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][251],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][252],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][253],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][254],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][255],"calendars":b[3][46]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][51]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][259],"calendars":b[3][52]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][264],"calendars":b[3][52]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][265],"calendars":b[3][52]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][259],"calendars":b[3][52]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][267],"calendars":b[3][53]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][271],"calendars":b[3][52]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][271],"calendars":b[3][52]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][273],"calendars":b[3][54]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][274],"calendars":b[3][55]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][278],"calendars":b[3][56]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][57]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][288],"calendars":b[3][58]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][288],"calendars":b[3][59]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][302],"calendars":b[3][60]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][306],"calendars":b[3][61]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][232],"calendars":b[3][62]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][316],"calendars":b[3][63]},{"ca":b[0][736],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][64]},{"ca":b[0][736],"hourNo0":true,"hour12":false,"formats":b[1][336],"calendars":b[3][64]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][337],"calendars":b[3][65]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][341],"calendars":b[3][66]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][345],"calendars":b[3][67]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][349],"calendars":b[3][68]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][353],"calendars":b[3][69]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][113],"calendars":b[3][70]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][71]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][72]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][362],"calendars":b[3][73]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][372],"calendars":b[3][74]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][375],"calendars":b[3][75]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][384],"calendars":b[3][76]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][394],"calendars":b[3][77]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][399],"calendars":b[3][78]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][413],"calendars":b[3][79]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][80]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][420],"calendars":b[3][81]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][82]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][83]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][434],"calendars":b[3][84]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][438],"calendars":b[3][85]},{"ca":b[0][736],"hourNo0":false,"hour12":false,"formats":b[1][445],"calendars":b[3][86]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][460],"calendars":b[3][87]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][88]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][467],"calendars":b[3][89]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][302],"calendars":b[3][90]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][91]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][92]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][484],"calendars":b[3][93]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][302],"calendars":b[3][94]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][95]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][492],"calendars":b[3][96]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][497],"calendars":b[3][97]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][500],"calendars":b[3][98]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][99]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][507],"calendars":b[3][100]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][512],"calendars":b[3][101]},{"ca":b[0][736],"hourNo0":true,"hour12":true,"formats":b[1][516],"calendars":b[3][102]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][103]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][130],"calendars":b[3][104]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][105]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][106]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][542],"calendars":b[3][107]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][108]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][549],"calendars":b[3][109]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][110]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][111]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][112]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][113]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][565],"calendars":b[3][114]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][581],"calendars":b[3][115]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][116]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][117]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][118]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][595],"calendars":b[3][119]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][120]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][121]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][302],"calendars":b[3][122]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][302],"calendars":b[3][123]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][614],"calendars":b[3][124]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][125]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][620],"calendars":b[3][126]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][624],"calendars":b[3][127]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][634],"calendars":b[3][128]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][638],"calendars":b[3][129]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][643],"calendars":b[3][130]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][131]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][132]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][654],"calendars":b[3][133]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][134]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][662],"calendars":b[3][135]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][136]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][669],"calendars":b[3][137]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][669],"calendars":b[3][138]},{"ca":b[0][736],"hourNo0":true,"hour12":false,"formats":b[1][507],"calendars":b[3][139]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][140]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][691],"calendars":b[3][141]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][695],"calendars":b[3][142]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][143]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][144]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][704],"calendars":b[3][145]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][31]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][146]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][147]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][713],"calendars":b[3][148]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][718],"calendars":b[3][149]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][718],"calendars":b[3][150]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][725],"calendars":b[3][151]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][733],"calendars":b[3][152]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][739],"calendars":b[3][153]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][745],"calendars":b[3][154]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][353],"calendars":b[3][155]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][156]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][757],"calendars":b[3][157]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][158]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][159]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][757],"calendars":b[3][160]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][161]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][162]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][163]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][788],"calendars":b[3][164]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][165]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][166]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][797],"calendars":b[3][167]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][302],"calendars":b[3][168]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][337],"calendars":b[3][169]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][337],"calendars":b[3][170]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][337],"calendars":b[3][171]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][811],"calendars":b[3][172]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][815],"calendars":b[3][173]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][819],"calendars":b[3][174]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][175]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][176]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][829],"calendars":b[3][177]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][833],"calendars":b[3][178]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][833],"calendars":b[3][179]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][833],"calendars":b[3][180]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][181]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][182]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][183]},{"ca":b[0][736],"hourNo0":true,"hour12":false,"formats":b[1][854],"calendars":b[3][184]},{"ca":b[0][736],"hourNo0":true,"hour12":false,"formats":b[1][854],"calendars":b[3][185]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][868],"calendars":b[3][186]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][187]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][874],"calendars":b[3][188]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][879],"calendars":b[3][189]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][190]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][191]},{"ca":b[0][736],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][192]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][193]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][194]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][195]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][196]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][913],"calendars":b[3][197]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][918],"calendars":b[3][198]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][199]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][200]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][201]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][934],"calendars":b[3][202]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][946],"calendars":b[3][203]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][946],"calendars":b[3][204]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][205]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][959],"calendars":b[3][206]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][207]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][208]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][788],"calendars":b[3][209]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][210]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][972],"calendars":b[3][211]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][212]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][980],"calendars":b[3][213]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][0],"calendars":b[3][214]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][0],"calendars":b[3][215]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][216]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][29],"calendars":b[3][217]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][218]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][362],"calendars":b[3][219]},{"ca":b[0][0],"hourNo0":true,"hour12":false,"formats":b[1][337],"calendars":b[3][220]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][1001],"calendars":b[3][221]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][1013],"calendars":b[3][221]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][1014],"calendars":b[3][221]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][1015],"calendars":b[3][222]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][1017],"calendars":b[3][221]},{"ca":b[0][736],"hourNo0":true,"hour12":true,"formats":b[1][1001],"calendars":b[3][223]},{"ca":b[0][0],"hourNo0":true,"hour12":true,"formats":b[1][167],"calendars":b[3][224]}];b[5]=[{"date":b[4][0],"number":b[2][13]},{"date":b[4][1],"number":b[2][15]},{"date":b[4][1],"number":b[2][16]},{"date":b[4][2],"number":b[2][18]},{"date":b[4][3],"number":b[2][19]},{"date":b[4][4],"number":b[2][21]},{"date":b[4][5],"number":b[2][23]},{"date":b[4][6],"number":b[2][28]},{"date":b[4][7],"number":b[2][38]},{"date":b[4][7],"number":b[2][39]},{"date":b[4][8],"number":b[2][41]},{"date":b[4][7],"number":b[2][42]},{"date":b[4][7],"number":b[2][43]},{"date":b[4][9],"number":b[2][38]},{"date":b[4][10],"number":b[2][38]},{"date":b[4][10],"number":b[2][46]},{"date":b[4][7],"number":b[2][41]},{"date":b[4][11],"number":b[2][41]},{"date":b[4][12],"number":b[2][49]},{"date":b[4][7],"number":b[2][50]},{"date":b[4][7],"number":b[2][51]},{"date":b[4][8],"number":b[2][52]},{"date":b[4][13],"number":b[2][54]},{"date":b[4][14],"number":b[2][56]},{"date":b[4][15],"number":b[2][58]},{"date":b[4][16],"number":b[2][60]},{"date":b[4][17],"number":b[2][62]},{"date":b[4][18],"number":b[2][64]},{"date":b[4][19],"number":b[2][66]},{"date":b[4][20],"number":b[2][68]},{"date":b[4][21],"number":b[2][70]},{"date":b[4][22],"number":b[2][75]},{"date":b[4][23],"number":b[2][77]},{"date":b[4][24],"number":b[2][81]},{"date":b[4][25],"number":b[2][83]},{"date":b[4][25],"number":b[2][84]},{"date":b[4][26],"number":b[2][86]},{"date":b[4][27],"number":b[2][89]},{"date":b[4][28],"number":b[2][92]},{"date":b[4][29],"number":b[2][103]},{"date":b[4][30],"number":b[2][105]},{"date":b[4][31],"number":b[2][109]},{"date":b[4][31],"number":b[2][110]},{"date":b[4][32],"number":b[2][112]},{"date":b[4][33],"number":b[2][114]},{"date":b[4][34],"number":b[2][117]},{"date":b[4][35],"number":b[2][77]},{"date":b[4][36],"number":b[2][121]},{"date":b[4][37],"number":b[2][123]},{"date":b[4][38],"number":b[2][125]},{"date":b[4][39],"number":b[2][127]},{"date":b[4][39],"number":b[2][128]},{"date":b[4][39],"number":b[2][129]},{"date":b[4][39],"number":b[2][130]},{"date":b[4][40],"number":b[2][132]},{"date":b[4][41],"number":b[2][64]},{"date":b[4][42],"number":b[2][135]},{"date":b[4][43],"number":b[2][137]},{"date":b[4][44],"number":b[2][123]},{"date":b[4][45],"number":b[2][148]},{"date":b[4][46],"number":b[2][152]},{"date":b[4][46],"number":b[2][153]},{"date":b[4][47],"number":b[2][156]},{"date":b[4][48],"number":b[2][58]},{"date":b[4][47],"number":b[2][157]},{"date":b[4][49],"number":b[2][158]},{"date":b[4][47],"number":b[2][159]},{"date":b[4][48],"number":b[2][160]},{"date":b[4][47],"number":b[2][161]},{"date":b[4][47],"number":b[2][162]},{"date":b[4][50],"number":b[2][163]},{"date":b[4][51],"number":b[2][164]},{"date":b[4][52],"number":b[2][165]},{"date":b[4][47],"number":b[2][166]},{"date":b[4][47],"number":b[2][167]},{"date":b[4][53],"number":b[2][157]},{"date":b[4][47],"number":b[2][169]},{"date":b[4][47],"number":b[2][170]},{"date":b[4][47],"number":b[2][171]},{"date":b[4][54],"number":b[2][175]},{"date":b[4][47],"number":b[2][176]},{"date":b[4][47],"number":b[2][177]},{"date":b[4][47],"number":b[2][178]},{"date":b[4][47],"number":b[2][179]},{"date":b[4][55],"number":b[2][181]},{"date":b[4][56],"number":b[2][157]},{"date":b[4][57],"number":b[2][183]},{"date":b[4][58],"number":b[2][184]},{"date":b[4][47],"number":b[2][185]},{"date":b[4][47],"number":b[2][186]},{"date":b[4][47],"number":b[2][187]},{"date":b[4][47],"number":b[2][188]},{"date":b[4][47],"number":b[2][189]},{"date":b[4][47],"number":b[2][190]},{"date":b[4][59],"number":b[2][191]},{"date":b[4][47],"number":b[2][192]},{"date":b[4][47],"number":b[2][193]},{"date":b[4][47],"number":b[2][194]},{"date":b[4][47],"number":b[2][195]},{"date":b[4][60],"number":b[2][196]},{"date":b[4][47],"number":b[2][197]},{"date":b[4][47],"number":b[2][198]},{"date":b[4][47],"number":b[2][199]},{"date":b[4][47],"number":b[2][200]},{"date":b[4][47],"number":b[2][201]},{"date":b[4][47],"number":b[2][202]},{"date":b[4][47],"number":b[2][203]},{"date":b[4][47],"number":b[2][204]},{"date":b[4][47],"number":b[2][205]},{"date":b[4][47],"number":b[2][206]},{"date":b[4][47],"number":b[2][207]},{"date":b[4][47],"number":b[2][208]},{"date":b[4][47],"number":b[2][209]},{"date":b[4][47],"number":b[2][210]},{"date":b[4][47],"number":b[2][211]},{"date":b[4][47],"number":b[2][212]},{"date":b[4][47],"number":b[2][213]},{"date":b[4][47],"number":b[2][214]},{"date":b[4][61],"number":b[2][215]},{"date":b[4][47],"number":b[2][216]},{"date":b[4][62],"number":b[2][157]},{"date":b[4][63],"number":b[2][86]},{"date":b[4][64],"number":b[2][220]},{"date":b[4][65],"number":b[2][221]},{"date":b[4][64],"number":b[2][222]},{"date":b[4][66],"number":b[2][223]},{"date":b[4][67],"number":b[2][224]},{"date":b[4][64],"number":b[2][225]},{"date":b[4][64],"number":b[2][226]},{"date":b[4][64],"number":b[2][227]},{"date":b[4][64],"number":b[2][228]},{"date":b[4][64],"number":b[2][229]},{"date":b[4][64],"number":b[2][230]},{"date":b[4][64],"number":b[2][231]},{"date":b[4][64],"number":b[2][232]},{"date":b[4][68],"number":b[2][234]},{"date":b[4][64],"number":b[2][235]},{"date":b[4][69],"number":b[2][236]},{"date":b[4][64],"number":b[2][237]},{"date":b[4][64],"number":b[2][238]},{"date":b[4][70],"number":b[2][228]},{"date":b[4][64],"number":b[2][239]},{"date":b[4][71],"number":b[2][241]},{"date":b[4][64],"number":b[2][242]},{"date":b[4][64],"number":b[2][243]},{"date":b[4][72],"number":b[2][245]},{"date":b[4][73],"number":b[2][249]},{"date":b[4][74],"number":b[2][135]},{"date":b[4][75],"number":b[2][255]},{"date":b[4][76],"number":b[2][255]},{"date":b[4][77],"number":b[2][135]},{"date":b[4][77],"number":b[2][259]},{"date":b[4][77],"number":b[2][260]},{"date":b[4][78],"number":b[2][264]},{"date":b[4][79],"number":b[2][198]},{"date":b[4][80],"number":b[2][267]},{"date":b[4][81],"number":b[2][277]},{"date":b[4][81],"number":b[2][278]},{"date":b[4][81],"number":b[2][279]},{"date":b[4][82],"number":b[2][280]},{"date":b[4][81],"number":b[2][281]},{"date":b[4][81],"number":b[2][282]},{"date":b[4][81],"number":b[2][283]},{"date":b[4][81],"number":b[2][284]},{"date":b[4][81],"number":b[2][285]},{"date":b[4][81],"number":b[2][286]},{"date":b[4][81],"number":b[2][287]},{"date":b[4][81],"number":b[2][288]},{"date":b[4][81],"number":b[2][289]},{"date":b[4][81],"number":b[2][290]},{"date":b[4][81],"number":b[2][291]},{"date":b[4][81],"number":b[2][292]},{"date":b[4][81],"number":b[2][293]},{"date":b[4][81],"number":b[2][294]},{"date":b[4][81],"number":b[2][295]},{"date":b[4][81],"number":b[2][296]},{"date":b[4][83],"number":b[2][298]},{"date":b[4][84],"number":b[2][300]},{"date":b[4][85],"number":b[2][114]},{"date":b[4][86],"number":b[2][303]},{"date":b[4][87],"number":b[2][305]},{"date":b[4][88],"number":b[2][307]},{"date":b[4][89],"number":b[2][123]},{"date":b[4][90],"number":b[2][77]},{"date":b[4][91],"number":b[2][311]},{"date":b[4][91],"number":b[2][312]},{"date":b[4][92],"number":b[2][114]},{"date":b[4][93],"number":b[2][318]},{"date":b[4][94],"number":b[2][325]},{"date":b[4][95],"number":b[2][328]},{"date":b[4][95],"number":b[2][329]},{"date":b[4][96],"number":b[2][340]},{"date":b[4][97],"number":b[2][342]},{"date":b[4][98],"number":b[2][298]},{"date":b[4][99],"number":b[2][349]},{"date":b[4][100],"number":b[2][351]},{"date":b[4][101],"number":b[2][353]},{"date":b[4][102],"number":b[2][355]},{"date":b[4][103],"number":b[2][359]},{"date":b[4][103],"number":b[2][360]},{"date":b[4][104],"number":b[2][372]},{"date":b[4][105],"number":b[2][298]},{"date":b[4][106],"number":b[2][375]},{"date":b[4][107],"number":b[2][380]},{"date":b[4][108],"number":b[2][382]},{"date":b[4][109],"number":b[2][123]},{"date":b[4][110],"number":b[2][375]},{"date":b[4][111],"number":b[2][386]},{"date":b[4][112],"number":b[2][132]},{"date":b[4][113],"number":b[2][123]},{"date":b[4][114],"number":b[2][391]},{"date":b[4][115],"number":b[2][298]},{"date":b[4][116],"number":b[2][394]},{"date":b[4][117],"number":b[2][123]},{"date":b[4][118],"number":b[2][398]},{"date":b[4][119],"number":b[2][77]},{"date":b[4][120],"number":b[2][408]},{"date":b[4][121],"number":b[2][89]},{"date":b[4][122],"number":b[2][412]},{"date":b[4][123],"number":b[2][70]},{"date":b[4][124],"number":b[2][135]},{"date":b[4][125],"number":b[2][416]},{"date":b[4][126],"number":b[2][77]},{"date":b[4][127],"number":b[2][419]},{"date":b[4][128],"number":b[2][421]},{"date":b[4][129],"number":b[2][423]},{"date":b[4][130],"number":b[2][425]},{"date":b[4][131],"number":b[2][427]},{"date":b[4][131],"number":b[2][428]},{"date":b[4][132],"number":b[2][439]},{"date":b[4][133],"number":b[2][441]},{"date":b[4][134],"number":b[2][443]},{"date":b[4][135],"number":b[2][445]},{"date":b[4][136],"number":b[2][447]},{"date":b[4][137],"number":b[2][451]},{"date":b[4][138],"number":b[2][123]},{"date":b[4][138],"number":b[2][453]},{"date":b[4][139],"number":b[2][123]},{"date":b[4][140],"number":b[2][456]},{"date":b[4][141],"number":b[2][458]},{"date":b[4][142],"number":b[2][460]},{"date":b[4][143],"number":b[2][462]},{"date":b[4][144],"number":b[2][464]},{"date":b[4][145],"number":b[2][468]},{"date":b[4][146],"number":b[2][470]},{"date":b[4][147],"number":b[2][472]},{"date":b[4][148],"number":b[2][475]},{"date":b[4][148],"number":b[2][476]},{"date":b[4][148],"number":b[2][477]},{"date":b[4][149],"number":b[2][77]},{"date":b[4][150],"number":b[2][480]},{"date":b[4][151],"number":b[2][482]},{"date":b[4][152],"number":b[2][484]},{"date":b[4][153],"number":b[2][486]},{"date":b[4][154],"number":b[2][77]},{"date":b[4][155],"number":b[2][490]},{"date":b[4][156],"number":b[2][490]},{"date":b[4][157],"number":b[2][500]},{"date":b[4][157],"number":b[2][501]},{"date":b[4][157],"number":b[2][502]},{"date":b[4][157],"number":b[2][503]},{"date":b[4][157],"number":b[2][504]},{"date":b[4][157],"number":b[2][505]},{"date":b[4][158],"number":b[2][135]},{"date":b[4][159],"number":b[2][508]},{"date":b[4][160],"number":b[2][298]},{"date":b[4][161],"number":b[2][19]},{"date":b[4][162],"number":b[2][19]},{"date":b[4][163],"number":b[2][77]},{"date":b[4][164],"number":b[2][112]},{"date":b[4][165],"number":b[2][16]},{"date":b[4][165],"number":b[2][514]},{"date":b[4][166],"number":b[2][89]},{"date":b[4][167],"number":b[2][517]},{"date":b[4][167],"number":b[2][518]},{"date":b[4][168],"number":b[2][89]},{"date":b[4][169],"number":b[2][521]},{"date":b[4][170],"number":b[2][527]},{"date":b[4][171],"number":b[2][530]},{"date":b[4][172],"number":b[2][534]},{"date":b[4][172],"number":b[2][535]},{"date":b[4][172],"number":b[2][536]},{"date":b[4][172],"number":b[2][537]},{"date":b[4][173],"number":b[2][540]},{"date":b[4][172],"number":b[2][541]},{"date":b[4][174],"number":b[2][543]},{"date":b[4][175],"number":b[2][545]},{"date":b[4][176],"number":b[2][551]},{"date":b[4][176],"number":b[2][552]},{"date":b[4][177],"number":b[2][375]},{"date":b[4][178],"number":b[2][462]},{"date":b[4][179],"number":b[2][563]},{"date":b[4][179],"number":b[2][564]},{"date":b[4][179],"number":b[2][565]},{"date":b[4][179],"number":b[2][566]},{"date":b[4][179],"number":b[2][567]},{"date":b[4][180],"number":b[2][569]},{"date":b[4][106],"number":b[2][70]},{"date":b[4][181],"number":b[2][571]},{"date":b[4][182],"number":b[2][123]},{"date":b[4][183],"number":b[2][70]},{"date":b[4][184],"number":b[2][575]},{"date":b[4][185],"number":b[2][575]},{"date":b[4][186],"number":b[2][578]},{"date":b[4][187],"number":b[2][132]},{"date":b[4][188],"number":b[2][581]},{"date":b[4][189],"number":b[2][21]},{"date":b[4][190],"number":b[2][21]},{"date":b[4][191],"number":b[2][585]},{"date":b[4][192],"number":b[2][587]},{"date":b[4][193],"number":b[2][589]},{"date":b[4][194],"number":b[2][77]},{"date":b[4][195],"number":b[2][592]},{"date":b[4][195],"number":b[2][593]},{"date":b[4][195],"number":b[2][594]},{"date":b[4][195],"number":b[2][595]},{"date":b[4][196],"number":b[2][597]},{"date":b[4][196],"number":b[2][598]},{"date":b[4][197],"number":b[2][103]},{"date":b[4][198],"number":b[2][103]},{"date":b[4][199],"number":b[2][610]},{"date":b[4][200],"number":b[2][612]},{"date":b[4][201],"number":b[2][105]},{"date":b[4][202],"number":b[2][615]},{"date":b[4][202],"number":b[2][19]},{"date":b[4][203],"number":b[2][625]},{"date":b[4][204],"number":b[2][625]},{"date":b[4][205],"number":b[2][453]},{"date":b[4][205],"number":b[2][628]},{"date":b[4][206],"number":b[2][630]},{"date":b[4][207],"number":b[2][633]},{"date":b[4][207],"number":b[2][634]},{"date":b[4][207],"number":b[2][635]},{"date":b[4][207],"number":b[2][636]},{"date":b[4][208],"number":b[2][114]},{"date":b[4][209],"number":b[2][640]},{"date":b[4][209],"number":b[2][112]},{"date":b[4][210],"number":b[2][642]},{"date":b[4][211],"number":b[2][653]},{"date":b[4][212],"number":b[2][15]},{"date":b[4][213],"number":b[2][16]},{"date":b[4][214],"number":b[2][105]},{"date":b[4][215],"number":b[2][658]},{"date":b[4][215],"number":b[2][19]},{"date":b[4][216],"number":b[2][660]},{"date":b[4][217],"number":b[2][667]},{"date":b[4][218],"number":b[2][19]},{"date":b[4][219],"number":b[2][132]},{"date":b[4][220],"number":b[2][135]},{"date":b[4][221],"number":b[2][679]},{"date":b[4][222],"number":b[2][682]},{"date":b[4][223],"number":b[2][684]},{"date":b[4][224],"number":b[2][687]},{"date":b[4][225],"number":b[2][689]},{"date":b[4][226],"number":b[2][692]},{"date":b[4][227],"number":b[2][694]},{"date":b[4][228],"number":b[2][694]},{"date":b[4][229],"number":b[2][19]},{"date":b[4][230],"number":b[2][699]},{"date":b[4][231],"number":b[2][462]},{"date":b[4][232],"number":b[2][702]},{"date":b[4][233],"number":b[2][704]},{"date":b[4][234],"number":b[2][19]},{"date":b[4][235],"number":b[2][707]},{"date":b[4][236],"number":b[2][135]},{"date":b[4][237],"number":b[2][351]},{"date":b[4][238],"number":b[2][351]},{"date":b[4][239],"number":b[2][712]},{"date":b[4][240],"number":b[2][721]},{"date":b[4][241],"number":b[2][722]},{"date":b[4][242],"number":b[2][723]},{"date":b[4][242],"number":b[2][724]},{"date":b[4][243],"number":b[2][722]},{"date":b[4][244],"number":b[2][726]},{"date":b[4][245],"number":b[2][736]},{"date":b[4][246],"number":b[2][738]}];addLocaleData(b[5][0], "aa-DJ");addLocaleData(b[5][1], "aa-ER");addLocaleData(b[5][2], "aa-ET");addLocaleData(b[5][2], "aa");addLocaleData(b[5][3], "af-NA");addLocaleData(b[5][4], "af-ZA");addLocaleData(b[5][4], "af");addLocaleData(b[5][5], "agq-CM");addLocaleData(b[5][5], "agq");addLocaleData(b[5][6], "ak-GH");addLocaleData(b[5][6], "ak");addLocaleData(b[5][7], "am-ET");addLocaleData(b[5][7], "am");addLocaleData(b[5][8], "ar-001");addLocaleData(b[5][8], "ar-AE");addLocaleData(b[5][8], "ar-BH");addLocaleData(b[5][9], "ar-DJ");addLocaleData(b[5][10], "ar-DZ");addLocaleData(b[5][8], "ar-EG");addLocaleData(b[5][11], "ar-EH");addLocaleData(b[5][12], "ar-ER");addLocaleData(b[5][8], "ar-IL");addLocaleData(b[5][13], "ar-IQ");addLocaleData(b[5][14], "ar-JO");addLocaleData(b[5][8], "ar-KM");addLocaleData(b[5][8], "ar-KW");addLocaleData(b[5][15], "ar-LB");addLocaleData(b[5][16], "ar-LY");addLocaleData(b[5][17], "ar-MA");addLocaleData(b[5][18], "ar-MR");addLocaleData(b[5][8], "ar-OM");addLocaleData(b[5][14], "ar-PS");addLocaleData(b[5][8], "ar-QA");addLocaleData(b[5][8], "ar-SA");addLocaleData(b[5][8], "ar-SD");addLocaleData(b[5][19], "ar-SO");addLocaleData(b[5][20], "ar-SS");addLocaleData(b[5][14], "ar-SY");addLocaleData(b[5][8], "ar-TD");addLocaleData(b[5][21], "ar-TN");addLocaleData(b[5][8], "ar-YE");addLocaleData(b[5][8], "ar");addLocaleData(b[5][22], "as-IN");addLocaleData(b[5][22], "as");addLocaleData(b[5][23], "asa-TZ");addLocaleData(b[5][23], "asa");addLocaleData(b[5][24], "ast-ES");addLocaleData(b[5][24], "ast");addLocaleData(b[5][25], "az-Cyrl-AZ");addLocaleData(b[5][26], "az-Cyrl");addLocaleData(b[5][25], "az-Latn-AZ");addLocaleData(b[5][25], "az-Latn");addLocaleData(b[5][25], "az");addLocaleData(b[5][27], "bas-CM");addLocaleData(b[5][27], "bas");addLocaleData(b[5][28], "be-BY");addLocaleData(b[5][28], "be");addLocaleData(b[5][29], "bem-ZM");addLocaleData(b[5][29], "bem");addLocaleData(b[5][30], "bez-TZ");addLocaleData(b[5][30], "bez");addLocaleData(b[5][31], "bg-BG");addLocaleData(b[5][31], "bg");addLocaleData(b[5][32], "bm-ML");addLocaleData(b[5][32], "bm");addLocaleData(b[5][33], "bn-BD");addLocaleData(b[5][33], "bn-IN");addLocaleData(b[5][33], "bn");addLocaleData(b[5][34], "bo-CN");addLocaleData(b[5][35], "bo-IN");addLocaleData(b[5][34], "bo");addLocaleData(b[5][36], "br-FR");addLocaleData(b[5][36], "br");addLocaleData(b[5][37], "brx-IN");addLocaleData(b[5][37], "brx");addLocaleData(b[5][38], "bs-Cyrl-BA");addLocaleData(b[5][39], "bs-Cyrl");addLocaleData(b[5][38], "bs-Latn-BA");addLocaleData(b[5][38], "bs-Latn");addLocaleData(b[5][38], "bs");addLocaleData(b[5][40], "byn-ER");addLocaleData(b[5][40], "byn");addLocaleData(b[5][41], "ca-AD");addLocaleData(b[5][41], "ca-ES");addLocaleData(b[5][42], "ca-FR");addLocaleData(b[5][41], "ca-IT");addLocaleData(b[5][41], "ca");addLocaleData(b[5][43], "cgg-UG");addLocaleData(b[5][43], "cgg");addLocaleData(b[5][44], "chr-US");addLocaleData(b[5][44], "chr");addLocaleData(b[5][45], "cs-CZ");addLocaleData(b[5][45], "cs");addLocaleData(b[5][46], "cy-GB");addLocaleData(b[5][46], "cy");addLocaleData(b[5][47], "da-DK");addLocaleData(b[5][47], "da-GL");addLocaleData(b[5][47], "da");addLocaleData(b[5][48], "dav-KE");addLocaleData(b[5][48], "dav");addLocaleData(b[5][49], "de-AT");addLocaleData(b[5][50], "de-BE");addLocaleData(b[5][51], "de-CH");addLocaleData(b[5][50], "de-DE");addLocaleData(b[5][52], "de-LI");addLocaleData(b[5][53], "de-LU");addLocaleData(b[5][50], "de");addLocaleData(b[5][54], "dje-NE");addLocaleData(b[5][54], "dje");addLocaleData(b[5][55], "dua-CM");addLocaleData(b[5][55], "dua");addLocaleData(b[5][56], "dyo-SN");addLocaleData(b[5][56], "dyo");addLocaleData(b[5][57], "dz-BT");addLocaleData(b[5][57], "dz");addLocaleData(b[5][58], "ebu-KE");addLocaleData(b[5][58], "ebu");addLocaleData(b[5][59], "ee-GH");addLocaleData(b[5][59], "ee-TG");addLocaleData(b[5][59], "ee");addLocaleData(b[5][60], "el-CY");addLocaleData(b[5][61], "el-GR");addLocaleData(b[5][61], "el");addLocaleData(b[5][62], "en-001");addLocaleData(b[5][63], "en-150");addLocaleData(b[5][64], "en-AG");addLocaleData(b[5][64], "en-AI");addLocaleData(b[5][64], "en-AS");addLocaleData(b[5][65], "en-AU");addLocaleData(b[5][66], "en-BB");addLocaleData(b[5][67], "en-BE");addLocaleData(b[5][68], "en-BM");addLocaleData(b[5][69], "en-BS");addLocaleData(b[5][70], "en-BW");addLocaleData(b[5][71], "en-BZ");addLocaleData(b[5][72], "en-CA");addLocaleData(b[5][73], "en-CC");addLocaleData(b[5][74], "en-CK");addLocaleData(b[5][64], "en-CM");addLocaleData(b[5][73], "en-CX");addLocaleData(b[5][64], "en-DG");addLocaleData(b[5][64], "en-DM");addLocaleData(b[5][64], "en-Dsrt-US");addLocaleData(b[5][75], "en-Dsrt");addLocaleData(b[5][76], "en-ER");addLocaleData(b[5][77], "en-FJ");addLocaleData(b[5][78], "en-FK");addLocaleData(b[5][64], "en-FM");addLocaleData(b[5][79], "en-GB");addLocaleData(b[5][64], "en-GD");addLocaleData(b[5][64], "en-GG");addLocaleData(b[5][80], "en-GH");addLocaleData(b[5][81], "en-GI");addLocaleData(b[5][82], "en-GM");addLocaleData(b[5][64], "en-GU");addLocaleData(b[5][83], "en-GY");addLocaleData(b[5][84], "en-HK");addLocaleData(b[5][85], "en-IE");addLocaleData(b[5][64], "en-IM");addLocaleData(b[5][86], "en-IN");addLocaleData(b[5][64], "en-IO");addLocaleData(b[5][64], "en-JE");addLocaleData(b[5][87], "en-JM");addLocaleData(b[5][88], "en-KE");addLocaleData(b[5][73], "en-KI");addLocaleData(b[5][64], "en-KN");addLocaleData(b[5][89], "en-KY");addLocaleData(b[5][64], "en-LC");addLocaleData(b[5][90], "en-LR");addLocaleData(b[5][91], "en-LS");addLocaleData(b[5][92], "en-MG");addLocaleData(b[5][64], "en-MH");addLocaleData(b[5][93], "en-MO");addLocaleData(b[5][64], "en-MP");addLocaleData(b[5][64], "en-MS");addLocaleData(b[5][94], "en-MT");addLocaleData(b[5][95], "en-MU");addLocaleData(b[5][96], "en-MW");addLocaleData(b[5][97], "en-NA");addLocaleData(b[5][73], "en-NF");addLocaleData(b[5][98], "en-NG");addLocaleData(b[5][73], "en-NR");addLocaleData(b[5][74], "en-NU");addLocaleData(b[5][99], "en-NZ");addLocaleData(b[5][100], "en-PG");addLocaleData(b[5][101], "en-PH");addLocaleData(b[5][102], "en-PK");addLocaleData(b[5][74], "en-PN");addLocaleData(b[5][64], "en-PR");addLocaleData(b[5][64], "en-PW");addLocaleData(b[5][103], "en-RW");addLocaleData(b[5][104], "en-SB");addLocaleData(b[5][105], "en-SC");addLocaleData(b[5][64], "en-SD");addLocaleData(b[5][106], "en-SG");addLocaleData(b[5][107], "en-SH");addLocaleData(b[5][108], "en-SL");addLocaleData(b[5][109], "en-SS");addLocaleData(b[5][110], "en-SX");addLocaleData(b[5][111], "en-SZ");addLocaleData(b[5][64], "en-TC");addLocaleData(b[5][74], "en-TK");addLocaleData(b[5][112], "en-TO");addLocaleData(b[5][113], "en-TT");addLocaleData(b[5][73], "en-TV");addLocaleData(b[5][114], "en-TZ");addLocaleData(b[5][115], "en-UG");addLocaleData(b[5][64], "en-UM");addLocaleData(b[5][64], "en-US");addLocaleData(b[5][64], "en-VC");addLocaleData(b[5][64], "en-VG");addLocaleData(b[5][64], "en-VI");addLocaleData(b[5][116], "en-VU");addLocaleData(b[5][117], "en-WS");addLocaleData(b[5][118], "en-ZA");addLocaleData(b[5][119], "en-ZM");addLocaleData(b[5][120], "en-ZW");addLocaleData(b[5][64], "en");addLocaleData(b[5][121], "eo-001");addLocaleData(b[5][121], "eo");addLocaleData(b[5][122], "es-419");addLocaleData(b[5][123], "es-AR");addLocaleData(b[5][124], "es-BO");addLocaleData(b[5][125], "es-CL");addLocaleData(b[5][126], "es-CO");addLocaleData(b[5][127], "es-CR");addLocaleData(b[5][128], "es-CU");addLocaleData(b[5][129], "es-DO");addLocaleData(b[5][130], "es-EA");addLocaleData(b[5][131], "es-EC");addLocaleData(b[5][130], "es-ES");addLocaleData(b[5][132], "es-GQ");addLocaleData(b[5][133], "es-GT");addLocaleData(b[5][134], "es-HN");addLocaleData(b[5][130], "es-IC");addLocaleData(b[5][135], "es-MX");addLocaleData(b[5][136], "es-NI");addLocaleData(b[5][137], "es-PA");addLocaleData(b[5][138], "es-PE");addLocaleData(b[5][139], "es-PH");addLocaleData(b[5][140], "es-PR");addLocaleData(b[5][141], "es-PY");addLocaleData(b[5][130], "es-SV");addLocaleData(b[5][142], "es-US");addLocaleData(b[5][143], "es-UY");addLocaleData(b[5][144], "es-VE");addLocaleData(b[5][130], "es");addLocaleData(b[5][145], "et-EE");addLocaleData(b[5][145], "et");addLocaleData(b[5][146], "eu-ES");addLocaleData(b[5][146], "eu");addLocaleData(b[5][147], "ewo-CM");addLocaleData(b[5][147], "ewo");addLocaleData(b[5][148], "fa-AF");addLocaleData(b[5][149], "fa-IR");addLocaleData(b[5][149], "fa");addLocaleData(b[5][150], "ff-CM");addLocaleData(b[5][151], "ff-GN");addLocaleData(b[5][152], "ff-MR");addLocaleData(b[5][150], "ff-SN");addLocaleData(b[5][150], "ff");addLocaleData(b[5][153], "fi-FI");addLocaleData(b[5][153], "fi");addLocaleData(b[5][154], "fil-PH");addLocaleData(b[5][154], "fil");addLocaleData(b[5][155], "fo-FO");addLocaleData(b[5][155], "fo");addLocaleData(b[5][156], "fr-BE");addLocaleData(b[5][157], "fr-BF");addLocaleData(b[5][158], "fr-BI");addLocaleData(b[5][157], "fr-BJ");addLocaleData(b[5][157], "fr-BL");addLocaleData(b[5][159], "fr-CA");addLocaleData(b[5][160], "fr-CD");addLocaleData(b[5][157], "fr-CF");addLocaleData(b[5][157], "fr-CG");addLocaleData(b[5][161], "fr-CH");addLocaleData(b[5][157], "fr-CI");addLocaleData(b[5][157], "fr-CM");addLocaleData(b[5][162], "fr-DJ");addLocaleData(b[5][163], "fr-DZ");addLocaleData(b[5][157], "fr-FR");addLocaleData(b[5][157], "fr-GA");addLocaleData(b[5][157], "fr-GF");addLocaleData(b[5][164], "fr-GN");addLocaleData(b[5][157], "fr-GP");addLocaleData(b[5][157], "fr-GQ");addLocaleData(b[5][165], "fr-HT");addLocaleData(b[5][166], "fr-KM");addLocaleData(b[5][167], "fr-LU");addLocaleData(b[5][157], "fr-MA");addLocaleData(b[5][157], "fr-MC");addLocaleData(b[5][157], "fr-MF");addLocaleData(b[5][168], "fr-MG");addLocaleData(b[5][157], "fr-ML");addLocaleData(b[5][157], "fr-MQ");addLocaleData(b[5][169], "fr-MR");addLocaleData(b[5][170], "fr-MU");addLocaleData(b[5][157], "fr-NC");addLocaleData(b[5][157], "fr-NE");addLocaleData(b[5][157], "fr-PF");addLocaleData(b[5][157], "fr-PM");addLocaleData(b[5][157], "fr-RE");addLocaleData(b[5][171], "fr-RW");addLocaleData(b[5][172], "fr-SC");addLocaleData(b[5][157], "fr-SN");addLocaleData(b[5][173], "fr-SY");addLocaleData(b[5][157], "fr-TD");addLocaleData(b[5][157], "fr-TG");addLocaleData(b[5][174], "fr-TN");addLocaleData(b[5][175], "fr-VU");addLocaleData(b[5][157], "fr-WF");addLocaleData(b[5][157], "fr-YT");addLocaleData(b[5][157], "fr");addLocaleData(b[5][176], "fur-IT");addLocaleData(b[5][176], "fur");addLocaleData(b[5][177], "ga-IE");addLocaleData(b[5][177], "ga");addLocaleData(b[5][178], "gd-GB");addLocaleData(b[5][178], "gd");addLocaleData(b[5][179], "gl-ES");addLocaleData(b[5][179], "gl");addLocaleData(b[5][180], "gsw-CH");addLocaleData(b[5][180], "gsw-LI");addLocaleData(b[5][180], "gsw");addLocaleData(b[5][181], "gu-IN");addLocaleData(b[5][181], "gu");addLocaleData(b[5][182], "guz-KE");addLocaleData(b[5][182], "guz");addLocaleData(b[5][183], "gv-GB");addLocaleData(b[5][183], "gv-IM");addLocaleData(b[5][183], "gv");addLocaleData(b[5][184], "ha-Latn-GH");addLocaleData(b[5][185], "ha-Latn-NE");addLocaleData(b[5][185], "ha-Latn-NG");addLocaleData(b[5][185], "ha-Latn");addLocaleData(b[5][185], "ha");addLocaleData(b[5][186], "haw-US");addLocaleData(b[5][186], "haw");addLocaleData(b[5][187], "he-IL");addLocaleData(b[5][187], "he");addLocaleData(b[5][188], "hi-IN");addLocaleData(b[5][188], "hi");addLocaleData(b[5][189], "hr-BA");addLocaleData(b[5][190], "hr-HR");addLocaleData(b[5][190], "hr");addLocaleData(b[5][191], "hu-HU");addLocaleData(b[5][191], "hu");addLocaleData(b[5][192], "hy-AM");addLocaleData(b[5][192], "hy");addLocaleData(b[5][193], "ia-FR");addLocaleData(b[5][193], "ia");addLocaleData(b[5][194], "id-ID");addLocaleData(b[5][194], "id");addLocaleData(b[5][195], "ig-NG");addLocaleData(b[5][195], "ig");addLocaleData(b[5][196], "ii-CN");addLocaleData(b[5][196], "ii");addLocaleData(b[5][197], "is-IS");addLocaleData(b[5][197], "is");addLocaleData(b[5][198], "it-CH");addLocaleData(b[5][199], "it-IT");addLocaleData(b[5][199], "it-SM");addLocaleData(b[5][199], "it");addLocaleData(b[5][200], "ja-JP");addLocaleData(b[5][200], "ja");addLocaleData(b[5][201], "jgo-CM");addLocaleData(b[5][201], "jgo");addLocaleData(b[5][202], "jmc-TZ");addLocaleData(b[5][202], "jmc");addLocaleData(b[5][203], "ka-GE");addLocaleData(b[5][203], "ka");addLocaleData(b[5][204], "kab-DZ");addLocaleData(b[5][204], "kab");addLocaleData(b[5][205], "kam-KE");addLocaleData(b[5][205], "kam");addLocaleData(b[5][206], "kde-TZ");addLocaleData(b[5][206], "kde");addLocaleData(b[5][207], "kea-CV");addLocaleData(b[5][207], "kea");addLocaleData(b[5][208], "khq-ML");addLocaleData(b[5][208], "khq");addLocaleData(b[5][209], "ki-KE");addLocaleData(b[5][209], "ki");addLocaleData(b[5][210], "kk-Cyrl-KZ");addLocaleData(b[5][210], "kk-Cyrl");addLocaleData(b[5][210], "kk");addLocaleData(b[5][211], "kkj-CM");addLocaleData(b[5][211], "kkj");addLocaleData(b[5][212], "kl-GL");addLocaleData(b[5][212], "kl");addLocaleData(b[5][213], "kln-KE");addLocaleData(b[5][213], "kln");addLocaleData(b[5][214], "km-KH");addLocaleData(b[5][214], "km");addLocaleData(b[5][215], "kn-IN");addLocaleData(b[5][215], "kn");addLocaleData(b[5][216], "ko-KP");addLocaleData(b[5][216], "ko-KR");addLocaleData(b[5][216], "ko");addLocaleData(b[5][217], "kok-IN");addLocaleData(b[5][217], "kok");addLocaleData(b[5][218], "ks-Arab-IN");addLocaleData(b[5][218], "ks-Arab");addLocaleData(b[5][218], "ks");addLocaleData(b[5][219], "ksb-TZ");addLocaleData(b[5][219], "ksb");addLocaleData(b[5][220], "ksf-CM");addLocaleData(b[5][220], "ksf");addLocaleData(b[5][221], "ksh-DE");addLocaleData(b[5][221], "ksh");addLocaleData(b[5][222], "kw-GB");addLocaleData(b[5][222], "kw");addLocaleData(b[5][223], "ky-Cyrl-KG");addLocaleData(b[5][223], "ky-Cyrl");addLocaleData(b[5][223], "ky-KG");addLocaleData(b[5][223], "ky");addLocaleData(b[5][224], "lag-TZ");addLocaleData(b[5][224], "lag");addLocaleData(b[5][225], "lg-UG");addLocaleData(b[5][225], "lg");addLocaleData(b[5][226], "lkt-US");addLocaleData(b[5][226], "lkt");addLocaleData(b[5][227], "ln-AO");addLocaleData(b[5][228], "ln-CD");addLocaleData(b[5][228], "ln-CF");addLocaleData(b[5][228], "ln-CG");addLocaleData(b[5][228], "ln");addLocaleData(b[5][229], "lo-LA");addLocaleData(b[5][229], "lo");addLocaleData(b[5][230], "lt-LT");addLocaleData(b[5][230], "lt");addLocaleData(b[5][231], "lu-CD");addLocaleData(b[5][231], "lu");addLocaleData(b[5][232], "luo-KE");addLocaleData(b[5][232], "luo");addLocaleData(b[5][233], "luy-KE");addLocaleData(b[5][233], "luy");addLocaleData(b[5][234], "lv-LV");addLocaleData(b[5][234], "lv");addLocaleData(b[5][235], "mas-KE");addLocaleData(b[5][236], "mas-TZ");addLocaleData(b[5][235], "mas");addLocaleData(b[5][237], "mer-KE");addLocaleData(b[5][237], "mer");addLocaleData(b[5][238], "mfe-MU");addLocaleData(b[5][238], "mfe");addLocaleData(b[5][239], "mg-MG");addLocaleData(b[5][239], "mg");addLocaleData(b[5][240], "mgh-MZ");addLocaleData(b[5][240], "mgh");addLocaleData(b[5][241], "mgo-CM");addLocaleData(b[5][241], "mgo");addLocaleData(b[5][242], "mk-MK");addLocaleData(b[5][242], "mk");addLocaleData(b[5][243], "ml-IN");addLocaleData(b[5][243], "ml");addLocaleData(b[5][244], "mn-Cyrl-MN");addLocaleData(b[5][244], "mn-Cyrl");addLocaleData(b[5][244], "mn");addLocaleData(b[5][245], "mr-IN");addLocaleData(b[5][245], "mr");addLocaleData(b[5][246], "ms-Latn-BN");addLocaleData(b[5][247], "ms-Latn-MY");addLocaleData(b[5][248], "ms-Latn-SG");addLocaleData(b[5][247], "ms-Latn");addLocaleData(b[5][247], "ms");addLocaleData(b[5][249], "mt-MT");addLocaleData(b[5][249], "mt");addLocaleData(b[5][250], "mua-CM");addLocaleData(b[5][250], "mua");addLocaleData(b[5][251], "my-MM");addLocaleData(b[5][251], "my");addLocaleData(b[5][252], "naq-NA");addLocaleData(b[5][252], "naq");addLocaleData(b[5][253], "nb-NO");addLocaleData(b[5][253], "nb-SJ");addLocaleData(b[5][253], "nb");addLocaleData(b[5][254], "nd-ZW");addLocaleData(b[5][254], "nd");addLocaleData(b[5][255], "ne-IN");addLocaleData(b[5][256], "ne-NP");addLocaleData(b[5][256], "ne");addLocaleData(b[5][257], "nl-AW");addLocaleData(b[5][258], "nl-BE");addLocaleData(b[5][259], "nl-BQ");addLocaleData(b[5][260], "nl-CW");addLocaleData(b[5][261], "nl-NL");addLocaleData(b[5][262], "nl-SR");addLocaleData(b[5][260], "nl-SX");addLocaleData(b[5][261], "nl");addLocaleData(b[5][263], "nmg-CM");addLocaleData(b[5][263], "nmg");addLocaleData(b[5][264], "nn-NO");addLocaleData(b[5][264], "nn");addLocaleData(b[5][265], "nnh-CM");addLocaleData(b[5][265], "nnh");addLocaleData(b[5][266], "nr-ZA");addLocaleData(b[5][266], "nr");addLocaleData(b[5][267], "nso-ZA");addLocaleData(b[5][267], "nso");addLocaleData(b[5][268], "nus-SD");addLocaleData(b[5][268], "nus");addLocaleData(b[5][269], "nyn-UG");addLocaleData(b[5][269], "nyn");addLocaleData(b[5][270], "om-ET");addLocaleData(b[5][271], "om-KE");addLocaleData(b[5][270], "om");addLocaleData(b[5][272], "or-IN");addLocaleData(b[5][272], "or");addLocaleData(b[5][273], "os-GE");addLocaleData(b[5][274], "os-RU");addLocaleData(b[5][273], "os");addLocaleData(b[5][275], "pa-Arab-PK");addLocaleData(b[5][276], "pa-Arab");addLocaleData(b[5][275], "pa-Guru-IN");addLocaleData(b[5][275], "pa-Guru");addLocaleData(b[5][275], "pa");addLocaleData(b[5][277], "pl-PL");addLocaleData(b[5][277], "pl");addLocaleData(b[5][278], "ps-AF");addLocaleData(b[5][278], "ps");addLocaleData(b[5][279], "pt-AO");addLocaleData(b[5][280], "pt-BR");addLocaleData(b[5][280], "pt-CV");addLocaleData(b[5][280], "pt-GW");addLocaleData(b[5][281], "pt-MO");addLocaleData(b[5][282], "pt-MZ");addLocaleData(b[5][283], "pt-PT");addLocaleData(b[5][284], "pt-ST");addLocaleData(b[5][280], "pt-TL");addLocaleData(b[5][280], "pt");addLocaleData(b[5][285], "rm-CH");addLocaleData(b[5][285], "rm");addLocaleData(b[5][286], "rn-BI");addLocaleData(b[5][286], "rn");addLocaleData(b[5][287], "ro-MD");addLocaleData(b[5][288], "ro-RO");addLocaleData(b[5][288], "ro");addLocaleData(b[5][289], "rof-TZ");addLocaleData(b[5][289], "rof");addLocaleData(b[5][290], "root");addLocaleData(b[5][291], "ru-BY");addLocaleData(b[5][292], "ru-KG");addLocaleData(b[5][293], "ru-KZ");addLocaleData(b[5][294], "ru-MD");addLocaleData(b[5][295], "ru-RU");addLocaleData(b[5][295], "ru-UA");addLocaleData(b[5][295], "ru");addLocaleData(b[5][296], "rw-RW");addLocaleData(b[5][296], "rw");addLocaleData(b[5][297], "rwk-TZ");addLocaleData(b[5][297], "rwk");addLocaleData(b[5][298], "sah-RU");addLocaleData(b[5][298], "sah");addLocaleData(b[5][299], "saq-KE");addLocaleData(b[5][299], "saq");addLocaleData(b[5][300], "sbp-TZ");addLocaleData(b[5][300], "sbp");addLocaleData(b[5][301], "se-FI");addLocaleData(b[5][302], "se-NO");addLocaleData(b[5][302], "se");addLocaleData(b[5][303], "seh-MZ");addLocaleData(b[5][303], "seh");addLocaleData(b[5][304], "ses-ML");addLocaleData(b[5][304], "ses");addLocaleData(b[5][305], "sg-CF");addLocaleData(b[5][305], "sg");addLocaleData(b[5][306], "shi-Latn-MA");addLocaleData(b[5][307], "shi-Latn");addLocaleData(b[5][306], "shi-Tfng-MA");addLocaleData(b[5][306], "shi-Tfng");addLocaleData(b[5][306], "shi");addLocaleData(b[5][308], "si-LK");addLocaleData(b[5][308], "si");addLocaleData(b[5][309], "sk-SK");addLocaleData(b[5][309], "sk");addLocaleData(b[5][310], "sl-SI");addLocaleData(b[5][310], "sl");addLocaleData(b[5][311], "sn-ZW");addLocaleData(b[5][311], "sn");addLocaleData(b[5][312], "so-DJ");addLocaleData(b[5][313], "so-ET");addLocaleData(b[5][314], "so-KE");addLocaleData(b[5][315], "so-SO");addLocaleData(b[5][315], "so");addLocaleData(b[5][316], "sq-AL");addLocaleData(b[5][317], "sq-MK");addLocaleData(b[5][316], "sq-XK");addLocaleData(b[5][316], "sq");addLocaleData(b[5][318], "sr-Cyrl-BA");addLocaleData(b[5][319], "sr-Cyrl-ME");addLocaleData(b[5][319], "sr-Cyrl-RS");addLocaleData(b[5][319], "sr-Cyrl-XK");addLocaleData(b[5][319], "sr-Cyrl");addLocaleData(b[5][319], "sr-Latn-BA");addLocaleData(b[5][319], "sr-Latn-ME");addLocaleData(b[5][319], "sr-Latn-RS");addLocaleData(b[5][319], "sr-Latn-XK");addLocaleData(b[5][320], "sr-Latn");addLocaleData(b[5][319], "sr");addLocaleData(b[5][321], "ss-SZ");addLocaleData(b[5][321], "ss-ZA");addLocaleData(b[5][321], "ss");addLocaleData(b[5][322], "ssy-ER");addLocaleData(b[5][322], "ssy");addLocaleData(b[5][323], "st-LS");addLocaleData(b[5][324], "st-ZA");addLocaleData(b[5][324], "st");addLocaleData(b[5][325], "sv-AX");addLocaleData(b[5][326], "sv-FI");addLocaleData(b[5][325], "sv-SE");addLocaleData(b[5][325], "sv");addLocaleData(b[5][327], "sw-KE");addLocaleData(b[5][327], "sw-TZ");addLocaleData(b[5][328], "sw-UG");addLocaleData(b[5][327], "sw");addLocaleData(b[5][329], "swc-CD");addLocaleData(b[5][329], "swc");addLocaleData(b[5][330], "ta-IN");addLocaleData(b[5][331], "ta-LK");addLocaleData(b[5][332], "ta-MY");addLocaleData(b[5][333], "ta-SG");addLocaleData(b[5][330], "ta");addLocaleData(b[5][334], "te-IN");addLocaleData(b[5][334], "te");addLocaleData(b[5][335], "teo-KE");addLocaleData(b[5][336], "teo-UG");addLocaleData(b[5][336], "teo");addLocaleData(b[5][337], "tg-Cyrl-TJ");addLocaleData(b[5][337], "tg-Cyrl");addLocaleData(b[5][337], "tg");addLocaleData(b[5][338], "th-TH");addLocaleData(b[5][338], "th");addLocaleData(b[5][339], "ti-ER");addLocaleData(b[5][340], "ti-ET");addLocaleData(b[5][340], "ti");addLocaleData(b[5][341], "tig-ER");addLocaleData(b[5][341], "tig");addLocaleData(b[5][342], "tn-BW");addLocaleData(b[5][343], "tn-ZA");addLocaleData(b[5][343], "tn");addLocaleData(b[5][344], "to-TO");addLocaleData(b[5][344], "to");addLocaleData(b[5][345], "tr-CY");addLocaleData(b[5][345], "tr-TR");addLocaleData(b[5][345], "tr");addLocaleData(b[5][346], "ts-ZA");addLocaleData(b[5][346], "ts");addLocaleData(b[5][347], "twq-NE");addLocaleData(b[5][347], "twq");addLocaleData(b[5][348], "tzm-Latn-MA");addLocaleData(b[5][348], "tzm-Latn");addLocaleData(b[5][348], "tzm");addLocaleData(b[5][349], "uk-UA");addLocaleData(b[5][349], "uk");addLocaleData(b[5][350], "ur-IN");addLocaleData(b[5][351], "ur-PK");addLocaleData(b[5][351], "ur");addLocaleData(b[5][352], "uz-Arab-AF");addLocaleData(b[5][353], "uz-Arab");addLocaleData(b[5][352], "uz-Cyrl-UZ");addLocaleData(b[5][354], "uz-Cyrl");addLocaleData(b[5][352], "uz-Latn-UZ");addLocaleData(b[5][352], "uz-Latn");addLocaleData(b[5][352], "uz");addLocaleData(b[5][355], "vai-Latn-LR");addLocaleData(b[5][356], "vai-Latn");addLocaleData(b[5][355], "vai-Vaii-LR");addLocaleData(b[5][355], "vai-Vaii");addLocaleData(b[5][355], "vai");addLocaleData(b[5][357], "ve-ZA");addLocaleData(b[5][357], "ve");addLocaleData(b[5][358], "vi-VN");addLocaleData(b[5][358], "vi");addLocaleData(b[5][359], "vo-001");addLocaleData(b[5][359], "vo");addLocaleData(b[5][202], "vun-TZ");addLocaleData(b[5][202], "vun");addLocaleData(b[5][360], "wae-CH");addLocaleData(b[5][360], "wae");addLocaleData(b[5][361], "wal-ET");addLocaleData(b[5][361], "wal");addLocaleData(b[5][362], "xh-ZA");addLocaleData(b[5][362], "xh");addLocaleData(b[5][363], "xog-UG");addLocaleData(b[5][363], "xog");addLocaleData(b[5][364], "yav-CM");addLocaleData(b[5][364], "yav");addLocaleData(b[5][365], "yo-BJ");addLocaleData(b[5][366], "yo-NG");addLocaleData(b[5][366], "yo");addLocaleData(b[5][367], "zgh-MA");addLocaleData(b[5][367], "zgh");addLocaleData(b[5][368], "zh-Hans-CN");addLocaleData(b[5][369], "zh-Hans-HK");addLocaleData(b[5][370], "zh-Hans-MO");addLocaleData(b[5][371], "zh-Hans-SG");addLocaleData(b[5][368], "zh-Hans");addLocaleData(b[5][372], "zh-Hant-HK");addLocaleData(b[5][373], "zh-Hant-MO");addLocaleData(b[5][368], "zh-Hant-TW");addLocaleData(b[5][374], "zh-Hant");addLocaleData(b[5][368], "zh");addLocaleData(b[5][375], "zu-ZA");addLocaleData(b[5][375], "zu");})();
return Intl;
});
