/*
 * Copyright 2014 the original author or authors
 * @license MIT, see LICENSE.txt for details
 *
 * @author Scott Andrews
 */

(function (define) {
	'use strict';

	define(function (require) {

		var Promise = require('when/lib/Promise'),
			when = require('when'),
			normalizeHeaderName = require('./normalizeHeaderName');

		// extend ResponsePromise from Promise
		function ResponsePromise() {
			return Promise.apply(this, arguments);
		}
		ResponsePromise.prototype = Object.create(Promise.prototype);

		// augment ResponsePromise with HTTP Response specific methods

		function property(promise, name) {
			return promise.then(
				function (value) {
					return value && value[name];
				},
				function (value) {
					return when.reject(value && value[name]);
				}
			);
		}

		/**
		 * Obtain the response entity
		 *
		 * @returns {Promise} for the response entity
		 */
		ResponsePromise.prototype.entity = function entity() {
			return property(this, 'entity');
		};

		/**
		 * Obtain the response status
		 *
		 * @returns {Promise} for the response status
		 */
		ResponsePromise.prototype.status = function status() {
			return property(property(this, 'status'), 'code');
		};

		/**
		 * Obtain the response headers map
		 *
		 * @returns {Promise} for the response headers map
		 */
		ResponsePromise.prototype.headers = function headers() {
			return property(this, 'headers');
		};

		/**
		 * Obtain a specific response header
		 *
		 * @param {String} headerName the header to retrieve
		 * @returns {Promise} for the response header's value
		 */
		ResponsePromise.prototype.header = function header(headerName) {
			headerName = normalizeHeaderName(headerName);
			return property(this.headers(), headerName);
		};

		/**
		 * Wrap a Promise as an ResponsePromise
		 *
		 * @param {Promise<Response>} promise the promise for an HTTP Response
		 * @returns {ResponsePromise<Response>} wrapped promise for Response with additional helper methods
		 */
		function makeResponsePromise(promise) {
			return new ResponsePromise(promise.then.bind(promise));
		}

		makeResponsePromise.ResponsePromise = ResponsePromise;

		return makeResponsePromise;

	});

}(
	typeof define === 'function' && define.amd ? define : function (factory) { module.exports = factory(require); }
	// Boilerplate for AMD and Node
));
