import { __extends, __read } from "tslib";
import { Flow } from "./Flow";
import { SankeyNodes } from "./SankeyNodes";
import { SankeyLink } from "./SankeyLink";
import { area, line } from "d3-shape";
import { curveMonotoneXTension } from "../../core/render/MonotoneXTension";
import { curveMonotoneYTension } from "../../core/render/MonotoneYTension";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as $utils from "../../core/util/Utils";
import * as d3sankey from "d3-sankey";
/**
 * Sankey series.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/flow-charts/} for more information
 * @important
 */
var Sankey = /** @class */ (function (_super) {
    __extends(Sankey, _super);
    function Sankey() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * List of link elements.
         *
         * @default new ListTemplate<SankeyLink>
         */
        Object.defineProperty(_this, "links", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return SankeyLink._new(_this._root, { themeTags: ["link", "shape"] }, [_this.links.template]); })
        });
        /**
         * A series representing sankey nodes.
         *
         * @default SankeyNodes.new()
         */
        Object.defineProperty(_this, "nodes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(SankeyNodes.new(_this._root, {}))
        });
        Object.defineProperty(_this, "_d3Sankey", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3sankey.sankey()
        });
        Object.defineProperty(_this, "_d3Graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_fillGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: area()
        });
        Object.defineProperty(_this, "_strokeGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: line()
        });
        return _this;
    }
    Object.defineProperty(Sankey.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["sankey", this._settings.orientation || "horizontal"]);
            this._fillGenerator.y0(function (p) {
                return p[3];
            });
            this._fillGenerator.x0(function (p) {
                return p[2];
            });
            this._fillGenerator.y1(function (p) {
                return p[1];
            });
            this._fillGenerator.x1(function (p) {
                return p[0];
            });
            _super.prototype._afterNew.call(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sankey.prototype, "makeLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var source = dataItem.get("source");
            var target = dataItem.get("target");
            var link = this.links.make();
            if (source.get("unknown")) {
                link.addTag("source");
                link.addTag("unknown");
            }
            if (target.get("unknown")) {
                link.addTag("target");
                link.addTag("unknown");
            }
            this.linksContainer.children.push(link);
            link._setDataItem(dataItem);
            link.set("source", source);
            link.set("target", target);
            link.series = this;
            this.links.push(link);
            return link;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sankey.prototype, "updateSankey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var d3Graph = this._d3Graph;
            if (d3Graph) {
                this._d3Sankey.update(d3Graph);
                $array.each(this.dataItems, function (dataItem) {
                    var link = dataItem.get("link");
                    link.setPrivate("orientation", _this.get("orientation"));
                    link.markDirty();
                });
            }
        }
    });
    Object.defineProperty(Sankey.prototype, "_updateLinkColor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._updateLinkColor.call(this, dataItem);
            var orientation = this.get("orientation");
            var fillGradient = dataItem.get("link")._fillGradient;
            var strokeGradient = dataItem.get("link")._strokeGradient;
            if (orientation == "vertical") {
                if (fillGradient) {
                    fillGradient.set("rotation", 90);
                }
                if (strokeGradient) {
                    strokeGradient.set("rotation", 90);
                }
            }
            else {
                if (fillGradient) {
                    fillGradient.set("rotation", 0);
                }
                if (strokeGradient) {
                    strokeGradient.set("rotation", 0);
                }
            }
        }
    });
    Object.defineProperty(Sankey.prototype, "_getBulletLocation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet) {
            if (this.get("orientation") == "vertical") {
                return bullet.get("locationY", 0);
            }
            else {
                return bullet.get("locationX", 0);
            }
        }
    });
    Object.defineProperty(Sankey.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _a;
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            var vertical = false;
            if (this.get("orientation") == "vertical") {
                vertical = true;
            }
            if (this.isDirty("orientation") || this.isDirty("linkTension")) {
                var linkTension = this.get("linkTension", 0.5);
                if (vertical) {
                    this._fillGenerator.curve(curveMonotoneYTension(linkTension));
                    this._strokeGenerator.curve(curveMonotoneYTension(linkTension));
                }
                else {
                    this._fillGenerator.curve(curveMonotoneXTension(linkTension));
                    this._strokeGenerator.curve(curveMonotoneXTension(linkTension));
                }
            }
            if (this._valuesDirty || this._sizeDirty || this.isDirty("nodePadding") || this.isDirty("nodeWidth") || this.isDirty("nodeAlign") || this.isDirty("nodeSort") || this.isDirty("orientation") || this.isDirty("linkTension")) {
                var d3Sankey = this._d3Sankey;
                var w = this.innerWidth();
                var h = this.innerHeight();
                if (vertical) {
                    _a = __read([h, w], 2), w = _a[0], h = _a[1];
                }
                d3Sankey.size([w, h]);
                d3Sankey.nodePadding(this.get("nodePadding", 10));
                d3Sankey.nodeWidth(this.get("nodeWidth", 10));
                d3Sankey.nodeSort(this.get("nodeSort", null));
                switch (this.get("nodeAlign")) {
                    case "right":
                        d3Sankey.nodeAlign(d3sankey.sankeyRight);
                        break;
                    case "justify":
                        d3Sankey.nodeAlign(d3sankey.sankeyJustify);
                        break;
                    case "center":
                        d3Sankey.nodeAlign(d3sankey.sankeyCenter);
                        break;
                    default:
                        d3Sankey.nodeAlign(d3sankey.sankeyLeft);
                        break;
                }
                this._d3Graph = d3Sankey({ nodes: this._nodesData, links: this._linksData });
                $array.each(this.dataItems, function (dataItem) {
                    var link = dataItem.get("link");
                    link.setPrivate("orientation", _this.get("orientation"));
                    link.markDirty();
                });
                var d3Graph = this._d3Graph;
                if (d3Graph) {
                    var nodes = d3Graph.nodes;
                    $array.each(nodes, function (d3SankeyNode) {
                        var dataItem = d3SankeyNode.dataItem;
                        var node = dataItem.get("node");
                        var x0;
                        var x1;
                        var y0;
                        var y1;
                        if (vertical) {
                            x0 = d3SankeyNode.y0;
                            x1 = d3SankeyNode.y1;
                            y0 = d3SankeyNode.x0;
                            y1 = d3SankeyNode.x1;
                        }
                        else {
                            x0 = d3SankeyNode.x0;
                            x1 = d3SankeyNode.x1;
                            y0 = d3SankeyNode.y0;
                            y1 = d3SankeyNode.y1;
                        }
                        if ($type.isNumber(x0) && $type.isNumber(x1) && $type.isNumber(y0) && $type.isNumber(y1)) {
                            node.setAll({ x: x0, y: y0, width: x1 - x0, height: y1 - y0 });
                            var rectangle = dataItem.get("rectangle");
                            rectangle.setAll({ width: x1 - x0, height: y1 - y0 });
                        }
                    });
                }
            }
        }
    });
    Object.defineProperty(Sankey, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Sankey"
    });
    Object.defineProperty(Sankey, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Flow.classNames.concat([Sankey.className])
    });
    return Sankey;
}(Flow));
export { Sankey };
//# sourceMappingURL=Sankey.js.map