import { __awaiter, __extends, __generator } from "tslib";
import { PercentSeries } from "../percent/PercentSeries";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import { FunnelSlice } from "./FunnelSlice";
import { Tick } from "../../core/render/Tick";
import { Label } from "../../core/render/Label";
import { percent, p50, p100 } from "../../core/util/Percent";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as $utils from "../../core/util/Utils";
/**
 * Creates a funnel series for use in a [[SlicedChart]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/percent-charts/sliced-chart/funnel-series/} for more info
 * @important
 */
var FunnelSeries = /** @class */ (function (_super) {
    __extends(FunnelSeries, _super);
    function FunnelSeries() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "funnel"
        });
        /**
         * A [[ListTemplate]] of all slice links in series.
         *
         * `links.template` can also be used to configure slice links.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/percent-charts/sliced-chart/funnel-series/#Slice_links} for more info
         */
        Object.defineProperty(_this, "links", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeLinks()
        });
        Object.defineProperty(_this, "_total", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_count", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_nextCoord", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_opposite", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(FunnelSeries.prototype, "_makeSlices", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new ListTemplate(Template.new({}), function () { return FunnelSlice._new(_this._root, {
                themeTags: $utils.mergeTags(_this.slices.template.get("themeTags", []), [_this._tag, "series", "slice", _this.get("orientation")])
            }, [_this.slices.template]); });
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_makeLabels", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new ListTemplate(Template.new({}), function () { return Label._new(_this._root, {
                themeTags: $utils.mergeTags(_this.labels.template.get("themeTags", []), [_this._tag, "series", "label", _this.get("orientation")])
            }, [_this.labels.template]); });
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_makeTicks", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new ListTemplate(Template.new({}), function () { return Tick._new(_this._root, {
                themeTags: $utils.mergeTags(_this.ticks.template.get("themeTags", []), [_this._tag, "series", "tick", _this.get("orientation")])
            }, [_this.ticks.template]); });
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_makeLinks", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new ListTemplate(Template.new({}), function () { return FunnelSlice._new(_this._root, {
                themeTags: $utils.mergeTags(_this.links.template.get("themeTags", []), [_this._tag, "series", "link", _this.get("orientation")])
            }, [_this.links.template]); });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(FunnelSeries.prototype, "makeLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var link = this.slicesContainer.children.push(this.links.make());
            link._setDataItem(dataItem);
            dataItem.set("link", link);
            this.links.push(link);
            return link;
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._afterNew.call(this);
            var slicesContainer = this.slicesContainer;
            slicesContainer.setAll({ isMeasured: true, position: "relative", width: percent(100), height: percent(100) });
            slicesContainer.onPrivate("width", function () {
                _this.markDirtySize();
            });
            slicesContainer.onPrivate("height", function () {
                _this.markDirtySize();
            });
            if (this.get("orientation") == "vertical") {
                this.set("layout", this._root.horizontalLayout);
            }
            else {
                this.set("layout", this._root.verticalLayout);
            }
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            _super.prototype.processDataItem.call(this, dataItem);
            var slice = this.makeSlice(dataItem);
            slice._setDataItem(dataItem);
            dataItem.set("slice", slice);
            this.makeLink(dataItem);
            var label = this.makeLabel(dataItem);
            label.on("x", function () {
                _this._updateTick(dataItem);
            });
            label.on("y", function () {
                _this._updateTick(dataItem);
            });
            this.makeTick(dataItem);
            slice.events.on("positionchanged", function () {
                label.markDirty();
            });
            slice.events.on("boundschanged", function () {
                var dataItem = slice.dataItem;
                if (dataItem) {
                    _this._updateTick(dataItem);
                }
            });
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._opposite = false;
            if (this.children.indexOf(this.labelsContainer) == 0) {
                this._opposite = true;
            }
            var total = 0;
            var count = 0;
            $array.each(this.dataItems, function (dataItem) {
                var value = dataItem.get("value");
                if ($type.isNumber(value)) {
                    count++;
                    if (value > 0) {
                        total += Math.abs(dataItem.get("valueWorking", value) / value);
                    }
                    else {
                        if (_this.get("ignoreZeroValues", false)) {
                            count--;
                        }
                        else {
                            if (dataItem.isHidden()) {
                                count--;
                            }
                            else {
                                total += 1;
                            }
                        }
                    }
                }
            });
            this._total = 1 / count * total;
            this._count = count;
            if (this.isDirty("alignLabels")) {
                this._fixLayout();
            }
            if (this._total > 0 && (this._valuesDirty || this._sizeDirty)) {
                var slicesContainer = this.slicesContainer;
                var h = void 0;
                if (this.get("orientation") == "vertical") {
                    h = slicesContainer.innerHeight();
                }
                else {
                    h = slicesContainer.innerWidth();
                }
                this._nextCoord = this.get("startLocation", 0) * h;
                this.markDirtyBounds();
                var i_1 = 0;
                $array.each(this._dataItems, function (dataItem) {
                    _this.updateLegendValue(dataItem);
                    dataItem.set("index", i_1);
                    i_1++;
                    var slice = dataItem.get("slice");
                    var tick = dataItem.get("tick");
                    var label = dataItem.get("label");
                    var link = dataItem.get("link");
                    var color = dataItem.get("fill");
                    slice._setDefault("fill", color);
                    slice._setDefault("stroke", color);
                    link._setDefault("fill", color);
                    link._setDefault("stroke", color);
                    var value = dataItem.get("value");
                    if ($type.isNumber(value)) {
                        if (value == 0 && _this.get("ignoreZeroValues")) {
                            slice.setPrivate("visible", false);
                            tick.setPrivate("visible", false);
                            label.setPrivate("visible", false);
                        }
                        else {
                            slice.setPrivate("visible", true);
                            tick.setPrivate("visible", true);
                            label.setPrivate("visible", true);
                            _this.decorateSlice(dataItem);
                            if (_this.isLast(dataItem)) {
                                link.setPrivate("visible", false);
                            }
                            else if (!dataItem.isHidden()) {
                                link.setPrivate("visible", true);
                            }
                        }
                    }
                });
            }
            _super.prototype._updateChildren.call(this);
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_fixLayout", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var orientation = this.get("orientation");
            var labelsContainer = this.labelsContainer;
            var labelsTemplate = this.labels.template;
            if (this.get("alignLabels")) {
                labelsContainer.set("position", "relative");
                labelsContainer.setAll({ isMeasured: true });
                if (orientation == "vertical") {
                    this.set("layout", this._root.horizontalLayout);
                    labelsTemplate.setAll({ centerX: p100, x: p100 });
                }
                else {
                    this.set("layout", this._root.verticalLayout);
                    labelsTemplate.setAll({ centerX: 0, x: 0 });
                }
            }
            else {
                labelsContainer.setAll({ isMeasured: false, position: "absolute" });
                if (orientation == "vertical") {
                    labelsContainer.setAll({ x: p50 });
                    labelsTemplate.setAll({ centerX: p50, x: 0 });
                }
                else {
                    labelsContainer.setAll({ y: p50 });
                    labelsTemplate.setAll({ centerX: p50, y: 0 });
                }
            }
            this.markDirtySize();
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "getNextValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var index = dataItem.get("index");
            var nextValue = dataItem.get("valueWorking", 0);
            if (index < this.dataItems.length - 1) {
                var nextItem = this.dataItems[index + 1];
                nextValue = nextItem.get("valueWorking", 0);
                if (nextItem.isHidden() || (nextItem.get("value") == 0 && this.get("ignoreZeroValues"))) {
                    return this.getNextValue(nextItem);
                }
            }
            return nextValue;
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "isLast", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var index = dataItem.get("index");
            if (index == this.dataItems.length - 1) {
                return true;
            }
            else {
                for (var i = index + 1; i < this.dataItems.length; i++) {
                    if (!this.dataItems[i].isHidden()) {
                        return false;
                    }
                }
            }
            return true;
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "decorateSlice", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var orientation = this.get("orientation");
            var slice = dataItem.get("slice");
            var label = dataItem.get("label");
            var link = dataItem.get("link");
            var slicesContainer = this.slicesContainer;
            var maxWidth = slicesContainer.innerWidth();
            var maxHeight = slicesContainer.innerHeight();
            var maxSize = maxWidth;
            if (orientation == "horizontal") {
                maxSize = maxHeight;
            }
            var nextValue = this.getNextValue(dataItem);
            var value = dataItem.get("value", 0);
            var workingValue = Math.abs(dataItem.get("valueWorking", value));
            var bottomRatio = this.get("bottomRatio", 0);
            var valueHigh = this.getPrivate("valueHigh", 0);
            var d = 1;
            if (value != 0) {
                d = workingValue / Math.abs(value);
            }
            else {
                if (dataItem.isHidden()) {
                    d = 0.000001;
                }
            }
            if (this._nextCoord == Infinity) {
                this._nextCoord = 0;
            }
            var topWidth = workingValue / valueHigh * maxSize;
            var bottomWidth = (workingValue - (workingValue - nextValue) * bottomRatio) / valueHigh * maxSize;
            slice.setAll({ topWidth: topWidth, bottomWidth: bottomWidth, orientation: orientation });
            link.setAll({ topWidth: bottomWidth, bottomWidth: (workingValue - (workingValue - nextValue)) / valueHigh * maxSize, orientation: orientation });
            var startLocation = this.get("startLocation", 0);
            var endLocation = this.get("endLocation", 1);
            if (orientation == "vertical") {
                var linkHeight = link.height() * d;
                maxHeight = maxHeight * (endLocation - startLocation) + linkHeight;
                slice.set("y", this._nextCoord);
                var height = Math.min(100000, Math.max(0, maxHeight / this._count * d / this._total - linkHeight));
                slice.setAll({ height: height, x: maxWidth / 2 });
                var labelY = this._nextCoord + height / 2;
                label.set("y", labelY);
                this._nextCoord += height + linkHeight;
                link.setAll({ y: this._nextCoord - linkHeight, x: maxWidth / 2 });
            }
            else {
                var linkHeight = link.width() * d;
                maxWidth = maxWidth * (endLocation - startLocation) + linkHeight;
                slice.set("x", this._nextCoord);
                var width = Math.min(100000, Math.max(0, maxWidth / this._count * d / this._total - linkHeight));
                slice.setAll({ width: width, y: maxHeight / 2 });
                var labelX = this._nextCoord + width / 2;
                label.set("x", labelX);
                this._nextCoord += width + linkHeight;
                link.setAll({ x: this._nextCoord - linkHeight, y: maxHeight / 2 });
            }
        }
    });
    /**
     * Hides series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(FunnelSeries.prototype, "hideDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                return __generator(this, function (_a) {
                    dataItem.get("link").hide(duration);
                    return [2 /*return*/, _super.prototype.hideDataItem.call(this, dataItem, duration)];
                });
            });
        }
    });
    /**
     * Shows series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(FunnelSeries.prototype, "showDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                return __generator(this, function (_a) {
                    dataItem.get("link").show(duration);
                    return [2 /*return*/, _super.prototype.showDataItem.call(this, dataItem, duration)];
                });
            });
        }
    });
    Object.defineProperty(FunnelSeries.prototype, "_updateTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            if (this.get("alignLabels")) {
                var tick = dataItem.get("tick");
                var label = dataItem.get("label");
                var slice = dataItem.get("slice");
                if (tick && slice && label) {
                    var labelsContainer = this.labelsContainer;
                    var slicesContainer = this.slicesContainer;
                    var tickLocation = tick.get("location", 0.5);
                    var lcw = labelsContainer.width();
                    var lch = labelsContainer.height();
                    var pl = labelsContainer.get("paddingLeft", 0);
                    var pr = labelsContainer.get("paddingRight", 0);
                    var pt = labelsContainer.get("paddingTop", 0);
                    var pb = labelsContainer.get("paddingBottom", 0);
                    var p0 = { x: 0, y: 0 };
                    var p1 = { x: 0, y: 0 };
                    var p2 = { x: 0, y: 0 };
                    if (this._opposite) {
                        tickLocation = 1 - tickLocation;
                    }
                    if (this.get("orientation") == "vertical") {
                        p0 = slice.getPoint(tickLocation, 0.5);
                        p0.x += slice.x() + slicesContainer.x();
                        p0.y += slice.y() + slicesContainer.y();
                        if (this._opposite) {
                            p1.x = lcw;
                            p1.y = label.y();
                            p2.x = lcw - pl;
                            p2.y = p1.y;
                        }
                        else {
                            p1.x = slicesContainer.x() + slicesContainer.width();
                            p1.y = label.y();
                            p2.x = p1.x + lcw - label.width() - pr;
                            p2.y = p1.y;
                        }
                    }
                    else {
                        p0 = slice.getPoint(0.5, tickLocation);
                        p0.x += slice.x() + slicesContainer.x();
                        p0.y += slice.y() + slicesContainer.y();
                        if (this._opposite) {
                            p1.y = lch;
                            p1.x = label.x();
                            p2.y = lch - pt;
                            p2.x = p1.x;
                        }
                        else {
                            p1.y = slicesContainer.y() + slicesContainer.height();
                            p1.x = label.x();
                            p2.y = p1.y + lch - label.height() - pb;
                            p2.x = p1.x;
                        }
                    }
                    tick.set("points", [p0, p1, p2]);
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(FunnelSeries.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.disposeDataItem.call(this, dataItem);
            var link = dataItem.get("link");
            if (link) {
                this.links.removeValue(link);
                link.dispose();
            }
        }
    });
    Object.defineProperty(FunnelSeries, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "FunnelSeries"
    });
    Object.defineProperty(FunnelSeries, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: PercentSeries.classNames.concat([FunnelSeries.className])
    });
    return FunnelSeries;
}(PercentSeries));
export { FunnelSeries };
//# sourceMappingURL=FunnelSeries.js.map