import { __extends } from "tslib";
import { LinkedHierarchy } from "./LinkedHierarchy";
import * as $array from "../../core/util/Array";
import * as $utils from "../../core/util/Utils";
import * as $type from "../../core/util/Type";
import * as d3Force from "d3-force";
;
/**
 * Creates a force-directed tree.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/hierarchy/force-directed/} for more info
 * @important
 */
var ForceDirected = /** @class */ (function (_super) {
    __extends(ForceDirected, _super);
    function ForceDirected() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "forcedirected"
        });
        /**
         * @ignore
         */
        Object.defineProperty(_this, "d3forceSimulation", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3Force.forceSimulation()
        });
        /**
         * @ignore
         */
        Object.defineProperty(_this, "collisionForce", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3Force.forceCollide(20)
        });
        /**
         * @ignore
         */
        Object.defineProperty(_this, "linkForce", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3Force.forceLink()
        });
        Object.defineProperty(_this, "_nodes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_links", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_tick", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_nodesDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(ForceDirected.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._afterNew.call(this);
            this.d3forceSimulation.on("tick", function () {
                _this._tick++;
                _this.updateNodePositions();
            });
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this.isDirty("showOnFrame")) {
                var showOnFrame = this.get("showOnFrame");
                if (showOnFrame > this._tick) {
                    this.nodesContainer.setPrivate("visible", false);
                    this.linksContainer.setPrivate("visible", false);
                }
            }
            var d3forceSimulation = this.d3forceSimulation;
            if (this.isDirty("velocityDecay")) {
                d3forceSimulation.velocityDecay(this.get("velocityDecay", 0));
            }
            if (this.isDirty("initialFrames")) {
                d3forceSimulation.alphaDecay(1 - Math.pow(0.001, 1 / this.get("initialFrames", 500)));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ForceDirected.prototype, "restartSimulation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (alpha) {
            var d3forceSimulation = this.d3forceSimulation;
            if (d3forceSimulation.alpha() < alpha) {
                d3forceSimulation.alpha(alpha);
                d3forceSimulation.restart();
            }
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_handleRadiusChange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._updateForces();
        }
    });
    Object.defineProperty(ForceDirected.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var d3ForceNode = { index: this._index, x: this.innerWidth() / 2, y: this.innerHeight() / 2, dataItem: dataItem };
            var index = this._nodes.push(d3ForceNode) - 1;
            d3ForceNode.index = index;
            this.d3forceSimulation.nodes(this._nodes);
            dataItem.set("d3ForceNode", d3ForceNode);
            _super.prototype.processDataItem.call(this, dataItem);
            var node = dataItem.get("node");
            node.on("scale", function () {
                _this._nodesDirty = true;
                _this.markDirty();
            });
            node.events.on("dragged", function () {
                d3ForceNode.fx = node.x();
                d3ForceNode.fy = node.y();
                _this._updateForces();
            });
            node.events.on("dragstop", function () {
                if (dataItem.get("x") == null) {
                    d3ForceNode.fx = undefined;
                }
                if (dataItem.get("y") == null) {
                    d3ForceNode.fy = undefined;
                }
            });
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_updateValues", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (d3HierarchyNode) {
            _super.prototype._updateValues.call(this, d3HierarchyNode);
            this._nodesDirty = true;
            var d3forceSimulation = this.d3forceSimulation;
            d3forceSimulation.force("collision", this.collisionForce);
            d3forceSimulation.nodes(this._nodes);
            this.linkForce = d3Force.forceLink(this._links);
            d3forceSimulation.force("link", this.linkForce);
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_updateVisuals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateVisuals.call(this);
            this.restartSimulation(1);
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateChildren.call(this);
            var d3forceSimulation = this.d3forceSimulation;
            if (this._sizeDirty) {
                var w = Math.max(50, this.innerWidth());
                var h = Math.max(50, this.innerHeight());
                var pt = this.get("paddingTop", 0);
                var pl = this.get("paddingLeft", 0);
                var centerStrength = this.get("centerStrength", 1);
                d3forceSimulation.force("x", d3Force.forceX().x(w / 2 + pl).strength(centerStrength * 100 / w));
                d3forceSimulation.force("y", d3Force.forceY().y(h / 2 + pt).strength(centerStrength * 100 / h));
            }
            if (this._nodesDirty) {
                this._updateForces();
            }
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_updateForces", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var d3forceSimulation = this.d3forceSimulation;
            d3forceSimulation.force("manybody", d3Force.forceManyBody().strength(function (d3node) {
                var dataItem = d3node.dataItem;
                var node = dataItem.get("node");
                var circle = dataItem.get("circle");
                var manyBodyStrength = _this.get("manyBodyStrength", -15);
                if (circle) {
                    return circle.get("radius", 1) * node.get("scale", 1) * manyBodyStrength;
                }
                return 0;
            }));
            this.collisionForce.radius(function (d3node) {
                var dataItem = d3node.dataItem;
                var node = dataItem.get("node");
                var circle = dataItem.get("circle");
                var outerCircle = dataItem.get("outerCircle");
                if (circle && outerCircle) {
                    var radius = circle.get("radius", 1);
                    if (!outerCircle.isHidden()) {
                        radius = radius * outerCircle.get("scale", 1.1);
                    }
                    radius *= node.get("scale", 1);
                    return radius + _this.get("nodePadding", 0);
                }
            });
            this.restartSimulation(1);
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_clearDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._clearDirty.call(this);
            this._nodesDirty = false;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ForceDirected.prototype, "updateNodePositions", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var linkForce = this.linkForce;
            if (linkForce) {
                linkForce.distance(function (linkDatum) {
                    return _this.getDistance(linkDatum);
                });
                linkForce.strength(function (linkDatum) {
                    return _this.getStrength(linkDatum);
                });
            }
            if (this._tick == this.get("showOnFrame")) {
                this.nodesContainer.setPrivate("visible", true);
                this.linksContainer.setPrivate("visible", true);
            }
            var d3Nodes = this.d3forceSimulation.nodes();
            $array.each(d3Nodes, function (d3Node) {
                var dataItem = d3Node.dataItem;
                var node = dataItem.get("node");
                node.set("x", d3Node.x);
                node.set("y", d3Node.y);
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ForceDirected.prototype, "updateLinkWith", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItems) {
            var _this = this;
            $array.each(dataItems, function (dataItem) {
                var linkWith = dataItem.get("linkWith");
                if (linkWith) {
                    $array.each(linkWith, function (id) {
                        var linkWithDataItem = _this._getDataItemById(_this.dataItems, id);
                        if (linkWithDataItem) {
                            _this.linkDataItems(dataItem, linkWithDataItem, _this.get("linkWithStrength"));
                        }
                    });
                }
                var children = dataItem.get("children");
                if (children) {
                    _this.updateLinkWith(children);
                }
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ForceDirected.prototype, "getDistance", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (linkDatum) {
            var sourceDataItem = linkDatum.sourceDataItem;
            var targetDataItem = linkDatum.targetDataItem;
            var distance = 0;
            if (sourceDataItem && targetDataItem) {
                var targetNode = targetDataItem.get("node");
                if (targetNode.isHidden()) {
                    return 0;
                }
                var link = linkDatum.link;
                if (link) {
                    distance = link.get("distance", 1);
                }
                var sourceNode = sourceDataItem.get("node");
                if (targetNode.isHidden()) {
                    distance = 1;
                }
                return (distance * (sourceDataItem.get("circle").get("radius", 1) * sourceNode.get("scale", 1) + targetDataItem.get("circle").get("radius", 1) * targetNode.get("scale", 1)));
            }
            return distance;
        }
    });
    /**
     * @ignore
     * @todo description
     */
    Object.defineProperty(ForceDirected.prototype, "getStrength", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (linkDatum) {
            var strength = 0;
            var link = linkDatum.link;
            if (link) {
                strength = link.get("strength", 1);
            }
            var targetDataItem = linkDatum.targetDataItem;
            strength *= targetDataItem.get("node").get("scale");
            return strength;
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_updateNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._updateNode.call(this, dataItem);
            this._updateRadius(dataItem);
            var x = dataItem.get("x");
            var y = dataItem.get("y");
            var d3Node = dataItem.get("d3ForceNode");
            if (x != null) {
                d3Node.fx = $utils.relativeToValue(x, this.innerWidth());
            }
            if (y != null) {
                d3Node.fy = $utils.relativeToValue(y, this.innerHeight());
            }
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_updateRadius", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var size = (this.innerWidth() + this.innerHeight()) / 2;
            var minRadius = $utils.relativeToValue(this.get("minRadius", 1), size);
            var maxRadius = $utils.relativeToValue(this.get("maxRadius", 5), size);
            var valueWorking = dataItem.get("sum");
            var radius = maxRadius;
            var min = this.getPrivate("valueLow", 0);
            var max = this.getPrivate("valueHigh", 0);
            if (max > 0) {
                radius = minRadius + (valueWorking - min) / (max - min) * (maxRadius - minRadius);
            }
            if (!$type.isNumber(radius)) {
                radius = minRadius;
            }
            var duration = this.get("animationDuration", 0);
            var easing = this.get("animationEasing");
            dataItem.get("circle").animate({ key: "radius", to: radius, duration: duration, easing: easing });
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_processLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (link, source, target) {
            var d3Link = { link: link, source: source.get("d3ForceNode").index, target: target.get("d3ForceNode").index, sourceDataItem: source, targetDataItem: target };
            this._links.push(d3Link);
            link.setPrivate("d3Link", d3Link);
            this.linkForce = d3Force.forceLink(this._links);
            this.d3forceSimulation.force("link", this.linkForce);
            this.restartSimulation(0.5);
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_disposeLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (link) {
            _super.prototype._disposeLink.call(this, link);
            $array.remove(this._links, link.getPrivate("d3Link"));
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_handleUnlink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.restartSimulation(0.5);
        }
    });
    Object.defineProperty(ForceDirected.prototype, "_onDataClear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._onDataClear.call(this);
            this._nodes = [];
            this._links = [];
        }
    });
    Object.defineProperty(ForceDirected, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "ForceDirected"
    });
    Object.defineProperty(ForceDirected, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: LinkedHierarchy.classNames.concat([ForceDirected.className])
    });
    return ForceDirected;
}(LinkedHierarchy));
export { ForceDirected };
//# sourceMappingURL=ForceDirected.js.map