import { __extends } from "tslib";
import { Hierarchy } from "./Hierarchy";
import { Container } from "../../core/render/Container";
import { LinkedHierarchyNode } from "./LinkedHierarchyNode";
import { HierarchyLink } from "./HierarchyLink";
import { Template } from "../../core/util/Template";
import { Circle } from "../../core/render/Circle";
import { ListTemplate } from "../../core/util/List";
import * as $array from "../../core/util/Array";
import * as $utils from "../../core/util/Utils";
;
/**
 * A base class for linked hierarchy series.
 */
var LinkedHierarchy = /** @class */ (function (_super) {
    __extends(LinkedHierarchy, _super);
    function LinkedHierarchy() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A list of nodes in a [[LinkedHierarchy]] chart.
         *
         * @default new ListTemplate<LinkedHierarchyNode>
         */
        Object.defineProperty(_this, "nodes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return LinkedHierarchyNode._new(_this._root, {
                themeTags: $utils.mergeTags(_this.nodes.template.get("themeTags", []), [_this._tag, "linkedhierarchy", "hierarchy", "node"]),
                x: _this.width() / 2,
                y: _this.height() / 2
            }, [_this.nodes.template]); })
        });
        /**
         * A list of node circle elements in a [[LinkedHierarchy]] chart.
         *
         * @default new ListTemplate<Circle>
         */
        Object.defineProperty(_this, "circles", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Circle._new(_this._root, {
                themeTags: $utils.mergeTags(_this.circles.template.get("themeTags", []), [_this._tag, "shape"])
            }, [_this.circles.template]); })
        });
        /**
         * A list of node outer circle elements in a [[LinkedHierarchy]] chart.
         *
         * @default new ListTemplate<Circle>
         */
        Object.defineProperty(_this, "outerCircles", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Circle._new(_this._root, {
                themeTags: $utils.mergeTags(_this.outerCircles.template.get("themeTags", []), [_this._tag, "outer", "shape"])
            }, [_this.outerCircles.template]); })
        });
        /**
         * A list of link elements in a [[LinkedHierarchy]] chart.
         *
         * @default new ListTemplate<HierarchyLink>
         */
        Object.defineProperty(_this, "links", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return HierarchyLink._new(_this._root, {
                themeTags: $utils.mergeTags(_this.links.template.get("themeTags", []), [_this._tag, "linkedhierarchy", "hierarchy", "link"])
            }, [_this.links.template]); })
        });
        /**
         * A [[Container]] that link elements are placed in.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "linksContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.moveValue(Container.new(_this._root, {}), 0)
        });
        return _this;
    }
    Object.defineProperty(LinkedHierarchy.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.fields.push("linkWith", "x", "y");
            _super.prototype._afterNew.call(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(LinkedHierarchy.prototype, "makeNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var node = _super.prototype.makeNode.call(this, dataItem);
            var circle = node.children.moveValue(this.circles.make(), 0);
            this.circles.push(circle);
            node.setPrivate("tooltipTarget", circle);
            dataItem.setRaw("circle", circle);
            var outerCircle = node.children.moveValue(this.outerCircles.make(), 0);
            this.outerCircles.push(outerCircle);
            dataItem.setRaw("outerCircle", outerCircle);
            var label = dataItem.get("label");
            circle.on("radius", function () {
                var d = circle.get("radius", _this.width()) * 2;
                label.setAll({ maxWidth: d, maxHeight: d });
                outerCircle.set("radius", d / 2);
                _this._handleRadiusChange();
            });
            var d = circle.get("radius", this.width()) * 2;
            label.setAll({ maxWidth: d, maxHeight: d });
            circle._setDataItem(dataItem);
            outerCircle._setDataItem(dataItem);
            return node;
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_handleRadiusChange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            dataItem.setRaw("childLinks", []);
            dataItem.setRaw("links", []);
            _super.prototype.processDataItem.call(this, dataItem);
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._processDataItem.call(this, dataItem);
            var parentDataItem = dataItem.get("parent");
            if (parentDataItem && parentDataItem.get("depth") >= this.get("topDepth")) {
                var link = this.linkDataItems(parentDataItem, dataItem);
                dataItem.setRaw("parentLink", link);
            }
            var node = dataItem.get("node");
            this.updateLinkWith(this.dataItems);
            node._updateLinks(0);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(LinkedHierarchy.prototype, "updateLinkWith", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItems) {
            var _this = this;
            $array.each(dataItems, function (dataItem) {
                var linkWith = dataItem.get("linkWith");
                if (linkWith) {
                    $array.each(linkWith, function (id) {
                        var linkWithDataItem = _this._getDataItemById(_this.dataItems, id);
                        if (linkWithDataItem) {
                            _this.linkDataItems(dataItem, linkWithDataItem);
                        }
                    });
                }
                var children = dataItem.get("children");
                if (children) {
                    _this.updateLinkWith(children);
                }
            });
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_getPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (hierarchyNode) {
            return { x: hierarchyNode.x, y: hierarchyNode.y };
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_updateNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            _super.prototype._updateNode.call(this, dataItem);
            var node = dataItem.get("node");
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var point = this._getPoint(hierarchyNode);
            var duration = this.get("animationDuration", 0);
            var easing = this.get("animationEasing");
            node.animate({ key: "x", to: point.x, duration: duration, easing: easing });
            node.animate({ key: "y", to: point.y, duration: duration, easing: easing });
            var hierarchyChildren = hierarchyNode.children;
            if (hierarchyChildren) {
                $array.each(hierarchyChildren, function (hierarchyChild) {
                    _this._updateNodes(hierarchyChild);
                });
            }
            var fill = dataItem.get("fill");
            var circle = dataItem.get("circle");
            var children = dataItem.get("children");
            if (circle) {
                circle._setDefault("fill", fill);
                circle._setDefault("stroke", fill);
            }
            var outerCircle = dataItem.get("outerCircle");
            if (outerCircle) {
                outerCircle._setDefault("fill", fill);
                outerCircle._setDefault("stroke", fill);
                if (!children || children.length == 0) {
                    outerCircle.setPrivate("visible", false);
                }
            }
        }
    });
    /**
     * Link two data items with a link element.
     *
     * @param   source    Source node data item
     * @param   target    Target node data item
     * @param   strength  Link strength
     * @return            Link element
     */
    Object.defineProperty(LinkedHierarchy.prototype, "linkDataItems", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (source, target, strength) {
            var link;
            var sourceLinks = source.get("links");
            if (sourceLinks) {
                $array.each(sourceLinks, function (lnk) {
                    if (lnk.get("target") == target) {
                        link = lnk;
                    }
                });
            }
            var targetLinks = target.get("links");
            if (targetLinks) {
                $array.each(targetLinks, function (lnk) {
                    if (lnk.get("target") == source) {
                        link = lnk;
                    }
                });
            }
            if (!link) {
                link = this.links.make();
                this.links.push(link);
                this.linksContainer.children.push(link);
                link.set("source", source);
                link.set("target", target);
                link._setDataItem(source);
                link.set("stroke", source.get("fill"));
                if (strength != null) {
                    link.set("strength", strength);
                }
                source.get("childLinks").push(link);
                $array.move(source.get("links"), link);
                $array.move(target.get("links"), link);
                this._processLink(link, source, target);
            }
            return link;
        }
    });
    /**
     * Unlink two linked data items.
     *
     * @param   source  Source node data item
     * @param   target  Target node data item
     */
    Object.defineProperty(LinkedHierarchy.prototype, "unlinkDataItems", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (source, target) {
            var link;
            var sourceLinks = source.get("links");
            if (sourceLinks) {
                $array.each(sourceLinks, function (lnk) {
                    if (lnk && lnk.get("target") == target) {
                        link = lnk;
                        $array.remove(sourceLinks, link);
                    }
                });
            }
            var targetLinks = target.get("links");
            if (targetLinks) {
                $array.each(targetLinks, function (lnk) {
                    if (lnk && lnk.get("target") == source) {
                        link = lnk;
                        $array.remove(targetLinks, link);
                    }
                });
            }
            if (link) {
                this._disposeLink(link);
            }
            this._handleUnlink();
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_handleUnlink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_disposeLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (link) {
            this.links.removeValue(link);
            link.dispose();
        }
    });
    /**
     * Returns `true` if two nodes are linked with each other.
     */
    Object.defineProperty(LinkedHierarchy.prototype, "areLinked", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (source, target) {
            var sourceLinks = source.get("links");
            var linked = false;
            if (sourceLinks) {
                $array.each(sourceLinks, function (lnk) {
                    if (lnk.get("target") == target) {
                        linked = true;
                    }
                });
            }
            var targetLinks = target.get("links");
            if (targetLinks) {
                $array.each(targetLinks, function (lnk) {
                    if (lnk.get("target") == source) {
                        linked = true;
                    }
                });
            }
            return linked;
        }
    });
    Object.defineProperty(LinkedHierarchy.prototype, "_processLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_link, _source, _target) {
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(LinkedHierarchy.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            _super.prototype.disposeDataItem.call(this, dataItem);
            var links = dataItem.get("links");
            if (links) {
                $array.each(links, function (link) {
                    _this._disposeLink(link);
                });
            }
        }
    });
    /**
     * Select a data item.
     * @param  dataItem  Data item
     */
    Object.defineProperty(LinkedHierarchy.prototype, "selectDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var parent = dataItem.get("parent");
            if (!dataItem.get("disabled")) {
                this.set("selectedDataItem", dataItem);
                this._selectDataItem(dataItem);
            }
            else {
                if (parent) {
                    this.setRaw("selectedDataItem", parent);
                    var type = "dataitemselected";
                    this.events.dispatch(type, { type: type, target: this, dataItem: parent });
                    this.disableDataItem(dataItem);
                }
            }
        }
    });
    Object.defineProperty(LinkedHierarchy, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "LinkedHierarchy"
    });
    Object.defineProperty(LinkedHierarchy, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Hierarchy.classNames.concat([LinkedHierarchy.className])
    });
    return LinkedHierarchy;
}(Hierarchy));
export { LinkedHierarchy };
//# sourceMappingURL=LinkedHierarchy.js.map